#!/usr/bin/perl -w

use strict;

use lib './lib';
use lib './share';

BEGIN {
    $ENV{DATETIMEX_LITE_LOCALE_SKIP} = 1;
}

use Carp::Always;
use Data::Dumper;
use DateTimeX::Lite qw(Arithmetic);
use DateTimeX::Lite::OlsonDB;
use File::Copy;
use File::Find::Rule;
use File::Path;
use File::Spec;
use Getopt::Long;
use YAML;

$Data::Dumper::Indent = 1;
$Data::Dumper::Sortkeys = 1;
$Data::Dumper::Terse = 1;

my $VERSION = "0.07";

my $INFINITY  = 100 ** 100 ** 100;

my %opts;
GetOptions( 'dir:s'     => \$opts{dir},
            'clean'     => \$opts{clean},
            'version:s' => \$opts{version},
            'old'       => \$opts{old},
            'file:s'    => \$opts{file},
            'name:s'    => \$opts{name},
            'help'      => \$opts{help},
          );


$opts{help} = 1
    unless defined $opts{dir} && -d $opts{dir};

$opts{help} = 1
    unless defined $opts{version} || $opts{file} || $opts{name};

$opts{version} ||= 'test';

if ( $opts{help} )
{
    print <<'EOF';

This script parses the Olson time zone database files and turns them
into a set of Perl modules.  It also generates the MANIFEST and the
DateTimeX::Lite::TimeZone::Catalog module, which contains a list of all the
available time zone names.

By default, it looks for files named africa, antarctica, asia,
australasia, europe, northamerica, pacificnew, southamerica, and
backward.  All other files are ignored.

It takes the following arguments:

  --dir      A directory containing Olson db files.

  --version  The version of the Olson data files being used.
             Required unless one of the debugging options is given.

  --clean    Remove old generated modules (which may not be valid with
             the latest Olson database)

  --file     Parse just the file with the given name.  For debugging.

  --name     Only create the specified time zone.  For debugging.

  --old      Also look for files named etcetera, factory, and systemv

  --help     What you are reading

If the --file or --name options are specified, the MANIFEST and
DateTimeX::Lite::TimeZone::Catalog files will not be generated.

EOF

    exit;
}

clean() if $opts{clean};

my @files;

if ( $opts{file} )
{
    @files = $opts{file};
}
else
{
    @files = qw( africa antarctica asia australasia
                 europe northamerica pacificnew
                 southamerica backward
               );

    push @files, qw( etcetera factory systemv )
        if $opts{old};
}

my $man;
unless ( $opts{name} || $opts{file} )
{
    copy( 'MANIFEST.base', 'MANIFEST' );
    open $man, ">>MANIFEST" or die "Cannot write to MANIFEST: $!";
}

my ( @zones, %categories, %links );

my $autogen_warning = <<"EOF";
# This file is auto-generated by the Perl DateTimeX Suite time zone
# code generator ($VERSION) This code generator comes with the
# DateTimeX::Lite::TimeZone module distribution in the tools/ directory
EOF

parse_file($_) for sort @files;

exit if $opts{name};

clean_links();

make_catalog_pm();

sub clean
{
    for my $f ( File::Find::Rule
                ->file
                ->name('*.pm')
                ->grep('This file is auto-generated' )
                ->in('lib'),
                File::Find::Rule
                ->file
                ->name('zd*.t')
                ->in('t')
              )
    {
        unlink $f or die "Cannot unlink $f: $!";
    }
}

sub parse_file
{
    my $file = File::Spec->catfile( $opts{dir}, shift );

    die "No such file $file\n" unless -e $file;

    print "Now parsing $file\n";

    my $odb = DateTimeX::Lite::OlsonDB->new;

    $odb->parse_file($file);

    %links = ( %links, $odb->links );

    foreach my $zone_name ( sort $odb->zone_names )
    {
        if ( $opts{name} )
        {
            next unless $zone_name eq $opts{name};
        }
        print "  creating zone $zone_name\n";

        push @zones, $zone_name;

        my $name;
        my @dir;
        if ( $zone_name =~ m{/} )
        {
            my $category;
            ( $category, $name ) = split /\//, $zone_name, 2;
            push @{ $categories{$category} }, $name;

            ($dir[0] = $category) =~ tr/-/_/;
        }
        else
        {
            $name = $zone_name;
        }

        (my $outfile1 = $name) =~ tr/-/_/;

        (my $mod_name = $zone_name) =~ s/\//::/g;
        $mod_name =~ tr/-/_/;

        my $max_year = (localtime)[5] + 1910;
        my $zone = $odb->expanded_zone( name => $zone_name,
                                        expand_to_year => $max_year,
                                      );

        my $spans = serialize_spans(zone_as_spans($zone));

=head1
        $spans =~ s/-inf/DateTimeX::Lite::TimeZone::NEG_INFINITY/g;
        $spans =~ s/inf/DateTimeX::Lite::TimeZone::INFINITY/g;

        $spans =~ s/('(?:start|end)_date'\s+=>\s+)'(\d+)'/$1$2/g;
=cut

        my %generator = zone_generator($zone);

        my $has_dst_changes = grep { $_->is_dst } $zone->sorted_changes;

        my $from = "Generated from $file.";
        $from .= "  Olson data version $opts{version}"
            if defined $opts{version};

        my $body = <<"EOF";
$autogen_warning
#
# $from
#
# Do not edit this file directly.
#
EOF
        $body .= Data::Dumper::Dumper({
            olson_version => $opts{version},
            name => $zone_name,
            has_dst_changes => $has_dst_changes,
            max_year => $max_year,
            spans => $spans,
            %generator,
        });

        my @name_pieces = split /\//, $outfile1;
        my $filename = (pop @name_pieces) . '.dat';

        my $outdir = File::Spec->catdir( qw( share DateTimeX Lite TimeZone ),
                                         @dir, @name_pieces  );

        mkpath( $outdir, 1, 0755 );

        my $outfile2 = File::Spec->catfile( $outdir, $filename );

        open my $fh, ">$outfile2" or die "Cannot write to $outfile2: $!";
        print $fh $body or die "Cannot write to $outfile2: $!";
        close $fh or die "Cannot write to $outfile2: $!";

        unless ( $opts{name} || $opts{file} )
        {
            print $man "$outfile2\n" or die "Cannot write to MANIFEST: $!"
        }
    }
}

sub zone_as_spans
{
    my $zone = shift;

    my @spans;

    my @changes = $zone->sorted_changes;

    for ( my $x = 1; $x < @changes; $x++ )
    {
        my $last_total_offset = $x > 1 ? $changes[ $x - 2 ]->total_offset : undef;

        my $span =
            DateTimeX::Lite::OlsonDB::Change::two_changes_as_span
                ( @changes[ $x - 1, $x ], $last_total_offset );

        push @spans, $span;

        if (@spans > 2)
        {
            die "Gap in UTC end/start datetime for " . $zone->name
                unless $spans[-2]{utc_end} == $spans[-1]{utc_start};
        }
    }

    unless ( $zone->infinite_rules )
    {
        my $last_change = $changes[-1];

        my $last_observance = $last_change->observance;

        if ( $last_change->utc_start_datetime )
        {
            push @spans, { utc_start   => $last_change->utc_start_datetime->utc_rd_as_seconds,
                           utc_end     => $INFINITY,
                           local_start => $last_change->local_start_datetime->utc_rd_as_seconds,
                           local_end   => $INFINITY,
                           short_name  => $last_change->short_name,
                           offset      => $last_change->total_offset,
                       is_dst      => $last_change->is_dst,
                         };
        }
        # This happens with zones that have only one rule and no real changes (Pacific/Johnston)
        else
        {
            my $utc_start =
                @spans ? $spans[-1]{utc_end} : -1 * $INFINITY;

            push @spans, { utc_start   => $utc_start,
                           utc_end     => $INFINITY,
                           local_start => $utc_start - $last_observance->total_offset,
                           local_end   => $INFINITY,
                           short_name  => sprintf( $last_observance->format, '' ),
                           offset      => $last_observance->total_offset,
                           is_dst      => 0,
                         };
        }
    }

    return \@spans;
}

sub serialize_spans
{
    my $spans = shift;

    return [ map { serialize_span($_) } @$spans ];
=head1
    my $string = "[\n";
    $string .= join "\n", map { serialize_span($_) } @$spans;
    $string .= "\n]";

    return $string;
=cut
}

sub serialize_span
{
    my $span = shift;
    # must correspond to constants in DT::TZ, and short_name is always last
    my @keys = qw( utc_start utc_end local_start local_end offset is_dst short_name );

    return [ map { 
        my $v = $span->{$_};
=head1
print "'$v' -> ";
        if ($v eq '-inf') {
print "(should set to NEG_INF) ";
            $v = &DateTimeX::Lite::NEG_INFINITY;
        } elsif ($v eq 'inf') {
print "(should set to INF) ";
            $v = &DateTimeX::Lite::INFINITY;
        }
print "'$v'\n";
=cut
        $v
    } @keys ];
=head1
    my $string = "    [\n";
    $string .= join ",\n", @$span{@keys};
    $string .= ",\n'$span->{short_name}'";
    $string .= "\n    ],";

    return $string;
=cut
}

sub zone_generator
{
    my $zone = shift;

    return () unless $zone->infinite_rules;

    my $generator = <<'EOF';
sub _last_offset { !OFFSET }

my $last_observance = !LAST_OBSERVANCE;
sub _last_observance { $last_observance }

my $rules = !RULES;
sub _rules { $rules }
EOF

    my $last_observance = ($zone->sorted_changes)[-1]->observance;

    # hack to trim size of dumped object
    delete $last_observance->{utc_start_datetime}{locale};
    delete $last_observance->{local_start_datetime}{locale};
    delete $last_observance->{utc_start_datetime}{local_c};
    delete $last_observance->{local_start_datetime}{local_c};
    delete $last_observance->{rules};
    delete $last_observance->{first_rule};

    # This assumes that there is only one observance from end of
    # changes til end of time, which should be guaranteed by code in
    # OlsonDB module.
    my $offset = $last_observance->total_offset;

    my @rules = $zone->infinite_rules;

=head1
    # This is cleaner than making the above a double-quoted string
    $generator =~ s/!RULES/Dumper \@rules/eg;
    $generator =~ s/!LAST_OBSERVANCE/Dumper $last_observance/eg;
    $generator =~
        s/\$VAR1->{'local_start_datetime'}{'tz'}/bless( {
      'name' => 'floating',
      'offset' => 0
    }, 'DateTimeX::Lite::TimeZone::Floating' )/;
    $generator =~
        s/\$VAR1->{'utc_start_datetime'}{'tz'}/bless( {
      'name' => 'floating',
      'offset' => 0
    }, 'DateTimeX::Lite::TimeZone::Floating' )/;
    $generator =~ s/!OFFSET/$offset/g;

    return $generator;
=cut
    return (
        rules => \@rules,
        last_observance => $last_observance,
        last_offset => $offset
    );

}

sub clean_links
{
    # override some links and add others
    %links =
        ( %links,
          'Etc/GMT'       => 'UTC',
          'Etc/GMT+0'     => 'UTC',
          'Etc/Universal' => 'UTC',
          'Etc/UCT'       => 'UTC',
          'Etc/UTC'       => 'UTC',
          'Etc/Zulu'      => 'UTC',
          'GMT0'          => 'UTC',
          'GMT'           => 'UTC',
          'AKST9AKDT'     => 'America/Anchorage',
          'JST-9'         => 'Asia/Tokyo',
        );

    delete $links{UTC};

    # Some links resolve to other links - chase them down until they point
    # to a real zone.
    while ( my @k = grep { $links{ $links{$_} } } keys %links )
    {
        for my $k (@k)
        {
            $links{$k} = $links{ $links{$k} };
        }
    }
}

sub make_catalog_pm
{
    my $links = Dumper(\%links);
    $links =~ s/{/(/;
    $links =~ s/}/)/;

    my $zones = join "\n", sort @zones;
    my $cat_names = join "\n", sort keys %categories;
    my $cat = '';
    foreach my $c ( sort keys %categories )
    {
        $cat .= join("\t", $c, sort @{ $categories{$c} }) . "\n";
    }

    my %countries = parse_zone_tab();
    # hard-code this alias per request of David Cantrell on the list.
    $countries{UK} = $countries{GB};

    my $countries = '';
    for my $c ( sort keys %countries )
    {
        $countries .= join("\t", lc $c, @{ $countries{$c} }) . "\n";
#        $countries .= qq|  '\L$c' => [ qw(\n|;
#        # We explicitly do not sort these because the order in
#        # zones.tab is by population.
#        $countries .= join "\n", map { "    $_" } @{ $countries{$c} };
#        $countries .= "\n) ],\n";
    }

    my $zonecatalog = <<"EOF";
$autogen_warning
#
# Do not edit this file directly.

package DateTimeX::Lite::TimeZone::Catalog;

use strict;

use vars qw( \@ALL \@CATEGORY_NAMES \%CATEGORIES \%ZONES_BY_COUNTRY \%LINKS \$LOADED_CATALOG );

sub OlsonVersion { '$opts{version}' }

%LINKS = $links;

sub load {
    my \$mode = '';
    while (<DATA>) {
        chomp;
        next unless length($_);

        if (/^__(.+)__\$/) {
            if (\$1 eq 'ZONES') {
                \$mode = 'zone';
            } elsif (\$1 eq 'CATEGORY_NAMES') {
                \$mode = 'category_names';
            } elsif (\$1 eq 'CATEGORIES') {
                \$mode = 'categories';
            } elsif (\$1 eq 'BY_COUNTRY') {
                \$mode = 'by_country';
            } elsif (\$1 eq 'END') {
                last;
            } else {
                \$mode = '';
            }
            next;
        }

        if (\$mode eq 'zone') {
            push \@ALL, \$_;
        } elsif (\$mode eq 'category_names') {
            push \@CATEGORY_NAMES, \$_;
        } elsif (\$mode eq 'by_country') {
            my (\$name, \@list) = split(/\\t/, \$_);
            \$ZONES_BY_COUNTRY{\$name} = \\\@list;
        } elsif (\$mode eq 'categories') {
            my (\$name, \@list) = split(/\\t/, \$_);
            \$CATEGORIES{\$name} = \\\@list;
        }
    }
    \$LOADED_CATALOG = 1;
}

1;

__DATA__
__ZONES__
$zones
__CATEGORY_NAMES__
$cat_names
__CATEGORIES__
$cat
__BY_COUNTRY__
$countries
__END__

=head1 NAME

DateTimeX::Lite::TimeZone::Catalog - Provides a list of all valid time zone names

=head1 SYNOPSIS

See DateTimeX::Lite::TimeZone for API details.

=head1 DESCRIPTION

This module contains an enumerated list of all known system timezones,
so that applications can easily present a list of timezones.

=head1 AVAILABLE ZONES

=head2 Zones by Region

EOF

    for my $category ( sort keys %categories )
    {
        $zonecatalog .= "=head3 $category\n\n";

        for my $zone ( @{ $categories{$category} } )
        {
            $zonecatalog .= "  $category/$zone\n";
        }

        $zonecatalog .= "\n";
    }

    open my $fh, ">lib/DateTimeX/Lite/TimeZone/Catalog.pm" or die $!;
    print $fh $zonecatalog or die $!;
    close $fh or die $!;
}


sub parse_zone_tab
{
    my $file = File::Spec->catfile( $opts{dir}, 'zone.tab' );

    open my $fh, "<$file" or die "Cannot read $file: $!";

    my %countries;
    while (<$fh>)
    {
        next if /^\#/;
        chomp;

        my ( $cc, undef, $tz, undef ) = split /\t/, $_;

        push @{ $countries{$cc} }, $tz;
    }

    return %countries;
}

