package OpenInteract2::Brick::ObjectActivity;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'object_activity-2.13.zip' => 'OBJECT_ACTIVITY213ZIP',
);

sub get_name {
    return 'object_activity';
}

sub get_resources {
    return (
        'object_activity-2.13.zip' => [ 'pkg object_activity-2.13.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::ObjectActivity - Base-64 encoded OI2 package 'object_activity-2.13.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "object_activity-2.13.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'object_activity' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<object_activity-2.13.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub OBJECT_ACTIVITY213ZIP {
    return <<'SOMELONGSTRING';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SOMELONGSTRING
}

