# Copyright (C) 2011-2021 A S Lewis
#
# This program is free software: you can redistribute it and/or modify it under the terms of the GNU
# General Public License as published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
# even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with this program. If not,
# see <http://www.gnu.org/licenses/>.
#
#
# Games::Axmud::Win::Map
# The Automapper window object (separate and independent from the automapper object, GA::Obj::Map)

{ package Games::Axmud::Win::Map;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::MapWin Games::Axmud::Generic::GridWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

    sub new {

        # Called by GA::Obj::Workspace->createGridWin and ->createSimpleGridWin
        # Creates an Automapper window
        #
        # Expected arguments
        #   $number     - Unique number for this window object
        #   $winType    - The window type, must be 'map'
        #   $winName    - The window name, must be 'map'
        #   $workspaceObj
        #               - The GA::Obj::Workspace object for the workspace in which this window is
        #                   created
        #
        # Optional arguments
        #   $owner      - The owner, if known ('undef' if not). Typically it's a GA::Session or a
        #                   task (inheriting from GA::Generic::Task); could also be GA::Client. It
        #                   should not be another window object (inheriting from GA::Generic::Win).
        #                   The owner should have its own ->del_winObj function which is called when
        #                   $self->winDestroy is called
        #   $session    - The owner's session. If $owner is a GA::Session, that session. If it's
        #                   something else (like a task), the task's session. If $owner is 'undef',
        #                   so is $session
        #   $workspaceGridObj
        #               - The GA::Obj::WorkspaceGrid object into whose grid this window has been
        #                   placed. 'undef' in $workspaceObj->gridEnableFlag = FALSE
        #   $areaObj    - The GA::Obj::Area (a region of a workspace grid zone) which handles this
        #                   window. 'undef' in $workspaceObj->gridEnableFlag = FALSE
        #   $winmap     - Ignored if set
        #
        # Return values
        #   'undef' on improper arguments or if no $session was specified
        #   Blessed reference to the newly-created object on success

        my (
            $class, $number, $winType, $winName, $workspaceObj, $owner, $session, $workspaceGridObj,
            $areaObj, $winmap, $check,
        ) = @_;

        # Check for improper arguments
        if (
            ! defined $class || ! defined $number || ! defined $winType || ! defined $winName
            || ! defined $workspaceObj || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($class . '->new', @_);
        }

        # Automapper windows are unique to their session. If no $session is specified, refuse to
        #   create a window object
        if (! $session) {

            return undef;
        }

        # Check that the $winType is valid
        if ($winType ne 'map') {

            return $axmud::CLIENT->writeError(
                'Internal window error: invalid \'map\' window type \'' . $winType . '\'',
                $class . '->new',
            );
        }

        # Setup
        my $self = {
            _objName                    => 'map_win_' . $number,
            _objClass                   => $class,
            _parentFile                 => undef,       # No parent file object
            _parentWorld                => undef,       # No parent file object
            _privFlag                   => TRUE,        # All IVs are private

            # Standard window object IVs
            # --------------------------

            # Unique number for this window object
            number                      => $number,
            # The window category - 'grid' or 'free'
            winCategory                 => 'grid',
            # The window type, must be 'map'
            winType                     => $winType,
            # The window name, must be 'map'
            winName                     => $winName,
            # The GA::Obj::Workspace object for the workspace in which this window is created
            workspaceObj                => $workspaceObj,
            # The owner, if known ('undef' if not). Typically it's a GA::Session or a task
            #   (inheriting from GA::Generic::Task); could also be GA::Client. It should not be
            #   another window object (inheriting from GA::Generic::Win). The owner must have its
            #   own ->del_winObj function which is called when $self->winDestroy is called
            owner                       => $owner,
            # The owner's session ('undef' if not). If ->owner is a GA::Session, that session. If
            #   it's something else (like a task), the task's sesssion. If ->owner is 'undef', so is
            #   ->session
            session                     => $session,
            # When GA::Session->pseudoCmd is called to execute a client command, the mode in which
            #   it should be called (usually 'win_error' or 'win_only', which causes errors to be
            #   displayed in a 'dialogue' window)
            pseudoCmdMode               => 'win_only',

            # The window widget. For most window objects, the Gtk3::Window. For pseudo-windows, the
            #   parent 'main' window's Gtk3::Window
            # The code should use this IV when it wants to do something to the window itself
            #   (minimise it, make it active, etc)
            winWidget                   => undef,
            # The window container. For most window objects, the Gtk3::Window. For pseudo-windows,
            #   the parent GA::Table::PseudoWin table object
            # The code should use this IV when it wants to add, modify or remove widgets inside the
            #   window itself
            winBox                      => undef,
            # Flag set to TRUE if the window actually exists (after a call to $self->winEnable),
            #   FALSE if not
            enabledFlag                 => FALSE,
            # Flag set to TRUE if the Gtk3 window itself is visible (after a call to
            #   $self->setVisible), FALSE if it is not visible (after a call to $self->setInvisible)
            visibleFlag                 => TRUE,
            # Registry hash of 'free' windows (excluding 'dialogue' windows) for which this window
            #   is the parent, a subset of GA::Obj::Desktop->freeWinHash. Hash in the form
            #       $childFreeWinHash{unique_number} = blessed_reference_to_window_object
            childFreeWinHash            => {},
            # When a child 'free' window (excluding 'dialogue' windows) is destroyed, this parent
            #   window is informed via a call to $self->del_childFreeWin
            # When the child is destroyed, this window might want to call some of its own functions
            #   to update various widgets and/or IVs, in which case this window adds an entry to
            #   this hash; a hash in the form
            #       $childDestroyHash{unique_number} = list_reference
            # ...where 'unique_number' is the child window's ->number, and 'list_reference' is a
            #   reference to a list in groups of 2, in the form
            #       (sub_name, argument_list_ref, sub_name, argument_list_ref...)
            childDestroyHash            => {},

            # The container widget into which all other widgets are packed (usually a Gtk3::VBox or
            #   Gtk3::HBox, but any container widget can be used; takes up the whole window client
            #   area)
            packingBox                  => undef,

            # Standard IVs for 'grid' windows

            # The GA::Obj::WorkspaceGrid object into whose grid this window has been placed. 'undef'
            #   if $workspaceObj->gridEnableFlag = FALSE
            workspaceGridObj            => $workspaceGridObj,
            # The GA::Obj::Area object for this window. An area object is a part of a zone's
            #   internal grid, handling a single window (this one). Set to 'undef' in
            #   $workspaceObj->gridEnableFlag = FALSE
            areaObj                     => $areaObj,
            # For pseudo-windows (in which a window object is created, but its widgets are drawn
            #   inside a GA::Table::PseudoWin table object), the table object created. 'undef' if
            #   this window object is a real 'grid' window
            pseudoWinTableObj           => undef,
            # The ->name of the GA::Obj::Winmap object (not used for 'map' windows)
            winmap                      => undef,

            # IVs for this kind of 'map' window

            # The parent automapper object (a GA::Obj::Map - set later)
            mapObj                      => undef,
            # The session's current world model object
            worldModelObj               => $session->worldModelObj,

            # The menu bar
            menuBar                     => undef,
            # The window can have several toolbars. Toolbars each have a button set, and there are
            #   several button sets to choose from
            # If the window component for toolbars is turned on, there is at least one toolbar,
            #   which has a switcher button and an adder button, followed by the default button set.
            #   The switcher button switches between button sets, and the adder button adds a new
            #   toolbar below the existing ones
            # Additional toolbars have a remove button, which removes that toolbar
            # Each button set can only be visible once; the switcher button switches to the next
            #   button set that isn't already in use
            # GA::Obj::WorldModel->buttonSetList stores which button sets should be used when the
            #   automapper window opens, and is updated as the user adds/removes sets (but not when
            #   the user clicks the switcher button; when the window opens, the first button set is
            #   always the default one)
            #
            # Constant list of names of button sets, in the order used by the switcher button
            constButtonSetList          => [
                'default',              # The default set, visible when the window first opens
                'exits',
                'painting',
                'quick',
                'background',
                'tracking',
                'misc',
                'flags',
                'interiors',
            ],
            # Constant hash of names of button sets and their corresponding (short) descriptions
            constButtonDescripHash      => {
                'default'               => 'Show the default button set',    # Never actually used
                'exits'                 => 'Show exit customisation buttons',
                'painting'              => 'Show room painting buttons',
                'quick'                 => 'Show quick painting buttons',
                'background'            => 'Show background colouring buttons',
                'tracking'              => 'Show room tracking buttons',
                'misc'                  => 'Show miscellaneous buttons',
                'flags'                 => 'Show room flag filter buttons',
                'interiors'             => 'Show room interior buttons',
            },
            # Hash of names of button sets, showing which are visible (TRUE) and which are not
            #   visible (FALSE)
            buttonSetHash               => {
                'default'               => FALSE,
                'exits'                 => FALSE,
                'painting'              => FALSE,
                'quick'                 => FALSE,
                'background'            => FALSE,
                'tracking'              => FALSE,
                'misc'                  => FALSE,
                'flags'                 => FALSE,
                'interiors'             => FALSE,
            },
            # Ordered list of toolbar widgets that are visible now, with the default toolbar always
            #   first in the list
            toolbarList                 => [],
            # Corresponding hash of toolbar widgets, in the form
            #   $toolbarHash{toolbar_widget} = name_of_button_set_visible_now
            toolbarHash                 => {},
            # A list of button widgets in the original toolbar (not including the add button, the
            #   switcher button and the separator that follows them); updated every time the user
            #   clicks the switcher icon
            # NB Buttons in additional toolbars aren't stored in an IV
            toolbarButtonList           => [],
            # The 'add' button in the original toolbar
            toolbarAddButton            => undef,
            # The 'switch' button in the original toolbar
            toolbarSwitchButton         => undef,
            # The default set (the first one drawn); this IV never changes
            constToolbarDefaultSet      => 'default',
            # Whenever the original (first) toolbar is drawn, this IV records the button set used
            #   (so that the same button set appears whenever $self->redrawWidgets is called)
            toolbarOriginalSet          => 'default',
            # Hash of room flags currently in use as preferred room flags, and whose toolbar button
            #   in the 'painter' set (if visible) is currently toggled on. The hash is reset every
            #   time the toolbar is drawn or redrawn, and is used to make sure that the toggled
            #   button(s) remain toggled after the redraw
            # Hash in the form
            #   toolbarRoomFlagHash{room_flag} = undef
            toolbarRoomFlagHash         => {},
            # When a colour button in the quick painting button set it toggled, the corresponding
            #   room flag is stored here
            # When this IV is defined, clicking a room toggles the room flag in that room. If
            #   multiple rooms are selected, and one of the selected rooms was the clicked one,
            #   the room flag is toggled in all of them
            toolbarQuickPaintColour     => undef,

            # Menu bar/toolbar items which will be sensitised or desensitised, depending on the
            #   context. Hash in the form
            #       $menuToolItemHash{'item_name'} = Gtk3_widget
            # ...where:
            #   'item_name' is a descriptive scalar, e.g. 'move_up_level'
            #   'Gtk3_widget' is the Gtk3::MenuItem or toolbar widget, typically Gtk3::ToolButton or
            #       Gtk3::RadioToolButton
            # NB Entries in this hash continue to exist, after the widgets are no longer visible.
            #   Doesn't matter, because there are a limited number of 'item_name' scalars, and so
            #   a limited size to the hash (and referencing no-longer visible widgets, for example
            #   to sensitise/desensitise them, doesn't have any ill effects)
            menuToolItemHash            => {},

            # A horizontal pane, dividing the treeview on the left from everything else on the right
            hPaned                      => undef,

            # The treeview widgets (on the left)
            treeViewModel               => undef,
            treeView                    => undef,
            treeViewScroller            => undef,
            treeViewWidthPixels         => 150,     # (Default width)
            # The currently selected line of the treeview (selected by single-clicking on it)
            treeViewSelectedLine        => undef,
            # A hash of regions in the treeview, which stores which rows containing parent regions
            #   have been expanded to reveal their child regions
            # Hash in the form
            #   $treeViewRegionHash{region_name} = flag
            # ...where 'flag' is TRUE when the row is expanded, FALSE when the row is not expanded
            treeViewRegionHash          => {},
            # A hash of pointers (iters) in the treeview, so we can look up each region's cell
            treeViewPointerHash         => {},

            # Canvas widgets (on the right)
            # ->canvas and ->canvasBackground store widgets for the current region and level (or the
            #   empty background map, if no region/level are visible)
            canvas                      => undef,
            canvasBackground            => undef,
            canvasFrame                 => undef,
            canvasScroller              => undef,
            canvasHAdjustment           => undef,
            canvasVAdjustment           => undef,

            # The size of the available area inside the scrolled window, set whenever the
            #   scrolled window's size-allocate signal is emitted (this is the only way to guarantee
            #   that the correct size is available to $self->setMapPosn)
            canvasScrollerWidth         => 1,
            canvasScrollerHeight        => 1,

            # Blessed reference of the currently displayed GA::Obj::Regionmap ('undef' if no region
            #   is displayed; not necessarily the same region as the character's current location)
            currentRegionmap            => undef,
            # Blessed reference of the currently displayed GA::Obj::Parchment ('undef' if no region
            #   is displayed; not necessarily the same region as the character's current location)
            currentParchment            => undef,
            # List of the names of regions that have been the current region recently. Does not
            #   include the current region, nor any duplicates, nor more than three regions. The
            #   most recent current region is the first one in the list. The list is modified
            #   whenever $self->setCurrentRegion is called
            recentRegionList            => [],

            # Flag set to TRUE if the visible map is the empty background map (created by a call to
            #   $self->resetMap). Set to FALSE if the visible map is a region (created by a call to
            #   $self->refreshMap). Set to FALSE if neither ->resetMap nor ->refreshMap have been
            #   called yet
            emptyMapFlag                => FALSE,
            # The first call to $self->winUpdate calls $self->preparePreDraw to compile a list of
            #   regions which should be drawn by background processes (i.e. regular calls to
            #   $self->winUpdate). It then sets this flag to TRUE so it knows no further calls to
            #   $self->preparePreDraw are necessary
            winUpdateCalledFlag         => FALSE,
            # If a call to $self->doDraw fails because a drawing cycle (i.e. another call to
            #   ->doDraw) is already in progress, then this flag is set to TRUE. When set to TRUE,
            #   $self->winUpdate knows that it must make another call to ->doDraw
            winUpdateForceFlag          => FALSE,
            # When $self->doDraw fails on a call from $self->setCurrentRegion or ->redrawRegions,
            #   this flag is also set to TRUE, as additional action is required. It remains set to
            #   FALSE when calls to ->doDraw fail for any other reason
            winUpdateShowFlag           => FALSE,

            # Hash of parchment objects (GA::Obj::Parchment), once for each region which has been
            #   drawn (or is being drawn)
            # Hash in the form
            #   $parchmentHash{region_name} = blessed_reference_to_parchment_object
            parchmentHash               => {},
            # Parchment objects can be in states - fully drawn, or partially drawn. Firstly, a hash
            #   of parchment objects which are fully drawn, in the form
            #   $parchmentReadyHash{region_name} = blessed_reference_to_parchment_object
            parchmentReadyHash          => {},
            # Secondly, a list of parchment objects which are partially drawn, Background processes
            #   draw canvas objects in the first parchment object in the list, before moving it to
            #   ->parchmentReadyHash; then they start drawing the next parchment object in the list
            #   until the list is empty
            parchmentQueueList          => [],

            # Tooltips
            # The current canvas object for which a tooltip is displayed ('undef' if no canvas
            #   object has a tooltip displayed)
            canvasTooltipObj            => undef,
            # What type of canvas object it is: 'room', 'room_tag', 'room_guild', 'exit', 'exit_tag'
            #   or 'label'
            canvasTooltipObjType        => undef,
            # When tooltips are visible, a useless 'leave-notify' event occurs. When the mouse moves
            #   over a canvas object, ->canvasTooltipObj is set and this IV is set to TRUE; if the
            #   next event is a 'leave-notify' event, it is ignored
            canvasTooltipFlag           => FALSE,

            # Objects on the map can be selected. There are three modes of selection:
            #   (1) There is a single room, OR a single room tag, OR a single room guild, OR a
            #           single exit, OR a single exit tag, OR a single label selected
            #   (2) Multiple objects are selected (including combinations of rooms, room tags, room
            #           guilds, exits, exit tags and labels)
            #   (3) Nothing is currently selected
            # In mode (1), one (or none) of the IVs ->selectedRoom, ->selectedRoomTag,
            #   ->selectedRoomGuild, ->selectedExit, ->selectedExitTag or ->selectedLabel is set
            #   (but the mode 2 IVs are empty)
            # In mode (2), the selected objects are in ->selectedRoomHash, ->selectedRoomTagHash,
            #   ->selectedRoomGuildHash ->selectedExitHash, ->selectedExitTagHash and
            #   ->selectedLabelHash (but the mode 1 IVs are set to 'undef')
            # In mode (3), all of the IVs below are not set
            #
            # Mode (1) IVs
            # Blessed reference of the currently selected location (a GA::ModelObj::Room), which
            #   might be the same as $self->mapObj->currentRoom or $self->mapObj->lastKnownRoom
            selectedRoom                => undef,
            # Blessed reference of the location (a GA::ModelObj::Room) whose room tag is selected
            selectedRoomTag             => undef,
            # Blessed reference of the location (a GA::ModelObj::Room) whose room guild is selected
            selectedRoomGuild           => undef,
            # Blessed reference of the currently selected exit (a GA::Obj::Exit)
            selectedExit                => undef,
            # Blessed reference of the exit (a GA::Obj::Exit) whose exit tag is selected
            selectedExitTag             => undef,
            # Blessed reference of the currently selected label (a GA::Obj::MapLabel object)
            selectedLabel               => undef,

            # Mode (2) IVs
            # Hash of selected locations, in the form
            #   $selectedRoomHash{model_number} = blessed_reference_to_room_object
            selectedRoomHash            => {},
            # Hash of locations whose room tags are selected, in the form
            #   $selectedRoomTagHash{model_number) = blessed_reference_to_room_object
            selectedRoomTagHash         => {},
            # Hash of locations whose room guilds are selected, in the form
            #   $selectedRoomGuildHash{model_number) = blessed_reference_to_room_object
            selectedRoomGuildHash       => {},
            # Hash of selected exits, in the form
            #   $selectedExitHash{exit_model_number} = blessed_reference_to_exit_object
            selectedExitHash            => {},
            # Hash of exits whose exit tags are selected, in the form
            #   $selectedExitTagHash{exit_model_number} = blessed_reference_to_exit_object
            selectedExitTagHash         => {},
            # Hash of selected labels, in the form
            #   $selectedLabelHash{label_id) = blessed_reference_to_map_label_object
            # ...where 'label_id' is in the form 'region-name_label_number', e.g. 'town_42'
            selectedLabelHash           => {},

            # When there is a single selected exit ($self->selectedExit is set), and if it's a
            #   broken or a region exit, the twin exit (and its parent room) are drawn a different
            #   colour
            # When the broken/region exit is selected, these IVs are set...
            # The blessed reference of the twin exit
            pairedTwinExit              => undef,
            # The blessed reference of the twin exit's parent room
            pairedTwinRoom              => undef,

            # Flag that can be set to TRUE by any code that wants to prevent a drawing operation
            #   from starting (temporarily); the operation will be able to start when the flag is
            #   set back to FALSE
            # Is set to TRUE at the beginning of calls to ->doDraw and ->doQuickDraw, so that a
            #   second call to either function can't be processed while an earlier one is still in
            #   progress
            delayDrawFlag               => FALSE,
            # When pre-drawing, objects are drawn in stack order, from bottom to top; in very large
            #   maps (thousands of rooms), GooCanvas2 can complete the drawing much more quickly
            #   when everything can be raised to the top of the stack, rather than being
            #   arbitrarily inserted somewhere in the middle
            # Flag set to TRUE at the beginning of calls to ->doQuickDraw, so that individual
            #   drawing functions like $self->drawRoomBox, ->drawIncompleteExit (etc) can raise the
            #   canvas object to the top of the drawing stack every time
            quickDrawFlag               => FALSE,

            # Drawing cycle IVs (each call to $self->doDraw is a single drawing cycle). Some of
            #   these IVs are also set in a call to ->doQuickDraw
            #
            # During the drawing cycle, regions are drawn one at a time. For each region, these IVs
            #   is set so individual drawing functions can quickly look up the regionmap and
            #   parchment object being drawn
            drawRegionmap               => undef,
            drawParchment               => undef,
            drawScheme                  => undef,
            # $self->drawCycleExitHash contains a list of exits drawn during a single drawing cycle.
            #   Before drawing an exit, we can check whether it has a twin exit (which occupies the
            #   same space) and, if so, we don't need to draw it a second time - thus each exit-twin
            #   exit pair is only drawn once for each call to $self->doDraw
            # Hash in the form
            #       $drawCycleExitHash{exit_model_number} = undef
            drawCycleExitHash           => {},
            # The size of room interior text. This value is set by $self->prepareDraw, at the start
            #   of every drawing cycle, to be a little bit smaller than half the width of the room
            #   (which depends on the draw exit mode in effect)
            drawRoomTextSize            => undef,
            # For room interior text, the size of the usable area (which depends on the draw exit
            #   mode in effect). The values are also set by $self->doDraw, once per drawing cycle
            drawRoomTextWidth           => undef,
            drawRoomTextHeight          => undef,
            # The size of other text drawn on the map, besides room interior text (includes room
            #   tags, room guilds, exit tags and labels). Also set by $self->prepareDraw, based on
            #   the size of a room when exits are being drawn
            drawOtherTextSize           => undef,
            # Hashes set by $self->preDrawPositions and $self->preDrawExits (see the comments in
            #   those functions for a longer explanation)
            # Calculates the position of each type of exit, and of a few room components, relative
            #   to their gridblocks, to make the drawing of rooms and exits much quicker
            blockCornerXPosPixels       => undef,
            blockCornerYPosPixels       => undef,
            blockCentreXPosPixels       => undef,
            blockCentreYPosPixels       => undef,
            borderCornerXPosPixels      => undef,
            borderCornerYPosPixels      => undef,
            preDrawnIncompleteExitHash  => {},
            preDrawnUncertainExitHash   => {},
            preDrawnLongExitHash        => {},
            preDrawnSquareExitHash      => {},
            # Also calculates which primary directions should be used, if counting checked/
            #   checkeable directions (i.e. when $self->worldModelObj->roomInteriorMode is set to
            #   'checked_count')
            preCountCheckedHash         => {},
            # When obscured rooms are enabled, exits are only drawn for rooms near the current room,
            #   or for selected rooms (and selected exits), and for rooms whose rooms flags match
            #   those in GA::Client->constRoomNoObscuredHash (e.g. 'main_route')
            # When obscured rooms are enabled, these hashes are emptied and then re-compiled by a
            #   call to ->doDraw, or (in anticipation of several calls to ->doQuickDraw), by
            #   ->preparePreDraw or ->redrawRegions
            # This hash contains any rooms which are due to be drawn, which should not be obscured.
            #   Hash in the form
            #       $noObscuredRoomHash{model_number} = undef
            noObscuredRoomHash          => {},
            # This hash contains any rooms which have been drawn un-obscured, but which are to be
            #   drawn re-obscured
            #   Hash in the form
            #       $reObscuredRoomHash{model_number} = undef
            reObscuredRoomHash          => {},

            # What happens when the user clicks on the map
            #   'default' - normal operation. Any selected objects are unselected
            #   'add_room' - 'Add room at click' menu option - when the user clicks on the map, a
            #       new room is added at that location
            #   'connect_exit' - 'Connect [exit] to click' menu option - when the user clicks on a
            #       room on the map (on any level, in any region), the exit is connected to that
            #       room
            #   'add_label' - 'Add label at click' menu option - when the user clicks on the map, a
            #       new label is added at that location
            #   'move_room' - 'Move selected rooms to click' menu option - when the user clicks on
            #       the map (probably in a new region), the selected rooms (and their room tags/room
            #       guilds/exits/exit tags) and labels are move to that position on the map
            #   'merge_room' - 'Merge/move rooms' menu option - when the user clicks on a room on
            #       the map, and it's one of the rooms specified by GA::Obj::Map->currentMatchList,
            #       the current room is merged with the clicked room
            # NB To set this IV, you must call $self->set_freeClickMode($mode) or
            #   $self->reset_freeClickMode() (which sets it to 'default')
            freeClickMode               => 'default',
            # A 'move selected rooms to click' operation can also be initiated by press CTRL+C.
            #   $self->setKeyPressEvent needs to know when the CTRL key is being held down, so this
            #   flag is set to TRUE when it's held down
            ctrlKeyFlag                 => FALSE,
            # Background colour mode can be applied whenever ->freeClickMode is 'default', and is
            #   used to colour in gridblocks in the background canvas:
            #   'default' - Normal operation. Clicks on the canvas don't affect background colour
            #   'square_start' - Clicks on a gridblock colours in that gridblock
            #   'rect_start' - The first click on a gridblock marks that gridblock as one corner in
            #       a rectangle. The mode is changed to 'rect_stop' in anticipation of the next
            #       click
            #   'rect_stop' - The first click on a gridblock marks that gridblock as the opposite
            #       corner in a rectangle. The mode is changed back to 'rect_start'
            bgColourMode                => 'default',
            # The colour to use when colouring in the background. If 'undef', colours are removed
            #   from gridblock(s) instead of being added. If defined, should be an RGB tag like
            #   '#ABCDEF' (case-insensitive)
            bgColourChoice              => undef,
            # When $self->bgColourMode is 'rect_start' and the user clicks on the map, the value
            #   is changed to 'rect_stop' and the coordinates of the click are stored in these
            #   IVs, while waiting for a second click
            bgRectXPos                  => undef,
            bgRectYPos                  => undef,
            # Flag set to TRUE when new coloured blocks/rectangles should be drawn on all levels,
            #   FALSE when new coloured blocks/squares should only be drawn on the current level
            bgAllLevelFlag              => FALSE,
            # When working out whether the user has clicked on an exit, how closely the angle of the
            #   drawn exit's gradient (relative to the x-axis) must match the gradient of a line
            #   from the exit's origin point, to the point on the map the user clicked (in degrees)
            exitSensitivity             => 30,
            # A value used to draw bends on bending exits
            exitBendSize                => 4,
            # When the user right-clicks on an exit, we need to record the position of the click, in
            #   case the user wants to add an exit bend at that point. These IVs are reset by a
            #   click on any other part of the canvas
            exitClickXPosn              => undef,
            exitClickYPosn              => undef,

            # GooCanvas2 signals don't recognise double clicks on canvas objects such as rooms,
            #   therefore we have to implement our own double-click detection code
            # These IVs are set when the user left-clicks on a room, and reset when a double-click
            #   is detected on the same room
            leftClickTime               => undef,
            leftClickObj                => undef,
            # The maximumd time (in seconds) between the two clicks
            leftClickWaitTime           => 0.3,

            # The operating mode:
            #   'wait'      - The automapper isn't doing anything
            #   'follow'    - The automapper is following the character's position, but not
            #                   updating the world model (except for the character visit count)
            #   'update'    - The automapper is following the character's position and updating the
            #                   world model when required
            mode                        => 'wait',

            # To show visits for a different character, this IV is set to a character's name (which
            #   matches the name of a character profile). If set to 'undef', the current character
            #   profile is used
            showChar                    => undef,
            # The painter is a non-model GA::ModelObj::Room object stored in the world model. When
            #   this flag is set to TRUE, the painter's IVs are used to create (or update) new room
            #   objects; if set to FALSE, the painter is ignored
            painterFlag                 => FALSE,

            # Flag set to TRUE when 'graffiti mode' is on
            graffitiModeFlag            => FALSE,
            # Every room the character visits while graffiti mode is on (when $self->mode is
            #   'follow' or 'update') is added to this hash, and is drawn differently. The hash is
            #   emptied when graffiti mode is turned off (or when the window is closed)
            # No changes are made to the world model because of graffiti mode (though the model is
            #   still updated in the normal way, with character visits and so on). If multiple
            #   sessions are showing the same map, other automapper windows are not affected
            graffitiHash                => {},

            # Primary vector hash - maps Axmud's primary directions onto a vector, expressed in a
            #   list reference as (x, y, z), showing the direction that each primary direction takes
            #   us on the Axmud map
            # (In the grid, the top-left corner at the highest level has coordinates 0, 0, 0)
            constVectorHash             => {
                north                   => [0, -1, 0],
                northnortheast          => [0.5, -1, 0],
                northeast               => [1, -1, 0],
                eastnortheast           => [1, -0.5, 0],
                east                    => [1, 0, 0],
                eastsoutheast           => [1, 0.5, 0],
                southeast               => [1, 1, 0],
                southsoutheast          => [0.5, 1, 0],
                south                   => [0, 1, 0],
                southsouthwest          => [-0.5, 1, 0],
                southwest               => [-1, 1, 0],
                westsouthwest           => [-1, 0.5, 0],
                west                    => [-1, 0, 0],
                westnorthwest           => [-1, -0.5, 0],
                northwest               => [-1, -1, 0],
                northnorthwest          => [-0.5, -1, 0],
                up                      => [0, 0, 1],
                down                    => [0, 0, -1],
            },
            # A second vector hash for drawing two-way exits (which are drawn as two parallel lines)
            # Each value is expressed as a list reference (x1, y1, x2, y2)
            # (x1, y1) are simply added to the coordinates of the start and stop pixels of the first
            #   line, and (x2, y2) are added to the start and stop pixels of the second line - this
            #   moves the two lines either side of where the line is normally drawn
            # NB 'up' and 'down' are never drawn with double lines, so their values are both
            #   [0, 0, 0, 0]
            constDoubleVectorHash       => {
                north                   => [-2, 0, 2, 0],
                northnortheast          => [-2, 0, 2, 0],   # Also on top of room box, so same as N
                northeast               => [-2, 0, 0, 2],
                eastnortheast           => [0, -2, 0, 2],   # Same as E
                east                    => [0, -2, 0, 2],
                eastsoutheast           => [0, -2, 0, 2],   # Same as E
                southeast               => [-2, 0, 0, -2],
                southsoutheast          => [-2, 0, 2, 0],   # Same as S
                south                   => [-2, 0, 2, 0],
                southsouthwest          => [-2, 0, 2, 0],   # Same as S
                southwest               => [0, -2, 2, 0],
                westsouthwest           => [0, -2, 0, 2],   # Same as W
                west                    => [0, -2, 0, 2],
                westnorthwest           => [0, -2, 0, 2],   # Same as W
                northwest               => [2, 0, 0, 2],
                northnorthwest          => [-2, 0, 2, 0],   # Same as N
                up                      => [0, 0, 0, 0],
                down                    => [0, 0, 0, 0],
            },
            # A third vector hash for drawing one-way exits (which are drawn as a single line, with
            #   an arrowhead at the edge of the block, showing the exit's direction
            # Each value is expressed as a list reference (x1, y1, x2, y2)
            # (x1, y1) is a vector showing the direction of one half of the arrowhead, starting at
            #   the edge of the block. (x2, y2) is a vector showing the direction of travel of the
            #   other half
            # NB 'up' and 'down' are never drawn with single lines, so their values are both
            #   [0, 0, 0, 0]
            constArrowVectorHash        => {
                north                   => [-1, 1, 1, 1],
                northnortheast          => [-0.8, 0.5, 0.5, 0.8], # Approx. a right-angled arrowhead
                northeast               => [-1, 0, 0, 1],
                eastnortheast           => [-0.8, -0.5, -0.5, 0.8],
                east                    => [-1, -1, -1, 1],
                eastsoutheast           => [-0.5, -0.8, -0.8, 0.5],
                southeast               => [-1, 0, 0, -1],
                southsoutheast          => [-0.8, -0.5, 0.5, -0.8],
                south                   => [-1, -1, 1, -1],
                southsouthwest          => [-0.5, -0.8, 0.8, -0.5],
                southwest               => [0, -1, 1, 0],
                westsouthwest           => [0.5, -0.8, 0.8, 0.5],
                west                    => [1, -1, 1, 1],
                westnorthwest           => [0.8, -0.5, 0.5, 0.8],
                northwest               => [1, 0, 0, 1],
                northnorthwest          => [0.8, 0.5, -0.5, 0.8],
                up                      => [0, 0, 0, 0],
                down                    => [0, 0, 0, 0],
            },
            # A fourth vector hash for drawing exit ornaments (which are drawn perpendicular to the
            #   exit line)
            # Each value is expressed as a list reference (x1, y1, x2, y2)
            # (x1, y1) is a vector showing the direction of one half of the ornament, generally
            #   starting in the middle of the exit line (and perpendicular to it). (x2, y2) is a
            #   vector showing the direction of the other half
            # NB 'up' and 'down' are never drawn with single lines, so their values are both
            #   [0, 0, 0, 0]
            constPerpVectorHash         => {    # 'perp' for 'perpendicular'
                north                   => [-1, 0, 1, 0],
                northnortheast          => [-0.8, -0.5, 0.8, 0.5],  # Approx perpendicular line
                northeast               => [-1, -1, 1, 1],
                eastnortheast           => [-0.5, -0.8, 0.5, 0.8],
                east                    => [0, -1, 0, 1],
                eastsoutheast           => [0.5, -0.8, -0.5, 0.8],
                southeast               => [-1, 1, 1, -1],
                southsoutheast          => [-0.8, 0.5, 0.8, -0.5],
                south                   => [-1, 0, 1, 0],
                southsouthwest          => [-0.8, -0.5, 0.8, 0.5],
                southwest               => [-1, -1, 1, 1],
                westsouthwest           => [-0.5, -0.8, 0.5, 0.8],
                west                    => [0, -1, 0, 1],
                westnorthwest           => [-0.5, 0,8, 0.5, -0.8],
                northwest               => [-1, 1, 1, -1],
                northnorthwest          => [-0.8, 0.5, 0.8, -0.5],
                up                      => [0, 0, 0, 0],
                down                    => [0, 0, 0, 0],
            },
            # A fifth vector hash, a slightly modified version of ->constVectorHash, used by
            #   GA::Obj::Map->moveKnownDirSeen for placing new rooms on the map.
            # Moves in the north-south west-east southwest-northeast and southeast-northwest
            #   directions are placed in adjacent gridblocks, but moves in the northnortheast (etc)
            #   direction have to be placed about 2 gridblocks away
            constSpecialVectorHash      => {
                north                   => [0, -1, 0],  # Same values used in ->constVectorHash
                northnortheast          => [1, -2, 0],  # Double values used in ->constVectorHash
                northeast               => [1, -1, 0],
                eastnortheast           => [2, -1, 0],
                east                    => [1, 0, 0],
                eastsoutheast           => [2, 1, 0],
                southeast               => [1, 1, 0],
                southsoutheast          => [1, 2, 0],
                south                   => [0, 1, 0],
                southsouthwest          => [-1, 2, 0],
                southwest               => [-1, 1, 0],
                westsouthwest           => [-2, 1, 0],
                west                    => [-1, 0, 0],
                westnorthwest           => [-2, -1, 0],
                northwest               => [-1, -1, 0],
                northnorthwest          => [-1, -2, 0],
                up                      => [0, 0, 1],
                down                    => [0, 0, -1],
            },
            # A hash for drawing triangles in a return exit. One of the triangle's points is at the
            #   pixel where an incomplete exit would start, touching the room box. The other two
            #   points are corners of the square used to draw broken/region exits
            # $self->preDrawnSquareExitHash describes the positions of opposite corners of this
            #   square as:
            #       (top_left_x, top_left_y, bottom_right_x, bottom_right_y)
            # This hash tells gives us four of these values, referred to as 0-3
            #       (0,          1,          2,              3)
            # The first pair describes the second corner of the triangle; the second pair describes
            #   the third corner of the triangle
            constTriangleCornerHash     => {
                north                   => [0, 1, 2, 1],
                northnortheast          => [0, 1, 2, 1],    # Same as N
                northeast               => [0, 1, 2, 3],
                eastnortheast           => [2, 1, 2, 3],    # Same as E
                east                    => [2, 1, 2, 3],
                eastsoutheast           => [2, 1, 2, 3],    # Same as E
                southeast               => [2, 1, 0, 3],
                southsoutheast          => [0, 3, 2, 3],    # Same as S
                south                   => [0, 3, 2, 3],
                southsouthwest          => [0, 3, 2, 3],    # Same as S
                southwest               => [0, 1, 2, 3],
                westsouthwest           => [0, 1, 0, 3],    # Same as W
                west                    => [0, 1, 0, 3],
                westnorthwest           => [0, 1, 0, 3],    # Same as W
                northwest               => [2, 1, 0, 3],
                northnorthwest          => [0, 1, 2, 1],    # Same as N
                up                      => [0, 0],
                down                    => [0, 0],
            },
            # Anchor hashes - converts a standard primary direction into a Gtk3 anchor constant, so
            #   that exit tags can be drawn in the right position
            constGtkAnchorHash          => {
                north                   => 'GOO_CANVAS_ANCHOR_S',
                # No GooCanvas2 constant for NNE, etc, so use the same as N
                northnortheast          => 'GOO_CANVAS_ANCHOR_S',  # Same as N
                northeast               => 'GOO_CANVAS_ANCHOR_SW',
                eastnortheast           => 'GOO_CANVAS_ANCHOR_W',  # Same as E
                east                    => 'GOO_CANVAS_ANCHOR_W',
                eastsoutheast           => 'GOO_CANVAS_ANCHOR_W',  # Same as E
                southeast               => 'GOO_CANVAS_ANCHOR_NW',
                southsoutheast          => 'GOO_CANVAS_ANCHOR_N',  # Same as S
                south                   => 'GOO_CANVAS_ANCHOR_N',
                southsouthwest          => 'GOO_CANVAS_ANCHOR_N',  # Same as S
                southwest               => 'GOO_CANVAS_ANCHOR_NE',
                westsouthwest           => 'GOO_CANVAS_ANCHOR_E',  # Same as W
                west                    => 'GOO_CANVAS_ANCHOR_E',
                westnorthwest           => 'GOO_CANVAS_ANCHOR_E',  # Same as w
                northwest               => 'GOO_CANVAS_ANCHOR_SE',
                northnorthwest          => 'GOO_CANVAS_ANCHOR_S',  # Same as N
            },

            # Magnfication list. A list of standard magnification factors used for zooming in or out
            #   from the map
            # Each GA::Obj::Regionmap object has its own ->magnification IV, so zooming on one
            #   region doesn't affect the magnification of others
            # When the user zooms in or out, ->magnification is set to one of the values in this
            #   list, and various IVs in GA::Obj::Regionmap (such as ->blockWidthPixels and
            #   ->roomHeightPixels) are changed. When the map is redrawn, everything in it is bigger
            #   (or smaller)
            constMagnifyList            => [
                0.01, 0.02, 0.04, 0.06, 0.08, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9,
                1,
                1.1, 1.2, 1.35, 1.5, 2, 3, 5, 7, 10,
            ],
            # A subset of these magnifications, used as menu items
            constShortMagnifyList       => [
                0.5, 0.8, 1, 1.2, 1.5, 1.75, 2
            ],
            # When some menu items are selected (e.g. View > Room filters > Release markers filter),
            #   a call is made to this session's GA::Obj::WorldModel, which in turn calls every
            #   Automapper window using the model, in order to update its menu. When this happens,
            #   the following flag is set to TRUE, so that updating the menu item doesn't cause
            #   further calls to GA::Obj::WorldModel
            ignoreMenuUpdateFlag        => FALSE,

            # IVs used during a drag operation
            # Flag set to TRUE during drag mode (set from the menu or the toolbar). Normally, it's
            #   necessary to hold down the Alt-Gr key to drag canvas objects; when drag mode is on,
            #   clicks on canvas objects are treated as the start of a drag, rather than a
            #   select/unselect operation)
            # NB During a drag operation initiated with the Alt-Gr key, ->dragModeFlag's value
            #   doesn't change
            dragModeFlag                => FALSE,
            # Flag set to TRUE when a dragging operation starts
            dragFlag                    => FALSE,
            # Flag set to TRUE when $self->continueDrag is called, and set back to FALSE at the
            #   end of that call. ->continueDrag does nothing if a previous call to the function
            #   hasn't been completed (happens a lot)
            dragContinueFlag            => FALSE,
            # The canvas object that was underneath the mouse cursor when the drag operation began
            #   (the object that was grabbed, when using Alt-Gr)
            dragCanvasObj               => undef,
            # A list of all canvas objects that are being dragged together. $self->dragCanvasObj is
            #   always the first item in the list
            # If $self->dragCanvasObj is a room, all selected rooms/labels in the same region are
            #   dragged together
            # If $self->dragCanvasObj is a label, both the label and its box (if drawn) are dragged
            #   together
            dragCanvasObjList           => [],
            # The GA::ModelObj::Room / GA::Obj::Exit / GA::Obj::MapLabel being dragged,
            #   corresponding to $self->dragCanvasObj
            dragModelObj                => undef,
            # The type of object being dragged - 'room', 'room_tag', 'room_guild', 'exit',
            #   'exit_tag' or 'label'
            dragModelObjType            => undef,
            # The canvas object's initial coordinates on the canvas
            dragInitXPos                => undef,
            dragInitYPos                => undef,
            # The canvas object's current coordinates on the canvas
            dragCurrentXPos             => undef,
            dragCurrentYPos             => undef,
            # When dragging a room(s), the fake room(s) drawn at the original location (so that the
            #   exits don't look messy)
            dragFakeRoomList            => [],
            # When dragging an exit bend, the bend's index in the exit's list of bends (the bend
            #   closest to the start of the exit has the index 0)
            dragBendNum                 => undef,
            # When dragging an exit bend, the initial position of the bend, relative to the start of
            #   the bending section of the exit
            dragBendInitXPos            => undef,
            dragBendInitYPos            => undef,
            # The corresponding IVs for the twin exit, when dragging an exit bend
            dragBendTwinNum             => undef,
            dragBendTwinInitXPos        => undef,
            dragBendTwinInitYPos        => undef,
            # When dragging an exit bend, the exit drawing mode (corresponds to
            #   GA::Obj::WorldModel->drawExitMode)
            dragExitDrawMode            => undef,
            # When dragging an exit bend, the draw ornaments flag (corresponds to
            #   GA:Obj::WorldModel->drawOrnamentsFlag
            dragExitOrnamentsFlag       => undef,

            # IVs used during a selection box operation
            # Flag set to TRUE when a selection box operation starts, but before the box has
            #   actually been drawn
            selectBoxFlag               => FALSE,
            # The selection box's canvas object, once it has been drawn
            selectBoxCanvasObj          => undef,
            # The canvas 's initial coordinates on the canvas
            selectBoxInitXPos           => undef,
            selectBoxInitYPos           => undef,
            # The canvas object's current coordinates on the canvas
            selectBoxCurrentXPos        => undef,
            selectBoxCurrentYPos        => undef,
        };

        # Bless the object into existence
        bless $self, $class;

        return $self;
    }

    ##################
    # Methods

    # Standard window object functions

    sub winSetup {

        # Called by GA::Obj::Workspace->createGridWin or ->createSimpleGridWin
        # Creates the Gtk3::Window itself
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $title      - The window title; ignored if specified ($self->setWinTitle sets the
        #                   window title)
        #   $listRef    - Reference to a list of functions to call, just after the Gtk3::Window is
        #                   created (can be used to set up further ->signal_connects, if this
        #                   window needs them)
        #
        # Return values
        #   'undef' on improper arguments or if the window can't be opened
        #   1 on success

        my ($self, $title, $listRef, $check) = @_;

        # Local variables
        my $iv;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->winSetup', @_);
        }

        # Don't create a new window, if it already exists
        if ($self->enabledFlag) {

            return undef;
        }

        # Create the Gtk3::Window
        my $winWidget = Gtk3::Window->new('toplevel');
        if (! $winWidget) {

            return undef;

        } else {

            # Store the IV now, as subsequent code needs it
            $self->ivPoke('winWidget', $winWidget);
            $self->ivPoke('winBox', $winWidget);
        }

        # Set up ->signal_connects (other ->signal_connects are set up in the call to
        #   $self->winEnable() )
        $self->setDeleteEvent();            # 'delete-event'
        $self->setKeyPressEvent();          # 'key-press-event'
        $self->setKeyReleaseEvent();        # 'key-release-event'
        $self->setFocusOutEvent();          # 'focus-out-event'
        # Set up ->signal_connects specified by the calling function, if any
        if ($listRef) {

            foreach my $func (@$listRef) {

                $self->$func();
            }
        }

        # Set the window title. If $title wasn't specified, use a suitable default title
        $self->setWinTitle();

        # Set the window's default size and position (this will almost certainly be changed before
        #   the call to $self->winEnable() )
        $winWidget->set_default_size(
            $axmud::CLIENT->customGridWinWidth,
            $axmud::CLIENT->customGridWinHeight,
        );

        $winWidget->set_border_width($axmud::CLIENT->constGridBorderPixels);

        # Set the icon list for this window
        $iv = $self->winType . 'WinIconList';
        $winWidget->set_icon_list($axmud::CLIENT->desktopObj->{$iv});

        # Draw the widgets used by this window
        if (! $self->drawWidgets()) {

            return undef;
        }

        # The calling function can now move the window into position, before calling
        #   $self->winEnable to make it visible, and to set up any more ->signal_connects()
        return 1;
    }

    sub winEnable {

        # Called by GA::Obj::Workspace->createGridWin or ->createSimpleGridWin
        # After the Gtk3::Window has been setup and moved into position, makes it visible and calls
        #   any further ->signal_connects that must be not be setup until the window is visible
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $listRef    - Reference to a list of functions to call, just after the Gtk3::Window is
        #                   created (can be used to set up further ->signal_connects, if this
        #                   window needs them)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 on success

        my ($self, $listRef, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->winEnable', @_);
        }

        # Make the window appear on the desktop
        $self->winShowAll($self->_objClass . '->winEnable');
        $self->ivPoke('enabledFlag', TRUE);

        # For windows about to be placed on a grid, briefly minimise the window so it doesn't
        #   appear in the centre of the desktop before being moved to its correct workspace, size
        #   and position
#        if ($self->workspaceGridObj && $self->winWidget eq $self->winBox) {
#
#            $self->minimise();
#        }

        # This type of window is unique to its GA::Session (only one can be open at any time, per
        #   session); inform the session it has opened
        $self->session->set_mapWin($self);

        # Set up ->signal_connects that must not be set up until the window is visible
        $self->setConfigureEvent();         # 'configure-event'
        # Set up ->signal_connects specified by the calling function, if any
        if ($listRef) {

            foreach my $func (@$listRef) {

                $self->$func();
            }
        }

        # If the automapper object is in 'track alone' mode, disable the mode
        $self->session->mapObj->set_trackAloneFlag(FALSE);

        return 1;
    }

    sub winDisengage {

        # Should not be called, in general (provides compatibility with other types of window,
        #   whose window objects can be destroyed without closing the windows themselves)
        # If called, this function just calls $self->winDestroy and returns the result
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the window can't be disengaged
        #   1 on success

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->winDisengage', @_);
        }

        return $self->winDestroy();
    }

    sub winDestroy {

        # Called by GA::Obj::WorkspaceGrid->stop or by any other function
        # Updates the automapper object (GA::Obj::Map), informs the parent workspace grid (if this
        #   'grid' window is on a workspace grid) and the desktop object, and then destroys the
        #   Gtk3::Window (if it is open)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the window can't be destroyed or if it has already
        #       been destroyed
        #   1 on success

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->winDestroy', @_);
        }

        if (! $self->winBox) {

            # Window already destroyed in a previous call to this function
            return undef;
        }

        # If the pause window is visible, destroy it
        if ($axmud::CLIENT->busyWin) {

            $self->hidePauseWin();
        }

        # If the automapper object knows the current world model room, and if the Locator task is
        #   running and knows about the current location, and if the world model flag that permits
        #   it is set, and if this Automapper window isn't currently in 'wait' mode, let the
        #   automapper go into 'track alone' mode
        if (
            $self->mapObj->currentRoom
            && $self->session->locatorTask
            && $self->session->locatorTask->roomObj
            && $self->worldModelObj->allowTrackAloneFlag
            && $self->mode ne 'wait'
        ) {
            # Go into 'track alone' mode
            $self->mapObj->set_trackAloneFlag(TRUE);
        }

        # Update the parent GA::Obj::Map in all cases
        $self->mapObj->set_mapWin();

        # Close any 'free' windows for which this window is a parent
        foreach my $winObj ($self->ivValues('childFreeWinHash')) {

            $winObj->winDestroy();
        }

        # Inform the parent workspace grid object (if any)
        if ($self->workspaceGridObj) {

            $self->workspaceGridObj->del_gridWin($self);
        }

        # Inform the desktop object
        $axmud::CLIENT->desktopObj->del_gridWin($self);

        # Destroy the Gtk3::Window
        eval { $self->winBox->destroy(); };
        if ($@) {

            # Window can't be destroyed
            return undef;

        } else {

            $self->ivUndef('winWidget');
            $self->ivUndef('winBox');
        }

        # Inform the ->owner, if there is one
        if ($self->owner) {

            $self->owner->del_winObj($self);
        }

        # This type of window is unique to its GA::Session (only one can be open at any time, per
        #   session); inform the session it has closed
        $self->session->set_mapWin();

        return 1;
    }

#   sub winShowAll {}           # Inherited from GA::Win::Generic

    sub drawWidgets {

        # Called by $self->winSetup
        # Sets up the Gtk3::Window by drawing its widgets
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 on success

        my ($self, $check) = @_;

        # Local variables
        my ($menuBar, $hPaned, $treeViewScroller, $canvasFrame);

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->drawWidgets', @_);
        }

        # Create a packing box
        my $packingBox = Gtk3::VBox->new(FALSE, 0);
        $self->winBox->add($packingBox);
        $packingBox->set_border_width(0);
        # Update IVs immediately
        $self->ivPoke('packingBox', $packingBox);

        # Create a menu (if allowed)
        if ($self->worldModelObj->showMenuBarFlag) {

            $menuBar = $self->enableMenu();
            if ($menuBar) {

                # Pack the widget
                $packingBox->pack_start($menuBar, FALSE, FALSE, 0);
            }
        }

        # Create toolbar(s) at the top of the window (if allowed)
        if ($self->worldModelObj->showToolbarFlag) {

            # Reset toolbar IVs to their default state; the subsequent call to $self->enableToolbar
            #   imports the list of button sets from the world model, and updates these IVs
            #   accordinly
            $self->resetToolbarIVs();

            foreach my $toolbar ($self->enableToolbar()) {

                # Pack the widget
                $packingBox->pack_start($toolbar, FALSE, FALSE, 0);
            }
        }

        # Create a horizontal pane to divide everything under the menu into two, with the treeview
        #   on the left, and everything else on the right (only if both the treeview and the canvas
        #   are shown)
        if ($self->worldModelObj->showTreeViewFlag && $self->worldModelObj->showCanvasFlag) {

            $hPaned = Gtk3::HPaned->new();
            if ($hPaned) {

                # Set the width of the space about to be filled with the treeview
                $hPaned->set_position($self->treeViewWidthPixels);

                # Pack the widget
                $packingBox->pack_start($hPaned, TRUE, TRUE, 0);
                $self->ivPoke('hPaned', $hPaned);
            }
        }

        # Create a treeview (if allowed)
        if ($self->worldModelObj->showTreeViewFlag) {

            $treeViewScroller = $self->enableTreeView();
            if ($treeViewScroller) {

                # Pack the widget
                if ($hPaned) {

                    # Add the treeview's scroller to the left pane
                    $hPaned->add1($treeViewScroller);

                } else {

                    # Pack the treeview directly into the packing box
                    $packingBox->pack_start($treeViewScroller, TRUE, TRUE, 0);
                }
            }
        }

        # Create a canvas (if allowed)
        if ($self->worldModelObj->showCanvasFlag) {

            $canvasFrame = $self->enableCanvas();
            if ($canvasFrame) {

                # Pack the widget
                if ($hPaned) {

                    # Add the frame to the right pane
                    $hPaned->add2($canvasFrame);

                } else {

                    # Pack the frame directly into the packing box
                    $packingBox->pack_start($canvasFrame, TRUE, TRUE, 0);
                }
            }
        }

        return 1;
    }

    sub redrawWidgets {

        # Can be called by any function
        # Redraws some or all of the menu bar, toolbar(s), treeview and canvas
        # The widgets redrawn are specified by the calling function, but are not redrawn if the
        #   right flags aren't set (e.g. the menu bar isn't redrawn if
        #   GA::Obj::WorldModel->showMenuBarFlag isn't set)
        #
        # Expected arguments
        #   @widgetList - A list of widget names. One or all of the following strings, in any order:
        #                   'menu_bar', 'toolbar', 'treeview', 'canvas'
        #
        # Return values
        #   'undef' on improper arguments or if any of the widgets in @widgetList are unrecognised
        #   1 otherwise

        my ($self, @widgetList) = @_;

        # Local variables
        my (
            $menuBar, $hPaned, $treeViewScroller, $canvasFrame,
            @toolbarList,
            %widgetHash,
        );

        # Check for improper arguments
        if (! @widgetList) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->redrawWidgets', @_);
        }

        # Check that the strings in @widgetList are valid, and add each string into a hash so that
        #   no widget is drawn more than once
        # Initialise the hash of allowed widgets
        %widgetHash = (
            'menu_bar'  => FALSE,
            'toolbar'   => FALSE,
            'treeview'  => FALSE,
            'canvas'    => FALSE,
        );

        # Check everything in @widgetList
        foreach my $name (@widgetList) {

            if (! exists $widgetHash{$name}) {

                return $self->session->writeError(
                    'Unrecognised widget \'' . $name . '\'',
                    $self->_objClass . '->redrawWidgets',
                );

            } else {

                # If the same string appears more than once in @widgetList, we only draw the widget
                #   once
                $widgetHash{$name} = TRUE;
            }
        }

        # Remove the old widgets from the vertical packing box
        if ($self->menuBar) {

            $axmud::CLIENT->desktopObj->removeWidget($self->packingBox, $self->menuBar);
        }

        foreach my $toolbar ($self->toolbarList) {

            $axmud::CLIENT->desktopObj->removeWidget($self->packingBox, $toolbar);
        }

        if ($self->hPaned) {

            foreach my $child ($self->hPaned->get_children()) {

                $self->hPaned->remove($child);
            }

            $axmud::CLIENT->desktopObj->removeWidget($self->packingBox, $self->hPaned);

        } else {

            if ($self->treeViewScroller) {

                $axmud::CLIENT->desktopObj->removeWidget(
                    $self->packingBox,
                    $self->treeViewScroller,
                );
            }

            if ($self->canvasFrame) {

                $axmud::CLIENT->desktopObj->removeWidget($self->packingBox, $self->canvasFrame);
            }
        }

        # Redraw the menu bar, if specified (and if allowed)
        if ($self->worldModelObj->showMenuBarFlag) {

            if ($widgetHash{'menu_bar'}) {

                $self->resetMenuBarIVs();

                my $menuBar = $self->enableMenu();
                if ($menuBar) {

                    # Pack the new widget
                    $self->packingBox->pack_start($menuBar,FALSE,FALSE,0);

                } else {

                    # After the error, stop trying to draw menu bars
                    $self->worldModelObj->set_showMenuBarFlag(FALSE);
                }

            # Otherwise, repack the old menu bar
            } elsif ($self->menuBar) {

                $self->packingBox->pack_start($self->menuBar,FALSE,FALSE,0);
            }
        }

        # Redraw the toolbar(s), if specified (and if allowed)
        if ($self->worldModelObj->showToolbarFlag) {

            if ($widgetHash{'toolbar'}) {

                # Reset toolbar IVs to their default state; the subsequent call to
                #   $self->enableToolbar imports the list of button sets from the world model, and
                #   updates these IVs accordinly
                $self->resetToolbarIVs();

                @toolbarList = $self->enableToolbar();
                if (@toolbarList) {

                    foreach my $toolbar (@toolbarList) {

                        # Pack the new widget
                        $self->packingBox->pack_start($toolbar, FALSE, FALSE, 0);
                    }

                } else {

                    # After the error, stop trying to draw toolbars
                    $self->worldModelObj->set_showToolbarFlag(FALSE);
                }

            # Otherwise, repack the old toolbar(s)
            } else {

                foreach my $toolbar ($self->toolbarList) {

                    $self->packingBox->pack_start($toolbar, FALSE, FALSE, 0);
                }
            }

        } else {

            # When the toolbars are next drawn, make sure the default button set is visible in the
            #   original (first) toolbar
            $self->ivPoke('toolbarOriginalSet', $self->constToolbarDefaultSet);
        }

        # Create a new horizontal pane (only if both the treeview and the canvas are allowed)
        if ($self->worldModelObj->showTreeViewFlag && $self->worldModelObj->showCanvasFlag) {

            $hPaned = Gtk3::HPaned->new();
            if ($hPaned) {

                # Set the width of the space about to be filled with the treeview
                $hPaned->set_position($self->treeViewWidthPixels);

                # Pack the widget
                $self->packingBox->pack_start($hPaned, TRUE, TRUE, 0);
                $self->ivPoke('hPaned', $hPaned);

            } else {

                # After the error, stop trying to draw either the treeview or the canvas
                $self->worldModelObj->set_showTreeViewFlag(FALSE);
                $self->worldModelObj->set_showCanvasFlag(FALSE);
            }

        } else {

            # Horizontal pane no longer required
            $self->ivUndef('hPaned');
        }

        # Redraw the treeview, if specified (and if allowed)
        if ($self->worldModelObj->showTreeViewFlag) {

            if ($widgetHash{'treeview'}) {

                $self->resetTreeViewIVs();

                $treeViewScroller = $self->enableTreeView();
                if ($treeViewScroller) {

                    # Pack the new widget
                    if ($hPaned) {

                        # Add the treeview's scroller to the left pane
                        $hPaned->add1($treeViewScroller);

                    } else {

                        # Pack the treeview directly into the packing box
                        $self->packingBox->pack_start($treeViewScroller, TRUE, TRUE, 0);
                    }

                } else {

                    # After the error, stop trying to draw treeviews
                    $self->worldModelObj->set_showTreeViewFlag(FALSE);
                }

            # Otherwise, repack the old treeview
            } elsif ($self->treeViewScroller) {

                if ($hPaned) {

                    # Add the treeview's scroller to the left-hand pane
                    $hPaned->add1($self->treeViewScroller);

                } else {

                    # Pack the treeview directly into the packing box
                    $self->packingBox->pack_start($self->treeViewScroller, TRUE, TRUE, 0);
                }
            }
        }

        # Redraw the canvas, if specified (and if allowed)
        if ($self->worldModelObj->showCanvasFlag) {

            if ($widgetHash{'canvas'}) {

                $self->resetCanvasIVs();

                $canvasFrame = $self->enableCanvas();
                if ($canvasFrame) {

                    # Pack the new widget
                    if ($hPaned) {

                        # Add the frame to the right pane
                        $hPaned->add2($canvasFrame);

                    } else {

                        # Pack the frame directly into the packing box
                        $self->packingBox->pack_start($canvasFrame, TRUE, TRUE, 0);
                    }

                } else {

                    # After the error, stop trying to draw canvases
                    $self->worldModelObj->set_showCanvasFlag(FALSE);
                }

            # Otherwise, repack the old canvas
            } elsif ($self->canvasFrame) {

                if ($hPaned) {

                    # Add the frame to the right-hand pane
                    $hPaned->add2($self->canvasFrame);

                } else {

                    # Pack the frame directly into the packing box
                    $self->packingBox->pack_start($self->canvasFrame, TRUE, TRUE, 0);
                }
            }
        }

        # Now, for each widget that is no longer drawn, set default IVs
        if (! $self->worldModelObj->showMenuBarFlag) {

            $self->resetMenuBarIVs();
        }

        if (! $self->worldModelObj->showToolbarFlag) {

            $self->resetToolbarIVs();
        }

        if (! $self->worldModelObj->showTreeViewFlag || ! $self->worldModelObj->showCanvasFlag) {

            $self->ivUndef('hPaned');
        }

        if (! $self->worldModelObj->showTreeViewFlag) {

            $self->resetTreeViewIVs();
        }

        if (! $self->worldModelObj->showCanvasFlag) {

            $self->resetCanvasIVs();
        }

        # Repack complete
        $self->winShowAll($self->_objClass . '->redrawWidgets');
        $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->redrawWidgets');

        return 1;
    }

    # Standard 'map' window object functions

    sub winReset {

        # Called by GA::Obj::Map->openWin to reset an existing Automapper window
        #
        # Expected arguments
        #   $mapObj     - The calling GA::Obj::Map object
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $mapObj, $check) = @_;

        # Check for improper arguments
        if (! defined $mapObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->winReset', @_);
        }

        # Set new Perl object component IVs
        $self->ivPoke('mapObj', $mapObj);
        $self->ivPoke('worldModelObj', $self->session->worldModelObj);

        # Reset the current region
        $self->ivUndef('currentRegionmap');
        $self->ivUndef('currentParchment');
        $self->ivEmpty('recentRegionList');

        # Reset parchment objects (which destroys all canvas widgets except the empty one created
        #   by the call to ->resetMap)
        $self->ivEmpty('parchmentHash');
        $self->ivEmpty('parchmentReadyHash');
        $self->ivEmpty('parchmentQueueList');

        # Reset selected objects
        $self->ivUndef('selectedRoom');
        $self->ivUndef('selectedExit');
        $self->ivUndef('selectedRoomTag');
        $self->ivUndef('selectedRoomGuild');
        $self->ivUndef('selectedLabel');
        $self->ivEmpty('selectedRoomHash');
        $self->ivEmpty('selectedExitHash');
        $self->ivEmpty('selectedRoomTagHash');
        $self->ivEmpty('selectedRoomGuildHash');
        $self->ivEmpty('selectedLabelHash');

        # Reset drawing cycle IVs
        $self->tidyUpDraw();
        $self->ivEmpty('drawCycleExitHash');

        # Reset other IVs to their default values
        $self->reset_freeClickMode();
        $self->ivPoke('mode', 'wait');
        $self->ivUndef('showChar');     # Show character visits for the current character
        $self->ivPoke('emptyMapFlag', FALSE);
        $self->ivPoke('winUpdateCalledFlag', FALSE);

        # Reset the title bar
        $self->setWinTitle();
        # Reset window components
        $self->redrawWidgets('menu_bar', 'toolbar', 'treeview', 'canvas');

        return 1;
    }

    sub winUpdate {

        # Called by GA::Session->spinMaintainLoop or by any other code
        # Check all of the automapper window's parchment objects (Games::Axmud::Obj::Parchment)
        # If there are any canvas objects waiting in queue to be drawn, mark a number of them to be
        #   drawn
        # Then draw everything that's been marked to be drawn
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if a drawing cycle (i.e. a call to $self->doDraw) is
        #       already in progress
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($parchmentObj);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->winUpdate', @_);
        }

        # If a drawing cycle (i.e. a call to $self->doDraw) is already in progress, don't do
        #   anything this time; wait for the next spin of the session's maintain loop
        if ($self->delayDrawFlag || ! $self->mapObj) {

            return undef;
        }

        # If this is the first call to this function since the window opened (or was reset),
        #   compile a list of regions that should be pre-drawn
        if (! $self->winUpdateCalledFlag) {

            $self->ivPoke('winUpdateCalledFlag', TRUE);
            $self->preparePreDraw();
        }

        # We only draw things from the first parchment object in the list (if any)
        $parchmentObj = $self->ivFirst('parchmentQueueList');
        if ($parchmentObj) {

            # We call ->doQuickDraw, rather than calling the standard ->doDraw function, as the
            #   former is optimised for drawing a whole region
            # The TRUE flag tells ->doQuickDraw to apply a limit to the number of rooms, exits
            #   and/or labels drawn
            $self->doQuickDraw($parchmentObj, TRUE);

            # If the parchment has no more queued drawing operations, we can remove it from the
            #   queue
            if (
                ! $parchmentObj->queueRoomEchoHash && ! $parchmentObj->queueRoomBoxHash
                && ! $parchmentObj->queueRoomTextHash && ! $parchmentObj->queueRoomExitHash
                && ! $parchmentObj->queueRoomInfoHash && ! $parchmentObj->queueLabelHash
            ) {
                $self->ivShift('parchmentQueueList');
                $self->ivAdd('parchmentReadyHash', $parchmentObj->name, $parchmentObj);

                # Show the next region to be pre-drawn (if any) in the window's title bar
                $self->setWinTitle();
            }
        }

        # If a recent call to $self->doDraw failed because a drawing cycle was already in progress,
        #   call ->doDraw now to complete that operation
        if ($self->winUpdateForceFlag) {

            if ($self->doDraw()) {

                $self->ivPoke('winUpdateForceFlag', FALSE);

                # If the failed call to ->doDraw came from ->setCurrentRegion, then we can make the
                #   current region's canvas widget visible, now that the map is fully drawn
                if ($self->winUpdateShowFlag) {

                    $self->ivPoke('winUpdateShowFlag', FALSE);
                    $self->swapCanvasWidget();
                }
            }
        }

        return 1;
    }

    # ->signal_connects

    sub setDeleteEvent {

        # Called by $self->winSetup
        # Set up a ->signal_connect to watch out for the user manually closing the 'map' window
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->setDeleteEvent', @_);
        }

        $self->winBox->signal_connect('delete-event' => sub {

            # Prevent Gtk3 from taking action directly. Instead redirect the request to
            #   $self->winDestroy, which does things like resetting a portion of the workspace
            #   grid, as well as actually destroying the window
            return $self->winDestroy();
        });

        return 1;
    }

    sub setKeyPressEvent {

        # Called by $self->winSetup
        # Set up a ->signal_connect to watch out for certain key presses
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the ->signal_connect doesn't interfere with the key
        #       press
        #   1 if the ->signal_connect does interfere with the key press, or when the
        #       ->signal_connect is first set up

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->setKeyPressEvent', @_);
        }

        $self->winBox->signal_connect('key-press-event' => sub {

            my ($widget, $event) = @_;

            # Local variables
            my ($keycode, $standard);

            # Get the system keycode for this keypress
            $keycode = Gtk3::Gdk::keyval_name($event->keyval);
            # Translate it into a standard Axmud keycode
            $standard = $axmud::CLIENT->reverseKeycode($keycode);

            # Respond to the keypress. The only key combination that interests the automapper is
            #   CTRL+C
            if ($standard eq 'ctrl') {

                $self->ivPoke('ctrlKeyFlag', TRUE);
            }

            if ($standard eq 'c' && $self->ctrlKeyFlag && $self->worldModelObj->allowCtrlCopyFlag) {

                # If there are one or more selected rooms, start a 'move selected room to click'
                #   operation
                if ($self->selectedRoom || $self->selectedRoomHash) {

                    $self->set_freeClickMode('move_room');
                }

                return 1;

            } else {

                return undef;
            }
        });

        return 1;
    }

    sub setKeyReleaseEvent {

        # Called by $self->winSetup
        # Set up a ->signal_connect to watch out for certain key releases
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the ->signal_connect doesn't interfere with the key
        #       release
        #   1 if the ->signal_connect does interfere with the key release, or when the
        #       ->signal_connect is first set up

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->setKeyReleaseEvent', @_);
        }

        $self->winBox->signal_connect('key-release-event' => sub {

            my ($widget, $event) = @_;

            # Local variables
            my ($keycode, $standard);

            # Get the system keycode for this keypress
            $keycode = Gtk3::Gdk::keyval_name($event->keyval);
            # Translate it into a standard Axmud keycode
            $standard = $axmud::CLIENT->reverseKeycode($keycode);

            # Respond to the key release. The only key combination that interests the automapper is
            #   CTRL+C
            if ($standard && $standard eq 'ctrl') {

                $self->ivPoke('ctrlKeyFlag', FALSE);
            }

            # Return 'undef' to show that we haven't interfered with this keypress
            return undef;
        });

        return 1;
    }

    sub setConfigureEvent {

        # Called by $self->winEnable
        # Set up a ->signal_connect to watch out for changes in the window size and position
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->setConfigureEvent', @_);
        }

        $self->winBox->signal_connect('configure-event' => sub {

            my ($widget, $event) = @_;

            # Let the GA::Client store the most recent size and position for a window of this
            #   ->winName, if it needs to
            if ($self->winWidget) {

                $axmud::CLIENT->add_storeGridPosn(
                    $self,
                    $self->winWidget->get_position(),
                    $self->winWidget->get_size(),
                );
            }

            # Without returning 'undef', the window's strip/table objects aren't resized along with
            #   the window
            return undef;
        });

        return 1;
    }

    sub setFocusOutEvent {

        # Called by $self->winSetup
        # Set up a ->signal_connect to watch out for the 'map' window losing the focus
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

             return $axmud::CLIENT->writeImproper($self->_objClass . '->setFocusInEvent', @_);
        }

        $self->winBox->signal_connect('focus-out-event' => sub {

            my ($widget, $event) = @_;

            # If the tooltips are visible, hide them
            if ($event->type eq 'focus-change' && $self->canvasTooltipFlag) {

               $self->hideTooltips();
            }
        });

        return 1;
    }

    # Other functions

    sub resetMenuBarIVs {

        # Called by $self->redrawWidgets at certain points, to reset the IVs storing details about
        #   the menu bar back to their defaults
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetMenuBarIVs', @_);
        }

        $self->ivUndef('menuBar');
        $self->ivEmpty('menuToolItemHash');

        return 1;
    }

    sub resetToolbarIVs {

        # Called by $self->drawWidgets and $self->redrawWidget to reset the IVs storing details
        #   about toolbars back to their defaults
        # (If $self->enableToolbar is then called, it's that function which imports a list of
        #   button sets from the world model and updates these IVs accordinly)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetToolbarIVs', @_);
        }

        foreach my $key ($self->ivKeys('buttonSetHash')) {

            $self->ivAdd('buttonSetHash', $key, FALSE);
        }

        $self->ivEmpty('toolbarList');
        $self->ivEmpty('toolbarHash');

        return 1;
    }

    sub resetTreeViewIVs {

        # Called by $self->redrawWidgets at certain points, to reset the IVs storing details about
        #   the treeview back to their defaults
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetTreeViewIVs', @_);
        }

        $self->ivUndef('treeViewModel');
        $self->ivUndef('treeView');
        $self->ivUndef('treeViewScroller');
        $self->ivUndef('treeViewSelectedLine');
        $self->ivEmpty('treeViewRegionHash');
        $self->ivEmpty('treeViewPointerHash');

        return 1;
    }

    sub resetCanvasIVs {

        # Called by $self->redrawWidgets at certain points, to reset the IVs storing details about
        #   the canvas back to their defaults
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetCanvasIVs', @_);
        }

        $self->ivUndef('canvas');
        $self->ivUndef('canvasBackground');
        # (For some reason, commenting out these lines decreases the draw time, during a call to
        #   $self->redrawWidgets, by about 40%. The IVs receive their correct values anyway when
        #   ->enableCanvas is called)
#        $self->ivUndef('canvasFrame');
#        $self->ivUndef('canvasScroller');
        $self->ivUndef('canvasHAdjustment');
        $self->ivUndef('canvasVAdjustment');

        $self->ivUndef('canvasTooltipObj');
        $self->ivUndef('canvasTooltipObjType');
        $self->ivUndef('canvasTooltipFlag');

        return 1;
    }

    # Menu widget methods

    sub enableMenu {

        # Called by $self->drawWidgets
        # Sets up the Automapper window's Gtk3::MenuBar widget
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::MenuBar created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableMenu', @_);
        }

        # Create the menu bar
        my $menuBar = Gtk3::MenuBar->new();
        if (! $menuBar) {

            return undef;
        }

        # 'File' column
        my $column_file = $self->enableFileColumn();
        my $item_file = Gtk3::MenuItem->new('_File');
        $item_file->set_submenu($column_file);
        $menuBar->append($item_file);

        # 'Edit' column
        my $column_edit = $self->enableEditColumn();
        my $item_edit = Gtk3::MenuItem->new('_Edit');
        $item_edit->set_submenu($column_edit);
        $menuBar->append($item_edit);

        # 'View' column
        my $column_view = $self->enableViewColumn();
        my $item_view = Gtk3::MenuItem->new('_View');
        $item_view->set_submenu($column_view);
        $menuBar->append($item_view);

        # 'Mode' column
        my $column_mode = $self->enableModeColumn();
        my $item_mode = Gtk3::MenuItem->new('_Mode');
        $item_mode->set_submenu($column_mode);
        $menuBar->append($item_mode);

        # 'Regions' column
        my $column_regions = $self->enableRegionsColumn();
        my $item_regions = Gtk3::MenuItem->new('_Regions');
        $item_regions->set_submenu($column_regions);
        $menuBar->append($item_regions);

        # 'Rooms' column
        my $column_rooms = $self->enableRoomsColumn();
        my $item_rooms = Gtk3::MenuItem->new('R_ooms');
        $item_rooms->set_submenu($column_rooms);
        $menuBar->append($item_rooms);

        # 'Exits' column
        my $column_exits = $self->enableExitsColumn();
        my $item_exits = Gtk3::MenuItem->new('E_xits');
        $item_exits->set_submenu($column_exits);
        $menuBar->append($item_exits);

        # 'Labels' column
        my $column_labels = $self->enableLabelsColumn();
        my $item_labels = Gtk3::MenuItem->new('_Labels');
        $item_labels->set_submenu($column_labels);
        $menuBar->append($item_labels);

        # Store the widget
        $self->ivPoke('menuBar', $menuBar);

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        # Setup complete
        return $menuBar;
    }

    sub enableFileColumn {

        # Called by $self->enableMenu
        # Sets up the 'File' column of the Automapper window's menu bar
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableFileColumn', @_);
        }

        # Set up column
        my $column_file = Gtk3::Menu->new();
        if (! $column_file) {

            return undef;
        }

        my $item_loadModel = Gtk3::MenuItem->new('_Load world model');
        $item_loadModel->signal_connect('activate' => sub {

            # $self->winReset will be called by $self->set_worldModelObj when the ';load' command
            #   has finished its work
            # NB Force pseudo command mode 'win_error' in this menu column (success system messages
            #   in the 'main' window; errors/improper arguments messages shown in a 'dialogue'
            #   window)
            $self->session->pseudoCmd('load -m', 'win_error');
        });
        $column_file->append($item_loadModel);

        my $item_loadAll = Gtk3::ImageMenuItem->new('L_oad all files');
        $item_loadAll->signal_connect('activate' => sub {

            # The ';load' command will  $self->winReset when finished

            $self->session->pseudoCmd('load', 'win_error');
        });
        my $img_loadAll = Gtk3::Image->new_from_stock('gtk-open', 'menu');
        $item_loadAll->set_image($img_loadAll);
        $column_file->append($item_loadAll);

        $column_file->append(Gtk3::SeparatorMenuItem->new());   # Separator

        my $item_saveModel = Gtk3::MenuItem->new('_Save world model');
        $item_saveModel->signal_connect('activate' => sub {

            # Do a forced save. The ';save' command sets $self->freeClickMode back to 'default'
            $self->session->pseudoCmd('save -m -f', 'win_error');
        });
        $column_file->append($item_saveModel);

        my $item_saveAll = Gtk3::ImageMenuItem->new('S_ave all files');
        $item_saveAll->signal_connect('activate' => sub {

            # Do a forced save. The ';save' command sets $self->freeClickMode back to 'default'
            $self->session->pseudoCmd('save -f', 'win_error');
        });
        my $img_saveAll = Gtk3::Image->new_from_stock('gtk-save', 'menu');
        $item_saveAll->set_image($img_saveAll);
        $column_file->append($item_saveAll);

        $column_file->append(Gtk3::SeparatorMenuItem->new());   # Separator

        my $item_importModel = Gtk3::MenuItem->new('_Import/load world model...');
        $item_importModel->signal_connect('activate' => sub {

            $self->importModelCallback();
        });
        $column_file->append($item_importModel);

        my $item_exportModel = Gtk3::MenuItem->new('Save/_export world model...');
        $item_exportModel->signal_connect('activate' => sub {

            $self->exportModelCallback();
        });
        $column_file->append($item_exportModel);

        $column_file->append(Gtk3::SeparatorMenuItem->new());   # Separator

        my $item_mergeModel = Gtk3::MenuItem->new('_Merge world models...');
        $item_mergeModel->signal_connect('activate' => sub {

            $self->session->pseudoCmd('mergemodel')
        });
        $column_file->append($item_mergeModel);

        $column_file->append(Gtk3::SeparatorMenuItem->new());   # Separator

        my $item_closeWindow = Gtk3::ImageMenuItem->new('_Close window');
        $item_closeWindow->signal_connect('activate' => sub {

            $self->winDestroy();
        });
        my $img_closeWindow = Gtk3::Image->new_from_stock('gtk-quit', 'menu');
        $item_closeWindow->set_image($img_closeWindow);
        $column_file->append($item_closeWindow);

        # Setup complete
        return $column_file;
    }

    sub enableEditColumn {

        # Called by $self->enableMenu
        # Sets up the 'Edit' column of the Automapper window's menu bar
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Local variables
        my $winObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableEditColumn', @_);
        }

        # Set up column
        my $column_edit = Gtk3::Menu->new();
        if (! $column_edit) {

            return undef;
        }

            # 'Select' submenu
            my $subMenu_select = Gtk3::Menu->new();

                # 'Select rooms' sub-submenu
                my $subSubMenu_selectRooms = Gtk3::Menu->new();

                my $item_selectNoTitle = Gtk3::MenuItem->new('Rooms with no _titles');
                $item_selectNoTitle->signal_connect('activate' => sub {

                    $self->selectRoomCallback('no_title');
                });
                $subSubMenu_selectRooms->append($item_selectNoTitle);

                my $item_selectNoDescrip = Gtk3::MenuItem->new('Rooms with no _descriptions');
                $item_selectNoDescrip->signal_connect('activate' => sub {

                    $self->selectRoomCallback('no_descrip');
                });
                $subSubMenu_selectRooms->append($item_selectNoDescrip);

                my $item_selectNoTitleDescrip = Gtk3::MenuItem->new('Rooms with _neither');
                $item_selectNoTitleDescrip->signal_connect('activate' => sub {

                    $self->selectRoomCallback('no_title_descrip');
                });
                $subSubMenu_selectRooms->append($item_selectNoTitleDescrip);

                my $item_selectTitleDescrip = Gtk3::MenuItem->new('Rooms with _both');
                $item_selectTitleDescrip->signal_connect('activate' => sub {

                    $self->selectRoomCallback('title_descrip');
                });
                $subSubMenu_selectRooms->append($item_selectTitleDescrip);

                $subSubMenu_selectRooms->append(Gtk3::SeparatorMenuItem->new());   # Separator

                my $item_selectNoVisitChar = Gtk3::MenuItem->new('Rooms not visited by _character');
                $item_selectNoVisitChar->signal_connect('activate' => sub {

                    $self->selectRoomCallback('no_visit_char');
                });
                $subSubMenu_selectRooms->append($item_selectNoVisitChar);

                my $item_selectNoVisitAllChar = Gtk3::MenuItem->new('Rooms not visited by _anyone');
                $item_selectNoVisitAllChar->signal_connect('activate' => sub {

                    $self->selectRoomCallback('no_visit_all');
                });
                $subSubMenu_selectRooms->append($item_selectNoVisitAllChar);

                my $item_selectVisitChar = Gtk3::MenuItem->new('Rooms visited by c_haracter');
                $item_selectVisitChar->signal_connect('activate' => sub {

                    $self->selectRoomCallback('visit_char');
                });
                $subSubMenu_selectRooms->append($item_selectVisitChar);

                my $item_selectVisitAllChar = Gtk3::MenuItem->new('Rooms visited by an_yone');
                $item_selectVisitAllChar->signal_connect('activate' => sub {

                    $self->selectRoomCallback('visit_all');
                });
                $subSubMenu_selectRooms->append($item_selectVisitAllChar);

                $subSubMenu_selectRooms->append(Gtk3::SeparatorMenuItem->new());   # Separator

                my $item_selectCheckable = Gtk3::MenuItem->new('Rooms with checkable d_irections');
                $item_selectCheckable->signal_connect('activate' => sub {

                    $self->selectRoomCallback('checkable');
                });
                $subSubMenu_selectRooms->append($item_selectCheckable);

            my $item_selectRooms = Gtk3::MenuItem->new('Select _rooms');
            $item_selectRooms->set_submenu($subSubMenu_selectRooms);
            $subMenu_select->append($item_selectRooms);

                # 'Select exits' sub-submenu
                my $subSubMenu_selectExits = Gtk3::Menu->new();

                my $item_selectInRooms = Gtk3::MenuItem->new('Exits in selected _rooms');
                $item_selectInRooms->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('in_rooms');
                });
                $subSubMenu_selectExits->append($item_selectInRooms);

                $subSubMenu_selectExits->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_selectUnallocated = Gtk3::MenuItem->new('_Unallocated exits');
                $item_selectUnallocated->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('unallocated');
                });
                $subSubMenu_selectExits->append($item_selectUnallocated);

                my $item_selectUnallocatable = Gtk3::MenuItem->new('U_nallocatable exits');
                $item_selectUnallocatable->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('unallocatable');
                });
                $subSubMenu_selectExits->append($item_selectUnallocatable);

                my $item_selectUncertain = Gtk3::MenuItem->new('Un_certain exits');
                $item_selectUncertain->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('uncertain');
                });
                $subSubMenu_selectExits->append($item_selectUncertain);

                my $item_selectIncomplete = Gtk3::MenuItem->new('_Incomplete exits');
                $item_selectIncomplete->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('incomplete');
                });
                $subSubMenu_selectExits->append($item_selectIncomplete);

                my $item_selectAllAbove = Gtk3::MenuItem->new('_All of the above');
                $item_selectAllAbove->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('all_above');
                });
                $subSubMenu_selectExits->append($item_selectAllAbove);

                $subSubMenu_selectExits->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_selectImpassable = Gtk3::MenuItem->new('I_mpassable exits');
                $item_selectImpassable->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('impass');
                });
                $subSubMenu_selectExits->append($item_selectImpassable);

                my $item_selectMystery = Gtk3::MenuItem->new('M_ystery exits');
                $item_selectMystery->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('mystery');
                });
                $subSubMenu_selectExits->append($item_selectMystery);

                $subSubMenu_selectExits->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_selectNonSuper = Gtk3::MenuItem->new('R_egion exits');
                $item_selectNonSuper->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('region');
                });
                $subSubMenu_selectExits->append($item_selectNonSuper);

                my $item_selectSuper = Gtk3::MenuItem->new('_Super-region exits');
                $item_selectSuper->signal_connect('activate' => sub {

                    $self->selectExitTypeCallback('super');
                });
                $subSubMenu_selectExits->append($item_selectSuper);

            my $item_selectExits = Gtk3::MenuItem->new('Select _exits');
            $item_selectExits->set_submenu($subSubMenu_selectExits);
            $subMenu_select->append($item_selectExits);

            $subMenu_select->append(Gtk3::SeparatorMenuItem->new());   # Separator

                # 'Select in region' sub-submenu
                my $subSubMenu_selectRegion = Gtk3::Menu->new();

                my $item_selectRegionRoom = Gtk3::MenuItem->new('Every _room');
                $item_selectRegionRoom->signal_connect('activate' => sub {

                    $self->selectInRegionCallback('room');
                });
                $subSubMenu_selectRegion->append($item_selectRegionRoom);

                my $item_selectRegionExit = Gtk3::MenuItem->new('Every _exit');
                $item_selectRegionExit->signal_connect('activate' => sub {

                    $self->selectInRegionCallback('exit');
                });
                $subSubMenu_selectRegion->append($item_selectRegionExit);

                my $item_selectRegionRoomTag = Gtk3::MenuItem->new('Every room _tag');
                $item_selectRegionRoomTag->signal_connect('activate' => sub {

                    $self->selectInRegionCallback('room_tag');
                });
                $subSubMenu_selectRegion->append($item_selectRegionRoomTag);

                my $item_selectRegionRoomGuild = Gtk3::MenuItem->new('Every room _guild');
                $item_selectRegionRoomGuild->signal_connect('activate' => sub {

                    $self->selectInRegionCallback('room_guild');
                });
                $subSubMenu_selectRegion->append($item_selectRegionRoomGuild);

                my $item_selectRegionLabel = Gtk3::MenuItem->new('Every _label');
                $item_selectRegionLabel->signal_connect('activate' => sub {

                    $self->selectInRegionCallback('label');
                });
                $subSubMenu_selectRegion->append($item_selectRegionLabel);

                $subSubMenu_selectRegion->append(Gtk3::SeparatorMenuItem->new());   # Separator

                my $item_selectRegionAbove = Gtk3::MenuItem->new('_All of the above');
                $item_selectRegionAbove->signal_connect('activate' => sub {

                    $self->selectInRegionCallback();
                });
                $subSubMenu_selectRegion->append($item_selectRegionAbove);

            my $item_selectRegion = Gtk3::MenuItem->new('Select in re_gion');
            $item_selectRegion->set_submenu($subSubMenu_selectRegion);
            $subMenu_select->append($item_selectRegion);

                # 'Select in map' sub-submenu
                my $subSubMenu_selectMap = Gtk3::Menu->new();

                my $item_selectMapRoom = Gtk3::MenuItem->new('Every _room');
                $item_selectMapRoom->signal_connect('activate' => sub {

                    $self->selectInMapCallback('room');
                });
                $subSubMenu_selectMap->append($item_selectMapRoom);

                my $item_selectMapExit = Gtk3::MenuItem->new('Every _exit');
                $item_selectMapExit->signal_connect('activate' => sub {

                    $self->selectInMapCallback('exit');
                });
                $subSubMenu_selectMap->append($item_selectMapExit);

                my $item_selectMapRoomTag = Gtk3::MenuItem->new('Every room _tag');
                $item_selectMapRoomTag->signal_connect('activate' => sub {

                    $self->selectInMapCallback('room_tag');
                });
                $subSubMenu_selectMap->append($item_selectMapRoomTag);

                my $item_selectMapRoomGuild = Gtk3::MenuItem->new('Every room _guild');
                $item_selectMapRoomGuild->signal_connect('activate' => sub {

                    $self->selectInMapCallback('room_guild');
                });
                $subSubMenu_selectMap->append($item_selectMapRoomGuild);

                my $item_selectMapLabel = Gtk3::MenuItem->new('Every _label');
                $item_selectMapLabel->signal_connect('activate' => sub {

                    $self->selectInMapCallback('label');
                });
                $subSubMenu_selectMap->append($item_selectMapLabel);

                $subSubMenu_selectMap->append(Gtk3::SeparatorMenuItem->new());   # Separator

                my $item_selectMapAbove = Gtk3::MenuItem->new('_All of the above');
                $item_selectMapAbove->signal_connect('activate' => sub {

                    $self->selectInMapCallback();
                });
                $subSubMenu_selectMap->append($item_selectMapAbove);

            my $item_selectMap = Gtk3::MenuItem->new('Select in _map');
            $item_selectMap->set_submenu($subSubMenu_selectMap);
            $subMenu_select->append($item_selectMap);

        my $item_select = Gtk3::MenuItem->new('_Select');
        $item_select->set_submenu($subMenu_select);
        $column_edit->append($item_select);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'select', $item_select);

            # 'Selected items' submenu
            my $subMenu_selectedObjs = Gtk3::Menu->new();

            my $item_identifyRoom = Gtk3::MenuItem->new('Identify _room(s)');
            $item_identifyRoom->signal_connect('activate' => sub {

                $self->identifyRoomsCallback();
            });
            $subMenu_selectedObjs->append($item_identifyRoom);
            # (Requires $self->currentRegionmap and EITHER $self->selectedRoom or
            #   $self->selectedRoomHash or $self->mapObj->currentRoom)
            $self->ivAdd('menuToolItemHash', 'identify_room', $item_identifyRoom);

            my $item_identifyExit = Gtk3::MenuItem->new('Identify _exit(s)');
            $item_identifyExit->signal_connect('activate' => sub {

                $self->identifyExitsCallback();
            });
            $subMenu_selectedObjs->append($item_identifyExit);
            # (Requires $self->currentRegionmap & either $self->selectedExit or
            #   $self->selectedExitHash)
            $self->ivAdd('menuToolItemHash', 'identify_exit', $item_identifyExit);

        my $item_selectedObjs = Gtk3::MenuItem->new('_Identify selected items');
        $item_selectedObjs->set_submenu($subMenu_selectedObjs);
        $column_edit->append($item_selectedObjs);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'selected_objs', $item_selectedObjs);

        my $item_unselectAll = Gtk3::MenuItem->new('_Unselect all');
        $item_unselectAll->signal_connect('activate' => sub {

            $self->setSelectedObj();
        });
        $column_edit->append($item_unselectAll);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'unselect_all', $item_unselectAll);

        $column_edit->append(Gtk3::SeparatorMenuItem->new());   # Separator

            # 'Search' submenu
            my $subMenu_search = Gtk3::Menu->new();

            my $item_searchModel = Gtk3::MenuItem->new('Search world _model...');
            $item_searchModel->signal_connect('activate' => sub {

                # Open a 'pref' window to conduct the search
                $self->createFreeWin(
                    'Games::Axmud::PrefWin::Search',
                    $self,
                    $self->session,
                    'World model search',
                );
            });
            $subMenu_search->append($item_searchModel);

            $subMenu_search->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_findRoom = Gtk3::MenuItem->new('Find _room...');
            $item_findRoom->signal_connect('activate' => sub {

                $self->findRoomCallback();
            });
            $subMenu_search->append($item_findRoom);

            my $item_findExit = Gtk3::MenuItem->new('Find _exit...');
            $item_findExit->signal_connect('activate' => sub {

                $self->findExitCallback();
            });
            $subMenu_search->append($item_findExit);

        my $item_search = Gtk3::ImageMenuItem->new('S_earch');
        my $img_search = Gtk3::Image->new_from_stock('gtk-find', 'menu');
        $item_search->set_image($img_search);
        $item_search->set_submenu($subMenu_search);
        $column_edit->append($item_search);

            # 'Generate reports' submenu
            my $subMenu_reports = Gtk3::Menu->new();

            my $item_showSummary = Gtk3::MenuItem->new('_Show general report');
            $item_showSummary->signal_connect('activate' => sub {

                # (Don't use $self->pseudoCmdMode - we want to see the footer messages)
                $self->session->pseudoCmd('modelreport', 'show_all');
            });
            $subMenu_reports->append($item_showSummary);

            my $item_showCurrentRegion = Gtk3::MenuItem->new('S_how current region');
            $item_showCurrentRegion->signal_connect('activate' => sub {

                $self->session->pseudoCmd(
                    'modelreport -r <' . $self->currentRegionmap->name . '>',
                    'show_all',
                );
            });
            $subMenu_reports->append($item_showCurrentRegion);
            # (Requires $self->currentRegionmap)
            $self->ivAdd('menuToolItemHash', 'report_region', $item_showCurrentRegion);

            $subMenu_reports->append(Gtk3::SeparatorMenuItem->new());  # Separator

                # 'Character visits' sub-submenu
                my $subSubMenu_visits = Gtk3::Menu->new();

                my $item_visits1 = Gtk3::MenuItem->new('_All regions/characters');
                $item_visits1->signal_connect('activate' => sub {

                    $self->session->pseudoCmd('modelreport -v', 'show_all');
                });
                $subSubMenu_visits->append($item_visits1);

                my $item_visits2 = Gtk3::MenuItem->new('Current _region');
                $item_visits2->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -v -r <' . $self->currentRegionmap->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_visits->append($item_visits2);
                # (Requires $self->currentRegionmap)
                $self->ivAdd('menuToolItemHash', 'report_visits_2', $item_visits2);

                my $item_visits3 = Gtk3::MenuItem->new('Current _character');
                $item_visits3->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -v -c <' . $self->session->currentChar->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_visits->append($item_visits3);
                # (Requires current character profile)
                $self->ivAdd('menuToolItemHash', 'report_visits_3', $item_visits3);

                my $item_visits4 = Gtk3::MenuItem->new('C_urrent region/character');
                $item_visits4->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -v -r <' . $self->currentRegionmap->name . '>' . ' -c <'
                        . $self->session->currentChar->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_visits->append($item_visits4);
                # (Requires $self->currentRegionmap and current character profile)
                $self->ivAdd('menuToolItemHash', 'report_visits_4', $item_visits4);

            my $item_visits = Gtk3::MenuItem->new('_Character visits');
            $item_visits->set_submenu($subSubMenu_visits);
            $subMenu_reports->append($item_visits);

                # 'Room guilds' sub-submenu
                my $subSubMenu_guilds = Gtk3::Menu->new();

                my $item_guilds1 = Gtk3::MenuItem->new('_All regions/guilds');
                $item_guilds1->signal_connect('activate' => sub {

                    $self->session->pseudoCmd('modelreport -g', 'show_all');
                });
                $subSubMenu_guilds->append($item_guilds1);

                my $item_guilds2 = Gtk3::MenuItem->new('Current _region');
                $item_guilds2->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -g -r <' . $self->currentRegionmap->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_guilds->append($item_guilds2);
                # (Requires $self->currentRegionmap)
                $self->ivAdd('menuToolItemHash', 'report_guilds_2', $item_guilds2);

                my $item_guilds3 = Gtk3::MenuItem->new('Current _guild');
                $item_guilds3->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -g -n <' . $self->session->currentGuild->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_guilds->append($item_guilds3);
                # (Requires current guild profile)
                $self->ivAdd('menuToolItemHash', 'report_guilds_3', $item_guilds3);

                my $item_guilds4 = Gtk3::MenuItem->new('C_urrent region/guild');
                $item_guilds4->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -g -r <' . $self->currentRegionmap->name . '>' . ' -n <'
                        . $self->session->currentGuild->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_guilds->append($item_guilds4);
                # (Requires $self->currentRegionmap and current guild profile)
                $self->ivAdd('menuToolItemHash', 'report_guilds_4', $item_guilds4);

            my $item_guilds = Gtk3::MenuItem->new('Room _guilds');
            $item_guilds->set_submenu($subSubMenu_guilds);
            $subMenu_reports->append($item_guilds);

                # 'Room flags' sub-submenu
                my $subSubMenu_roomFlags = Gtk3::Menu->new();

                my $item_roomFlags1 = Gtk3::MenuItem->new('_All regions/flags');
                $item_roomFlags1->signal_connect('activate' => sub {

                    $self->session->pseudoCmd('modelreport -f', 'show_all');
                });
                $subSubMenu_roomFlags->append($item_roomFlags1);

                my $item_roomFlags2 = Gtk3::MenuItem->new('Current _region');
                $item_roomFlags2->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -f -r <' . $self->currentRegionmap->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_roomFlags->append($item_roomFlags2);
                # (Requires $self->currentRegionmap)
                $self->ivAdd('menuToolItemHash', 'report_flags_2', $item_roomFlags2);

                my $item_roomFlags3 = Gtk3::MenuItem->new('_Specify flag...');
                $item_roomFlags3->signal_connect('activate' => sub {

                    my (
                        $choice,
                        @list,
                    );

                    @list = $self->worldModelObj->roomFlagOrderedList;

                    $choice = $self->showComboDialogue(
                        'Select room flag',
                        'Select one of the world model\'s room flags',
                        \@list,
                    );

                    if ($choice) {

                        $self->session->pseudoCmd(
                            'modelreport -f -l <' . $choice . '>',
                            'show_all',
                        );
                    }
                });
                $subSubMenu_roomFlags->append($item_roomFlags3);

                my $item_roomFlags4 = Gtk3::MenuItem->new('C_urrent region/specify flag...');
                $item_roomFlags4->signal_connect('activate' => sub {

                    my (
                        $choice,
                        @list,
                    );

                    @list = $self->worldModelObj->roomFlagOrderedList;

                    $choice = $self->showComboDialogue(
                        'Select room flag',
                        'Select one of the world model\'s room flags',
                        \@list,
                    );

                    if ($choice) {

                        $self->session->pseudoCmd(
                            'modelreport -f -r <' . $self->currentRegionmap->name . '>' . ' -l <'
                            . $choice . '>',
                            'show_all',
                        );
                    }
                });
                $subSubMenu_roomFlags->append($item_roomFlags4);
                # (Requires $self->currentRegionmap)
                $self->ivAdd('menuToolItemHash', 'report_flags_4', $item_roomFlags4);

            my $item_roomFlags = Gtk3::MenuItem->new('Room _flags');
            $item_roomFlags->set_submenu($subSubMenu_roomFlags);
            $subMenu_reports->append($item_roomFlags);

                 # 'Rooms' sub-submenu
                my $subSubMenu_rooms = Gtk3::Menu->new();

                my $item_rooms1 = Gtk3::MenuItem->new('_All regions');
                $item_rooms1->signal_connect('activate' => sub {

                    $self->session->pseudoCmd('modelreport -m', 'show_all');
                });
                $subSubMenu_rooms->append($item_rooms1);

                my $item_rooms2 = Gtk3::MenuItem->new('_Current region');
                $item_rooms2->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -m -r <' . $self->currentRegionmap->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_rooms->append($item_rooms2);
                # (Requires $self->currentRegionmap)
                $self->ivAdd('menuToolItemHash', 'report_rooms_2', $item_rooms2);

            my $item_rooms = Gtk3::MenuItem->new('_Rooms');
            $item_rooms->set_submenu($subSubMenu_rooms);
            $subMenu_reports->append($item_rooms);

                 # 'Exits' sub-submenu
                my $subSubMenu_exits = Gtk3::Menu->new();

                my $item_exits1 = Gtk3::MenuItem->new('_All regions');
                $item_exits1->signal_connect('activate' => sub {

                    $self->session->pseudoCmd('modelreport -x', 'show_all');
                });
                $subSubMenu_exits->append($item_exits1);

                my $item_exits2 = Gtk3::MenuItem->new('_Current region');
                $item_exits2->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -x -r <' . $self->currentRegionmap->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_exits->append($item_exits2);
                # (Requires $self->currentRegionmap)
                $self->ivAdd('menuToolItemHash', 'report_exits_2', $item_exits2);

            my $item_exits = Gtk3::MenuItem->new('_Exits');
            $item_exits->set_submenu($subSubMenu_exits);
            $subMenu_reports->append($item_exits);

                # 'Checked directions' sub-submenu
                my $subSubMenu_checked = Gtk3::Menu->new();

                my $item_checked1 = Gtk3::MenuItem->new('_All regions');
                $item_checked1->signal_connect('activate' => sub {

                    $self->session->pseudoCmd('modelreport -h', 'show_all');
                });
                $subSubMenu_checked->append($item_checked1);

                my $item_checked2 = Gtk3::MenuItem->new('_Current region');
                $item_checked2->signal_connect('activate' => sub {

                    $self->session->pseudoCmd(
                        'modelreport -h -r <' . $self->currentRegionmap->name . '>',
                        'show_all',
                    );
                });
                $subSubMenu_checked->append($item_checked2);
                # (Requires $self->currentRegionmap)
                $self->ivAdd('menuToolItemHash', 'report_checked_2', $item_checked2);

            my $item_checked = Gtk3::MenuItem->new('Checked _directions');
            $item_checked->set_submenu($subSubMenu_checked);
            $subMenu_reports->append($item_checked);

        my $item_reports = Gtk3::MenuItem->new('_Generate reports');
        $item_reports->set_submenu($subMenu_reports);
        $column_edit->append($item_reports);

        $column_edit->append(Gtk3::SeparatorMenuItem->new());   # Separator

            # 'Reset' sub-submenu
            my $subMenu_reset = Gtk3::Menu->new();

            my $item_resetRoomData = Gtk3::MenuItem->new('Reset _room data...');
            $item_resetRoomData->signal_connect('activate' => sub {

                $self->resetRoomDataCallback();
            });
            $subMenu_reset->append($item_resetRoomData);

            my $item_resetCharVisits = Gtk3::MenuItem->new('Reset _visits by character...');
            $item_resetCharVisits->signal_connect('activate' => sub {

                $self->resetVisitsCallback();
            });
            $subMenu_reset->append($item_resetCharVisits);

        my $item_reset = Gtk3::MenuItem->new('_Reset');
        $item_reset->set_submenu($subMenu_reset);
        $column_edit->append($item_reset);

        $column_edit->append(Gtk3::SeparatorMenuItem->new());   # Separator

        my $item_editDict = Gtk3::ImageMenuItem->new('Edit current _dictionary...');
        my $img_editDict = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editDict->set_image($img_editDict);
        $item_editDict->signal_connect('activate' => sub {

            # Open an 'edit' window for the current dictionary
            $self->createFreeWin(
                'Games::Axmud::EditWin::Dict',
                $self,
                $self->session,
                'Edit dictionary \'' . $self->session->currentDict->name . '\'',
                $self->session->currentDict,
                FALSE,          # Not temporary
            );
        });
        $column_edit->append($item_editDict);

        my $item_addWords = Gtk3::MenuItem->new('Add dictionary _words...');
        $item_addWords->signal_connect('activate' => sub {

            $self->createFreeWin(
                'Games::Axmud::OtherWin::QuickWord',
                $self,
                $self->session,
                'Quick word adder',
            );
        });
        $column_edit->append($item_addWords);

        my $item_updateModel = Gtk3::MenuItem->new('U_pdate model words');
        $item_updateModel->signal_connect('activate' => sub {

            # Use pseudo-command mode 'win_error' - show success messages in the 'main' window,
            #   error messages in 'dialogue' window
            $self->session->pseudoCmd('updatemodel -t', 'win_error');
        });
        $column_edit->append($item_updateModel);

        $column_edit->append(Gtk3::SeparatorMenuItem->new());   # Separator

        my $item_setupWizard = Gtk3::ImageMenuItem->new('Run _Locator wizard...');
        my $img_setupWizard = Gtk3::Image->new_from_stock('gtk-page-setup', 'menu');
        $item_setupWizard->set_image($img_setupWizard);
        $item_setupWizard->signal_connect('activate' => sub {

            if ($self->session->wizWin) {

                # Some kind of 'wiz' window is already open
                $self->session->wizWin->restoreFocus();

            } else {

                # Open the Locator wizard window
                $self->session->pseudoCmd('locatorwizard', $self->pseudoCmdMode);
            }
        });
        $column_edit->append($item_setupWizard);

        my $item_editModel = Gtk3::ImageMenuItem->new('Edit world _model...');
        my $img_editModel = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editModel->set_image($img_editModel);
        $item_editModel->signal_connect('activate' => sub {

            # Open an 'edit' window for the world model
            $self->createFreeWin(
                'Games::Axmud::EditWin::WorldModel',
                $self,
                $self->session,
                'Edit world model',
                $self->session->worldModelObj,
                FALSE,                          # Not temporary
            );
        });
        $column_edit->append($item_editModel);

        # Setup complete
        return $column_edit;
    }

    sub enableViewColumn {

        # Sets up the 'View' column of the Automapper window's menu bar
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Local variables
        my (
            $item_group,
            @magList, @shortMagList, @initList, @interiorList,
            %interiorHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableViewColumn', @_);
        }

        # Set up column
        my $column_view = Gtk3::Menu->new();
        if (! $column_view) {

            return undef;
        }

            # 'Window components' submenu
            my $subMenu_winComponents = Gtk3::Menu->new();

            my $item_showMenuBar = Gtk3::CheckMenuItem->new('Show menu_bar');
            $item_showMenuBar->set_active($self->worldModelObj->showMenuBarFlag);
            $item_showMenuBar->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleWinComponents(
                    'showMenuBarFlag',
                    $item_showMenuBar->get_active(),
                );
            });
            $subMenu_winComponents->append($item_showMenuBar);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'show_menu_bar', $item_showMenuBar);

            my $item_showToolbar = Gtk3::CheckMenuItem->new('Show _toolbar');
            $item_showToolbar->set_active($self->worldModelObj->showToolbarFlag);
            $item_showToolbar->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleWinComponents(
                    'showToolbarFlag',
                    $item_showToolbar->get_active(),
                );
            });
            $subMenu_winComponents->append($item_showToolbar);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'show_toolbar', $item_showToolbar);

            my $item_showTreeView = Gtk3::CheckMenuItem->new('Show _regions');
            $item_showTreeView->set_active($self->worldModelObj->showTreeViewFlag);
            $item_showTreeView->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleWinComponents(
                    'showTreeViewFlag',
                    $item_showTreeView->get_active(),
                );
            });
            $subMenu_winComponents->append($item_showTreeView);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'show_treeview', $item_showTreeView);

            my $item_showCanvas = Gtk3::CheckMenuItem->new('Show _map');
            $item_showCanvas->set_active($self->worldModelObj->showCanvasFlag);
            $item_showCanvas->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleWinComponents(
                    'showCanvasFlag',
                    $item_showCanvas->get_active(),
                );
            });
            $subMenu_winComponents->append($item_showCanvas);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'show_canvas', $item_showCanvas);

            $subMenu_winComponents->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_redrawWindow = Gtk3::MenuItem->new('Re_draw window');
            $item_redrawWindow->signal_connect('activate' => sub {

                $self->redrawWidgets('menu_bar', 'toolbar', 'treeview', 'canvas');
            });
            $subMenu_winComponents->append($item_redrawWindow);

        my $item_windowComponents = Gtk3::MenuItem->new('_Window components');
        $item_windowComponents->set_submenu($subMenu_winComponents);
        $column_view->append($item_windowComponents);

            # 'Current room' submenu
            my $subMenu_currentRoom = Gtk3::Menu->new();

            my $item_radio1 = Gtk3::RadioMenuItem->new_with_mnemonic(undef, 'Draw _normal room');
            $item_radio1->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio1->get_active()) {

                    $self->worldModelObj->switchMode(
                        'currentRoomMode',
                        'single',           # New value of ->currentRoomMode
                        FALSE,              # No call to ->redrawRegions; current room is redrawn
                        'normal_current_mode',
                    );
                }
            });
            my $item_group0 = $item_radio1->get_group();
            $subMenu_currentRoom->append($item_radio1);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'normal_current_mode', $item_radio1);

            my $item_radio2 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group0,
                'Draw _emphasised room',
            );
            if ($self->worldModelObj->currentRoomMode eq 'double') {

                $item_radio2->set_active(TRUE);
            }
            $item_radio2->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio2->get_active()) {

                    $self->worldModelObj->switchMode(
                        'currentRoomMode',
                        'double',           # New value of ->currentRoomMode
                        FALSE,              # No call to ->redrawRegions; current room is redrawn
                        'empahsise_current_room',
                    );
                }
            });
            $subMenu_currentRoom->append($item_radio2);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'empahsise_current_room', $item_radio2);

            my $item_radio3 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group0,
                'Draw _filled-in room',
            );
            if ($self->worldModelObj->currentRoomMode eq 'interior') {

                $item_radio3->set_active(TRUE);
            }
            $item_radio3->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio3->get_active()) {

                    $self->worldModelObj->switchMode(
                        'currentRoomMode',
                        'interior',         # New value of ->currentRoomMode
                        FALSE,              # No call to ->redrawRegions; current room is redrawn
                        'fill_in_current_room',
                    );
                }
            });
            $subMenu_currentRoom->append($item_radio3);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'fill_in_current_room', $item_radio3);

        my $item_currentRoom = Gtk3::MenuItem->new('_Draw current room');
        $item_currentRoom->set_submenu($subMenu_currentRoom);
        $column_view->append($item_currentRoom);

            # 'Room filters' submenu
            my $subMenu_roomFilters = Gtk3::Menu->new();

            my $item_releaseAllFilters = Gtk3::CheckMenuItem->new('_Release all filters');
            $item_releaseAllFilters->set_active($self->worldModelObj->allRoomFiltersFlag);
            $item_releaseAllFilters->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'allRoomFiltersFlag',
                        $item_releaseAllFilters->get_active(),
                        TRUE,      # Do call $self->redrawRegions
                        'release_all_filters',
                        'icon_release_all_filters',
                    );
                }
            });
            $subMenu_roomFilters->append($item_releaseAllFilters);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'release_all_filters', $item_releaseAllFilters);

            $subMenu_roomFilters->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my @shortcutList = $axmud::CLIENT->constRoomFilterKeyList;
            foreach my $filter ($axmud::CLIENT->constRoomFilterList) {

                my $shortcut = shift @shortcutList;

                my $menuItem = Gtk3::CheckMenuItem->new('Release ' . $shortcut . ' filter');
                $menuItem->set_active($self->worldModelObj->ivShow('roomFilterApplyHash', $filter));
                $menuItem->signal_connect('toggled' => sub {

                    if (! $self->ignoreMenuUpdateFlag) {

                        $self->worldModelObj->toggleFilter(
                            $filter,
                            $menuItem->get_active(),
                        );
                    }
                });
                $subMenu_roomFilters->append($menuItem);
                # (Never desensitised)
                $self->ivAdd('menuToolItemHash', $filter . '_filter', $menuItem);
            }

        my $item_roomFilters = Gtk3::MenuItem->new('Room _filters');
        $item_roomFilters->set_submenu($subMenu_roomFilters);
        $column_view->append($item_roomFilters);

            # 'Room interiors' submenu
            my $subMenu_roomInteriors = Gtk3::Menu->new();

            @initList = (
                'none' => '_Don\'t draw counts',
                'shadow_count' => 'Draw _unallocated/shadow exits',
                'region_count' => 'Draw re_gion/super region exits',
                'checked_count' => 'Draw _checked/checkable directions',
                'room_content' => 'Draw _room contents',
                'hidden_count' => 'Draw _hidden contents',
                'temp_count' => 'Draw _temporary contents',
                'word_count' => 'Draw r_ecognised words',
                'room_tag' => 'Draw room t_ag',
                'room_flag' => 'Draw r_oom flag text',
                'visit_count' => 'Draw character _visits',
                'compare_count' => 'Draw _matching rooms',
                'profile_count' => 'Draw e_xclusive profiles',
                'title_descrip' => 'Draw t_itles/descriptions',
                'exit_pattern' => 'Draw exit _patterns',
                'source_code' => 'Draw room _source code',
                'grid_posn' => 'Dra_w grid coordinates',
                'vnum' => 'Draw world\'s room v_num',
            );

            do {

                my ($mode, $descrip);

                $mode = shift @initList;
                $descrip = shift @initList;

                push (@interiorList, $mode);
                $interiorHash{$mode} = $descrip;

            } until (! @initList);

            for (my $count = 0; $count < (scalar @interiorList); $count++) {

                my ($icon, $mode);

                $mode = $interiorList[$count];

                # (For $count = 0, $item_group is 'undef')
                my $item_radio = Gtk3::RadioMenuItem->new_with_mnemonic(
                    $item_group,
                    $interiorHash{$mode},
                );
                if ($self->worldModelObj->roomInteriorMode eq $mode) {

                    $item_radio->set_active(TRUE);
                }

                $item_radio->signal_connect('toggled' => sub {

                    if (! $self->ignoreMenuUpdateFlag && $item_radio->get_active()) {

                        $self->worldModelObj->switchRoomInteriorMode($mode);
                    }
                });
                $item_group = $item_radio->get_group();
                $subMenu_roomInteriors->append($item_radio);
                # (Never desensitised)
                $self->ivAdd('menuToolItemHash', 'interior_mode_' . $mode, $item_radio);
            }

            $subMenu_roomInteriors->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_changeCharDrawn = Gtk3::MenuItem->new('Ch_ange character drawn...');
            $item_changeCharDrawn->signal_connect('activate' => sub {

                # (Callback func has no dependencies)
                $self->changeCharDrawnCallback();
            });
            $subMenu_roomInteriors->append($item_changeCharDrawn);

        my $item_roomInteriors = Gtk3::MenuItem->new('R_oom interiors');
        $item_roomInteriors->set_submenu($subMenu_roomInteriors);
        $column_view->append($item_roomInteriors);

            # 'All exits' submenu
            my $subMenu_allExits = Gtk3::Menu->new();

            my $item_radio11 = Gtk3::RadioMenuItem->new_with_mnemonic(
                undef,
                '_Use region exit settings',
            );
            $item_radio11->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio11->get_active()) {

                    $self->worldModelObj->switchMode(
                        'drawExitMode',
                        'ask_regionmap',    # New value of ->drawExitMode
                        TRUE,               # Do call $self->redrawRegions
                        'draw_defer_exits',
                        'icon_draw_defer_exits',
                    );
                }
            });
            my $item_group1 = $item_radio11->get_group();
            $subMenu_allExits->append($item_radio11);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_defer_exits', $item_radio11);

            my $item_radio12 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group1,
                'Draw _no exits',
            );
            if ($self->worldModelObj->drawExitMode eq 'no_exit') {

                $item_radio12->set_active(TRUE);
            }
            $item_radio12->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio12->get_active()) {

                    $self->worldModelObj->switchMode(
                        'drawExitMode',
                        'no_exit',          # New value of ->drawExitMode
                        TRUE,               # Do call $self->redrawRegions
                        'draw_no_exits',
                        'icon_draw_no_exits',
                    );
                }
            });
            $subMenu_allExits->append($item_radio12);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_no_exits', $item_radio12);

            my $item_radio13 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group1,
                'Draw _simple exits',
            );
            if ($self->worldModelObj->drawExitMode eq 'simple_exit') {

                $item_radio13->set_active(TRUE);
            }
            $item_radio13->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio13->get_active()) {

                    $self->worldModelObj->switchMode(
                        'drawExitMode',
                        'simple_exit',      # New value of ->drawExitMode
                        TRUE,               # Do call $self->redrawRegions
                        'draw_simple_exits',
                        'icon_draw_simple_exits',
                    );
                }
            });
            $subMenu_allExits->append($item_radio13);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_simple_exits', $item_radio13);

            my $item_radio14 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group1,
                'Draw _complex exits',
            );
            if ($self->worldModelObj->drawExitMode eq 'complex_exit') {

                $item_radio14->set_active(TRUE);
            }
            $item_radio14->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio14->get_active()) {

                    $self->worldModelObj->switchMode(
                        'drawExitMode',
                        'complex_exit',     # New value of ->drawExitMode
                        TRUE,               # Do call $self->redrawRegions
                        'draw_complex_exits',
                        'icon_draw_complex_exits',
                    );
                }
            });
            $subMenu_allExits->append($item_radio14);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_complex_exits', $item_radio14);

            $subMenu_allExits->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_obscuredExits = Gtk3::CheckMenuItem->new('_Obscure unimportant exits');
            $item_obscuredExits->set_active($self->worldModelObj->obscuredExitFlag);
            $item_obscuredExits->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'obscuredExitFlag',
                        $item_obscuredExits->get_active(),
                        TRUE,      # Do call $self->redrawRegions
                        'obscured_exits',
                        'icon_obscured_exits',
                    );
                }
            });
            $subMenu_allExits->append($item_obscuredExits);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'obscured_exits', $item_obscuredExits);

            my $item_autoRedraw = Gtk3::CheckMenuItem->new('_Auto-redraw obscured exits');
            $item_autoRedraw->set_active($self->worldModelObj->obscuredExitRedrawFlag);
            $item_autoRedraw->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'obscuredExitRedrawFlag',
                        $item_autoRedraw->get_active(),
                        TRUE,      # Do call $self->redrawRegions
                        'auto_redraw_obscured',
                        'icon_auto_redraw_obscured',
                    );
                }
            });
            $subMenu_allExits->append($item_autoRedraw);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_redraw_obscured', $item_autoRedraw);

            my $item_obscuredExitRadius = Gtk3::MenuItem->new('Set obscure _radius...');
            $item_obscuredExitRadius->signal_connect('activate' => sub {

                $self->obscuredRadiusCallback();
            });
            $subMenu_allExits->append($item_obscuredExitRadius);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'obscured_exit_radius', $item_obscuredExitRadius);

            $subMenu_allExits->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_drawOrnaments = Gtk3::CheckMenuItem->new('Draw exit orna_ments');
            $item_drawOrnaments->set_active($self->worldModelObj->drawOrnamentsFlag);
            $item_drawOrnaments->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'drawOrnamentsFlag',
                        $item_drawOrnaments->get_active(),
                        TRUE,      # Do call $self->redrawRegions
                        'draw_ornaments',
                        'icon_draw_ornaments',
                    );
                }
            });
            $subMenu_allExits->append($item_drawOrnaments);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_ornaments', $item_drawOrnaments);

        my $item_allExits = Gtk3::MenuItem->new('Exits (_all regions)');
        $item_allExits->set_submenu($subMenu_allExits);
        $column_view->append($item_allExits);

            # 'Region exits' submenu
            my $subMenu_regionExits = Gtk3::Menu->new();

            my $item_radio21 = Gtk3::RadioMenuItem->new_with_mnemonic(undef, 'Draw _no exits');
            $item_radio21->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio21->get_active()) {

                    $self->worldModelObj->switchRegionDrawExitMode(
                        $self->currentRegionmap,
                        'no_exit',
                    );
                }
            });
            my $item_group2 = $item_radio21->get_group();
            $subMenu_regionExits->append($item_radio21);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'region_draw_no_exits', $item_radio21);

            my $item_radio22 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group2,
                'Draw _simple exits',
            );
            if ($self->currentRegionmap && $self->currentRegionmap->drawExitMode eq 'simple_exit') {

                $item_radio22->set_active(TRUE);
            }
            $item_radio22->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio22->get_active()) {

                    $self->worldModelObj->switchRegionDrawExitMode(
                        $self->currentRegionmap,
                        'simple_exit',
                    );
                }
            });
            $subMenu_regionExits->append($item_radio22);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'region_draw_simple_exits', $item_radio22);

            my $item_radio23 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group2,
                'Draw _complex exits',
            );
            if (
                $self->currentRegionmap
                && $self->currentRegionmap->drawExitMode eq 'complex_exit'
            ) {
                $item_radio23->set_active(TRUE);
            }
            $item_radio23->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio23->get_active()) {

                    $self->worldModelObj->switchRegionDrawExitMode(
                        $self->currentRegionmap,
                        'complex_exit',
                    );
                }
            });
            $subMenu_regionExits->append($item_radio23);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'region_draw_complex_exits', $item_radio23);

            $subMenu_regionExits->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_obscuredExitsRegion = Gtk3::CheckMenuItem->new('_Obscure unimportant exits');
            if ($self->currentRegionmap) {

                $item_obscuredExitsRegion->set_active($self->currentRegionmap->obscuredExitFlag);
            }
            $item_obscuredExitsRegion->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleObscuredExitFlag($self->currentRegionmap);
                }
            });
            $subMenu_regionExits->append($item_obscuredExitsRegion);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'obscured_exits_region', $item_obscuredExitsRegion);

            my $item_autoRedrawRegion = Gtk3::CheckMenuItem->new('_Auto-redraw obscured exits');
            if ($self->currentRegionmap) {

                $item_autoRedrawRegion->set_active($self->currentRegionmap->obscuredExitRedrawFlag);
            }
            $item_autoRedrawRegion->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleObscuredExitRedrawFlag($self->currentRegionmap);
                }
            });
            $subMenu_regionExits->append($item_autoRedrawRegion);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_redraw_obscured_region', $item_autoRedrawRegion);

            my $item_obscuredExitRadiusRegion = Gtk3::MenuItem->new('Set obscure _radius...');
            $item_obscuredExitRadiusRegion->signal_connect('activate' => sub {

                $self->obscuredRadiusCallback($self->currentRegionmap);
            });
            $subMenu_regionExits->append($item_obscuredExitRadiusRegion);
            # (Never desensitised)
            $self->ivAdd(
                'menuToolItemHash',
                'obscured_exit_radius_region',
                $item_obscuredExitRadiusRegion,
            );

            $subMenu_regionExits->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_drawOrnamentsRegion = Gtk3::CheckMenuItem->new('Draw exit orna_ments');
            if ($self->currentRegionmap) {

                $item_drawOrnamentsRegion->set_active($self->currentRegionmap->drawOrnamentsFlag);
            }
            $item_drawOrnamentsRegion->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleDrawOrnamentsFlag($self->currentRegionmap);
                }
            });
            $subMenu_regionExits->append($item_drawOrnamentsRegion);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_ornaments_region', $item_drawOrnamentsRegion);

        my $item_regionExits = Gtk3::MenuItem->new('Exits (_current region)');
        $item_regionExits->set_submenu($subMenu_regionExits);
        $column_view->append($item_regionExits);
        # (Requires $self->currentRegionmap and $self->worldModelObj->drawExitMode is
        #   'ask_regionmap')
        $self->ivAdd('menuToolItemHash', 'draw_region_exits', $item_regionExits);

        $column_view->append(Gtk3::SeparatorMenuItem->new());   # Separator

        my $item_zoomIn = Gtk3::ImageMenuItem->new('Zoom i_n');
        my $img_zoomIn = Gtk3::Image->new_from_stock('gtk-zoom-in', 'menu');
        $item_zoomIn->set_image($img_zoomIn);
        $item_zoomIn->signal_connect('activate' => sub {

            $self->zoomCallback('in');
        });
        $column_view->append($item_zoomIn);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'zoom_in', $item_zoomIn);

        my $item_zoomOut = Gtk3::ImageMenuItem->new('Zoom _out');
        my $img_zoomOut = Gtk3::Image->new_from_stock('gtk-zoom-out', 'menu');
        $item_zoomOut->set_image($img_zoomOut);
        $item_zoomOut->signal_connect('activate' => sub {

            $self->zoomCallback('out');
        });
        $column_view->append($item_zoomOut);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'zoom_out', $item_zoomOut);

            # 'Zoom' submenu
            my $subMenu_zoom = Gtk3::Menu->new();

            # Import the list of magnifications
            @magList = $self->constMagnifyList;
            # Use a subset of magnifications from $self->constMagnifyList (and in reverse order to
            #   that found in $self->constMagnifyList)
            @shortMagList = reverse $self->constShortMagnifyList;

            foreach my $mag (@shortMagList) {

                my $menuItem = Gtk3::MenuItem->new('Zoom ' . $mag * 100 . '%');
                $menuItem->signal_connect('activate' => sub {

                    # No argument causes the called function to prompt the user
                    $self->zoomCallback($mag);
                });
                $subMenu_zoom->append($menuItem);
            }

            $subMenu_zoom->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_zoomMax = Gtk3::MenuItem->new('Zoom _in max');
            $item_zoomMax->signal_connect('activate' => sub {

                $self->zoomCallback($magList[-1]);
            });
            $subMenu_zoom->append($item_zoomMax);

            my $item_zoomMin = Gtk3::MenuItem->new('Zoom _out max');
            $item_zoomMin->signal_connect('activate' => sub {

                $self->zoomCallback($magList[0]);
            });
            $subMenu_zoom->append($item_zoomMin);

            $subMenu_zoom->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_zoomPrompt = Gtk3::MenuItem->new('O_ther...');
            $item_zoomPrompt->signal_connect('activate' => sub {

                # No argument causes the called function to prompt the user
                $self->zoomCallback();
            });
            $subMenu_zoom->append($item_zoomPrompt);

        my $item_zoom = Gtk3::ImageMenuItem->new('_Zoom');
        my $img_zoom = Gtk3::Image->new_from_stock('gtk-zoom-fit', 'menu');
        $item_zoom->set_image($img_zoom);
        $item_zoom->set_submenu($subMenu_zoom);
        $column_view->append($item_zoom);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'zoom_sub', $item_zoom);

        $column_view->append(Gtk3::SeparatorMenuItem->new());   # Separator

            # 'Level' submenu
            my $subMenu_level = Gtk3::Menu->new();

            my $item_moveUpLevel = Gtk3::MenuItem->new('Move _up level');
            $item_moveUpLevel->signal_connect('activate' => sub {

                $self->setCurrentLevel($self->currentRegionmap->currentLevel + 1);

                # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
                $self->restrictWidgets();
            });
            $subMenu_level->append($item_moveUpLevel);
            # (Requires $self->currentRegionmap)
            $self->ivAdd('menuToolItemHash', 'move_up_level', $item_moveUpLevel);

            my $item_moveDownLevel = Gtk3::MenuItem->new('Move _down level');
            $item_moveDownLevel->signal_connect('activate' => sub {

                $self->setCurrentLevel($self->currentRegionmap->currentLevel - 1);

                # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
                $self->restrictWidgets();
            });
            $subMenu_level->append($item_moveDownLevel);
            # (Requires $self->currentRegionmap)
            $self->ivAdd('menuToolItemHash', 'move_down_level', $item_moveDownLevel);

            $subMenu_level->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_changeLevel = Gtk3::MenuItem->new('_Change level...');
            $item_changeLevel->signal_connect('activate' => sub {

                $self->changeLevelCallback();
            });
            $subMenu_level->append($item_changeLevel);

        my $item_level = Gtk3::MenuItem->new('_Level');
        $item_level->set_submenu($subMenu_level);
        $column_view->append($item_level);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'level_sub', $item_level);

        $column_view->append(Gtk3::SeparatorMenuItem->new());   # Separator

            # 'Centre map' submenu
            my $subMenu_centreMap = Gtk3::Menu->new();

            my $item_centreMap_currentRoom = Gtk3::MenuItem->new('_Current room');
            $item_centreMap_currentRoom->signal_connect('activate' => sub {

                $self->centreMapOverRoom($self->mapObj->currentRoom);
            });
            $subMenu_centreMap->append($item_centreMap_currentRoom);
            # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
            $self->ivAdd(
                'menuToolItemHash',
                'centre_map_current_room',
                $item_centreMap_currentRoom,
            );

            my $item_centreMap_selectRoom = Gtk3::MenuItem->new('_Selected room');
            $item_centreMap_selectRoom->signal_connect('activate' => sub {

                $self->centreMapOverRoom($self->selectedRoom);
            });
            $subMenu_centreMap->append($item_centreMap_selectRoom);
            # (Requires $self->currentRegionmap & $self->selectedRoom)
            $self->ivAdd(
                'menuToolItemHash',
                'centre_map_select_room',
                $item_centreMap_selectRoom,
            );

            my $item_centreMap_lastKnownRoom = Gtk3::MenuItem->new('_Last known room');
            $item_centreMap_lastKnownRoom->signal_connect('activate' => sub {

                $self->centreMapOverRoom($self->mapObj->lastKnownRoom);
            });
            $subMenu_centreMap->append($item_centreMap_lastKnownRoom);
            # (Requires $self->currentRegionmap & $self->mapObj->lastknownRoom)
            $self->ivAdd(
                'menuToolItemHash',
                'centre_map_last_known_room',
                $item_centreMap_lastKnownRoom,
            );

            $subMenu_centreMap->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_centreMap_middleGrid = Gtk3::MenuItem->new('_Middle of grid');
            $item_centreMap_middleGrid->signal_connect('activate' => sub {

                $self->setMapPosn(0.5, 0.5);
            });
            $subMenu_centreMap->append($item_centreMap_middleGrid);
            # (Requires $self->currentRegionmap)
            $self->ivAdd('menuToolItemHash', 'centre_map_middle_grid', $item_centreMap_middleGrid);

        my $item_centreMap = Gtk3::MenuItem->new('Centre _map');
        $item_centreMap->set_submenu($subMenu_centreMap);
        $column_view->append($item_centreMap);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'centre_map_sub', $item_centreMap);

        my $item_repositionAllMaps = Gtk3::MenuItem->new('_Reposition all maps');
        $item_repositionAllMaps->signal_connect('activate' => sub {

            $self->worldModelObj->repositionMaps();
        });
        $column_view->append($item_repositionAllMaps);

            # 'Tracking' submenu
            my $subMenu_tracking = Gtk3::Menu->new();

            my $item_trackCurrentRoom = Gtk3::CheckMenuItem->new('_Track current room');
            $item_trackCurrentRoom->set_active($self->worldModelObj->trackPosnFlag);
            $item_trackCurrentRoom->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'trackPosnFlag',
                        $item_trackCurrentRoom->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'track_current_room',
                        'icon_track_current_room',
                    );
                }
            });
            $subMenu_tracking->append($item_trackCurrentRoom);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'track_current_room', $item_trackCurrentRoom);

            $subMenu_tracking->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_radio31 = Gtk3::RadioMenuItem->new_with_mnemonic(undef, '_Always track');
            if (
                $self->worldModelObj->trackingSensitivity != 0.33
                && $self->worldModelObj->trackingSensitivity != 0.66
                && $self->worldModelObj->trackingSensitivity != 1
            ) {
                # Only the sensitivity values 0, 0.33, 0.66 and 1 are curently allowed; act as
                #   though the IV was set to 0
                $item_radio31->set_active(TRUE);
            }
            $item_radio31->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio31->get_active()) {

                    $self->worldModelObj->setTrackingSensitivity(0);
                }
            });
            my $item_group3 = $item_radio31->get_group();
            $subMenu_tracking->append($item_radio31);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'track_always', $item_radio31);

            my $item_radio32 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group3,
                'Track near _centre',
            );
            if ($self->worldModelObj->trackingSensitivity == 0.33) {

                $item_radio32->set_active(TRUE);
            }
            $item_radio32->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio32->get_active()) {

                    $self->worldModelObj->setTrackingSensitivity(0.33);
                }
            });
            $subMenu_tracking->append($item_radio32);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'track_near_centre', $item_radio32);

            my $item_radio33 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group3,
                'Track near _edge',
            );
            if ($self->worldModelObj->trackingSensitivity == 0.66) {

                $item_radio33->set_active(TRUE);
            }
            $item_radio33->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio33->get_active()) {

                    $self->worldModelObj->setTrackingSensitivity(0.66);
                }
            });
            $subMenu_tracking->append($item_radio33);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'track_near_edge', $item_radio33);

            my $item_radio34 = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_group3,
                'Track if not _visible',
            );
            if ($self->worldModelObj->trackingSensitivity == 1) {

                $item_radio34->set_active(TRUE);
            }
            $item_radio34->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $item_radio34->get_active()) {

                    $self->worldModelObj->setTrackingSensitivity(1);
                }
            });
            $subMenu_tracking->append($item_radio34);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'track_not_visible', $item_radio34);

        my $item_tracking = Gtk3::MenuItem->new('_Tracking');
        $item_tracking->set_submenu($subMenu_tracking);
        $column_view->append($item_tracking);

        # Setup complete
        return $column_view;
    }

    sub enableModeColumn {

        # Called by $self->enableMenu
        # Sets up the 'Mode' column of the Automapper window's menu bar
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableModeColumn', @_);
        }

        # Set up column
        my $column_mode = Gtk3::Menu->new();
        if (! $column_mode) {

            return undef;
        }

        # (Save each radio menu item in a hash IV, so that when $self->setMode is called, the radio
        #   group can be toggled)
        my $item_radio1 = Gtk3::RadioMenuItem->new_with_mnemonic(undef, '_Wait mode');
        $item_radio1->signal_connect('toggled' => sub {

            # (To stop the equivalent toolbar icon from being toggled by the call to ->setMode,
            #   make use of $self->ignoreMenuUpdateFlag)
            if ($item_radio1->get_active && ! $self->ignoreMenuUpdateFlag) {

                $self->setMode('wait');
            }
        });
        my $item_group = $item_radio1->get_group();
        $column_mode->append($item_radio1);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'set_wait_mode', $item_radio1);

        my $item_radio2 = Gtk3::RadioMenuItem->new_with_mnemonic($item_group, '_Follow mode');
        $item_radio2->signal_connect('toggled' => sub {

            if ($item_radio2->get_active && ! $self->ignoreMenuUpdateFlag) {

                $self->setMode('follow');
            }
        });
        $column_mode->append($item_radio2);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'set_follow_mode', $item_radio2);

        my $item_radio3 = Gtk3::RadioMenuItem->new_with_mnemonic($item_group, '_Update mode');
        $item_radio3->signal_connect('toggled' => sub {

            if ($item_radio3->get_active && ! $self->ignoreMenuUpdateFlag) {

                $self->setMode('update');
            }
        });
        $column_mode->append($item_radio3);
        # (Requires $self->currentRegionmap, GA::Obj::WorldModel->disableUpdateModeFlag set to
        #   FALSE and a session not in 'connect offline' mode
        $self->ivAdd('menuToolItemHash', 'set_update_mode', $item_radio3);

        $column_mode->append(Gtk3::SeparatorMenuItem->new());   # Separator

        my $item_dragMode = Gtk3::CheckMenuItem->new('_Drag mode');
        $item_dragMode->set_active($self->dragModeFlag);
        $item_dragMode->signal_connect('toggled' => sub {

            if ($item_dragMode->get_active()) {
                $self->ivPoke('dragModeFlag', TRUE);
            } else {
                $self->ivPoke('dragModeFlag', FALSE);
            }

            # Set the equivalent toolbar button
            if ($self->ivExists('menuToolItemHash', 'icon_drag_mode')) {

                my $menuItem = $self->ivShow('menuToolItemHash', 'icon_drag_mode');
                $menuItem->set_active($item_dragMode->get_active());
            }
        });
        $column_mode->append($item_dragMode);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'drag_mode', $item_dragMode);

        my $item_graffitMode = Gtk3::CheckMenuItem->new('_Graffiti mode');
        $item_graffitMode->set_active($self->graffitiModeFlag);
        $item_graffitMode->signal_connect('toggled' => sub {

            my @redrawList;

            if ($item_graffitMode->get_active()) {

                $self->ivPoke('graffitiModeFlag', TRUE);

                # Tag current room, if any
                if ($self->mapObj->currentRoom) {

                    $self->ivAdd('graffitiHash', $self->mapObj->currentRoom->number);
                    $self->markObjs('room', $self->mapObj->currentRoom);
                    $self->doDraw();
                }

                # Initialise graffitied room counts
                $self->setWinTitle();

            } else {

                $self->ivPoke('graffitiModeFlag', FALSE);

                foreach my $num ($self->ivKeys('graffitiHash')) {

                    my $roomObj = $self->worldModelObj->ivShow('modelHash', $num);
                    if ($roomObj) {

                        push (@redrawList, 'room', $self->worldModelObj->ivShow('modelHash', $num));
                    }
                }

                $self->ivEmpty('graffitiHash');

                # Redraw any graffitied rooms
                if (@redrawList) {

                    $self->markObjs(@redrawList);
                    $self->doDraw();
                }

                # Remove graffitied room counts
                $self->setWinTitle();
            }

            # Set the equivalent toolbar button
            if ($self->ivExists('menuToolItemHash', 'icon_graffiti_mode')) {

                my $menuItem = $self->ivShow('menuToolItemHash', 'icon_graffiti_mode');
                $menuItem->set_active($item_graffitMode->get_active());
            }

            # The menu items which toggle graffiti in selected rooms are desensitised if
            #   ->graffitiModeFlag is FALSE
            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $self->restrictWidgets();
        });
        $column_mode->append($item_graffitMode);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'graffiti_mode', $item_graffitMode);

        $column_mode->append(Gtk3::SeparatorMenuItem->new());   # Separator

            # 'Match rooms' submenu
            my $subMenu_matchRooms = Gtk3::Menu->new();

            my $item_matchTitle = Gtk3::CheckMenuItem->new('Match room _titles');
            $item_matchTitle->set_active($self->worldModelObj->matchTitleFlag);
            $item_matchTitle->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'matchTitleFlag',
                        $item_matchTitle->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'match_title',
                    );
                }
            });
            $subMenu_matchRooms->append($item_matchTitle);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'match_title', $item_matchTitle);

            my $item_matchDescrip = Gtk3::CheckMenuItem->new('Match room _descriptions');
            $item_matchDescrip->set_active($self->worldModelObj->matchDescripFlag);
            $item_matchDescrip->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'matchDescripFlag',
                        $item_matchDescrip->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'match_descrip',
                    );
                }
            });
            $subMenu_matchRooms->append($item_matchDescrip);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'match_descrip', $item_matchDescrip);

            my $item_matchExit = Gtk3::CheckMenuItem->new('Match _exits');
            $item_matchExit->set_active($self->worldModelObj->matchExitFlag);
            $item_matchExit->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'matchExitFlag',
                        $item_matchExit->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'match_exit',
                    );
                }
            });
            $subMenu_matchRooms->append($item_matchExit);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'match_exit', $item_matchExit);

            my $item_matchSource = Gtk3::CheckMenuItem->new('Match _source code');
            $item_matchSource->set_active($self->worldModelObj->matchSourceFlag);
            $item_matchSource->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'matchSourceFlag',
                        $item_matchSource->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'match_source',
                    );
                }
            });
            $subMenu_matchRooms->append($item_matchSource);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'match_source', $item_matchSource);

            my $item_matchVNum = Gtk3::CheckMenuItem->new('Match room _vnum');
            $item_matchVNum->set_active($self->worldModelObj->matchVNumFlag);
            $item_matchVNum->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'matchVNumFlag',
                        $item_matchVNum->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'match_vnum',
                    );
                }
            });
            $subMenu_matchRooms->append($item_matchVNum);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'match_vnum', $item_matchVNum);

            $subMenu_matchRooms->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_verboseChars = Gtk3::MenuItem->new('Set description _length...');
            $item_verboseChars->signal_connect('activate' => sub {

                $self->verboseCharsCallback();
            });
            $subMenu_matchRooms->append($item_verboseChars);

        my $item_matchRooms = Gtk3::MenuItem->new('_Match rooms');
        $item_matchRooms->set_submenu($subMenu_matchRooms);
        $column_mode->append($item_matchRooms);

            # 'Update rooms' submenu
            my $subMenu_updateRooms = Gtk3::Menu->new();

            my $item_updateTitle = Gtk3::CheckMenuItem->new('Update room _titles');
            $item_updateTitle->set_active($self->worldModelObj->updateTitleFlag);
            $item_updateTitle->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'updateTitleFlag',
                        $item_updateTitle->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'update_title',
                    );
                }
            });
            $subMenu_updateRooms->append($item_updateTitle);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'update_title', $item_updateTitle);

            my $item_updateDescrip = Gtk3::CheckMenuItem->new('Update room _descriptions');
            $item_updateDescrip->set_active($self->worldModelObj->updateDescripFlag);
            $item_updateDescrip->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'updateDescripFlag',
                        $item_updateDescrip->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'update_descrip',
                    );
                }
            });
            $subMenu_updateRooms->append($item_updateDescrip);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'update_descrip', $item_updateDescrip);

            my $item_updateExit = Gtk3::CheckMenuItem->new('Update _exits');
            $item_updateExit->set_active($self->worldModelObj->updateExitFlag);
            $item_updateExit->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'updateExitFlag',
                        $item_updateExit->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'update_exit',
                    );
                }
            });
            $subMenu_updateRooms->append($item_updateExit);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'update_exit', $item_updateExit);

            my $item_updateOrnament
                = Gtk3::CheckMenuItem->new('Update _ornaments from exit state');
            $item_updateOrnament->set_active($self->worldModelObj->updateOrnamentFlag);
            $item_updateOrnament->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'updateOrnamentFlag',
                        $item_updateOrnament->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'update_ornament',
                    );
                }
            });
            $subMenu_updateRooms->append($item_updateOrnament);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'update_ornament', $item_updateOrnament);

            my $item_updateSource = Gtk3::CheckMenuItem->new('Update _source code');
            $item_updateSource->set_active($self->worldModelObj->updateSourceFlag);
            $item_updateSource->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'updateSourceFlag',
                        $item_updateSource->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'update_source',
                    );
                }
            });
            $subMenu_updateRooms->append($item_updateSource);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'update_source', $item_updateSource);

            my $item_updateVNum = Gtk3::CheckMenuItem->new('Update room _vnum, etc');
            $item_updateVNum->set_active($self->worldModelObj->updateVNumFlag);
            $item_updateVNum->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'updateVNumFlag',
                        $item_updateVNum->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'update_vnum',
                    );
                }
            });
            $subMenu_updateRooms->append($item_updateVNum);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'update_vnum', $item_updateVNum);

            my $item_updateRoomCmd = Gtk3::CheckMenuItem->new('Update room _commands');
            $item_updateRoomCmd->set_active($self->worldModelObj->updateRoomCmdFlag);
            $item_updateRoomCmd->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'updateRoomCmdFlag',
                        $item_updateRoomCmd->get_active(),
                        FALSE,          # Do call $self->redrawRegions
                        'update_room_cmd',
                    );
                }
            });
            $subMenu_updateRooms->append($item_updateRoomCmd);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'update_room_cmd', $item_updateRoomCmd);

            $subMenu_updateRooms->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_analyseDescrip = Gtk3::CheckMenuItem->new('_Analyse room descrips');
            $item_analyseDescrip->set_active($self->worldModelObj->analyseDescripFlag);
            $item_analyseDescrip->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'analyseDescripFlag',
                        $item_analyseDescrip->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'analyse_descrip',
                    );
                }
            });
            $subMenu_updateRooms->append($item_analyseDescrip);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'analyse_descrip', $item_analyseDescrip);

        my $item_updateRooms = Gtk3::MenuItem->new('Update _rooms');
        $item_updateRooms->set_submenu($subMenu_updateRooms);
        $column_mode->append($item_updateRooms);

        $column_mode->append(Gtk3::SeparatorMenuItem->new());   # Separator

            # 'Painter' submenu
            my $subMenu_painter = Gtk3::Menu->new();

            my $item_painterEnabled = Gtk3::CheckMenuItem->new('_Painter enabled');
            $item_painterEnabled->set_active($self->painterFlag);
            $item_painterEnabled->signal_connect('toggled' => sub {

                my $item;

                # Toggle the flag
                if ($item_painterEnabled->get_active()) {
                    $self->ivPoke('painterFlag', TRUE);
                } else {
                    $self->ivPoke('painterFlag', FALSE);
                }

                # Update the corresponding toolbar icon
                $item = $self->ivShow('menuToolItemHash', 'icon_enable_painter');
                if ($item) {

                    $item->set_active($self->painterFlag);
                }
            });
            $subMenu_painter->append($item_painterEnabled);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'enable_painter', $item_painterEnabled);

            $subMenu_painter->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_paintAll = Gtk3::RadioMenuItem->new_with_mnemonic(undef, 'Paint _all rooms');
            $item_paintAll->signal_connect('toggled' => sub {

                if ($item_paintAll->get_active) {

                    $self->worldModelObj->set_paintAllRoomsFlag(TRUE);

                    # Set the equivalent toolbar button
                    if ($self->ivExists('menuToolItemHash', 'icon_paint_all')) {

                        $self->ivShow('menuToolItemHash', 'icon_paint_all')->set_active(TRUE);
                    }
                }
            });
            my $item_paintGroup = $item_paintAll->get_group();
            $subMenu_painter->append($item_paintAll);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'paint_all', $item_paintAll);

            my $item_paintNew = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_paintGroup,
                'Paint _only new rooms',
            );
            if (! $self->worldModelObj->paintAllRoomsFlag) {

                $item_paintNew->set_active(TRUE);
            }
            $item_paintNew->signal_connect('toggled' => sub {

                if ($item_paintNew->get_active) {

                    $self->worldModelObj->set_paintAllRoomsFlag(FALSE);

                    # Set the equivalent toolbar button
                    if ($self->ivExists('menuToolItemHash', 'icon_paint_new')) {

                        $self->ivShow('menuToolItemHash', 'icon_paint_new')->set_active(TRUE);
                    }
                }
            });
            $subMenu_painter->append($item_paintNew);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'paint_new', $item_paintNew);

            $subMenu_painter->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_paintNormal = Gtk3::RadioMenuItem->new_with_mnemonic(
                undef,
                'Paint _normal rooms',
            );
            $item_paintNormal->signal_connect('toggled' => sub {

                if ($item_paintNormal->get_active) {

                    $self->worldModelObj->painterObj->ivPoke('wildMode', 'normal');

                    # Set the equivalent toolbar button
                    if ($self->ivExists('menuToolItemHash', 'icon_paint_normal')) {

                        $self->ivShow('menuToolItemHash', 'icon_paint_normal')->set_active(TRUE);
                    }
                }
            });
            my $item_paintGroup2 = $item_paintNormal->get_group();
            $subMenu_painter->append($item_paintNormal);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'paint_normal', $item_paintNormal);

            my $item_paintWild = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_paintGroup2,
                'Paint _wilderness rooms',
            );
            if ($self->worldModelObj->painterObj->wildMode eq 'wild') {

                $item_paintWild->set_active(TRUE);
            }
            $item_paintWild->signal_connect('toggled' => sub {

                if ($item_paintWild->get_active) {

                    $self->worldModelObj->painterObj->ivPoke('wildMode', 'wild');

                    # Set the equivalent toolbar button
                    if ($self->ivExists('menuToolItemHash', 'icon_paint_wild')) {

                        $self->ivShow('menuToolItemHash', 'icon_paint_wild')->set_active(TRUE);
                    }
                }
            });
            $subMenu_painter->append($item_paintWild);
            # (Requires $self->session->currentWorld->basicMappingFlag to be FALSE)
            $self->ivAdd('menuToolItemHash', 'paint_wild', $item_paintWild);

            my $item_paintBorder = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_paintGroup2,
                'Paint wilderness _border rooms',
            );
            if ($self->worldModelObj->painterObj->wildMode eq 'border') {

                $item_paintBorder->set_active(TRUE);
            }
            $item_paintBorder->signal_connect('toggled' => sub {

                if ($item_paintBorder->get_active) {

                    $self->worldModelObj->painterObj->ivPoke('wildMode', 'border');

                    # Set the equivalent toolbar button
                    if ($self->ivExists('menuToolItemHash', 'icon_paint_border')) {

                        $self->ivShow('menuToolItemHash', 'icon_paint_border')->set_active(TRUE);
                    }
                }
            });
            $subMenu_painter->append($item_paintBorder);
            # (Requires $self->session->currentWorld->basicMappingFlag to be FALSE)
            $self->ivAdd('menuToolItemHash', 'paint_border', $item_paintBorder);

            $subMenu_painter->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_repaintCurrentRoom = Gtk3::MenuItem->new('Repaint _current room');
            $item_repaintCurrentRoom->signal_connect('activate' => sub {

                if ($self->mapObj->currentRoom) {

                    # Repaint the current room. The TRUE argument instructs the function to tell
                    #   the world model to redraw the room in every Automapper window
                    $self->paintRoom($self->mapObj->currentRoom, TRUE);
                }
            });
            $subMenu_painter->append($item_repaintCurrentRoom);
            # (Requires $self->currentRegionmap and $self->mapObj->currentRoom)
            $self->ivAdd('menuToolItemHash', 'repaint_current', $item_repaintCurrentRoom);

            my $item_repaintSelectedRooms = Gtk3::MenuItem->new('Repaint _selected rooms');
            $item_repaintSelectedRooms->signal_connect('activate' => sub {

                $self->repaintSelectedRoomsCallback();
            });
            $subMenu_painter->append($item_repaintSelectedRooms);
            # (Requires $self->currentRegionmap and either $self->selectedRoom or
            #   $self->selectedRoomHash)
            $self->ivAdd('menuToolItemHash', 'repaint_selected', $item_repaintSelectedRooms);

            $subMenu_painter->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_editPainter = Gtk3::ImageMenuItem->new('_Edit painter...');
            my $img_editPainter = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
            $item_editPainter->set_image($img_editPainter);
            $item_editPainter->signal_connect('activate' => sub {

                # Open an 'edit' window for the painter object
                $self->createFreeWin(
                    'Games::Axmud::EditWin::Painter',
                    $self,
                    $self->session,
                    'Edit world model painter',
                    $self->worldModelObj->painterObj,
                    FALSE,          # Not temporary
                );
            });
            $subMenu_painter->append($item_editPainter);

            $subMenu_painter->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_resetPainter = Gtk3::MenuItem->new('_Reset painter');
            $item_resetPainter->signal_connect('activate' => sub {

                $self->worldModelObj->resetPainter($self->session);

                $self->showMsgDialogue(
                    'Painter',
                    'info',
                    'The painter object has been reset',
                    'ok',
                );
            });
            $subMenu_painter->append($item_resetPainter);

        my $item_painter = Gtk3::ImageMenuItem->new('_Painter');
        my $img_painter = Gtk3::Image->new_from_stock('gtk-select-color', 'menu');
        $item_painter->set_image($img_painter);
        $item_painter->set_submenu($subMenu_painter);
        $column_mode->append($item_painter);

        $column_mode->append(Gtk3::SeparatorMenuItem->new());   # Separator

            # 'Auto-compare' submenu
            my $subMenu_autoCompare = Gtk3::Menu->new();

            my $item_compareDefault = Gtk3::RadioMenuItem->new_with_mnemonic(
                undef,
                '_Don\'t auto-compare current room',
            );
            $item_compareDefault->signal_connect('toggled' => sub {

                if ($item_compareDefault->get_active) {

                    $self->worldModelObj->setAutoCompareMode('default');
                }
            });
            my $item_compareGroup = $item_compareDefault->get_group();
            $subMenu_autoCompare->append($item_compareDefault);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_compare_default', $item_compareDefault);

            my $item_compareNew = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_compareGroup,
                'Auto-compare _new rooms',
            );
            if ($self->worldModelObj->autoCompareMode eq 'new') {

                $item_compareNew->set_active(TRUE);
            }
            $item_compareNew->signal_connect('toggled' => sub {

                if ($item_compareNew->get_active) {

                    $self->worldModelObj->setAutoCompareMode('new');
                }
            });
            $subMenu_autoCompare->append($item_compareNew);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_compare_new', $item_compareNew);

            my $item_compareCurrent = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_compareGroup,
                'Auto-compare the _current room',
            );
            if ($self->worldModelObj->autoCompareMode eq 'current') {

                $item_compareCurrent->set_active(TRUE);
            }
            $item_compareCurrent->signal_connect('toggled' => sub {

                if ($item_compareCurrent->get_active) {

                    $self->worldModelObj->setAutoCompareMode('current');
                }
            });
            $subMenu_autoCompare->append($item_compareCurrent);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_compare_current', $item_compareCurrent);

            $subMenu_autoCompare->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_compareRegion = Gtk3::RadioMenuItem->new_with_mnemonic(
                undef,
                'Compare with rooms in _same region',
            );
            $item_compareRegion->signal_connect('toggled' => sub {

                if ($item_compareRegion->get_active) {

                    $self->worldModelObj->toggleAutoCompareAllFlag(FALSE);
                }
            });
            my $item_compareRegionGroup = $item_compareRegion->get_group();
            $subMenu_autoCompare->append($item_compareRegion);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_compare_region', $item_compareRegion);

            my $item_compareWhole = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_compareRegionGroup,
                'Compare with rooms in _whole world',
            );
            if ($self->worldModelObj->autoCompareAllFlag) {

                $item_compareWhole->set_active(TRUE);
            }
            $item_compareWhole->signal_connect('toggled' => sub {

                if ($item_compareWhole->get_active) {

                    $self->worldModelObj->toggleAutoCompareAllFlag(TRUE);
                }
            });
            $subMenu_autoCompare->append($item_compareWhole);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_compare_model', $item_compareWhole);

            $subMenu_autoCompare->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_compareMax = Gtk3::MenuItem->new('Set _limit on room comparisons...');
            $item_compareMax->signal_connect('activate' => sub {

                $self->autoCompareMaxCallback();
            });
            $subMenu_autoCompare->append($item_compareMax);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_compare_max', $item_compareMax);

        my $item_autoCompare = Gtk3::MenuItem->new('_Auto-compare');
        $item_autoCompare->set_submenu($subMenu_autoCompare);
        $column_mode->append($item_autoCompare);

            # 'Auto-rescue mode' submenu
            my $subMenu_autoRescue = Gtk3::Menu->new();

            my $item_autoRescueEnable = Gtk3::CheckMenuItem->new('_Enable auto-rescue mode');
            $item_autoRescueEnable->set_active($self->worldModelObj->autoRescueFlag);
            $item_autoRescueEnable->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'autoRescueFlag',
                        $item_autoRescueEnable->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'auto_rescue',
                    );
                }
            });
            $subMenu_autoRescue->append($item_autoRescueEnable);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_rescue', $item_autoRescueEnable);

            $subMenu_autoRescue->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_autoRescueFirst = Gtk3::CheckMenuItem->new(
                '_Merge at first matching room',
            );
            $item_autoRescueFirst->set_active($self->worldModelObj->autoRescueFirstFlag);
            $item_autoRescueFirst->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'autoRescueFirstFlag',
                        $item_autoRescueFirst->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'auto_rescue_prompt',
                    );
                }
            });
            $subMenu_autoRescue->append($item_autoRescueFirst);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_rescue_first', $item_autoRescueFirst);

            my $item_autoRescuePrompt = Gtk3::CheckMenuItem->new('_Prompt before merging');
            $item_autoRescuePrompt->set_active($self->worldModelObj->autoRescuePromptFlag);
            $item_autoRescuePrompt->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'autoRescuePromptFlag',
                        $item_autoRescuePrompt->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'auto_rescue_prompt',
                    );
                }
            });
            $subMenu_autoRescue->append($item_autoRescuePrompt);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_rescue_prompt', $item_autoRescuePrompt);

            $subMenu_autoRescue->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_autoRescueNoMove = Gtk3::CheckMenuItem->new('_Don\'t move non-matching rooms');
            $item_autoRescueNoMove->set_active($self->worldModelObj->autoRescueNoMoveFlag);
            $item_autoRescueNoMove->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'autoRescueNoMoveFlag',
                        $item_autoRescueNoMove->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'auto_rescue_no_move',
                    );
                }
            });
            $subMenu_autoRescue->append($item_autoRescueNoMove);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_rescue_no_move', $item_autoRescueNoMove);

            my $item_autoRescueVisits = Gtk3::CheckMenuItem->new(
                '_Only update visits in merged rooms',
            );
            $item_autoRescueVisits->set_active($self->worldModelObj->autoRescueVisitsFlag);
            $item_autoRescueVisits->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'autoRescueVisitsFlag',
                        $item_autoRescueVisits->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'auto_rescue_visits',
                    );
                }
            });
            $subMenu_autoRescue->append($item_autoRescueVisits);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_rescue_visits', $item_autoRescueVisits);

            my $item_autoRescueForce = Gtk3::CheckMenuItem->new(
                '_Temporarily switch to \'update\' mode',
            );
            $item_autoRescueForce->set_active($self->worldModelObj->autoRescueForceFlag);
            $item_autoRescueForce->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'autoRescueForceFlag',
                        $item_autoRescueForce->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'auto_rescue_force',
                    );
                }
            });
            $subMenu_autoRescue->append($item_autoRescueForce);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_rescue_force', $item_autoRescueForce);

        my $item_autoRescue = Gtk3::MenuItem->new('Auto-r_escue mode');
        $item_autoRescue->set_submenu($subMenu_autoRescue);
        $column_mode->append($item_autoRescue);

            # 'Auto-slide mode' submenu
            my $subMenu_autoSlide = Gtk3::Menu->new();

            my $item_slideMode = Gtk3::RadioMenuItem->new_with_mnemonic(
                undef,
                '_Don\'t auto-slide new rooms',
            );
            $item_slideMode->signal_connect('toggled' => sub {

                if ($item_slideMode->get_active) {

                    $self->worldModelObj->setAutoSlideMode('default');
                }
            });
            my $item_slideGroup = $item_slideMode->get_group();
            $subMenu_autoSlide->append($item_slideMode);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'slide_default', $item_slideMode);

            my $item_slideOrigPull = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_slideGroup,
                'Slide original room _backwards',
            );
            if ($self->worldModelObj->autoSlideMode eq 'orig_pull') {

                $item_slideOrigPull->set_active(TRUE);
            }
            $item_slideOrigPull->signal_connect('toggled' => sub {

                if ($item_slideOrigPull->get_active) {

                    $self->worldModelObj->setAutoSlideMode('orig_pull');
                }
            });
            $subMenu_autoSlide->append($item_slideOrigPull);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'slide_orig_pull', $item_slideOrigPull);

            my $item_slideOrigPush = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_slideGroup,
                'Slide original room _forwards',
            );
            if ($self->worldModelObj->autoSlideMode eq 'orig_push') {

                $item_slideOrigPush->set_active(TRUE);
            }
            $item_slideOrigPush->signal_connect('toggled' => sub {

                if ($item_slideOrigPush->get_active) {

                    $self->worldModelObj->setAutoSlideMode('orig_push');
                }
            });
            $subMenu_autoSlide->append($item_slideOrigPush);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'slide_orig_pull', $item_slideOrigPush);

            my $item_slideOtherPull = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_slideGroup,
                'Slide blocking room b_ackwards',
            );
            if ($self->worldModelObj->autoSlideMode eq 'other_pull') {

                $item_slideOtherPull->set_active(TRUE);
            }
            $item_slideOtherPull->signal_connect('toggled' => sub {

                if ($item_slideOtherPull->get_active) {

                    $self->worldModelObj->setAutoSlideMode('other_pull');
                }
            });
            $subMenu_autoSlide->append($item_slideOtherPull);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'slide_orig_pull', $item_slideOtherPull);

            my $item_slideOtherPush = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_slideGroup,
                'Slide blocking room f_orwards',
            );
            if ($self->worldModelObj->autoSlideMode eq 'other_push') {

                $item_slideOtherPush->set_active(TRUE);
            }
            $item_slideOtherPush->signal_connect('toggled' => sub {

                if ($item_slideOtherPush->get_active) {

                    $self->worldModelObj->setAutoSlideMode('other_push');
                }
            });
            $subMenu_autoSlide->append($item_slideOtherPush);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'slide_orig_pull', $item_slideOtherPush);

            my $item_slideDestPull = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_slideGroup,
                'Slide new room ba_ckwards',
            );
            if ($self->worldModelObj->autoSlideMode eq 'dest_pull') {

                $item_slideDestPull->set_active(TRUE);
            }
            $item_slideDestPull->signal_connect('toggled' => sub {

                if ($item_slideDestPull->get_active) {

                    $self->worldModelObj->setAutoSlideMode('dest_pull');
                }
            });
            $subMenu_autoSlide->append($item_slideDestPull);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'slide_orig_pull', $item_slideDestPull);

            my $item_slideDestPush = Gtk3::RadioMenuItem->new_with_mnemonic(
                $item_slideGroup,
                'Slide new room fo_rwards',
            );
            if ($self->worldModelObj->autoSlideMode eq 'dest_push') {

                $item_slideDestPush->set_active(TRUE);
            }
            $item_slideDestPush->signal_connect('toggled' => sub {

                if ($item_slideDestPush->get_active) {

                    $self->worldModelObj->setAutoSlideMode('dest_push');
                }
            });
            $subMenu_autoSlide->append($item_slideDestPush);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'slide_orig_pull', $item_slideDestPush);

            $subMenu_autoSlide->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_slideMax = Gtk3::MenuItem->new('Set _limit on slide distance...');
            $item_slideMax->signal_connect('activate' => sub {

                $self->autoSlideMaxCallback();
            });
            $subMenu_autoSlide->append($item_slideMax);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'slide_max', $item_slideMax);

        my $item_autoSlide = Gtk3::MenuItem->new('Auto-s_lide mode');
        $item_autoSlide->set_submenu($subMenu_autoSlide);
        $column_mode->append($item_autoSlide);

        $column_mode->append(Gtk3::SeparatorMenuItem->new());   # Separator

            # 'Start-up flags' submenu
            my $subMenu_startUpFlags = Gtk3::Menu->new();

            my $item_autoOpenWindow = Gtk3::CheckMenuItem->new('Open _automapper on startup');
            $item_autoOpenWindow->set_active($self->worldModelObj->autoOpenWinFlag);
            $item_autoOpenWindow->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'autoOpenWinFlag',
                        $item_autoOpenWindow->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'auto_open_win',
                    );
                }
            });
            $subMenu_startUpFlags->append($item_autoOpenWindow);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'auto_open_win', $item_autoOpenWindow);

            my $item_pseudoWin = Gtk3::CheckMenuItem->new('Open as _pseudo-window');
            $item_pseudoWin->set_active($self->worldModelObj->pseudoWinFlag);
            $item_pseudoWin->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'pseudoWinFlag',
                        $item_pseudoWin->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'pseudo_win',
                    );
                }
            });
            $subMenu_startUpFlags->append($item_pseudoWin);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'pseudo_win', $item_pseudoWin);

            my $item_allowTrackAlone = Gtk3::CheckMenuItem->new('_Follow character after closing');
            $item_allowTrackAlone->set_active($self->worldModelObj->allowTrackAloneFlag);
            $item_allowTrackAlone->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'allowTrackAloneFlag',
                        $item_allowTrackAlone->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'keep_following',
                    );
                }
            });
            $subMenu_startUpFlags->append($item_allowTrackAlone);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'keep_following', $item_allowTrackAlone);

        my $item_startUpFlags = Gtk3::MenuItem->new('S_tart-up flags');
        $item_startUpFlags->set_submenu($subMenu_startUpFlags);
        $column_mode->append($item_startUpFlags);

            # 'Drawing flags' submenu
            my $subMenu_drawingFlags = Gtk3::Menu->new();

            my $item_roomTagsInCaps = Gtk3::CheckMenuItem->new('_Capitalise room tags');
            $item_roomTagsInCaps->set_active($self->worldModelObj->capitalisedRoomTagFlag);
            $item_roomTagsInCaps->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'capitalisedRoomTagFlag',
                        $item_roomTagsInCaps->get_active(),
                        TRUE,      # Do call $self->redrawRegions
                        'room_tags_capitalised',
                    );
                }
            });
            $subMenu_drawingFlags->append($item_roomTagsInCaps);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'room_tags_capitalised', $item_roomTagsInCaps);

            my $item_drawBentExits = Gtk3::CheckMenuItem->new('Draw _bent broken exits');
            $item_drawBentExits->set_active($self->worldModelObj->drawBentExitsFlag);
            $item_drawBentExits->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'drawBentExitsFlag',
                        $item_drawBentExits->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'draw_bent_exits',
                    );
                }
            });
            $subMenu_drawingFlags->append($item_drawBentExits);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_bent_exits', $item_drawBentExits);

            my $item_drawRoomEcho = Gtk3::CheckMenuItem->new('Draw _room echos');
            $item_drawRoomEcho->set_active($self->worldModelObj->drawRoomEchoFlag);
            $item_drawRoomEcho->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'drawRoomEchoFlag',
                        $item_drawRoomEcho->get_active(),
                        TRUE,      # Do call $self->redrawRegions
                        'draw_room_echo',
                    );
                }
            });
            $subMenu_drawingFlags->append($item_drawRoomEcho);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_room_echo', $item_drawRoomEcho);

            my $item_showTooltips = Gtk3::CheckMenuItem->new('Show _tooltips');
            $item_showTooltips->set_active($self->worldModelObj->showTooltipsFlag);
            $item_showTooltips->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleShowTooltipsFlag(
                    $item_showTooltips->get_active(),
                );
            });
            $subMenu_drawingFlags->append($item_showTooltips);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'show_tooltips', $item_showTooltips);

            my $item_showNotes = Gtk3::CheckMenuItem->new('Show room _notes in tooltips');
            $item_showNotes->set_active($self->worldModelObj->showNotesFlag);
            $item_showNotes->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'showNotesFlag',
                        $item_showNotes->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'show_notes',
                    );
                }
            });
            $subMenu_drawingFlags->append($item_showNotes);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'show_notes', $item_showNotes);

        my $item_drawingFlags = Gtk3::MenuItem->new('Draw_ing flags');
        $item_drawingFlags->set_submenu($subMenu_drawingFlags);
        $column_mode->append($item_drawingFlags);

            # 'Movement flags' submenu
            my $subMenu_moves = Gtk3::Menu->new();

            my $item_allowAssisted = Gtk3::CheckMenuItem->new('_Allow assisted moves');
            $item_allowAssisted->set_active($self->worldModelObj->assistedMovesFlag);
            $item_allowAssisted->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'assistedMovesFlag',
                        $item_allowAssisted->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'allow_assisted_moves',
                    );

                    # The menu items below which set ->protectedMovesFlag and
                    #   ->superProtectedMovesFlag are desensitised if ->assistedMovesFlag is FALSE
                    # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
                    $self->restrictWidgets();
                }
            });
            $subMenu_moves->append($item_allowAssisted);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'allow_assisted_moves', $item_allowAssisted);

            $subMenu_moves->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_assistedBreak = Gtk3::CheckMenuItem->new('_Break doors before move');
            $item_assistedBreak->set_active($self->worldModelObj->assistedBreakFlag);
            $item_assistedBreak->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'assistedBreakFlag',
                        $item_assistedBreak->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'break_before_move',
                    );
                }
            });
            $subMenu_moves->append($item_assistedBreak);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'break_before_move', $item_assistedBreak);

            my $item_assistedPick = Gtk3::CheckMenuItem->new('_Pick doors before move');
            $item_assistedPick->set_active($self->worldModelObj->assistedPickFlag);
            $item_assistedPick->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'assistedPickFlag',
                        $item_assistedPick->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'pick_before_move',
                    );
                }
            });
            $subMenu_moves->append($item_assistedPick);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'pick_before_move', $item_assistedPick);

            my $item_assistedUnlock = Gtk3::CheckMenuItem->new('_Unlock doors before move');
            $item_assistedUnlock->set_active($self->worldModelObj->assistedUnlockFlag);
            $item_assistedUnlock->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'assistedUnlockFlag',
                        $item_assistedUnlock->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'unlock_before_move',
                    );
                }
            });
            $subMenu_moves->append($item_assistedUnlock);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'unlock_before_move', $item_assistedUnlock);

            my $item_assistedOpen = Gtk3::CheckMenuItem->new('_Open doors before move');
            $item_assistedOpen->set_active($self->worldModelObj->assistedOpenFlag);
            $item_assistedOpen->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'assistedOpenFlag',
                        $item_assistedOpen->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'open_before_move',
                    );
                }
            });
            $subMenu_moves->append($item_assistedOpen);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'open_before_move', $item_assistedOpen);

            my $item_assistedClose = Gtk3::CheckMenuItem->new('_Close doors after move');
            $item_assistedClose->set_active($self->worldModelObj->assistedCloseFlag);
            $item_assistedClose->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'assistedCloseFlag',
                        $item_assistedClose->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'close_after_move',
                    );
                }
            });
            $subMenu_moves->append($item_assistedClose);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'close_after_move', $item_assistedClose);

            my $item_assistedLock = Gtk3::CheckMenuItem->new('_Lock doors after move');
            $item_assistedLock->set_active($self->worldModelObj->assistedLockFlag);
            $item_assistedLock->signal_connect('toggled' => sub {

                if (! $self->assistedLockFlag) {

                    $self->worldModelObj->toggleFlag(
                        'assistedLockFlag',
                        $item_assistedLock->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'lock_after_move',
                    );
                }
            });
            $subMenu_moves->append($item_assistedLock);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'lock_after_move', $item_assistedLock);

            $subMenu_moves->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_allowProtected = Gtk3::CheckMenuItem->new('Allow p_rotected moves');
            $item_allowProtected->set_active($self->worldModelObj->protectedMovesFlag);
            $item_allowProtected->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'protectedMovesFlag',
                        $item_allowProtected->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'allow_protected_moves',
                    );

                    # The menu item below which sets ->crafyMovesFlag is desensitised if
                    #   ->assistedMovesFlag is false
                    # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
                    $self->restrictWidgets();
                }
            });
            $subMenu_moves->append($item_allowProtected);
            # (Requires $self->worldModelObj->assistedMovesFlag)
            $self->ivAdd('menuToolItemHash', 'allow_protected_moves', $item_allowProtected);

            my $item_allowSuper = Gtk3::CheckMenuItem->new('Ca_ncel commands when overruled');
            $item_allowSuper->set_active($self->worldModelObj->superProtectedMovesFlag);
            $item_allowSuper->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'superProtectedMovesFlag',
                        $item_allowSuper->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'allow_super_protected_moves',
                    );
                }
            });
            $subMenu_moves->append($item_allowSuper);
            # (Requires $self->worldModelObj->assistedMovesFlag)
            $self->ivAdd('menuToolItemHash', 'allow_super_protected_moves', $item_allowSuper);

            $subMenu_moves->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_allowCrafty = Gtk3::CheckMenuItem->new('Allow crafty _moves');
            $item_allowCrafty->set_active($self->worldModelObj->craftyMovesFlag);
            $item_allowCrafty->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'craftyMovesFlag',
                        $item_allowCrafty->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'allow_crafty_moves',
                    );
                }
            });
            $subMenu_moves->append($item_allowCrafty);
            # (Requires $self->worldModelObj->protectedMovesFlag set to be FALSE)
            $self->ivAdd('menuToolItemHash', 'allow_crafty_moves', $item_allowCrafty);

        my $item_moves = Gtk3::MenuItem->new('Mo_vement flags');
        $item_moves->set_submenu($subMenu_moves);
        $column_mode->append($item_moves);

            # 'Other flags' submenu
            my $subMenu_otherFlags = Gtk3::Menu->new();

            my $item_allowModelScripts = Gtk3::CheckMenuItem->new('_Allow model-wide scripts');
            $item_allowModelScripts->set_active($self->worldModelObj->allowModelScriptFlag);
            $item_allowModelScripts->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'allowModelScriptFlag',
                        $item_allowModelScripts->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'allow_model_scripts',
                    );
                }
            });
            $subMenu_otherFlags->append($item_allowModelScripts);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'allow_model_scripts', $item_allowModelScripts);

            my $item_allowRoomScripts = Gtk3::CheckMenuItem->new(
                'Allow ' . $axmud::BASIC_NAME . ' _scripts',
            );
            $item_allowRoomScripts->set_active($self->worldModelObj->allowRoomScriptFlag);
            $item_allowRoomScripts->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'allowRoomScriptFlag',
                        $item_allowRoomScripts->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'allow_room_scripts',
                    );
                }
            });
            $subMenu_otherFlags->append($item_allowRoomScripts);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'allow_room_scripts', $item_allowRoomScripts);

            my $item_countVisits = Gtk3::CheckMenuItem->new('_Count character visits');
            $item_countVisits->set_active($self->worldModelObj->countVisitsFlag);
            $item_countVisits->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'countVisitsFlag',
                        $item_countVisits->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'count_char_visits',
                    );
                }
            });
            $subMenu_otherFlags->append($item_countVisits);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'count_char_visits', $item_countVisits);

            my $item_disableUpdate = Gtk3::CheckMenuItem->new('_Disable update mode');
            $item_disableUpdate->set_active($self->worldModelObj->disableUpdateModeFlag);
            $item_disableUpdate->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleDisableUpdateModeFlag(
                        $item_disableUpdate->get_active(),
                    );
                }
            });
            $subMenu_otherFlags->append($item_disableUpdate);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'disable_update_mode', $item_disableUpdate);

            my $item_explainGetLost = Gtk3::CheckMenuItem->new('_Explain when getting lost');
            $item_explainGetLost->set_active($self->worldModelObj->explainGetLostFlag);
            $item_explainGetLost->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'explainGetLostFlag',
                        $item_explainGetLost->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'explain_get_lost',
                    );
                }
            });
            $subMenu_otherFlags->append($item_explainGetLost);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'explain_get_lost', $item_explainGetLost);

            my $item_followAnchor = Gtk3::CheckMenuItem->new('New exits for _follow anchors');
            $item_followAnchor->set_active($self->worldModelObj->followAnchorFlag);
            $item_followAnchor->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'followAnchorFlag',
                        $item_followAnchor->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'follow_anchor',
                    );
                }
            });
            $subMenu_otherFlags->append($item_followAnchor);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'follow_anchor', $item_followAnchor);

            my $item_allowCtrlCopy = Gtk3::CheckMenuItem->new('_Move rooms to click with CTRL+C');
            $item_allowCtrlCopy->set_active($self->worldModelObj->allowCtrlCopyFlag);
            $item_allowCtrlCopy->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'allowCtrlCopyFlag',
                        $item_allowCtrlCopy->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'allow_ctrl_copy',
                    );
                }
            });
            $subMenu_otherFlags->append($item_allowCtrlCopy);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'allow_ctrl_copy', $item_allowCtrlCopy);

            my $item_showAllPrimary = Gtk3::CheckMenuItem->new('S_how all directions in dialogues');
            $item_showAllPrimary->set_active($self->worldModelObj->showAllPrimaryFlag);
            $item_showAllPrimary->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'showAllPrimaryFlag',
                        $item_showAllPrimary->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'show_all_primary',
                    );
                }
            });
            $subMenu_otherFlags->append($item_showAllPrimary);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'show_all_primary', $item_showAllPrimary);

        my $item_otherFlags = Gtk3::MenuItem->new('_Other flags');
        $item_otherFlags->set_submenu($subMenu_otherFlags);
        $column_mode->append($item_otherFlags);

        # Setup complete
        return $column_mode;
    }

    sub enableRegionsColumn {

        # Called by $self->enableMenu
        # Sets up the 'Regions' column of the Automapper window's menu bar
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableRegionsColumn', @_);
        }

        # Set up column
        my $column_regions = Gtk3::Menu->new();
        if (! $column_regions) {

            return undef;
        }

        my $item_newRegion = Gtk3::ImageMenuItem->new('_New region...');
        my $img_newRegion = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_newRegion->set_image($img_newRegion);
        $item_newRegion->signal_connect('activate' => sub {

            $self->newRegionCallback(FALSE);
        });
        $column_regions->append($item_newRegion);

        my $item_newTempRegion = Gtk3::ImageMenuItem->new('New _temporary region...');
        my $img_newTempRegion = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_newTempRegion->set_image($img_newTempRegion);
        $item_newTempRegion->signal_connect('activate' => sub {

            $self->newRegionCallback(TRUE);
        });
        $column_regions->append($item_newTempRegion);

        $column_regions->append(Gtk3::SeparatorMenuItem->new());    # Separator

        my $item_editRegion = Gtk3::ImageMenuItem->new('_Edit region...');
        my $img_editRegion = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editRegion->set_image($img_editRegion);
        $item_editRegion->signal_connect('activate' => sub {

            $self->editRegionCallback();
        });
        $column_regions->append($item_editRegion);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'edit_region', $item_editRegion);

        my $item_editRegionmap = Gtk3::ImageMenuItem->new('Edit _regionmap...');
        my $img_editRegionmap = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editRegionmap->set_image($img_editRegionmap);
        $item_editRegionmap->signal_connect('activate' => sub {

            # Open an 'edit' window for the regionmap
            $self->createFreeWin(
                'Games::Axmud::EditWin::Regionmap',
                $self,
                $self->session,
                'Edit \'' . $self->currentRegionmap->name . '\' regionmap',
                $self->currentRegionmap,
                FALSE,          # Not temporary
            );
        });
        $column_regions->append($item_editRegionmap);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'edit_regionmap', $item_editRegionmap);

        $column_regions->append(Gtk3::SeparatorMenuItem->new());    # Separator

            # 'Region list' submenu
            my $subMenu_regionsTree = Gtk3::Menu->new();

            my $item_resetList = Gtk3::MenuItem->new('_Reset region list');
            $item_resetList->signal_connect('activate' => sub {

                $self->worldModelObj->resetRegionList();
            });
            $subMenu_regionsTree->append($item_resetList);

            my $item_reverseList = Gtk3::MenuItem->new('Re_verse region list');
            $item_reverseList->signal_connect('activate' => sub {

                $self->worldModelObj->reverseRegionList();
            });
            $subMenu_regionsTree->append($item_reverseList);

            $subMenu_regionsTree->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_moveCurrentRegion = Gtk3::MenuItem->new('_Move current region to top');
            $item_moveCurrentRegion->signal_connect('activate' => sub {

                $self->worldModelObj->moveRegionToTop($self->currentRegionmap);
            });
            $subMenu_regionsTree->append($item_moveCurrentRegion);
            # (Requires $self->currentRegionmap for a region that doesn't have a parent region)
            $self->ivAdd('menuToolItemHash', 'move_region_top', $item_moveCurrentRegion);

            $subMenu_regionsTree->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_identifyRegion = Gtk3::MenuItem->new('_Identify highlighted region');
            $item_identifyRegion->signal_connect('activate' => sub {

                $self->identifyRegionCallback();
            });
            $subMenu_regionsTree->append($item_identifyRegion);
            # (Requires $self->treeViewSelectedLine)
            $self->ivAdd('menuToolItemHash', 'identify_region', $item_identifyRegion);

        my $item_regionsTree = Gtk3::MenuItem->new('Region _list');
        $item_regionsTree->set_submenu($subMenu_regionsTree);
        $column_regions->append($item_regionsTree);

            # 'Current region' submenu
            my $subMenu_currentRegion = Gtk3::Menu->new();

            my $item_renameRegion = Gtk3::MenuItem->new('_Rename region...');
            $item_renameRegion->signal_connect('activate' => sub {

                $self->renameRegionCallback();
            });
            $subMenu_currentRegion->append($item_renameRegion);

            my $item_changeParent = Gtk3::MenuItem->new('_Set parent region...');
            $item_changeParent->signal_connect('activate' => sub {

                $self->changeRegionParentCallback();
            });
            $subMenu_currentRegion->append($item_changeParent);

            $subMenu_currentRegion->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_convertRegionExit = Gtk3::MenuItem->new('_Convert all region exits');
            $item_convertRegionExit->signal_connect('activate' => sub {

                $self->convertRegionExitCallback(TRUE);
            });
            $subMenu_currentRegion->append($item_convertRegionExit);

            my $item_deconvertRegionExit = Gtk3::MenuItem->new(
                '_Deconvert all super-region exits',
            );
            $item_deconvertRegionExit->signal_connect('activate' => sub {

                $self->convertRegionExitCallback(FALSE);
            });
            $subMenu_currentRegion->append($item_deconvertRegionExit);

            $subMenu_currentRegion->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_resetObjectCounts = Gtk3::MenuItem->new('Reset _object counts');
            $item_resetObjectCounts->signal_connect('activate' => sub {

                 # Empty the hashes which store temporary object counts and redraw the region
                 $self->worldModelObj->resetRegionCounts($self->currentRegionmap);
            });
            $subMenu_currentRegion->append($item_resetObjectCounts);

            my $item_removeRoomFlags = Gtk3::MenuItem->new('Remove room _flags...');
            $item_removeRoomFlags->signal_connect('activate' => sub {

                $self->removeRoomFlagsCallback();
            });
            $subMenu_currentRegion->append($item_removeRoomFlags);

        my $item_currentRegion = Gtk3::MenuItem->new('C_urrent region');
        $item_currentRegion->set_submenu($subMenu_currentRegion);
        $column_regions->append($item_currentRegion);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'current_region', $item_currentRegion);

            # 'Pre-drawn regions' submenu
            my $subMenu_preDrawRegion = Gtk3::Menu->new();

            my $item_allowPreDraw = Gtk3::CheckMenuItem->new('_Allow pre-drawing of maps');
            $item_allowPreDraw->set_active($self->worldModelObj->preDrawAllowFlag);
            $item_allowPreDraw->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleFlag(
                    'preDrawAllowFlag',
                    $item_allowPreDraw->get_active(),
                    FALSE,      # Don't call $self->redrawRegions
                    'allow_pre_draw',
                );
            });
            $subMenu_preDrawRegion->append($item_allowPreDraw);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'allow_pre_draw', $item_allowPreDraw);

            my $item_setPreDrawSize = Gtk3::MenuItem->new('_Set minimum region size');
            $item_setPreDrawSize->signal_connect('activate' => sub {

                $self->preDrawSizeCallback();
            });
            $subMenu_preDrawRegion->append($item_setPreDrawSize);

            my $item_setRetainSize = Gtk3::MenuItem->new('Set minimum retention size');
            $item_setRetainSize->signal_connect('activate' => sub {

                $self->preDrawRetainCallback();
            });
            $subMenu_preDrawRegion->append($item_setRetainSize);

            my $item_setPreDrawSpeed = Gtk3::MenuItem->new('Set pre-draw speed');
            $item_setPreDrawSpeed->signal_connect('activate' => sub {

                $self->preDrawSpeedCallback();
            });
            $subMenu_preDrawRegion->append($item_setPreDrawSpeed);

            $subMenu_preDrawRegion->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_redrawRegion = Gtk3::MenuItem->new('Re_draw this region');
            $item_redrawRegion->signal_connect('activate' => sub {

                $self->redrawRegions($self->currentRegionmap, TRUE);
            });
            $subMenu_preDrawRegion->append($item_redrawRegion);
            # (Requires $self->currentRegionmap)
            $self->ivAdd('menuToolItemHash', 'redraw_region', $item_redrawRegion);

            my $item_redrawAllRegions = Gtk3::MenuItem->new('Redraw _all drawn regions');
            $item_redrawAllRegions->signal_connect('activate' => sub {

                $self->redrawRegionsCallback();
            });
            $subMenu_preDrawRegion->append($item_redrawAllRegions);

        my $item_preDrawRegion = Gtk3::MenuItem->new('_Pre-drawn regions');
        $item_preDrawRegion->set_submenu($subMenu_preDrawRegion);
        $column_regions->append($item_preDrawRegion);

        $column_regions->append(Gtk3::SeparatorMenuItem->new());    # Separator

            # 'Colour schemes' submenu
            my $subMenu_regionScheme = Gtk3::Menu->new();

            my $item_addScheme = Gtk3::MenuItem->new('_Add new colour scheme...');
            $item_addScheme->signal_connect('activate' => sub {

                $self->addRegionSchemeCallback();
            });
            $subMenu_regionScheme->append($item_addScheme);

            my $item_editScheme = Gtk3::MenuItem->new('_Edit colour scheme...');
            $item_editScheme->signal_connect('activate' => sub {

                $self->doRegionSchemeCallback('edit');
            });
            $subMenu_regionScheme->append($item_editScheme);

            my $item_renameScheme = Gtk3::MenuItem->new('_Rename colour scheme...');
            $item_renameScheme->signal_connect('activate' => sub {

                $self->doRegionSchemeCallback('rename');
            });
            $subMenu_regionScheme->append($item_renameScheme);

            my $item_deleteScheme = Gtk3::MenuItem->new('_Delete colour scheme...');
            $item_deleteScheme->signal_connect('activate' => sub {

                $self->doRegionSchemeCallback('delete');
            });
            $subMenu_regionScheme->append($item_deleteScheme);

            $subMenu_regionScheme->append(Gtk3::SeparatorMenuItem->new());  # Separator

                # 'This region' sub-submenu
                my $subSubMenu_thisRegionScheme = Gtk3::Menu->new();

                my $item_attachScheme = Gtk3::MenuItem->new('_Attach colour scheme...');
                $item_attachScheme->signal_connect('activate' => sub {

                    $self->attachRegionSchemeCallback();
                });
                $subSubMenu_thisRegionScheme->append($item_attachScheme);
                # (Requires $self->currentRegionmap and at least one non-default region colour
                #   schemes)
                $self->ivAdd('menuToolItemHash', 'attach_region_scheme', $item_attachScheme);

                my $item_detachScheme = Gtk3::MenuItem->new('_Detach colour scheme');
                $item_detachScheme->signal_connect('activate' => sub {

                    $self->detachRegionSchemeCallback();
                });
                $subSubMenu_thisRegionScheme->append($item_detachScheme);
                # (Requires $self->currentRegionmap with a defined ->regionScheme IV)
                $self->ivAdd('menuToolItemHash', 'detach_region_scheme', $item_detachScheme);

                $subSubMenu_thisRegionScheme->append(Gtk3::SeparatorMenuItem->new());  # Separator

                my $item_editThisScheme = Gtk3::MenuItem->new('_Edit colour scheme...');
                $item_editThisScheme->signal_connect('activate' => sub {

                    $self->doRegionSchemeCallback('edit', $self->currentRegionmap);
                });
                $subSubMenu_thisRegionScheme->append($item_editThisScheme);

            my $item_thisRegionScheme = Gtk3::MenuItem->new('_Current region');
            $item_thisRegionScheme->set_submenu($subSubMenu_thisRegionScheme);
            $subMenu_regionScheme->append($item_thisRegionScheme);
            # (Requires $self->currentRegionmap)
            $self->ivAdd('menuToolItemHash', 'this_region_scheme', $item_thisRegionScheme);

        my $item_colourScheme = Gtk3::MenuItem->new('Colour sc_hemes');
        $item_colourScheme->set_submenu($subMenu_regionScheme);
        $column_regions->append($item_colourScheme);

            # 'Background colours' submenu
            my $subMenu_bgColours = Gtk3::Menu->new();

            my $item_removeBGAll = Gtk3::MenuItem->new('_Remove colour...');
            $item_removeBGAll->signal_connect('activate' => sub {

                $self->removeBGColourCallback();
            });
            $subMenu_bgColours->append($item_removeBGAll);

            my $item_removeBGColour = Gtk3::MenuItem->new('Remove _all colours');
            $item_removeBGColour->signal_connect('activate' => sub {

                $self->removeBGAllCallback();
            });
            $subMenu_bgColours->append($item_removeBGColour);

        my $item_bgColours = Gtk3::MenuItem->new('_Background colours');
        $item_bgColours->set_submenu($subMenu_bgColours);
        $column_regions->append($item_bgColours);
        # (Requires $self->currentRegionmap whose ->gridColourBlockHash and/or ->gridColourObjHash
        #   is not empty)
        $self->ivAdd('menuToolItemHash', 'empty_bg_colours', $item_bgColours);

        $column_regions->append(Gtk3::SeparatorMenuItem->new());    # Separator

            # 'Recalculate paths' submenu
            my $subMenu_recalculatePaths = Gtk3::Menu->new();

            my $item_recalculateInCurrentRegion = Gtk3::MenuItem->new('In _current region');
            $item_recalculateInCurrentRegion->signal_connect('activate' => sub {

                $self->recalculatePathsCallback('current');
            });
            $subMenu_recalculatePaths->append($item_recalculateInCurrentRegion);
            # (Requires $self->currentRegionmap and a non-empty
            #   self->currentRegionmap->gridRoomHash)
            $self->ivAdd(
                'menuToolItemHash',
                'recalculate_in_region',
                $item_recalculateInCurrentRegion,
            );

            my $item_recalculateSelectRegion = Gtk3::MenuItem->new('In _region...');
            $item_recalculateSelectRegion->signal_connect('activate' => sub {

                $self->recalculatePathsCallback('select');
            });
            $subMenu_recalculatePaths->append($item_recalculateSelectRegion);

            my $item_recalculateAllRegions = Gtk3::MenuItem->new('In _all regions');
            $item_recalculateAllRegions->signal_connect('activate' => sub {

                $self->recalculatePathsCallback('all');
            });
            $subMenu_recalculatePaths->append($item_recalculateAllRegions);

            $subMenu_recalculatePaths->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_recalculateFromExit = Gtk3::MenuItem->new('For selected _exit');
            $item_recalculateFromExit->signal_connect('activate' => sub {

                $self->recalculatePathsCallback('exit');
            });
            $subMenu_recalculatePaths->append($item_recalculateFromExit);
            # (Requires $self->currentRegionmap and a $self->selectedExit which is a super-region
            #   exit)
            $self->ivAdd('menuToolItemHash', 'recalculate_from_exit', $item_recalculateFromExit);

        my $item_recalculatePaths = Gtk3::MenuItem->new('Re_calculate region paths');
        $item_recalculatePaths->set_submenu($subMenu_recalculatePaths);
        $column_regions->append($item_recalculatePaths);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'recalculate_paths', $item_recalculatePaths);

            # 'Locate current room' submenu
            my $subMenu_locateCurrentRoom = Gtk3::Menu->new();

            my $item_locateInCurrentRegion = Gtk3::MenuItem->new('In _current region');
            $item_locateInCurrentRegion->signal_connect('activate' => sub {

                $self->locateCurrentRoomCallback('current');
            });
            $subMenu_locateCurrentRoom->append($item_locateInCurrentRegion);
            # (Requires $self->currentRegionmap and a non-empty GA::Obj::Regionmap->gridRoomHash)
            $self->ivAdd('menuToolItemHash', 'locate_room_in_current', $item_locateInCurrentRegion);

            my $item_locateInSelectRegion = Gtk3::MenuItem->new('In _region...');
            $item_locateInSelectRegion->signal_connect('activate' => sub {

                $self->locateCurrentRoomCallback('select');
            });
            $subMenu_locateCurrentRoom->append($item_locateInSelectRegion);

            my $item_locateInAllRegions = Gtk3::MenuItem->new('In _all regions');
            $item_locateInAllRegions->signal_connect('activate' => sub {

                $self->locateCurrentRoomCallback('all');
            });
            $subMenu_locateCurrentRoom->append($item_locateInAllRegions);

        my $item_locateCurrentRoom = Gtk3::ImageMenuItem->new('L_ocate current room');
        my $img_locateCurrentRoom = Gtk3::Image->new_from_stock('gtk-find', 'menu');
        $item_locateCurrentRoom->set_image($img_locateCurrentRoom);
        $item_locateCurrentRoom->set_submenu($subMenu_locateCurrentRoom);
        $column_regions->append($item_locateCurrentRoom);

            # 'Screenshots' submenu
            my $subMenu_screenshots = Gtk3::Menu->new();

            my $item_visibleScreenshot = Gtk3::MenuItem->new('_Visible map');
            $item_visibleScreenshot->signal_connect('activate' => sub {

                $self->regionScreenshotCallback('visible');
            });
            $subMenu_screenshots->append($item_visibleScreenshot);

            my $item_occupiedScreenshot = Gtk3::MenuItem->new('_Occupied portion');
            $item_occupiedScreenshot->signal_connect('activate' => sub {

                $self->regionScreenshotCallback('occupied');
            });
            $subMenu_screenshots->append($item_occupiedScreenshot);

            my $item_wholeScreenshot = Gtk3::MenuItem->new('_Whole region');
            $item_wholeScreenshot->signal_connect('activate' => sub {

                $self->regionScreenshotCallback('whole');
            });
            $subMenu_screenshots->append($item_wholeScreenshot);

        my $item_screenshots = Gtk3::MenuItem->new('Take _screenshot');
        $item_screenshots->set_submenu($subMenu_screenshots);
        $column_regions->append($item_screenshots);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'screenshots', $item_screenshots);

        $column_regions->append(Gtk3::SeparatorMenuItem->new());    # Separator

        my $item_emptyRegion = Gtk3::MenuItem->new('E_mpty region');
        $item_emptyRegion->signal_connect('activate' => sub {

            $self->emptyRegionCallback();
        });
        $column_regions->append($item_emptyRegion);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'empty_region', $item_emptyRegion);

        my $item_deleteRegion = Gtk3::ImageMenuItem->new('_Delete region');
        my $img_deleteRegion = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_deleteRegion->set_image($img_deleteRegion);
        $item_deleteRegion->signal_connect('activate' => sub {

            $self->deleteRegionCallback();
        });
        $column_regions->append($item_deleteRegion);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'delete_region', $item_deleteRegion);

        my $item_deleteTempRegion = Gtk3::ImageMenuItem->new('Delete temporar_y regions');
        my $img_deleteTempRegion = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_deleteTempRegion->set_image($img_deleteTempRegion);
        $item_deleteTempRegion->signal_connect('activate' => sub {

            $self->deleteTempRegionsCallback();
        });
        $column_regions->append($item_deleteTempRegion);

        # Setup complete
        return $column_regions;
    }

    sub enableRoomsColumn {

        # Called by $self->enableMenu
        # Sets up the 'Rooms' column of the Automapper window's menu bar
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableRoomsColumn', @_);
        }

        # Set up column
        my $column_rooms = Gtk3::Menu->new();
        if (! $column_rooms) {

            return undef;
        }

        my $item_setCurrentRoom = Gtk3::MenuItem->new('_Set current room');
        $item_setCurrentRoom->signal_connect('activate' => sub {

            $self->mapObj->setCurrentRoom($self->selectedRoom);
        });
        $column_rooms->append($item_setCurrentRoom);
        # (Requires $self->currentRegionmap & $self->selectedRoom)
        $self->ivAdd('menuToolItemHash', 'set_current_room', $item_setCurrentRoom);

        my $item_unsetCurrentRoom = Gtk3::MenuItem->new('_Unset current room');
        $item_unsetCurrentRoom->signal_connect('activate' => sub {

            # This function automatically redraws the room
            $self->mapObj->setCurrentRoom();
        });
        $column_rooms->append($item_unsetCurrentRoom);
        # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
        $self->ivAdd('menuToolItemHash', 'unset_current_room', $item_unsetCurrentRoom);

        $column_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

            # 'Locator task' submenu
            my $subMenu_locatorTask = Gtk3::Menu->new();

            my $item_resetLocator = Gtk3::MenuItem->new('_Reset Locator');
            $item_resetLocator->signal_connect('activate' => sub {

                $self->resetLocatorCallback();
            });
            $subMenu_locatorTask->append($item_resetLocator);
            # (Requires $self->currentRegionmap)
            $self->ivAdd('menuToolItemHash', 'reset_locator', $item_resetLocator);

            my $item_updateLocator = Gtk3::MenuItem->new('_Update Locator');
            $item_updateLocator->signal_connect('activate' => sub {

                # Update the Locator task
                $self->mapObj->updateLocator();
            });
            $subMenu_locatorTask->append($item_updateLocator);
            # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
            $self->ivAdd('menuToolItemHash', 'update_locator', $item_updateLocator);

            $subMenu_locatorTask->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_setFacing = Gtk3::MenuItem->new('_Set facing direction...');
            $item_setFacing->signal_connect('activate' => sub {

                $self->setFacingCallback();
            });
            $subMenu_locatorTask->append($item_setFacing);

            my $item_resetFacing = Gtk3::MenuItem->new('R_eset facing direction...');
            $item_resetFacing->signal_connect('activate' => sub {

                $self->resetFacingCallback();
            });
            $subMenu_locatorTask->append($item_resetFacing);

            $subMenu_locatorTask->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_viewLocatorRoom = Gtk3::MenuItem->new('_View Locator room...');
            $item_viewLocatorRoom->signal_connect('activate' => sub {

                $self->editLocatorRoomCallback();
            });
            $subMenu_locatorTask->append($item_viewLocatorRoom);

        my $item_locatorTask = Gtk3::MenuItem->new('_Locator task');
        $item_locatorTask->set_submenu($subMenu_locatorTask);
        $column_rooms->append($item_locatorTask);

            # 'Pathfinding' submenu
            my $subMenu_pathFinding = Gtk3::Menu->new();

            my $item_highlightPath = Gtk3::MenuItem->new('_Highlight path');
            $item_highlightPath->signal_connect('activate' => sub {

                $self->processPathCallback('select_room');
            });
            $subMenu_pathFinding->append($item_highlightPath);
            # (Requires $self->currentRegionmap, $self->mapObj->currentRoom and $self->selectedRoom)
            $self->ivAdd('menuToolItemHash', 'path_finding_highlight', $item_highlightPath);

            my $item_displayPath = Gtk3::MenuItem->new('_Edit path...');
            $item_displayPath->signal_connect('activate' => sub {

                $self->processPathCallback('pref_win');
            });
            $subMenu_pathFinding->append($item_displayPath);
            # (Requires $self->currentRegionmap, $self->mapObj->currentRoom and $self->selectedRoom)
            $self->ivAdd('menuToolItemHash', 'path_finding_edit', $item_displayPath);

            my $item_goToRoom = Gtk3::MenuItem->new('_Go to room');
            $item_goToRoom->signal_connect('activate' => sub {

                $self->processPathCallback('send_char');
            });
            $subMenu_pathFinding->append($item_goToRoom);
            # (Requires $self->currentRegionmap, $self->mapObj->currentRoom and $self->selectedRoom)
            $self->ivAdd('menuToolItemHash', 'path_finding_go', $item_goToRoom);

            $subMenu_pathFinding->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_allowPostProcessing = Gtk3::CheckMenuItem->new('_Allow post-processing');
            $item_allowPostProcessing->set_active($self->worldModelObj->postProcessingFlag);
            $item_allowPostProcessing->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleFlag(
                    'postProcessingFlag',
                    $item_allowPostProcessing->get_active(),
                    FALSE,      # Don't call $self->redrawRegions
                    'allow_post_process',
                );
            });
            $subMenu_pathFinding->append($item_allowPostProcessing);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'allow_post_process', $item_allowPostProcessing);

            my $item_avoidHazardousRooms = Gtk3::CheckMenuItem->new('A_void hazardous rooms');
            $item_avoidHazardousRooms->set_active($self->worldModelObj->avoidHazardsFlag);
            $item_avoidHazardousRooms->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleFlag(
                    'avoidHazardsFlag',
                    $item_avoidHazardousRooms->get_active(),
                    FALSE,      # Don't call $self->redrawRegions
                    'allow_hazard_rooms',
                );
            });
            $subMenu_pathFinding->append($item_avoidHazardousRooms);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'allow_hazard_rooms', $item_avoidHazardousRooms);

            my $item_doubleClickPathFind = Gtk3::CheckMenuItem->new(
                'Allow _double-click moves',
            );
            $item_doubleClickPathFind->set_active($self->worldModelObj->quickPathFindFlag);
            $item_doubleClickPathFind->signal_connect('toggled' => sub {

                $self->worldModelObj->toggleFlag(
                    'quickPathFindFlag',
                    $item_doubleClickPathFind->get_active(),
                    FALSE,      # Don't call $self->redrawRegions
                    'allow_quick_path_find',
                );
            });
            $subMenu_pathFinding->append($item_doubleClickPathFind);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'allow_quick_path_find', $item_doubleClickPathFind);

            $subMenu_pathFinding->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_adjacentMode = Gtk3::MenuItem->new('_Set adjacent regions mode...');
            $item_adjacentMode->signal_connect('activate' => sub {

                $self->adjacentModeCallback();
            });
            $subMenu_pathFinding->append($item_adjacentMode);

        my $item_pathFinding = Gtk3::MenuItem->new('_Pathfinding');
        $item_pathFinding->set_submenu($subMenu_pathFinding);
        $column_rooms->append($item_pathFinding);

            # 'Move rooms/labels' submenu
            my $subMenu_moveRooms = Gtk3::Menu->new();

            my $item_moveSelected = Gtk3::MenuItem->new('Move in _direction...');
            $item_moveSelected->signal_connect('activate' => sub {

                $self->moveSelectedRoomsCallback();
            });
            $subMenu_moveRooms->append($item_moveSelected);
            # (Requires $self->currentRegionmap and one or more selected rooms)
            $self->ivAdd('menuToolItemHash', 'move_rooms_dir', $item_moveSelected);

            my $item_moveSelectedToClick = Gtk3::MenuItem->new('Move to _click');
            $item_moveSelectedToClick->signal_connect('activate' => sub {

                # Set the free clicking mode: $self->mouseClickEvent will move the objects when the
                #   user next clicks on an empty part of the map
                $self->set_freeClickMode('move_room');
            });
            $subMenu_moveRooms->append($item_moveSelectedToClick);
            # (Requires $self->currentRegionmap and one or more selected rooms)
            $self->ivAdd('menuToolItemHash', 'move_rooms_click', $item_moveSelectedToClick);

            $subMenu_moveRooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

                # 'Transfer to region' sub-submenu
                my $subSubMenu_transferRegion = Gtk3::Menu->new();

                if ($self->recentRegionList) {

                    foreach my $name ($self->recentRegionList) {

                        my $item_regionName = Gtk3::MenuItem->new($name);
                        $item_regionName->signal_connect('activate' => sub {

                            $self->transferSelectedRoomsCallback($name);
                        });
                        $subSubMenu_transferRegion->append($item_regionName);
                    }

                } else {

                    my $item_regionNone = Gtk3::MenuItem->new('(No recent regions)');
                    $item_regionNone->set_sensitive(FALSE);
                    $subSubMenu_transferRegion->append($item_regionNone);
                }

                $subSubMenu_transferRegion->append(Gtk3::SeparatorMenuItem->new());  # Separator

                my $item_transferSelect = Gtk3::MenuItem->new('Select region...');
                $item_transferSelect->signal_connect('activate' => sub {

                    $self->transferSelectedRoomsCallback();
                });
                $subSubMenu_transferRegion->append($item_transferSelect);

            my $item_transferRegion = Gtk3::MenuItem->new('_Transfer to region');
            $item_transferRegion->set_submenu($subSubMenu_transferRegion);
            $subMenu_moveRooms->append($item_transferRegion);
            # (Requires $self->currentRegionmap, one or more selected rooms and at least two regions
            #   in the world model)
            $self->ivAdd('menuToolItemHash', 'transfer_to_region', $item_transferRegion);

            $subMenu_moveRooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_mergeMoveRooms = Gtk3::MenuItem->new('_Merge/move rooms');
            $item_mergeMoveRooms->signal_connect('activate' => sub {

                $self->doMerge($self->mapObj->currentRoom);
            });
            $subMenu_moveRooms->append($item_mergeMoveRooms);
            # (Requires $self->currentRegionmap, a current room and the automapper object being set
            #   up to perform a merge)
            $self->ivAdd('menuToolItemHash', 'move_merge_rooms', $item_mergeMoveRooms);

        my $item_moveRooms = Gtk3::MenuItem->new('_Move rooms/labels');
        $item_moveRooms->set_submenu($subMenu_moveRooms);
        $column_rooms->append($item_moveRooms);
        # (Requires $self->currentRegionmap and EITHER one or more selected rooms OR a current room
        #   and the automapper being set up to perform a merge)
        $self->ivAdd('menuToolItemHash', 'move_rooms_labels', $item_moveRooms);

        $column_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

            # 'Add room' submenu
            my $subMenu_addRoom = Gtk3::Menu->new();

            my $item_addFirstRoom = Gtk3::MenuItem->new('Add _first room');
            $item_addFirstRoom->signal_connect('activate' => sub {

                $self->addFirstRoomCallback();
            });
            $subMenu_addRoom->append($item_addFirstRoom);
            # (Requires $self->currentRegionmap & an empty $self->currentRegionmap->gridRoomHash)
            $self->ivAdd('menuToolItemHash', 'add_first_room', $item_addFirstRoom);

            my $item_addRoomAtClick = Gtk3::MenuItem->new('Add room at _click');
            $item_addRoomAtClick->signal_connect('activate' => sub {

                # Set the free clicking mode: $self->mouseClickEvent will create the new room when
                #   the user next clicks on an empty part of the map
                if ($self->currentRegionmap) {

                    $self->set_freeClickMode('add_room');
                }
            });
            $subMenu_addRoom->append($item_addRoomAtClick);

            my $item_addRoomAtBlock = Gtk3::MenuItem->new('Add room at _block...');
            $item_addRoomAtBlock->signal_connect('activate' => sub {

                $self->addRoomAtBlockCallback();
            });
            $subMenu_addRoom->append($item_addRoomAtBlock);

        my $item_addRoom = Gtk3::ImageMenuItem->new('Add _room');
        my $img_addRoom = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_addRoom->set_image($img_addRoom);
        $item_addRoom->set_submenu($subMenu_addRoom);
        $column_rooms->append($item_addRoom);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'add_room', $item_addRoom);

            # 'Add pattern' submenu
            my $subMenu_exitPatterns = Gtk3::Menu->new();

            my $item_addFailedExitWorld = Gtk3::MenuItem->new('Add failed exit to _world...');
            $item_addFailedExitWorld->signal_connect('activate' => sub {

                $self->addFailedExitCallback(TRUE);
            });
            $subMenu_exitPatterns->append($item_addFailedExitWorld);

            my $item_addFailedExitRoom = Gtk3::MenuItem->new('Add failed exit to current _room...');
            $item_addFailedExitRoom->signal_connect('activate' => sub {

                $self->addFailedExitCallback(FALSE, $self->mapObj->currentRoom);
            });
            $subMenu_exitPatterns->append($item_addFailedExitRoom);
            # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
            $self->ivAdd('menuToolItemHash', 'add_failed_room', $item_addFailedExitRoom);

            $subMenu_exitPatterns->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_addInvoluntaryExitRoom = Gtk3::MenuItem->new(
                'Add _involuntary exit to current room...',
            );
            $item_addInvoluntaryExitRoom->signal_connect('activate' => sub {

                $self->addInvoluntaryExitCallback($self->mapObj->currentRoom);
            });
            $subMenu_exitPatterns->append($item_addInvoluntaryExitRoom);
            # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
            $self->ivAdd('menuToolItemHash', 'add_involuntary_exit', $item_addInvoluntaryExitRoom);

            my $item_addRepulseExitRoom = Gtk3::MenuItem->new(
                'Add r_epulse exit to current room...',
            );
            $item_addRepulseExitRoom->signal_connect('activate' => sub {

                $self->addRepulseExitCallback($self->mapObj->currentRoom);
            });
            $subMenu_exitPatterns->append($item_addRepulseExitRoom);
            # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
            $self->ivAdd('menuToolItemHash', 'add_repulse_exit', $item_addRepulseExitRoom);

            $subMenu_exitPatterns->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_addSpecialDepartRoom = Gtk3::MenuItem->new(
                'Add _special departure to current room...',
            );
            $item_addSpecialDepartRoom->signal_connect('activate' => sub {

                $self->addSpecialDepartureCallback($self->mapObj->currentRoom);
            });
            $subMenu_exitPatterns->append($item_addSpecialDepartRoom);
            # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
            $self->ivAdd('menuToolItemHash', 'add_special_depart', $item_addSpecialDepartRoom);

            my $item_addUnspecifiedRoom = Gtk3::MenuItem->new(
                'Add _unspecified room pattern...',
            );
            $item_addUnspecifiedRoom->signal_connect('activate' => sub {

                $self->addUnspecifiedPatternCallback($self->mapObj->currentRoom);
            });
            $subMenu_exitPatterns->append($item_addUnspecifiedRoom);
            # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
            $self->ivAdd('menuToolItemHash', 'add_unspecified_pattern', $item_addUnspecifiedRoom);

        my $item_exitPatterns = Gtk3::MenuItem->new('Add p_attern');
        $item_exitPatterns->set_submenu($subMenu_exitPatterns);
        $column_rooms->append($item_exitPatterns);

            # 'Add to model' submenu
            my $subMenu_addToModel = Gtk3::Menu->new();

            my $item_addRoomContents = Gtk3::MenuItem->new('Add _contents...');
            $item_addRoomContents->signal_connect('activate' => sub {

                $self->addContentsCallback(FALSE);
            });
            $subMenu_addToModel->append($item_addRoomContents);
            # Requires $self->currentRegionmap, $self->mapObj->currentRoom
            $self->ivAdd('menuToolItemHash', 'add_room_contents', $item_addRoomContents);

            my $item_addContentsString = Gtk3::MenuItem->new('Add c_ontents from string...');
            $item_addContentsString->signal_connect('activate' => sub {

                $self->addContentsCallback(TRUE);
            });
            $subMenu_addToModel->append($item_addContentsString);
            # Requires $self->currentRegionmap, $self->selectedRoom
            $self->ivAdd('menuToolItemHash', 'add_contents_string', $item_addContentsString);

            $subMenu_addToModel->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_addHiddenObj = Gtk3::MenuItem->new('Add _hidden object...');
            $item_addHiddenObj->signal_connect('activate' => sub {

                $self->addHiddenObjCallback(FALSE);
            });
            $subMenu_addToModel->append($item_addHiddenObj);
            # Requires $self->currentRegionmap, $self->mapObj->currentRoom
            $self->ivAdd('menuToolItemHash', 'add_hidden_object', $item_addHiddenObj);

            my $item_addHiddenString = Gtk3::MenuItem->new('Add h_idden object from string...');
            $item_addHiddenString->signal_connect('activate' => sub {

                $self->addHiddenObjCallback(TRUE);
            });
            $subMenu_addToModel->append($item_addHiddenString);
            # Requires $self->currentRegionmap, $self->selectedRoom
            $self->ivAdd('menuToolItemHash', 'add_hidden_string', $item_addHiddenString);

            $subMenu_addToModel->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_addSearchResult = Gtk3::MenuItem->new('Add _search result...');
            $item_addSearchResult->signal_connect('activate' => sub {

                $self->addSearchResultCallback();
            });
            $subMenu_addToModel->append($item_addSearchResult);
            # Requires $self->currentRegionmap and $self->mapObj->currentRoom
            $self->ivAdd('menuToolItemHash', 'add_search_result', $item_addSearchResult);

        my $item_addToModel = Gtk3::MenuItem->new('Add to m_odel');
        $item_addToModel->set_submenu($subMenu_addToModel);
        $column_rooms->append($item_addToModel);
        # Requires $self->currentRegionmap and either $self->mapObj->currentRoom or
        #   $self->selectedRoom
        $self->ivAdd('menuToolItemHash', 'add_to_model', $item_addToModel);

        $column_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

            # 'Add/set exits' submenu
            my $subMenu_setExits = Gtk3::Menu->new();

            my $item_addNormal = Gtk3::MenuItem->new('Add _normal exit...');
            $item_addNormal->signal_connect('activate' => sub {

                $self->addExitCallback(FALSE);  # FALSE - not a hidden exit
            });
            $subMenu_setExits->append($item_addNormal);
            # (Requires $self->currentRegionmap and a $self->selectedRoom whose ->wildMode is not
            #   'wild' - the value 'border' is ok, though)
            $self->ivAdd('menuToolItemHash', 'add_normal_exit', $item_addNormal);

            my $item_addHiddenExit = Gtk3::MenuItem->new('Add _hidden exit...');
            $item_addHiddenExit->signal_connect('activate' => sub {

                $self->addExitCallback(TRUE);   # TRUE - a hidden exit
            });
            $subMenu_setExits->append($item_addHiddenExit);
            # (Requires $self->currentRegionmap and a $self->selectedRoom whose ->wildMode is not
            #   'wild' - the value 'border' is ok, though)
            $self->ivAdd('menuToolItemHash', 'add_hidden_exit', $item_addHiddenExit);

            $subMenu_setExits->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_addMultiple = Gtk3::MenuItem->new('Add _multiple exits...');
            $item_addMultiple->signal_connect('activate' => sub {

                $self->addMultipleExitsCallback();
            });
            $subMenu_setExits->append($item_addMultiple);
            # (Requires $self->currentRegionmap and one or more selected rooms)
            $self->ivAdd('menuToolItemHash', 'add_multiple_exits', $item_addMultiple);

            $subMenu_setExits->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_removeChecked = Gtk3::MenuItem->new('Remove _checked direction...');
            $item_removeChecked->signal_connect('activate' => sub {

                $self->removeCheckedDirCallback(FALSE);
            });
            $subMenu_setExits->append($item_removeChecked);
            # (Require a current regionmap, a single selected room that has one or more checked
            #   directions)
            $self->ivAdd('menuToolItemHash', 'remove_checked', $item_removeChecked);

            my $item_removeCheckedAll = Gtk3::MenuItem->new('Remove _all checked directions');
            $item_removeCheckedAll->signal_connect('activate' => sub {

                $self->removeCheckedDirCallback(TRUE);
            });
            $subMenu_setExits->append($item_removeCheckedAll);
            # (Require a current regionmap, a single selected room that has one or more checked
            #   directions)
            $self->ivAdd('menuToolItemHash', 'remove_checked_all', $item_removeCheckedAll);

            $subMenu_setExits->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_markNormal = Gtk3::MenuItem->new('Mark room(s) as n_ormal');
            $item_markNormal->signal_connect('activate' => sub {

                $self->setWildCallback('normal');
            });
            $subMenu_setExits->append($item_markNormal);
            # (Require a current regionmap and one or more selected rooms)
            $self->ivAdd('menuToolItemHash', 'wilderness_normal', $item_markNormal);

            my $item_markWild = Gtk3::MenuItem->new('Mark room(s) as _wilderness');
            $item_markWild->signal_connect('activate' => sub {

                $self->setWildCallback('wild');
            });
            $subMenu_setExits->append($item_markWild);
            # (Require a current regionmap, one or more selected rooms and
            #   $self->session->currentWorld->basicMappingFlag to be FALSE)
            $self->ivAdd('menuToolItemHash', 'wilderness_wild', $item_markWild);

            my $item_markBorder = Gtk3::MenuItem->new('Mark room(s) as wilderness _border');
            $item_markBorder->signal_connect('activate' => sub {

                $self->setWildCallback('border');
            });
            $subMenu_setExits->append($item_markBorder);
            # (Require a current regionmap, one or more selected rooms and
            #   $self->session->currentWorld->basicMappingFlag to be FALSE)
            $self->ivAdd('menuToolItemHash', 'wilderness_border', $item_markBorder);

        my $item_setExits = Gtk3::ImageMenuItem->new('Add/set _exits');
        my $img_setExits = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_setExits->set_image($img_setExits);
        $item_setExits->set_submenu($subMenu_setExits);
        $column_rooms->append($item_setExits);
        # (Require a current regionmap and one or more selected rooms)
        $self->ivAdd('menuToolItemHash', 'set_exits', $item_setExits);

        my $item_selectExit = Gtk3::MenuItem->new('Select e_xit in room...');
        $item_selectExit->signal_connect('activate' => sub {

            $self->selectExitCallback();
        });
        $column_rooms->append($item_selectExit);
        # (Requires $self->currentRegionmap & $self->selectedRoom)
        $self->ivAdd('menuToolItemHash', 'select_exit', $item_selectExit);

        $column_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_editRoom = Gtk3::ImageMenuItem->new('Ed_it room...');
        my $img_editRoom = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editRoom->set_image($img_editRoom);
        $item_editRoom->signal_connect('activate' => sub {

            # Open the room's 'edit' window
            $self->createFreeWin(
                'Games::Axmud::EditWin::ModelObj::Room',
                $self,
                $self->session,
                'Edit ' . $self->selectedRoom->category . ' model object',
                $self->selectedRoom,
                FALSE,                          # Not temporary
            );
        });
        $column_rooms->append($item_editRoom);
        # (Requires $self->currentRegionmap & $self->selectedRoom)
        $self->ivAdd('menuToolItemHash', 'edit_room', $item_editRoom);

            # 'Room text' submenu
            my $subMenu_roomText = Gtk3::Menu->new();

            my $item_setRoomTag = Gtk3::MenuItem->new('Set room _tag...');
            $item_setRoomTag->signal_connect('activate' => sub {

                $self->setRoomTagCallback();
            });
            $subMenu_roomText->append($item_setRoomTag);
            # (Requires $self->currentRegionmap and either $self->selectedRoom or
            #   $self->selectedRoomTag)
            $self->ivAdd('menuToolItemHash', 'set_room_tag', $item_setRoomTag);

            my $item_setGuild = Gtk3::MenuItem->new('Set room _guild...');
            $item_setGuild->signal_connect('activate' => sub {

                $self->setRoomGuildCallback();
            });
            $subMenu_roomText->append($item_setGuild);
            # (Requires $self->currentRegionmap and one or more of $self->selectedRoom,
            #   $self->selectedRoomHash, $self->selectedRoomGuild, $self->selectedRoomGuildHash)
            $self->ivAdd('menuToolItemHash', 'set_room_guild', $item_setGuild);

            $subMenu_roomText->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_resetPositions = Gtk3::MenuItem->new('_Reset text positions');
            $item_resetPositions->signal_connect('activate' => sub {

                $self->resetRoomOffsetsCallback();
            });
            $subMenu_roomText->append($item_resetPositions);
            # (Requires $self->currentRegionmap & $self->selectedRoom)
            $self->ivAdd('menuToolItemHash', 'reset_positions', $item_resetPositions);

            $subMenu_roomText->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_roomText = Gtk3::MenuItem->new('Set room _text');
        $item_roomText->set_submenu($subMenu_roomText);
        $column_rooms->append($item_roomText);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'room_text', $item_roomText);

            # 'Toggle room flag' submenu
            my $subMenu_toggleRoomFlag = Gtk3::Menu->new();

            if ($self->worldModelObj->roomFlagShowMode eq 'default') {

                # Show all room flags, sorted by filter
                foreach my $filter ($axmud::CLIENT->constRoomFilterList) {

                    # A sub-sub menu for $filter
                    my $subSubMenu_filter = Gtk3::Menu->new();

                    my @nameList = $self->worldModelObj->getRoomFlagsInFilter($filter);
                    foreach my $name (@nameList) {

                        my $obj = $self->worldModelObj->ivShow('roomFlagHash', $name);
                        if ($obj) {

                            my $menuItem = Gtk3::MenuItem->new($obj->descrip);
                            $menuItem->signal_connect('activate' => sub {

                                # Toggle the flags for all selected rooms, redraw them and (if the
                                #   flag is one of the hazardous room flags) recalculate the
                                #   regionmap's paths. The TRUE argument tells the world model to
                                #   redraw the rooms
                                $self->worldModelObj->toggleRoomFlags(
                                    $self->session,
                                    TRUE,
                                    $obj->name,
                                    $self->compileSelectedRooms(),
                                );
                            });
                            $subSubMenu_filter->append($menuItem);
                        }
                    }

                    if (! @nameList) {

                        my $menuItem = Gtk3::MenuItem->new('(No flags in this filter)');
                        $menuItem->set_sensitive(FALSE);
                        $subSubMenu_filter->append($menuItem);
                    }

                    my $menuItem = Gtk3::MenuItem->new(ucfirst($filter));
                    $menuItem->set_submenu($subSubMenu_filter);
                    $subMenu_toggleRoomFlag->append($menuItem);
                }

            } else {

                # Show selected room flags, sorted only by priority
                my %showHash = $self->worldModelObj->getVisibleRoomFlags();
                if (%showHash) {

                    foreach my $obj (sort {$a->priority <=> $b->priority} (values %showHash)) {

                        my $menuItem = Gtk3::MenuItem->new($obj->descrip);
                        $menuItem->signal_connect('activate' => sub {

                            # Toggle the flags for all selected rooms, redraw them and (if the
                            #   flag is one of the hazardous room flags) recalculate the
                            #   regionmap's paths. The TRUE argument tells the world model to
                            #   redraw the rooms
                            $self->worldModelObj->toggleRoomFlags(
                                $self->session,
                                TRUE,
                                $obj->name,
                                $self->compileSelectedRooms(),
                            );
                        });
                        $subMenu_toggleRoomFlag->append($menuItem);
                    }

                } else {

                    my $menuItem = Gtk3::MenuItem->new('(None are marked visible)');
                    $menuItem->set_sensitive(FALSE);
                    $subMenu_toggleRoomFlag->append($menuItem);
                }
            }

        my $item_toggleRoomFlag = Gtk3::MenuItem->new('Toggle room _flags');
        $item_toggleRoomFlag->set_submenu($subMenu_toggleRoomFlag);
        $column_rooms->append($item_toggleRoomFlag);
        # (Requires $self->currentRegionmap & either $self->selectedRoom or
        #   $self->selectedRoomHash)
        $self->ivAdd('menuToolItemHash', 'toggle_room_flag_sub', $item_toggleRoomFlag);

            # 'Other room features' submenu
            my $subMenu_roomFeatures = Gtk3::Menu->new();

                # 'Update character visits' sub-submenu
                my $subSubMenu_updateVisits = Gtk3::Menu->new();

                my $item_increaseSetCurrent = Gtk3::MenuItem->new('Increase & set _current');
                $item_increaseSetCurrent->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('increase');
                    $self->mapObj->setCurrentRoom($self->selectedRoom);
                });
                $subSubMenu_updateVisits->append($item_increaseSetCurrent);
                # (Requires $self->currentRegionmap and $self->selectedRoom)
                $self->ivAdd('menuToolItemHash', 'increase_set_current', $item_increaseSetCurrent);

                $subSubMenu_updateVisits->append(Gtk3::SeparatorMenuItem->new()); # Separator

                my $item_increaseVisits = Gtk3::MenuItem->new('_Increase by one');
                $item_increaseVisits->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('increase');
                });
                $subSubMenu_updateVisits->append($item_increaseVisits);

                my $item_decreaseVisits = Gtk3::MenuItem->new('_Decrease by one');
                $item_decreaseVisits->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('decrease');
                });
                $subSubMenu_updateVisits->append($item_decreaseVisits);

                my $item_manualVisits = Gtk3::MenuItem->new('Set _manually');
                $item_manualVisits->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('manual');
                });
                $subSubMenu_updateVisits->append($item_manualVisits);

                my $item_resetVisits = Gtk3::MenuItem->new('_Reset to zero');
                $item_resetVisits->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('reset');
                });
                $subSubMenu_updateVisits->append($item_resetVisits);

                $subSubMenu_updateVisits->append(Gtk3::SeparatorMenuItem->new()); # Separator

                my $item_toggleGraffiti = Gtk3::MenuItem->new('Toggle _graffiti');
                $item_toggleGraffiti->signal_connect('activate' => sub {

                    $self->toggleGraffitiCallback();
                });
                $subSubMenu_updateVisits->append($item_toggleGraffiti);
                # (Requires $self->currentRegionmap, $self->graffitiModeFlag & one or more selected
                #   rooms)
                $self->ivAdd('menuToolItemHash', 'toggle_graffiti', $item_toggleGraffiti);

            my $item_updateVisits = Gtk3::MenuItem->new('Update character _visits');
            $item_updateVisits->set_submenu($subSubMenu_updateVisits);
            $subMenu_roomFeatures->append($item_updateVisits);
            # (Requires $self->currentRegionmap & either $self->selectedRoom or
            #   $self->selectedRoomHash)
            $self->ivAdd('menuToolItemHash', 'update_visits', $item_updateVisits);

                # 'Room exclusivity' sub-submenu
                my $subSubMenu_exclusivity = Gtk3::Menu->new();

                my $item_toggleExclusivity = Gtk3::MenuItem->new('_Toggle exclusivity');
                $item_toggleExclusivity->signal_connect('activate' => sub {

                    $self->toggleExclusiveProfileCallback();
                });
                $subSubMenu_exclusivity->append($item_toggleExclusivity);
                # (Requires $self->currentRegionmap & either $self->selectedRoom or
                #   $self->selectedRoomHash)
                $self->ivAdd('menuToolItemHash', 'toggle_exclusivity', $item_toggleExclusivity);

                $subSubMenu_exclusivity->append(Gtk3::SeparatorMenuItem->new());  # Separator

                my $item_addExclusiveProf = Gtk3::MenuItem->new('_Add exclusive profile...');
                $item_addExclusiveProf->signal_connect('activate' => sub {

                    $self->addExclusiveProfileCallback();
                });
                $subSubMenu_exclusivity->append($item_addExclusiveProf);
                # (Requires $self->currentRegionmap & $self->selectedRoom)
                $self->ivAdd('menuToolItemHash', 'add_exclusive_prof', $item_addExclusiveProf);

                my $item_clearExclusiveProf = Gtk3::MenuItem->new('_Clear exclusive profiles');
                $item_clearExclusiveProf->signal_connect('activate' => sub {

                    $self->resetExclusiveProfileCallback();
                });
                $subSubMenu_exclusivity->append($item_clearExclusiveProf);
                # (Requires $self->currentRegionmap & either $self->selectedRoom or
                #   $self->selectedRoomHash)
                $self->ivAdd('menuToolItemHash', 'clear_exclusive_profs', $item_clearExclusiveProf);

            my $item_exclusivity = Gtk3::MenuItem->new('Room _exclusivity');
            $item_exclusivity->set_submenu($subSubMenu_exclusivity);
            $subMenu_roomFeatures->append($item_exclusivity);
            # (Requires $self->currentRegionmap & either $self->selectedRoom or
            #   $self->selectedRoomHash)
            $self->ivAdd('menuToolItemHash', 'room_exclusivity', $item_exclusivity);

                # 'Source code' sub-submenu
                my $subSubMenu_sourceCode = Gtk3::Menu->new();

                my $item_setFilePath = Gtk3::MenuItem->new('_Set file path...');
                $item_setFilePath->signal_connect('activate' => sub {

                    $self->setFilePathCallback();
                });
                $subSubMenu_sourceCode->append($item_setFilePath);
                # (Requires $self->currentRegionmap and $self->selectedRoom)
                $self->ivAdd('menuToolItemHash', 'set_file_path', $item_setFilePath);

                my $item_setVirtualArea = Gtk3::MenuItem->new('Set virtual _area...');
                $item_setVirtualArea->signal_connect('activate' => sub {

                    $self->setVirtualAreaCallback(TRUE);
                });
                $subSubMenu_sourceCode->append($item_setVirtualArea);
                # (Requires $self->currentRegionmap & either $self->selectedRoom or
                #   $self->selectedRoomHash)
                $self->ivAdd('menuToolItemHash', 'set_virtual_area', $item_setVirtualArea);

                my $item_resetVirtualArea = Gtk3::MenuItem->new('_Reset virtual area...');
                $item_resetVirtualArea->signal_connect('activate' => sub {

                    $self->setVirtualAreaCallback(FALSE);
                });
                $subSubMenu_sourceCode->append($item_resetVirtualArea);
                # (Requires $self->currentRegionmap & either $self->selectedRoom or
                #   $self->selectedRoomHash)
                $self->ivAdd('menuToolItemHash', 'reset_virtual_area', $item_resetVirtualArea);

                my $item_showSourceCode = Gtk3::MenuItem->new('S_how file paths');
                $item_showSourceCode->signal_connect('activate' => sub {

                    # (Don't use $self->pseudoCmdMode - we want to see the footer messages)
                    $self->session->pseudoCmd('listsourcecode', 'show_all');
                });
                $subSubMenu_sourceCode->append($item_showSourceCode);

                $subSubMenu_sourceCode->append(Gtk3::SeparatorMenuItem->new()); # Separator

                my $item_viewSourceCode = Gtk3::MenuItem->new('_View file...');
                $item_viewSourceCode->signal_connect('activate' => sub {

                    $self->quickFreeWin(
                        'Games::Axmud::OtherWin::SourceCode',
                        $self->session,
                        # Config
                        'model_obj' => $self->selectedRoom,
                    );
                });
                $subSubMenu_sourceCode->append($item_viewSourceCode);
                # (Requires $self->currentRegionmap, $self->selectedRoom &
                #   $self->selectedRoom->sourceCodePath & empty
                #   $self->selectedRoom->virtualAreaPath)
                $self->ivAdd('menuToolItemHash', 'view_source_code', $item_viewSourceCode);

                my $item_editSourceCode = Gtk3::MenuItem->new('_Edit file...');
                $item_editSourceCode->signal_connect('activate' => sub {

                    $self->editFileCallback();
                });
                $subSubMenu_sourceCode->append($item_editSourceCode);
                # (Requires $self->currentRegionmap, $self->selectedRoom &
                #   $self->selectedRoom->sourceCodePath & empty
                #   $self->selectedRoom->virtualAreaPath)
                $self->ivAdd('menuToolItemHash', 'edit_source_code', $item_editSourceCode);

                $subSubMenu_sourceCode->append(Gtk3::SeparatorMenuItem->new()); # Separator

                my $item_viewVirtualArea = Gtk3::MenuItem->new('View virtual area _file...');
                $item_viewVirtualArea->signal_connect('activate' => sub {

                    $self->quickFreeWin(
                        'Games::Axmud::OtherWin::SourceCode',
                        $self->session,
                        # Config
                        'model_obj' => $self->selectedRoom,
                        'virtual_flag' => TRUE,
                    );
                });
                $subSubMenu_sourceCode->append($item_viewVirtualArea);
                # (Requires $self->currentRegionmap, $self->selectedRoom &
                #   $self->selectedRoom->virtualAreaPath
                $self->ivAdd('menuToolItemHash', 'view_virtual_area', $item_viewVirtualArea);

                my $item_editVirtualArea = Gtk3::MenuItem->new('E_dit virtual area file...');
                $item_editVirtualArea->signal_connect('activate' => sub {

                    # Use TRUE to specify that the virtual area file should be opened
                    $self->editFileCallback(TRUE);
                });
                $subSubMenu_sourceCode->append($item_editVirtualArea);
                # (Requires $self->currentRegionmap, $self->selectedRoom &
                #   $self->selectedRoom->virtualAreaPath
                $self->ivAdd('menuToolItemHash', 'edit_virtual_area', $item_editVirtualArea);

            my $item_sourceCode = Gtk3::MenuItem->new('Source _code');
            $item_sourceCode->set_submenu($subSubMenu_sourceCode);
            $subMenu_roomFeatures->append($item_sourceCode);

            $subMenu_roomFeatures->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_setInteriorOffsets = Gtk3::MenuItem->new('_Synchronise grid coordinates...');
            $item_setInteriorOffsets->signal_connect('activate' => sub {

                $self->setInteriorOffsetsCallback();
            });
            $subMenu_roomFeatures->append($item_setInteriorOffsets);

            my $item_resetInteriorOffsets = Gtk3::MenuItem->new('_Reset grid coordinates');
            $item_resetInteriorOffsets->signal_connect('activate' => sub {

                $self->resetInteriorOffsetsCallback();
            });
            $subMenu_roomFeatures->append($item_resetInteriorOffsets);

        my $item_roomFeatures = Gtk3::MenuItem->new('Ot_her room features');
        $item_roomFeatures->set_submenu($subMenu_roomFeatures);
        $column_rooms->append($item_roomFeatures);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'other_room_features', $item_roomFeatures);

        $column_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_deleteRoom = Gtk3::ImageMenuItem->new('_Delete rooms');
        my $img_deleteRoom = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_deleteRoom->set_image($img_deleteRoom);
        $item_deleteRoom->signal_connect('activate' => sub {

            $self->deleteRoomsCallback();
        });
        $column_rooms->append($item_deleteRoom);
        # (Requires $self->currentRegionmap & either $self->selectedRoom or
        #   $self->selectedRoomHash)
        $self->ivAdd('menuToolItemHash', 'delete_room', $item_deleteRoom);

        # Setup complete
        return $column_rooms;
    }

    sub enableExitsColumn {

        # Called by $self->enableMenu
        # Sets up the 'Exits' column of the Automapper window's menu bar
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Local variables
        my @titleList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableExitsColumn', @_);
        }

        # Set up column
        my $column_exits = Gtk3::Menu->new();
        if (! $column_exits) {

            return undef;
        }

            # 'Set direction' submenu
            my $subMenu_setDir = Gtk3::Menu->new();

            my $item_changeDir = Gtk3::MenuItem->new('_Change direction...');
            $item_changeDir->signal_connect('activate' => sub {

                $self->changeDirCallback();
            });
            $subMenu_setDir->append($item_changeDir);
            # (Requires $self->currentRegionmap and $self->selectedExit and
            #   $self->selectedExit->drawMode is 'primary' or 'perm_alloc')
            $self->ivAdd('menuToolItemHash', 'change_direction', $item_changeDir);

            my $item_altDir = Gtk3::MenuItem->new('Set _alternative direction(s)...');
            $item_altDir->signal_connect('activate' => sub {

                $self->setAltDirCallback();
            });
            $subMenu_setDir->append($item_altDir);

        my $item_setDir = Gtk3::MenuItem->new('Set di_rection');
        $item_setDir->set_submenu($subMenu_setDir);
        $column_exits->append($item_setDir);
        # (Requires $self->currentRegionmap and $self->selectedExit)
        $self->ivAdd('menuToolItemHash', 'set_exit_dir', $item_setDir);

        my $item_setAssisted = Gtk3::MenuItem->new('Set assisted _move...');
        $item_setAssisted->signal_connect('activate' => sub {

            $self->setAssistedMoveCallback();
        });
        $column_exits->append($item_setAssisted);
        # (Requires $self->currentRegionmap and $self->selectedExit and
        #   $self->selectedExit->drawMode is 'primary', 'temp_unalloc' or 'perm_alloc')
        $self->ivAdd('menuToolItemHash', 'set_assisted_move', $item_setAssisted);

            # 'Allocate map direction' submenu
            my $subMenu_allocateMapDir = Gtk3::Menu->new();

            my $item_allocatePrimary = Gtk3::MenuItem->new('Choose _direction...');
            $item_allocatePrimary->signal_connect('activate' => sub {

                $self->allocateMapDirCallback();
            });
            $subMenu_allocateMapDir->append($item_allocatePrimary);

            my $item_confirmTwoWay = Gtk3::MenuItem->new('Confirm _two-way exit...');
            $item_confirmTwoWay->signal_connect('activate' => sub {

                $self->confirmTwoWayCallback();
            });
            $subMenu_allocateMapDir->append($item_confirmTwoWay);

        my $item_allocateMapDir = Gtk3::MenuItem->new('_Allocate map direction');
        $item_allocateMapDir->set_submenu($subMenu_allocateMapDir);
        $column_exits->append($item_allocateMapDir);
        # (Requires $self->currentRegionmap and $self->selectedExit and
        #   $self->selectedExit->drawMode is 'temp_alloc' or 'temp_unalloc')
        $self->ivAdd('menuToolItemHash', 'allocate_map_dir', $item_allocateMapDir);

        my $item_allocateShadow = Gtk3::MenuItem->new('Allocate _shadow...');
        $item_allocateShadow->signal_connect('activate' => sub {

            $self->allocateShadowCallback();
        });
        $column_exits->append($item_allocateShadow);
        # (Requires $self->currentRegionmap and $self->selectedExit and
        #   $self->selectedExit->drawMode is 'temp_alloc' or 'temp_unalloc')
        $self->ivAdd('menuToolItemHash', 'allocate_shadow', $item_allocateShadow);

        $column_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_connectExitToClick = Gtk3::MenuItem->new('_Connect to click');
        $item_connectExitToClick->signal_connect('activate' => sub {

            $self->connectToClickCallback();
        });
        $column_exits->append($item_connectExitToClick);
        # (Requires $self->currentRegionmap, $self->selectedExit and
        #   $self->selectedExit->drawMode 'primary', 'temp_unalloc' or 'perm_alloc')
        $self->ivAdd('menuToolItemHash', 'connect_to_click', $item_connectExitToClick);

        my $item_disconnectExit = Gtk3::MenuItem->new('D_isconnect exit');
        $item_disconnectExit->signal_connect('activate' => sub {

            $self->disconnectExitCallback();
        });
        $column_exits->append($item_disconnectExit);
        # (Requires $self->currentRegionmap and $self->selectedExit)
        $self->ivAdd('menuToolItemHash', 'disconnect_exit', $item_disconnectExit);

        $column_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

            # 'Set ornaments' submenu
            my $subMenu_setOrnament = Gtk3::Menu->new();

            # Create a list of exit ornament types, in groups of two, in the form
            #   (menu_item_title, exit_ornament_type)
            @titleList = (
                '_No ornament', 'none',
                '_Openable exit', 'open',
                '_Lockable exit', 'lock',
                '_Pickable exit', 'pick',
                '_Breakable exit', 'break',
                '_Impassable exit', 'impass',
                '_Mystery exit', 'mystery',
            );

            do {

                my ($title, $type);

                $title = shift @titleList;
                $type = shift @titleList;

                my $menuItem = Gtk3::MenuItem->new($title);
                $menuItem->signal_connect('activate' => sub {

                    $self->exitOrnamentCallback($type);
                });
                $subMenu_setOrnament->append($menuItem);

            } until (! @titleList);

            $subMenu_setOrnament->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_setTwinOrnament = Gtk3::CheckMenuItem->new('Also set _twin exits');
            $item_setTwinOrnament->set_active($self->worldModelObj->setTwinOrnamentFlag);
            $item_setTwinOrnament->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'setTwinOrnamentFlag',
                        $item_setTwinOrnament->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'also_set_twin_exits',
                    );
                }
            });
            $subMenu_setOrnament->append($item_setTwinOrnament);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'also_set_twin_exits', $item_setTwinOrnament);

        my $item_setOrnament = Gtk3::MenuItem->new('Set _ornaments');
        $item_setOrnament->set_submenu($subMenu_setOrnament);
        $column_exits->append($item_setOrnament);
        # (Requires $self->currentRegionmap & either $self->selectedExit or
        #   $self->selectedExitHash)
        $self->ivAdd('menuToolItemHash', 'set_ornament_sub', $item_setOrnament);

            # 'Set exit type' submenu
            my $subMenu_setExitType = Gtk3::Menu->new();

                # 'Set hidden' sub-submenu
                my $subSubMenu_setHidden = Gtk3::Menu->new();

                my $item_setHiddenExit = Gtk3::MenuItem->new('Mark exit _hidden');
                $item_setHiddenExit->signal_connect('activate' => sub {

                    $self->hiddenExitCallback(TRUE);
                });
                $subSubMenu_setHidden->append($item_setHiddenExit);

                my $item_setNotHiddenExit = Gtk3::MenuItem->new('Mark exit _not hidden');
                $item_setNotHiddenExit->signal_connect('activate' => sub {

                    $self->hiddenExitCallback(FALSE);
                });
                $subSubMenu_setHidden->append($item_setNotHiddenExit);

            my $item_setHidden = Gtk3::MenuItem->new('Set _hidden');
            $item_setHidden->set_submenu($subSubMenu_setHidden);
            $subMenu_setExitType->append($item_setHidden);
            # (Requires $self->currentRegionmap and $self->selectedExit)
            $self->ivAdd('menuToolItemHash', 'set_hidden_sub', $item_setHidden);

                # 'Set broken' sub-submenu
                my $subSubMenu_setBroken = Gtk3::Menu->new();

                my $item_markBrokenExit = Gtk3::MenuItem->new('_Mark exit as broken');
                $item_markBrokenExit->signal_connect('activate' => sub {

                    $self->markBrokenExitCallback();
                });
                $subSubMenu_setBroken->append($item_markBrokenExit);

                my $item_toggleBrokenExit = Gtk3::MenuItem->new('_Toggle bent broken exit');
                $item_toggleBrokenExit->signal_connect('activate' => sub {

                    $self->worldModelObj->toggleBentExit(
                        TRUE,                       # Update Automapper windows now
                        $self->selectedExit,
                    );
                });
                $subSubMenu_setBroken->append($item_toggleBrokenExit);
                # (Requires $self->currentRegionmap and a $self->selectedExit which is a broken
                #   exit)
                $self->ivAdd('menuToolItemHash', 'toggle_bent_exit', $item_toggleBrokenExit);

                $subSubMenu_setBroken->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_restoreBrokenExit = Gtk3::MenuItem->new('_Restore unbroken exit');
                $item_restoreBrokenExit->signal_connect('activate' => sub {

                    $self->restoreBrokenExitCallback();
                });
                $subSubMenu_setBroken->append($item_restoreBrokenExit);

            my $item_setBroken = Gtk3::MenuItem->new('Set _broken');
            $item_setBroken->set_submenu($subSubMenu_setBroken);
            $subMenu_setExitType->append($item_setBroken);
            # (Requires $self->currentRegionmap and $self->selectedExit)
            $self->ivAdd('menuToolItemHash', 'set_broken_sub', $item_setBroken);

                # 'Set one-way' sub-submenu
                my $subSubMenu_setOneWay = Gtk3::Menu->new();

                my $item_markOneWayExit = Gtk3::MenuItem->new('_Mark exit as one-way');
                $item_markOneWayExit->signal_connect('activate' => sub {

                    $self->markOneWayExitCallback();
                });
                $subSubMenu_setOneWay->append($item_markOneWayExit);

                $subSubMenu_setOneWay->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_restoreUncertainExit = Gtk3::MenuItem->new('Restore _uncertain exit');
                $item_restoreUncertainExit->signal_connect('activate' => sub {

                    $self->restoreOneWayExitCallback(FALSE);
                });
                $subSubMenu_setOneWay->append($item_restoreUncertainExit);

                my $item_restoreTwoWayExit = Gtk3::MenuItem->new('Restore _two-way exit');
                $item_restoreTwoWayExit->signal_connect('activate' => sub {

                    $self->restoreOneWayExitCallback(TRUE);
                });
                $subSubMenu_setOneWay->append($item_restoreTwoWayExit);

                $subSubMenu_setOneWay->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_setIncomingDir = Gtk3::MenuItem->new('Set incoming _direction...');
                $item_setIncomingDir->signal_connect('activate' => sub {

                    $self->setIncomingDirCallback();
                });
                $subSubMenu_setOneWay->append($item_setIncomingDir);
                # (Requires $self->currentRegionmap and a $self->selectedExit which is a one-way
                #   exit)
                $self->ivAdd('menuToolItemHash', 'set_incoming_dir', $item_setIncomingDir);

            my $item_setOneWay = Gtk3::MenuItem->new('Set _one-way');
            $item_setOneWay->set_submenu($subSubMenu_setOneWay);
            $subMenu_setExitType->append($item_setOneWay);
            # (Requires $self->currentRegionmap and $self->selectedExit)
            $self->ivAdd('menuToolItemHash', 'set_oneway_sub', $item_setOneWay);

                # 'Set retracing' sub-submenu
                my $subSubMenu_setRetracing = Gtk3::Menu->new();

                my $item_markRetracingExit = Gtk3::MenuItem->new('_Mark exit as retracing');
                $item_markRetracingExit->signal_connect('activate' => sub {

                    $self->markRetracingExitCallback();
                });
                $subSubMenu_setRetracing->append($item_markRetracingExit);

                $subSubMenu_setRetracing->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_restoreRetracingExit = Gtk3::MenuItem->new('_Restore incomplete exit');
                $item_restoreRetracingExit->signal_connect('activate' => sub {

                    $self->restoreRetracingExitCallback();
                });
                $subSubMenu_setRetracing->append($item_restoreRetracingExit);

            my $item_setRetracing = Gtk3::MenuItem->new('Set _retracing');
            $item_setRetracing->set_submenu($subSubMenu_setRetracing);
            $subMenu_setExitType->append($item_setRetracing);
            # (Requires $self->currentRegionmap and $self->selectedExit)
            $self->ivAdd('menuToolItemHash', 'set_retracing_sub', $item_setRetracing);

                # 'Set random' sub-submenu
                my $subSubMenu_setRandomExit = Gtk3::Menu->new();

                my $item_markRandomRegion = Gtk3::MenuItem->new(
                    'Set random destination in same _region',
                );
                $item_markRandomRegion->signal_connect('activate' => sub {

                    $self->markRandomExitCallback('same_region');
                });
                $subSubMenu_setRandomExit->append($item_markRandomRegion);

                my $item_markRandomAnywhere
                    = Gtk3::MenuItem->new('Set random destination _anywhere');
                $item_markRandomAnywhere->signal_connect('activate' => sub {

                    $self->markRandomExitCallback('any_region');
                });
                $subSubMenu_setRandomExit->append($item_markRandomAnywhere);

                my $item_randomTempRegion
                    = Gtk3::MenuItem->new('_Create destination in temporary region');
                $item_randomTempRegion->signal_connect('activate' => sub {

                    $self->markRandomExitCallback('temp_region');
                });
                $subSubMenu_setRandomExit->append($item_randomTempRegion);

                my $item_markRandomList = Gtk3::MenuItem->new('_Use list of random destinations');
                $item_markRandomList->signal_connect('activate' => sub {

                    $self->markRandomExitCallback('room_list');
                });
                $subSubMenu_setRandomExit->append($item_markRandomList);

                $subSubMenu_setRandomExit->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_restoreRandomExit = Gtk3::MenuItem->new('Restore _incomplete exit');
                $item_restoreRandomExit->signal_connect('activate' => sub {

                    $self->restoreRandomExitCallback();
                });
                $subSubMenu_setRandomExit->append($item_restoreRandomExit);

            my $item_setRandomExit = Gtk3::MenuItem->new('Set r_andom');
            $item_setRandomExit->set_submenu($subSubMenu_setRandomExit);
            $subMenu_setExitType->append($item_setRandomExit);
            # (Requires $self->currentRegionmap and $self->selectedExit)
            $self->ivAdd('menuToolItemHash', 'set_random_sub', $item_setRandomExit);

                # 'Set super' sub-submenu
                my $subSubMenu_setSuperExit = Gtk3::Menu->new();

                my $item_markSuper = Gtk3::MenuItem->new('Mark exit as _super-region exit');
                $item_markSuper->signal_connect('activate' => sub {

                    $self->markSuperExitCallback(FALSE);
                });
                $subSubMenu_setSuperExit->append($item_markSuper);

                my $item_markSuperExcl = Gtk3::MenuItem->new(
                    'Mark exit as _exclusive super-region exit',
                );
                $item_markSuperExcl->signal_connect('activate' => sub {

                    $self->markSuperExitCallback(TRUE);
                });
                $subSubMenu_setSuperExit->append($item_markSuperExcl);

                $subSubMenu_setSuperExit->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_markNotSuper = Gtk3::MenuItem->new('Mark exit as _normal region exit');
                $item_markNotSuper->signal_connect('activate' => sub {

                    $self->restoreSuperExitCallback();
                });
                $subSubMenu_setSuperExit->append($item_markNotSuper);

            my $item_setSuperExit = Gtk3::MenuItem->new('Set _super');
            $item_setSuperExit->set_submenu($subSubMenu_setSuperExit);
            $subMenu_setExitType->append($item_setSuperExit);
            # (Requires $self->currentRegionmap and $self->selectedExit which is a region exit)
            $self->ivAdd('menuToolItemHash', 'set_super_sub', $item_setSuperExit);

            $subMenu_setExitType->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_setExitTwin = Gtk3::MenuItem->new('Set exit _twin...');
            $item_setExitTwin->signal_connect('activate' => sub {

                $self->setExitTwinCallback();
            });
            $subMenu_setExitType->append($item_setExitTwin);
            # (Requires $self->currentRegionmap and a $self->selectedExit which is either a one-way
            #   exit or an uncertain exit)
            $self->ivAdd('menuToolItemHash', 'set_exit_twin', $item_setExitTwin);

        my $item_setExitType = Gtk3::MenuItem->new('Set _exit type');
        $item_setExitType->set_submenu($subMenu_setExitType);
        $column_exits->append($item_setExitType);
        # (Requires $self->currentRegionmap and $self->selectedExit)
        $self->ivAdd('menuToolItemHash', 'set_exit_type', $item_setExitType);

            # 'Exit tags' submenu
            my $subMenu_exitTags = Gtk3::Menu->new();

            my $item_setExitText = Gtk3::MenuItem->new('_Edit tag text');
            $item_setExitText->signal_connect('activate' => sub {

                $self->editExitTagCallback();
            });
            $subMenu_exitTags->append($item_setExitText);
            # (Requires $self->currentRegionmap and either a $self->selectedExit which is a region
            #   exit, or a $self->selectedExitTag)
            $self->ivAdd('menuToolItemHash', 'edit_tag_text', $item_setExitText);

            my $item_toggleExitTag = Gtk3::MenuItem->new('_Toggle exit tag');
            $item_toggleExitTag->signal_connect('activate' => sub {

                $self->toggleExitTagCallback();
            });
            $subMenu_exitTags->append($item_toggleExitTag);
            # (Requires $self->currentRegionmap and either a $self->selectedExit which is a region
            #   exit, or a $self->selectedExitTag)
            $self->ivAdd('menuToolItemHash', 'toggle_exit_tag', $item_toggleExitTag);

            $subMenu_exitTags->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_resetPositions = Gtk3::MenuItem->new('_Reset tag positions');
            $item_resetPositions->signal_connect('activate' => sub {

                $self->resetExitOffsetsCallback();
            });
            $subMenu_exitTags->append($item_resetPositions);
            # (Requires $self->currentRegionmap and one or more of $self->selectedExit,
            #   $self->selectedExitHash, $self->selectedExitTag and $self->selectedExitTagHash)
            $self->ivAdd('menuToolItemHash', 'reset_exit_tags', $item_resetPositions);

            $subMenu_exitTags->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_applyExitTags = Gtk3::MenuItem->new('_Apply all tags in region');
            $item_applyExitTags->signal_connect('activate' => sub {

                $self->applyExitTagsCallback(TRUE);
            });
            $subMenu_exitTags->append($item_applyExitTags);


            my $item_cancelExitTags = Gtk3::MenuItem->new('_Cancel all tags in region');
            $item_cancelExitTags->signal_connect('activate' => sub {

                $self->applyExitTagsCallback(FALSE);
            });
            $subMenu_exitTags->append($item_cancelExitTags);

        my $item_exitTags = Gtk3::MenuItem->new('Exit _tags');
        $item_exitTags->set_submenu($subMenu_exitTags);
        $column_exits->append($item_exitTags);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'exit_tags', $item_exitTags);

        $column_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_editExit = Gtk3::ImageMenuItem->new('Edit e_xit...');
        my $img_editExit = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editExit->set_image($img_editExit);
        $item_editExit->signal_connect('activate' => sub {

            $self->editExitCallback();
        });
        $column_exits->append($item_editExit);
        # (Requires $self->currentRegionmap and $self->selectedExit)
        $self->ivAdd('menuToolItemHash', 'edit_exit', $item_editExit);

        $column_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

            # 'Exit options' submenu
            my $subMenu_exitOptions = Gtk3::Menu->new();

            my $item_completeSelected = Gtk3::MenuItem->new('_Complete selected uncertain exits');
            $item_completeSelected->signal_connect('activate' => sub {

                $self->completeExitsCallback();
            });
            $subMenu_exitOptions->append($item_completeSelected);

            my $item_connectAdjacent = Gtk3::MenuItem->new('C_onnect selected adjacent rooms');
            $item_connectAdjacent->signal_connect('activate' => sub {

                $self->connectAdjacentCallback();
            });
            $subMenu_exitOptions->append($item_connectAdjacent);
            # (Requires $self->currentRegionmap and one or more selected rooms)
            $self->ivAdd('menuToolItemHash', 'connect_adjacent', $item_connectAdjacent);

            $subMenu_exitOptions->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_autocomplete = Gtk3::CheckMenuItem->new('_Autocomplete uncertain exits');
            $item_autocomplete->set_active($self->worldModelObj->autocompleteExitsFlag);
            $item_autocomplete->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'autocompleteExitsFlag',
                        $item_autocomplete->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'autcomplete_uncertain',
                    );
                }
            });
            $subMenu_exitOptions->append($item_autocomplete);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'autocomplete_uncertain', $item_autocomplete);

            my $item_intUncertain = Gtk3::CheckMenuItem->new('_Intelligent uncertain exits');
            $item_intUncertain->set_active(
                $self->worldModelObj->intelligentExitsFlag,
            );
            $item_intUncertain->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'intelligentExitsFlag',
                        $item_intUncertain->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'intelligent_uncertain',
                    );
                }
            });
            $subMenu_exitOptions->append($item_intUncertain);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'intelligent_uncertain', $item_intUncertain);

            $subMenu_exitOptions->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_collectChecked = Gtk3::CheckMenuItem->new('Co_llect checked directions');
            $item_collectChecked->set_active(
                $self->worldModelObj->collectCheckedDirsFlag,
            );
            $item_collectChecked->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'collectCheckedDirsFlag',
                        $item_collectChecked->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'collect_checked_dirs',
                    );
                }
            });
            $subMenu_exitOptions->append($item_collectChecked);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'collect_checked_dirs', $item_collectChecked);

            my $item_drawChecked = Gtk3::CheckMenuItem->new('_Draw checked directions');
            $item_drawChecked->set_active(
                $self->worldModelObj->drawCheckedDirsFlag,
            );
            $item_drawChecked->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'drawCheckedDirsFlag',
                        $item_drawChecked->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'draw_checked_dirs',
                    );
                }

                # Redraw the region, if one is visible
                if ($self->currentRegionmap) {

                    $self->redrawRegions();
                }
            });
            $subMenu_exitOptions->append($item_drawChecked);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'draw_checked_dirs', $item_drawChecked);

            $subMenu_exitOptions->append(Gtk3::SeparatorMenuItem->new());    # Separator

                 # 'Checkable directions' sub-submenu
                my $subSubMenu_checkable = Gtk3::Menu->new();

                my $item_checkableSimple
                    = Gtk3::RadioMenuItem->new_with_mnemonic(undef, 'Count _NSEW');
                $item_checkableSimple->signal_connect('toggled' => sub {

                    if ($item_checkableSimple->get_active) {

                        $self->worldModelObj->setCheckableDirMode('simple');
                    }
                });
                my $item_checkableGroup = $item_checkableSimple->get_group();
                $subSubMenu_checkable->append($item_checkableSimple);
                # (Never desensitised)
                $self->ivAdd('menuToolItemHash', 'checkable_dir_simple', $item_checkableSimple);

                my $item_checkableDiku = Gtk3::RadioMenuItem->new_with_mnemonic(
                    $item_checkableGroup,
                    'Count NSEW_UD',
                );
                if ($self->worldModelObj->checkableDirMode eq 'diku') {

                    $item_checkableDiku->set_active(TRUE);
                }
                $item_checkableDiku->signal_connect('toggled' => sub {

                    if ($item_checkableDiku->get_active) {

                        $self->worldModelObj->setCheckableDirMode('diku');
                    }
                });
                $subSubMenu_checkable->append($item_checkableDiku);
                # (Never desensitised)
                $self->ivAdd('menuToolItemHash', 'checkable_dir_diku', $item_checkableDiku);

                my $item_checkableLP = Gtk3::RadioMenuItem->new_with_mnemonic(
                    $item_checkableGroup,
                    'Count NSEWUD, N_E/NW/SE/SW',
                );
                if ($self->worldModelObj->checkableDirMode eq 'lp') {

                    $item_checkableLP->set_active(TRUE);
                }
                $item_checkableLP->signal_connect('toggled' => sub {

                    if ($item_checkableLP->get_active) {

                        $self->worldModelObj->setCheckableDirMode('lp');
                    }
                });
                $subSubMenu_checkable->append($item_checkableLP);
                # (Never desensitised)
                $self->ivAdd('menuToolItemHash', 'checkable_dir_lp', $item_checkableLP);

                my $item_checkableComplex = Gtk3::RadioMenuItem->new_with_mnemonic(
                    $item_checkableGroup,
                    'Count _all primary directions',
                );
                if ($self->worldModelObj->checkableDirMode eq 'complex') {

                    $item_checkableComplex->set_active(TRUE);
                }
                $item_checkableComplex->signal_connect('toggled' => sub {

                    if ($item_checkableComplex->get_active) {

                        $self->worldModelObj->setCheckableDirMode('complex');
                    }
                });
                $subSubMenu_checkable->append($item_checkableComplex);
                # (Never desensitised)
                $self->ivAdd('menuToolItemHash', 'checkable_dir_complex', $item_checkableComplex);

            my $item_exits = Gtk3::MenuItem->new('C_heckable directions');
            $item_exits->set_submenu($subSubMenu_checkable);
            $subMenu_exitOptions->append($item_exits);

        my $item_exitOptions = Gtk3::MenuItem->new('Exit o_ptions');
        $item_exitOptions->set_submenu($subMenu_exitOptions);
        $column_exits->append($item_exitOptions);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'exit_options', $item_exitOptions);

            # 'Exit lengths' submenu
            my $subMenu_exitLengths = Gtk3::Menu->new();

            my $item_horizontalLength = Gtk3::MenuItem->new('Set _horizontal length...');
            $item_horizontalLength->signal_connect('activate' => sub {

                $self->setExitLengthCallback('horizontal');
            });
            $subMenu_exitLengths->append($item_horizontalLength);

            my $item_verticalLength = Gtk3::MenuItem->new('Set _vertical length...');
            $item_verticalLength->signal_connect('activate' => sub {

                $self->setExitLengthCallback('vertical');
            });
            $subMenu_exitLengths->append($item_verticalLength);

            $subMenu_exitLengths->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_resetLength = Gtk3::MenuItem->new('_Reset exit lengths');
            $item_resetLength->signal_connect('activate' => sub {

                $self->resetExitLengthCallback();
            });
            $subMenu_exitLengths->append($item_resetLength);

        my $item_exitLengths = Gtk3::MenuItem->new('Exit _lengths');
        $item_exitLengths->set_submenu($subMenu_exitLengths);
        $column_exits->append($item_exitLengths);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'exit_lengths', $item_exitLengths);

        $column_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_deleteExit = Gtk3::ImageMenuItem->new('_Delete exit');
        my $img_deleteExit = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_deleteExit->set_image($img_deleteExit);
        $item_deleteExit->signal_connect('activate' => sub {

            $self->deleteExitCallback();
        });
        $column_exits->append($item_deleteExit);
        # (Requires $self->currentRegionmap and $self->selectedExit)
        $self->ivAdd('menuToolItemHash', 'delete_exit', $item_deleteExit);

        # Setup complete
        return $column_exits;
    }

    sub enableLabelsColumn {

        # Called by $self->enableMenu
        # Sets up the 'Labels' column of the Automapper window's menu bar
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Local variables
        my $alignFlag;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableLabelsColumn', @_);
        }

        # Set up column
        my $column_labels = Gtk3::Menu->new();
        if (! $column_labels) {

            return undef;
        }

        my $item_addLabelAtClick = Gtk3::ImageMenuItem->new('Add label at _click');
        my $img_addLabelAtClick = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_addLabelAtClick->set_image($img_addLabelAtClick);
        $item_addLabelAtClick->signal_connect('activate' => sub {

            # Set the free click mode; $self->canvasEventHandler will create the new label when the
            #   user next clicks on an empty part of the map
            if ($self->currentRegionmap) {

                $self->set_freeClickMode('add_label');
            }
        });
        $column_labels->append($item_addLabelAtClick);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'add_label_at_click', $item_addLabelAtClick);

        my $item_addLabelAtBlock = Gtk3::ImageMenuItem->new('Add label at _block');
        my $img_addLabelAtBlock = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_addLabelAtBlock->set_image($img_addLabelAtBlock);
        $item_addLabelAtBlock->signal_connect('activate' => sub {

            $self->addLabelAtBlockCallback();
        });
        $column_labels->append($item_addLabelAtBlock);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'add_label_at_block', $item_addLabelAtBlock);

        $column_labels->append(Gtk3::SeparatorMenuItem->new()); # Separator

        my $item_setLabel = Gtk3::ImageMenuItem->new('_Set label...');
        my $img_setLabel = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_setLabel->set_image($img_setLabel);
        $item_setLabel->signal_connect('activate' => sub {

            $self->setLabelCallback(FALSE);
        });
        $column_labels->append($item_setLabel);
        # (Requires $self->currentRegionmap and $self->selectedLabel)
        $self->ivAdd('menuToolItemHash', 'set_label', $item_setLabel);

        my $item_customiseLabel = Gtk3::ImageMenuItem->new('C_ustomise label...');
        my $img_customiseLabel = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_customiseLabel->set_image($img_customiseLabel);
        $item_customiseLabel->signal_connect('activate' => sub {

            $self->setLabelCallback(TRUE);
        });
        $column_labels->append($item_customiseLabel);
        # (Requires $self->currentRegionmap and $self->selectedLabel)
        $self->ivAdd('menuToolItemHash', 'customise_label', $item_customiseLabel);

            my $item_useMultiLine = Gtk3::CheckMenuItem->new('Use _multiline labels');
            $item_useMultiLine->set_active($self->worldModelObj->mapLabelTextViewFlag);
            $item_useMultiLine->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'mapLabelTextViewFlag',
                        $item_useMultiLine->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'use_multi_line',
                    );
                }
            });
            $column_labels->append($item_useMultiLine);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'use_multi_line', $item_useMultiLine);

        $column_labels->append(Gtk3::SeparatorMenuItem->new()); # Separator

        my $item_selectLabel = Gtk3::MenuItem->new('S_elect label...');
        $item_selectLabel->signal_connect('activate' => sub {

            $self->selectLabelCallback();
        });
        $column_labels->append($item_selectLabel);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'select_label', $item_selectLabel);

        $column_labels->append(Gtk3::SeparatorMenuItem->new()); # Separator

        my $item_addStyle = Gtk3::ImageMenuItem->new('_Add label style...');
        my $img_addStyle = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_addStyle->set_image($img_addStyle);
        $item_addStyle->signal_connect('activate' => sub {

            $self->addStyleCallback();
        });
        $column_labels->append($item_addStyle);

        my $item_editStyle = Gtk3::ImageMenuItem->new('Ed_it label style...');
        my $img_editStyle = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editStyle->set_image($img_editStyle);
        $item_editStyle->signal_connect('activate' => sub {

            $self->editStyleCallback();
        });
        $column_labels->append($item_editStyle);
        # (Requires at least one label style in $self->worldModelObj->mapLabelStyleHash)
        $self->ivAdd('menuToolItemHash', 'edit_style', $item_selectLabel);

            # 'Label alignment' submenu
            my $subMenu_alignment = Gtk3::Menu->new();

            my $item_alignHorizontal = Gtk3::CheckMenuItem->new('Align _horizontally');
            $item_alignHorizontal->set_active($self->worldModelObj->mapLabelAlignXFlag);
            $item_alignHorizontal->signal_connect('toggled' => sub {

                # Use $alignFlag to avoid an infinite loop, if we have to toggle the button back to
                #   its original state because the user declined to confirm the operation
                if (! $alignFlag) {

                    if (! $self->toggleAlignCallback('horizontal')) {

                        $alignFlag = TRUE;
                        if (! $item_alignHorizontal->get_active()) {
                            $item_alignHorizontal->set_active(TRUE);
                        } else {
                            $item_alignHorizontal->set_active(FALSE);
                        }

                        $alignFlag = FALSE;
                    }
                }
            });
            $subMenu_alignment->append($item_alignHorizontal);

            my $item_alignVertical = Gtk3::CheckMenuItem->new('Align _vertically');
            $item_alignVertical->set_active($self->worldModelObj->mapLabelAlignYFlag);
            $item_alignVertical->signal_connect('toggled' => sub {

                # Use $alignFlag to avoid an infinite loop, if we have to toggle the button back to
                #   its original state because the user declined to confirm the operation
                if (! $alignFlag) {

                    if (! $self->toggleAlignCallback('vertical')) {

                        $alignFlag = TRUE;
                        if (! $item_alignVertical->get_active()) {
                            $item_alignVertical->set_active(TRUE);
                        } else {
                            $item_alignVertical->set_active(FALSE);
                        }

                        $alignFlag = FALSE;
                    }
                }
            });
            $subMenu_alignment->append($item_alignVertical);

        my $item_alignment = Gtk3::MenuItem->new('_Label alignment');
        $item_alignment->set_submenu($subMenu_alignment);
        $column_labels->append($item_alignment);

        $column_labels->append(Gtk3::SeparatorMenuItem->new()); # Separator

        my $item_deleteLabel = Gtk3::ImageMenuItem->new('_Delete labels');
        my $img_deleteLabel = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_deleteLabel->set_image($img_deleteLabel);
        $item_deleteLabel->signal_connect('activate' => sub {

            # Callback to prompt for confirmation, before deleting multiple labels
            $self->deleteLabelsCallback();
        });
        $column_labels->append($item_deleteLabel);
        # (Requires $self->currentRegionmap & either $self->selectedLabel or
        #   $self->selectedLabelHash)
        $self->ivAdd('menuToolItemHash', 'delete_label', $item_deleteLabel);

        my $item_quickDelete = Gtk3::ImageMenuItem->new('_Quick label deletion...');
        my $img_quickDelete = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_quickDelete->set_image($img_quickDelete);
        $item_quickDelete->signal_connect('activate' => sub {

            $self->session->pseudoCmd('quicklabeldelete', $self->pseudoCmdMode);
        });
        $column_labels->append($item_quickDelete);

        # Setup complete
        return $column_labels;
    }

    # Popup menu widget methods

    sub enableCanvasPopupMenu {

        # Called by $self->canvasEventHandler
        # Creates a popup-menu for the Gtk3::Canvas when no rooms, exits, room tags or labels are
        #   selected
        #
        # Expected arguments
        #   $clickXPosPixels, $clickYPosPixels
        #       - Coordinates of the pixel that was right-clicked on the map
        #   $clickXPosBlocks, $clickYPosBlocks
        #       - Coordinates of the gridblock that was right-clicked on the map
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my (
            $self, $clickXPosPixels, $clickYPosPixels, $clickXPosBlocks, $clickYPosBlocks, $check,
        ) = @_;

        # Check for improper arguments
        if (
            ! defined $clickXPosPixels || ! defined $clickYPosPixels
            || ! defined $clickXPosBlocks || ! defined $clickYPosBlocks || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableCanvasPopupMenu', @_);
        }

        # Set up the popup menu
        my $menu_canvas = Gtk3::Menu->new();
        if (! $menu_canvas) {

            return undef;
        }

        # (Everything here assumes $self->currentRegionmap)

        my $item_addFirstRoom = Gtk3::ImageMenuItem->new('Add _first room');
        my $img_addFirstRoom = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_addFirstRoom->set_image($img_addFirstRoom);
        $item_addFirstRoom->signal_connect('activate' => sub {

            $self->addFirstRoomCallback();
        });
        $menu_canvas->append($item_addFirstRoom);
        # (Also requires empty $self->currentRegionmap->gridRoomHash)
        if ($self->currentRegionmap->gridRoomHash) {

            $item_addFirstRoom->set_sensitive(FALSE);
        }

        my $item_addRoomHere = Gtk3::ImageMenuItem->new('Add _room here');
        my $img_addRoomHere = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_addRoomHere->set_image($img_addRoomHere);
        $item_addRoomHere->signal_connect('activate' => sub {

            my $roomObj;

            # The 'Add room at click' operation from the main menu resets the value of
            #   ->freeClickMode; we must do the same here
            $self->reset_freeClickMode();

            # Create the room
            $roomObj = $self->mapObj->createNewRoom(
                $self->currentRegionmap,
                $clickXPosBlocks,
                $clickYPosBlocks,
                $self->currentRegionmap->currentLevel,
            );

            # When using the 'Add room at block' menu item, the new room is selected to make it
            #   easier to see where it was drawn. To make things consistent, select this new room,
            #   too
            if ($roomObj) {

                $self->setSelectedObj(
                    [$roomObj, 'room'],
                    FALSE,      # Select this object; unselect all other objects
                );
            }
        });
        $menu_canvas->append($item_addRoomHere);

        $menu_canvas->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_addLabelHere = Gtk3::ImageMenuItem->new('Add _label here');
        my $img_addLabelHere = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_addLabelHere->set_image($img_addLabelHere);
        $item_addLabelHere->signal_connect('activate' => sub {

            $self->addLabelAtClickCallback($clickXPosPixels, $clickYPosPixels);
        });
        $menu_canvas->append($item_addLabelHere);

        $menu_canvas->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_centreMap = Gtk3::MenuItem->new('_Centre map here');
        $item_centreMap->signal_connect('activate' => sub {

            $self->centreMapOverRoom(
                undef,              # Centre the map, not over a room...
                $clickXPosBlocks,   # ...but over this gridblock
                $clickYPosBlocks,
            );
        });
        $menu_canvas->append($item_centreMap);

        $menu_canvas->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_editRegionmap = Gtk3::ImageMenuItem->new('_Edit regionmap...');
        my $img_editRegionmap = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editRegionmap->set_image($img_editRegionmap);
        $item_editRegionmap->signal_connect('activate' => sub {

            # Open an 'edit' window for the regionmap
            $self->createFreeWin(
                'Games::Axmud::EditWin::Regionmap',
                $self,
                $self->session,
                'Edit \'' . $self->currentRegionmap->name . '\' regionmap',
                $self->currentRegionmap,
                FALSE,                          # Not temporary
            );
        });
        $menu_canvas->append($item_editRegionmap);

        my $item_preferences = Gtk3::ImageMenuItem->new('Edit world _model...');
        my $img_preferences = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_preferences->set_image($img_preferences);
        $item_preferences->signal_connect('activate' => sub {

            # Open an 'edit' window for the world model
            $self->createFreeWin(
                'Games::Axmud::EditWin::WorldModel',
                $self,
                $self->session,
                'Edit world model',
                $self->session->worldModelObj,
                FALSE,                          # Not temporary
            );
        });
        $menu_canvas->append($item_preferences);

        # Setup complete
        $menu_canvas->show_all();

        return $menu_canvas;
    }

    sub enableRoomsPopupMenu {

        # Called by $self->canvasObjEventHandler
        # Creates a popup-menu for the selected room
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableRoomsPopupMenu', @_);
        }

        # Set up the popup menu
        my $menu_rooms = Gtk3::Menu->new();
        if (! $menu_rooms) {

            return undef;
        }

        # (Everything here assumes $self->currentRegionmap and $self->selectedRoom)

        my $item_setCurrentRoom = Gtk3::MenuItem->new('_Set current room');
        $item_setCurrentRoom->signal_connect('activate' => sub {

            $self->mapObj->setCurrentRoom($self->selectedRoom);
        });
        $menu_rooms->append($item_setCurrentRoom);

        my $item_centreMap = Gtk3::MenuItem->new('_Centre map over room');
        $item_centreMap->signal_connect('activate' => sub {

            $self->centreMapOverRoom($self->selectedRoom);
        });
        $menu_rooms->append($item_centreMap);

        my $item_executeScripts = Gtk3::MenuItem->new('Run _Axbasic scripts');
        $item_executeScripts->signal_connect('activate' => sub {

            $self->executeScriptsCallback();
        });
        $menu_rooms->append($item_executeScripts);
        # (Also requires $self->mapObj->currentRoom that's the same as $self->selectedRoom)
        if (! $self->mapObj->currentRoom || $self->mapObj->currentRoom ne $self->selectedRoom) {

            $item_executeScripts->set_sensitive(FALSE);
        }

        $menu_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

            # 'Pathfinding' submenu
            my $subMenu_pathFinding = Gtk3::Menu->new();

            my $item_highlightPath = Gtk3::MenuItem->new('_Highlight path');
            $item_highlightPath->signal_connect('activate' => sub {

                $self->processPathCallback('select_room');
            });
            $subMenu_pathFinding->append($item_highlightPath);

            my $item_displayPath = Gtk3::MenuItem->new('_Edit path...');
            $item_displayPath->signal_connect('activate' => sub {

                $self->processPathCallback('pref_win');
            });
            $subMenu_pathFinding->append($item_displayPath);

            my $item_goToRoom = Gtk3::MenuItem->new('_Go to room');
            $item_goToRoom->signal_connect('activate' => sub {

                $self->processPathCallback('send_char');
            });
            $subMenu_pathFinding->append($item_goToRoom);

        my $item_pathFinding = Gtk3::MenuItem->new('_Pathfinding');
        $item_pathFinding->set_submenu($subMenu_pathFinding);
        $menu_rooms->append($item_pathFinding);
        # (Also requires $self->mapObj->currentRoom)
        if (! $self->mapObj->currentRoom) {

            $item_pathFinding->set_sensitive(FALSE);
        }

            # 'Moves rooms/labels' submenu
            my $subMenu_moveRooms = Gtk3::Menu->new();

            my $item_moveSelected = Gtk3::MenuItem->new('Move in _direction...');
            $item_moveSelected->signal_connect('activate' => sub {

                $self->moveSelectedRoomsCallback();
            });
            $subMenu_moveRooms->append($item_moveSelected);

            my $item_moveSelectedToClick = Gtk3::MenuItem->new('Move to _click');
            $item_moveSelectedToClick->signal_connect('activate' => sub {

                # Set the free clicking mode: $self->mouseClickEvent will move the objects when the
                #   user next clicks on an empty part of the map
                $self->set_freeClickMode('move_room');
            });
            $subMenu_moveRooms->append($item_moveSelectedToClick);

            $subMenu_moveRooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

                # 'Transfer to region' sub-submenu
                my $subSubMenu_transferRegion = Gtk3::Menu->new();

                if ($self->recentRegionList) {

                    foreach my $name ($self->recentRegionList) {

                        my $item_regionName = Gtk3::MenuItem->new($name);
                        $item_regionName->signal_connect('activate' => sub {

                            $self->transferSelectedRoomsCallback($name);
                        });
                        $subSubMenu_transferRegion->append($item_regionName);
                    }

                } else {

                    my $item_regionNone = Gtk3::MenuItem->new('(No recent regions)');
                    $item_regionNone->set_sensitive(FALSE);
                    $subSubMenu_transferRegion->append($item_regionNone);
                }

                $subSubMenu_transferRegion->append(Gtk3::SeparatorMenuItem->new());  # Separator

                my $item_transferSelect = Gtk3::MenuItem->new('Select region...');
                $item_transferSelect->signal_connect('activate' => sub {

                    $self->transferSelectedRoomsCallback();
                });
                $subSubMenu_transferRegion->append($item_transferSelect);

            my $item_transferRegion = Gtk3::MenuItem->new('_Transfer to region');
            $item_transferRegion->set_submenu($subSubMenu_transferRegion);
            $subMenu_moveRooms->append($item_transferRegion);
            # (Also requires at least two regions in the world model)
            if ($self->worldModelObj->ivPairs('regionmapHash') <= 1) {

                $item_transferRegion->set_sensitive(FALSE);
            }

            $subMenu_moveRooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_mergeRoom = Gtk3::MenuItem->new('_Merge room');
            $item_mergeRoom->signal_connect('activate' => sub {

                $self->doMerge($self->mapObj->currentRoom);
            });
            $subMenu_moveRooms->append($item_mergeRoom);
            # (Also requires this to be the current room and the automapper being set up to perform
            #   a merge)
            if (
                ! $self->mapObj->currentRoom
                || $self->mapObj->currentRoom ne $self->selectedRoom
                || ! $self->mapObj->currentMatchFlag
            ) {
                $item_mergeRoom->set_sensitive(FALSE);
            }

                # 'Compare room' sub-submenu
                my $subSubMenu_compareRoom = Gtk3::Menu->new();

                my $item_compareRoomRegion = Gtk3::MenuItem->new('...with rooms in region');
                $item_compareRoomRegion->signal_connect('activate' => sub {

                    $self->compareRoomCallback(FALSE);
                });
                $subSubMenu_compareRoom->append($item_compareRoomRegion);

                my $item_compareRoomModel = Gtk3::MenuItem->new('...with rooms in whole world');
                $item_compareRoomModel->signal_connect('activate' => sub {

                    $self->compareRoomCallback(TRUE);
                });
                $subSubMenu_compareRoom->append($item_compareRoomModel);

            my $item_compareRoom = Gtk3::MenuItem->new('_Compare room');
            $item_compareRoom->set_submenu($subSubMenu_compareRoom);
            $subMenu_moveRooms->append($item_compareRoom);

        my $item_moveRooms = Gtk3::MenuItem->new('_Move rooms/labels');
        $item_moveRooms->set_submenu($subMenu_moveRooms);
        $menu_rooms->append($item_moveRooms);

            # 'Add pattern' submenu
            my $subMenu_exitPatterns = Gtk3::Menu->new();

            my $item_addFailedExitRoom = Gtk3::MenuItem->new('Add _failed exit...');
            $item_addFailedExitRoom->signal_connect('activate' => sub {

                $self->addFailedExitCallback(FALSE, $self->selectedRoom);
            });
            $subMenu_exitPatterns->append($item_addFailedExitRoom);

            $subMenu_exitPatterns->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_addInvoluntaryExitRoom = Gtk3::MenuItem->new('Add _involuntary exit...');
            $item_addInvoluntaryExitRoom->signal_connect('activate' => sub {

                $self->addInvoluntaryExitCallback($self->selectedRoom);
            });
            $subMenu_exitPatterns->append($item_addInvoluntaryExitRoom);

            my $item_addRepulseExitRoom = Gtk3::MenuItem->new('Add _repulse exit...');
            $item_addRepulseExitRoom->signal_connect('activate' => sub {

                $self->addRepulseExitCallback($self->selectedRoom);
            });
            $subMenu_exitPatterns->append($item_addRepulseExitRoom);

            $subMenu_exitPatterns->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_addSpecialDepartRoom = Gtk3::MenuItem->new('Add _special departure...');
            $item_addSpecialDepartRoom->signal_connect('activate' => sub {

                $self->addSpecialDepartureCallback($self->selectedRoom);
            });
            $subMenu_exitPatterns->append($item_addSpecialDepartRoom);

            my $item_addUnspecifiedRoom = Gtk3::MenuItem->new('Add _unspecified room pattern...');
            $item_addUnspecifiedRoom->signal_connect('activate' => sub {

                $self->addUnspecifiedPatternCallback($self->selectedRoom);
            });
            $subMenu_exitPatterns->append($item_addUnspecifiedRoom);

        $menu_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_patterns = Gtk3::MenuItem->new('Add pa_ttern');
        $item_patterns->set_submenu($subMenu_exitPatterns);
        $menu_rooms->append($item_patterns);

            # 'Add to model' submenu
            my $subMenu_addToModel = Gtk3::Menu->new();

            my $item_addRoomContents = Gtk3::MenuItem->new('Add _contents...');
            $item_addRoomContents->signal_connect('activate' => sub {

                $self->addContentsCallback(FALSE);
            });
            $subMenu_addToModel->append($item_addRoomContents);
            # (Also requires $self->mapObj->currentRoom that's the same as $self->selectedRoom
            if (! $self->mapObj->currentRoom || $self->mapObj->currentRoom ne $self->selectedRoom) {

                $item_addRoomContents->set_sensitive(FALSE);
            }

            my $item_addContentsString = Gtk3::MenuItem->new('Add c_ontents from string...');
            $item_addContentsString->signal_connect('activate' => sub {

                $self->addContentsCallback(TRUE);
            });
            $subMenu_addToModel->append($item_addContentsString);

            $subMenu_addToModel->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_addHiddenObj = Gtk3::MenuItem->new('Add _hidden object...');
            $item_addHiddenObj->signal_connect('activate' => sub {

                $self->addHiddenObjCallback(FALSE);
            });
            $subMenu_addToModel->append($item_addHiddenObj);
            # (Also requires $self->mapObj->currentRoom that's the same as $self->selectedRoom
            if (! $self->mapObj->currentRoom || $self->mapObj->currentRoom ne $self->selectedRoom) {

                $item_addHiddenObj->set_sensitive(FALSE);
            }

            my $item_addHiddenString = Gtk3::MenuItem->new('Add h_idden object from string...');
            $item_addHiddenString->signal_connect('activate' => sub {

                $self->addHiddenObjCallback(TRUE);
            });
            $subMenu_addToModel->append($item_addHiddenString);

            $subMenu_addToModel->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_addSearchResult = Gtk3::MenuItem->new('Add _search result...');
            $item_addSearchResult->signal_connect('activate' => sub {

                $self->addSearchResultCallback();
            });
            $subMenu_addToModel->append($item_addSearchResult);
            # (Also requires $self->mapObj->currentRoom that's the same as $self->selectedRoom)
            if (! $self->mapObj->currentRoom || $self->mapObj->currentRoom ne $self->selectedRoom) {

                $item_addSearchResult->set_sensitive(FALSE);
            }

        my $item_addToModel = Gtk3::MenuItem->new('Add to m_odel');
        $item_addToModel->set_submenu($subMenu_addToModel);
        $menu_rooms->append($item_addToModel);

        $menu_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

             # 'Add/set exits' submenu
            my $subMenu_setExits = Gtk3::Menu->new();

            my $item_addExit = Gtk3::MenuItem->new('Add _normal exit...');
            $item_addExit->signal_connect('activate' => sub {

                $self->addExitCallback(FALSE);      # FALSE - not a hidden exit
            });
            $subMenu_setExits->append($item_addExit);
            # (Also requires the selected room's ->wildMode to be 'normal' or 'border')
            if ($self->selectedRoom->wildMode eq 'wild') {

                $item_addExit->set_sensitive(FALSE);
            }

            my $item_addHiddenExit = Gtk3::MenuItem->new('Add _hidden exit...');
            $item_addHiddenExit->signal_connect('activate' => sub {

                $self->addExitCallback(TRUE);       # TRUE - a hidden exit
            });
            $subMenu_setExits->append($item_addHiddenExit);
            # (Also requires the selected room's ->wildMode to be 'normal' or 'border')
            if ($self->selectedRoom->wildMode eq 'wild') {

                $item_addHiddenExit->set_sensitive(FALSE);
            }

            $subMenu_setExits->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_addMultiple = Gtk3::MenuItem->new('Add _multiple exits...');
            $item_addMultiple->signal_connect('activate' => sub {

                $self->addMultipleExitsCallback();
            });
            $subMenu_setExits->append($item_addMultiple);

            $subMenu_setExits->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_removeChecked = Gtk3::MenuItem->new('Remove _checked direction...');
            $item_removeChecked->signal_connect('activate' => sub {

                $self->removeCheckedDirCallback(FALSE);
            });
            $subMenu_setExits->append($item_removeChecked);
            # (Also requires the selected room's ->checkedDirHash to be non-empty)
            if (! $self->selectedRoom->checkedDirHash) {

                $item_removeChecked->set_sensitive(FALSE);
            }

            my $item_removeCheckedAll = Gtk3::MenuItem->new('Remove _all checked directions');
            $item_removeCheckedAll->signal_connect('activate' => sub {

                $self->removeCheckedDirCallback(TRUE);
            });
            $subMenu_setExits->append($item_removeCheckedAll);
            # (Also requires the selected room's ->checkedDirHash to be non-empty)
            if (! $self->selectedRoom->checkedDirHash) {

                $item_removeCheckedAll->set_sensitive(FALSE);
            }

            $subMenu_setExits->append(Gtk3::SeparatorMenuItem->new()); # Separator

            my $item_markNormal = Gtk3::MenuItem->new('Mark room as n_ormal');
            $item_markNormal->signal_connect('activate' => sub {

                $self->setWildCallback('normal');
            });
            $subMenu_setExits->append($item_markNormal);

            my $item_markWild = Gtk3::MenuItem->new('Mark room as _wilderness');
            $item_markWild->signal_connect('activate' => sub {

                $self->setWildCallback('wild');
            });
            $subMenu_setExits->append($item_markWild);
            # (Also requires $self->session->currentWorld->basicMappingFlag to be FALSE)
            if ($self->session->currentWorld->basicMappingFlag) {

                $item_markWild->set_sensitive(FALSE);
            }

            my $item_markBorder = Gtk3::MenuItem->new('Mark room as wilderness _border');
            $item_markBorder->signal_connect('activate' => sub {

                $self->setWildCallback('border');
            });
            $subMenu_setExits->append($item_markBorder);
            # (Also requires $self->session->currentWorld->basicMappingFlag to be FALSE)
            if ($self->session->currentWorld->basicMappingFlag) {

                $item_markBorder->set_sensitive(FALSE);
            }

        my $item_setExits = Gtk3::ImageMenuItem->new('Add/set _exits');
        my $img_setExits = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_setExits->set_image($img_setExits);
        $item_setExits->set_submenu($subMenu_setExits);
        $menu_rooms->append($item_setExits);

        my $item_selectExit = Gtk3::MenuItem->new('Se_lect exit...');
        $item_selectExit->signal_connect('activate' => sub {

            $self->selectExitCallback();
        });
        $menu_rooms->append($item_selectExit);

        $menu_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_editRoom = Gtk3::ImageMenuItem->new('Ed_it room...');
        my $img_editRoom = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editRoom->set_image($img_editRoom);
        $item_editRoom->signal_connect('activate' => sub {

            if ($self->selectedRoom) {

                # Open the room's 'edit' window
                $self->createFreeWin(
                    'Games::Axmud::EditWin::ModelObj::Room',
                    $self,
                    $self->session,
                    'Edit ' . $self->selectedRoom->category . ' model object',
                    $self->selectedRoom,
                    FALSE,                          # Not temporary
                );
            }
        });
        $menu_rooms->append($item_editRoom);

            # 'Set room text' submenu
            my $subMenu_setRoomText = Gtk3::Menu->new();

            my $item_setRoomTag = Gtk3::MenuItem->new('Set room _tag...');
            $item_setRoomTag->signal_connect('activate' => sub {

                $self->setRoomTagCallback();
            });
            $subMenu_setRoomText->append($item_setRoomTag);

            my $item_setGuild = Gtk3::MenuItem->new('Set room _guild...');
            $item_setGuild->signal_connect('activate' => sub {

                $self->setRoomGuildCallback();
            });
            $subMenu_setRoomText->append($item_setGuild);

            $subMenu_setRoomText->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_resetPositions = Gtk3::MenuItem->new('_Reset text posit_ions');
            $item_resetPositions->signal_connect('activate' => sub {

                $self->resetRoomOffsetsCallback();
            });
            $subMenu_setRoomText->append($item_resetPositions);

        my $item_setRoomText = Gtk3::MenuItem->new('Set _room text');
        $item_setRoomText->set_submenu($subMenu_setRoomText);
        $menu_rooms->append($item_setRoomText);

            # 'Toggle room flag' submenu
            my $subMenu_toggleRoomFlag = Gtk3::Menu->new();

            if ($self->worldModelObj->roomFlagShowMode eq 'default') {

                # Show all room flags, sorted by filter
                foreach my $filter ($axmud::CLIENT->constRoomFilterList) {

                    # A sub-sub menu for $filter
                    my $subSubMenu_filter = Gtk3::Menu->new();

                    my @nameList = $self->worldModelObj->getRoomFlagsInFilter($filter);
                    foreach my $name (@nameList) {

                        my $obj = $self->worldModelObj->ivShow('roomFlagHash', $name);
                        if ($obj) {

                            my $menuItem = Gtk3::MenuItem->new($obj->descrip);
                            $menuItem->signal_connect('activate' => sub {

                                # Toggle the flags for all selected rooms, redraw them and (if the
                                #   flag is one of the hazardous room flags) recalculate the
                                #   regionmap's paths. The TRUE argument tells the world model to
                                #   redraw the rooms
                                $self->worldModelObj->toggleRoomFlags(
                                    $self->session,
                                    TRUE,
                                    $obj->name,
                                    $self->compileSelectedRooms(),
                                );
                            });
                            $subSubMenu_filter->append($menuItem);
                        }
                    }

                    if (! @nameList) {

                        my $menuItem = Gtk3::MenuItem->new('(No flags in this filter)');
                        $menuItem->set_sensitive(FALSE);
                        $subSubMenu_filter->append($menuItem);
                    }

                    my $menuItem = Gtk3::MenuItem->new(ucfirst($filter));
                    $menuItem->set_submenu($subSubMenu_filter);
                    $subMenu_toggleRoomFlag->append($menuItem);
                }

            } else {

                # Show selected room flags, sorted only by priority
                my %showHash = $self->worldModelObj->getVisibleRoomFlags();
                if (%showHash) {

                    foreach my $obj (sort {$a->priority <=> $b->priority} (values %showHash)) {

                        my $menuItem = Gtk3::MenuItem->new($obj->descrip);
                        $menuItem->signal_connect('activate' => sub {

                            # Toggle the flags for all selected rooms, redraw them and (if the
                            #   flag is one of the hazardous room flags) recalculate the
                            #   regionmap's paths. The TRUE argument tells the world model to
                            #   redraw the rooms
                            $self->worldModelObj->toggleRoomFlags(
                                $self->session,
                                TRUE,
                                $obj->name,
                                $self->compileSelectedRooms(),
                            );
                        });
                        $subMenu_toggleRoomFlag->append($menuItem);
                    }

                } else {

                    my $menuItem = Gtk3::MenuItem->new('(None are marked visible)');
                    $menuItem->set_sensitive(FALSE);
                    $subMenu_toggleRoomFlag->append($menuItem);
                }
            }

        my $item_toggleRoomFlag = Gtk3::MenuItem->new('To_ggle room flags');
        $item_toggleRoomFlag->set_submenu($subMenu_toggleRoomFlag);
        $menu_rooms->append($item_toggleRoomFlag);

            # 'Other room features' submenu
            my $subMenu_roomFeatures = Gtk3::Menu->new();

                # 'Update character visits' sub-submenu
                my $subSubMenu_updateVisits = Gtk3::Menu->new();

                my $item_increaseSetCurrent = Gtk3::MenuItem->new('Increase & set _current');
                $item_increaseSetCurrent->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('increase');
                    $self->mapObj->setCurrentRoom($self->selectedRoom);
                });
                $subSubMenu_updateVisits->append($item_increaseSetCurrent);

                $subSubMenu_updateVisits->append(Gtk3::SeparatorMenuItem->new()); # Separator

                my $item_increaseVisits = Gtk3::MenuItem->new('_Increase by one');
                $item_increaseVisits->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('increase');
                });
                $subSubMenu_updateVisits->append($item_increaseVisits);

                my $item_decreaseVisits = Gtk3::MenuItem->new('_Decrease by one');
                $item_decreaseVisits->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('decrease');
                });
                $subSubMenu_updateVisits->append($item_decreaseVisits);

                my $item_manualVisits = Gtk3::MenuItem->new('Set _manually');
                $item_manualVisits->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('manual');
                });
                $subSubMenu_updateVisits->append($item_manualVisits);

                my $item_resetVisits = Gtk3::MenuItem->new('_Reset to zero');
                $item_resetVisits->signal_connect('activate' => sub {

                    $self->updateVisitsCallback('reset');
                });
                $subSubMenu_updateVisits->append($item_resetVisits);

                $subSubMenu_updateVisits->append(Gtk3::SeparatorMenuItem->new()); # Separator

                my $item_toggleGraffiti = Gtk3::MenuItem->new('Toggle _graffiti');
                $item_toggleGraffiti->signal_connect('activate' => sub {

                    $self->toggleGraffitiCallback();
                });
                $subSubMenu_updateVisits->append($item_toggleGraffiti);
                # (Also requires $self->graffitiModeFlag)
                if (! $self->graffitiModeFlag) {

                    $item_toggleGraffiti->set_sensitive(FALSE);
                }

            my $item_updateVisits = Gtk3::MenuItem->new('Update character _visits');
            $item_updateVisits->set_submenu($subSubMenu_updateVisits);
            $subMenu_roomFeatures->append($item_updateVisits);

            # 'Room exclusivity' submenu
            my $subMenu_exclusivity = Gtk3::Menu->new();

                my $item_toggleExclusivity = Gtk3::MenuItem->new('_Toggle exclusivity');
                $item_toggleExclusivity->signal_connect('activate' => sub {

                    $self->toggleExclusiveProfileCallback();
                });
                $subMenu_exclusivity->append($item_toggleExclusivity);

                my $item_addExclusiveProf = Gtk3::MenuItem->new('_Add exclusive profile...');
                $item_addExclusiveProf->signal_connect('activate' => sub {

                    $self->addExclusiveProfileCallback();
                });
                $subMenu_exclusivity->append($item_addExclusiveProf);

                my $item_clearExclusiveProf = Gtk3::MenuItem->new('_Clear exclusive profiles');
                $item_clearExclusiveProf->signal_connect('activate' => sub {

                    $self->resetExclusiveProfileCallback();
                });
                $subMenu_exclusivity->append($item_clearExclusiveProf);

            my $item_exclusivity = Gtk3::MenuItem->new('Room _exclusivity');
            $item_exclusivity->set_submenu($subMenu_exclusivity);
            $subMenu_roomFeatures->append($item_exclusivity);

                # 'Source code' sub-submenu
                my $subSubMenu_sourceCode = Gtk3::Menu->new();

                my $item_setFilePath = Gtk3::MenuItem->new('_Set file path...');
                $item_setFilePath->signal_connect('activate' => sub {

                    $self->setFilePathCallback();
                });
                $subSubMenu_sourceCode->append($item_setFilePath);

                my $item_setVirtualArea = Gtk3::MenuItem->new('Set virtual _area...');
                $item_setVirtualArea->signal_connect('activate' => sub {

                    $self->setVirtualAreaCallback(TRUE);
                });
                $subSubMenu_sourceCode->append($item_setVirtualArea);

                my $item_resetVirtualArea = Gtk3::MenuItem->new('_Reset virtual area...');
                $item_resetVirtualArea->signal_connect('activate' => sub {

                    $self->setVirtualAreaCallback(FALSE);
                });
                $subSubMenu_sourceCode->append($item_resetVirtualArea);

                $subSubMenu_sourceCode->append(Gtk3::SeparatorMenuItem->new());  # Separator

                my $item_viewSource = Gtk3::MenuItem->new('_View source file...');
                $item_viewSource->signal_connect('activate' => sub {

                    my $flag;

                    if ($self->selectedRoom) {

                        if (! $self->selectedRoom->virtualAreaPath) {
                            $flag = FALSE;
                        } else {
                            $flag = TRUE;
                        }

                        # Show source code file
                        $self->quickFreeWin(
                            'Games::Axmud::OtherWin::SourceCode',
                            $self->session,
                            # Config
                            'model_obj' => $self->selectedRoom,
                            'virtual_flag' => $flag,
                        );
                    }
                });
                $subSubMenu_sourceCode->append($item_viewSource);
                # (Also requires either $self->selectedRoom->sourceCodePath or
                #   $self->selectedRoom->virtualAreaPath)
                if (
                    ! $self->selectedRoom->sourceCodePath
                    && ! $self->selectedRoom->virtualAreaPath
                ) {
                    $item_viewSource->set_sensitive(FALSE);
                }

                my $item_editSource = Gtk3::MenuItem->new('Edit so_urce file...');
                $item_editSource->signal_connect('activate' => sub {

                    if ($self->selectedRoom) {

                        if (! $self->selectedRoom->virtualAreaPath) {

                            # Edit source code file
                            $self->editFileCallback();

                        } else {

                            # Edit virtual area file
                            $self->editFileCallback(TRUE);
                        }
                    }
                });
                $subSubMenu_sourceCode->append($item_editSource);
                # (Also requires either $self->selectedRoom->sourceCodePath or
                #   $self->selectedRoom->virtualAreaPath)
                if (
                    ! $self->selectedRoom->sourceCodePath
                    && ! $self->selectedRoom->virtualAreaPath
                ) {
                    $item_editSource->set_sensitive(FALSE);
                }

            my $item_sourceCode = Gtk3::MenuItem->new('Source _code');
            $item_sourceCode->set_submenu($subSubMenu_sourceCode);
            $subMenu_roomFeatures->append($item_sourceCode);

            $subMenu_roomFeatures->append(Gtk3::SeparatorMenuItem->new());  # Separator

            my $item_setInteriorOffsets = Gtk3::MenuItem->new('_Synchronise grid coordinates...');
            $item_setInteriorOffsets->signal_connect('activate' => sub {

                $self->setInteriorOffsetsCallback();
            });
            $subMenu_roomFeatures->append($item_setInteriorOffsets);

            my $item_resetInteriorOffsets = Gtk3::MenuItem->new('_Reset grid coordinates');
            $item_resetInteriorOffsets->signal_connect('activate' => sub {

                $self->resetInteriorOffsetsCallback();
            });
            $subMenu_roomFeatures->append($item_resetInteriorOffsets);

        my $item_roomFeatures = Gtk3::MenuItem->new('Ot_her room features');
        $item_roomFeatures->set_submenu($subMenu_roomFeatures);
        $menu_rooms->append($item_roomFeatures);

        $menu_rooms->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_deleteRoom = Gtk3::ImageMenuItem->new('_Delete room');
        my $img_deleteRoom = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_deleteRoom->set_image($img_deleteRoom);
        $item_deleteRoom->signal_connect('activate' => sub {

            $self->deleteRoomsCallback();
        });
        $menu_rooms->append($item_deleteRoom);

        # Setup complete
        $menu_rooms->show_all();

        return $menu_rooms;
    }

    sub enableRoomTagsPopupMenu {

        # Called by $self->canvasObjEventHandler
        # Creates a popup-menu for the selected room tag
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->enableRoomTagsPopupMenu',
                @_,
            );
        }

        # Set up the popup menu
        my $menu_tags = Gtk3::Menu->new();
        if (! $menu_tags) {

            return undef;
        }

        # (Everything here assumes $self->currentRegionmap and $self->selectedRoomTag)

        my $item_editTag = Gtk3::MenuItem->new('_Set room tag...');
        $item_editTag->signal_connect('activate' => sub {

            $self->setRoomTagCallback();
        });
        $menu_tags->append($item_editTag);

        my $item_resetPosition = Gtk3::MenuItem->new('_Reset position');
        $item_resetPosition->signal_connect('activate' => sub {

            if ($self->selectedRoomTag) {

                $self->worldModelObj->resetRoomOffsets(
                    TRUE,                       # Update Automapper windows now
                    1,                          # Mode 1 - reset room tag only
                    $self->selectedRoomTag,     # Set to the parent room's blessed reference
                );
            }
        });
        $menu_tags->append($item_resetPosition);

        # Setup complete
        $menu_tags->show_all();

        return $menu_tags;
    }

    sub enableRoomGuildsPopupMenu {

        # Called by $self->canvasObjEventHandler
        # Creates a popup-menu for the selected room guild
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->enableRoomGuildsPopupMenu',
                @_,
            );
        }

        # Set up the popup menu
        my $menu_guilds = Gtk3::Menu->new();
        if (! $menu_guilds) {

            return undef;
        }

        # (Everything here assumes $self->currentRegionmap and $self->selectedRoomGuild)

        my $item_editGuild = Gtk3::MenuItem->new('_Set room guild...');
        $item_editGuild->signal_connect('activate' => sub {

            $self->setRoomGuildCallback();
        });
        $menu_guilds->append($item_editGuild);

        my $item_resetPosition = Gtk3::MenuItem->new('_Reset position');
        $item_resetPosition->signal_connect('activate' => sub {

            if ($self->selectedRoomGuild) {

                $self->worldModelObj->resetRoomOffsets(
                    TRUE,                       # Update Automapper windows now
                    2,                          # Mode 2 - reset room guild only
                    $self->selectedRoomGuild,   # Set to the parent room's blessed reference
                );
            }
        });
        $menu_guilds->append($item_resetPosition);

        # Setup complete
        $menu_guilds->show_all();

        return $menu_guilds;
    }

    sub enableExitsPopupMenu {

        # Called by $self->canvasObjEventHandler
        # Creates a popup-menu for the selected exit
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Local variables
        my @titleList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableExitsPopupMenu', @_);
        }

        # Set up the popup menu
        my $menu_exits = Gtk3::Menu->new();
        if (! $menu_exits) {

            return undef;
        }

        # (Everything here assumes $self->currentRegionmap and $self->selectedExit)

            # 'Allocate map direction' submenu
            my $subMenu_setDir = Gtk3::Menu->new();

            my $item_changeDir = Gtk3::MenuItem->new('_Change direction...');
            $item_changeDir->signal_connect('activate' => sub {

                $self->changeDirCallback();
            });
            $subMenu_setDir->append($item_changeDir);
            # (Also requires $self->selectedExit->drawMode is 'primary' or 'perm_alloc'
            if (
                $self->selectedExit->drawMode ne 'primary'
                && $self->selectedExit->drawMode ne 'perm_alloc'
            ) {
                $item_changeDir->set_sensitive(FALSE);
            }

            my $item_altDir = Gtk3::MenuItem->new('Set _alternative direction(s)...');
            $item_altDir->signal_connect('activate' => sub {

                $self->setAltDirCallback();
            });
            $subMenu_setDir->append($item_altDir);

        my $item_setDir = Gtk3::MenuItem->new('Set di_rection');
        $item_setDir->set_submenu($subMenu_setDir);
        $menu_exits->append($item_setDir);

        my $item_setAssisted = Gtk3::MenuItem->new('Set assisted _move...');
        $item_setAssisted->signal_connect('activate' => sub {

            $self->setAssistedMoveCallback();
        });
        $menu_exits->append($item_setAssisted);
        # (Also requires $self->selectedExit->drawMode 'primary', 'temp_unalloc' or 'perm_unalloc')
        if ($self->selectedExit->drawMode eq 'temp_alloc') {

            $item_setAssisted->set_sensitive(FALSE);
        }

            # 'Allocate map direction' submenu
            my $subMenu_allocateMapDir = Gtk3::Menu->new();

            my $item_allocatePrimary = Gtk3::MenuItem->new('Choose _direction...');
            $item_allocatePrimary->signal_connect('activate' => sub {

                $self->allocateMapDirCallback();
            });
            $subMenu_allocateMapDir->append($item_allocatePrimary);


            my $item_confirmTwoWay = Gtk3::MenuItem->new('Confirm _two-way exit...');
            $item_confirmTwoWay->signal_connect('activate' => sub {

                $self->confirmTwoWayCallback();
            });
            $subMenu_allocateMapDir->append($item_confirmTwoWay);

        my $item_allocateMapDir = Gtk3::MenuItem->new('_Allocate map direction...');
        $item_allocateMapDir->set_submenu($subMenu_allocateMapDir);
        $menu_exits->append($item_allocateMapDir);
        # (Also requires $self->selectedExit->drawMode is 'temp_alloc' or 'temp_unalloc')
        if (
            $self->selectedExit->drawMode ne 'temp_alloc'
            && $self->selectedExit->drawMode ne 'temp_unalloc'
        ) {
            $item_allocateMapDir->set_sensitive(FALSE);
        }

        my $item_allocateShadow = Gtk3::MenuItem->new('Allocate _shadow...');
        $item_allocateShadow->signal_connect('activate' => sub {

            $self->allocateShadowCallback();
        });
        $menu_exits->append($item_allocateShadow);
        # (Also requires $self->selectedExit->drawMode is 'temp_alloc' or 'temp_unalloc')
        if (
            $self->selectedExit->drawMode ne 'temp_alloc'
            && $self->selectedExit->drawMode ne 'temp_unalloc'
        ) {
            $item_allocateShadow->set_sensitive(FALSE);
        }

        $menu_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_connectExitToClick = Gtk3::MenuItem->new('_Connect to click');
        $item_connectExitToClick->signal_connect('activate' => sub {

            $self->connectToClickCallback();
        });
        $menu_exits->append($item_connectExitToClick);
        # (Also requires $self->selectedExit->drawMode 'primary', 'temp_unalloc' or 'perm_unalloc')
        if ($self->selectedExit->drawMode eq 'temp_alloc') {

            $item_connectExitToClick->set_sensitive(FALSE);
        }

        my $item_disconnectExit = Gtk3::MenuItem->new('D_isconnect exit');
        $item_disconnectExit->signal_connect('activate' => sub {

            $self->disconnectExitCallback();
        });
        $menu_exits->append($item_disconnectExit);

        $menu_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_addExitBend = Gtk3::MenuItem->new('Add _bend');
        $item_addExitBend->signal_connect('activate' => sub {

            $self->addBendCallback();
        });
        $menu_exits->append($item_addExitBend);
        # (Also requires a $self->selectedExit that's a one-way or two-way broken exit, not a region
        #   exit, and also defined values for $self->exitClickXPosn and $self->exitClickYPosn)
        if (
            (! $self->selectedExit->oneWayFlag && ! $self->selectedExit->twinExit)
            || $self->selectedExit->regionFlag
            || ! defined $self->exitClickXPosn
            || ! defined $self->exitClickYPosn
        ) {
            $item_addExitBend->set_sensitive(FALSE);
        }

        my $item_removeExitBend = Gtk3::MenuItem->new('Remo_ve bend');
        $item_removeExitBend->signal_connect('activate' => sub {

            $self->removeBendCallback();
        });
        $menu_exits->append($item_removeExitBend);
        # (Also requires a $self->selectedExit that's a one-way or two-way exit with a bend, and
        #   also defined values for $self->exitClickXPosn and $self->exitClickYPosn)
        if (
            (! $self->selectedExit->oneWayFlag && ! $self->selectedExit->twinExit)
            || ! $self->selectedExit->bendOffsetList
            || ! defined $self->exitClickXPosn
            || ! defined $self->exitClickYPosn
        ) {
            $item_removeExitBend->set_sensitive(FALSE);
        }

        $menu_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

            # 'Set ornaments' submenu
            my $subMenu_setOrnament = Gtk3::Menu->new();

            # Create a list of exit ornament types, in groups of two, in the form
            #   (menu_item_title, exit_ornament_type)
            @titleList = (
                '_No ornament', 'none',
                '_Openable exit', 'open',
                '_Lockable exit', 'lock',
                '_Pickable exit', 'pick',
                '_Breakable exit', 'break',
                '_Impassable exit', 'impass',
                '_Mystery exit', 'mystery',
            );

            do {

                my ($title, $type);

                $title = shift @titleList;
                $type = shift @titleList;

                my $menuItem = Gtk3::MenuItem->new($title);
                $menuItem->signal_connect('activate' => sub {

                    $self->exitOrnamentCallback($type);
                });
                $subMenu_setOrnament->append($menuItem);

            } until (! @titleList);

            $subMenu_setOrnament->append(Gtk3::SeparatorMenuItem->new());   # Separator

            my $item_setTwinOrnament = Gtk3::CheckMenuItem->new('Also set _twin exits');
            $item_setTwinOrnament->set_active($self->worldModelObj->setTwinOrnamentFlag);
            $item_setTwinOrnament->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFlag(
                        'setTwinOrnamentFlag',
                        $item_setTwinOrnament->get_active(),
                        FALSE,      # Don't call $self->redrawRegions
                        'also_set_twin_exits',
                    );
                }
            });
            $subMenu_setOrnament->append($item_setTwinOrnament);

        my $item_setOrnament = Gtk3::MenuItem->new('Set _ornaments');
        $item_setOrnament->set_submenu($subMenu_setOrnament);
        $menu_exits->append($item_setOrnament);

            # 'Set exit type' submenu
            my $subMenu_setExitType = Gtk3::Menu->new();

                # 'Set hidden' sub-submenu
                my $subSubMenu_setHidden = Gtk3::Menu->new();

                my $item_setHiddenExit = Gtk3::MenuItem->new('Mark exit _hidden');
                $item_setHiddenExit->signal_connect('activate' => sub {

                    $self->hiddenExitCallback(TRUE);
                });
                $subSubMenu_setHidden->append($item_setHiddenExit);

                my $item_setNotHiddenExit = Gtk3::MenuItem->new('Mark exit _not hidden');
                $item_setNotHiddenExit->signal_connect('activate' => sub {

                    $self->hiddenExitCallback(FALSE);
                });
                $subSubMenu_setHidden->append($item_setNotHiddenExit);

            my $item_setHidden = Gtk3::MenuItem->new('Set _hidden');
            $item_setHidden->set_submenu($subSubMenu_setHidden);
            $subMenu_setExitType->append($item_setHidden);

                # 'Set broken' sub-submenu
                my $subSubMenu_setBroken = Gtk3::Menu->new();

                my $item_markBrokenExit = Gtk3::MenuItem->new('_Mark exit as broken');
                $item_markBrokenExit->signal_connect('activate' => sub {

                    $self->markBrokenExitCallback();
                });
                $subSubMenu_setBroken->append($item_markBrokenExit);

                my $item_toggleBrokenExit = Gtk3::MenuItem->new('_Toggle bent broken exit');
                $item_toggleBrokenExit->signal_connect('activate' => sub {

                    $self->worldModelObj->toggleBentExit(
                        TRUE,                       # Update Automapper windows now
                        $self->selectedExit,
                    );
                });
                $subSubMenu_setBroken->append($item_toggleBrokenExit);
                # (Also requires $self->selectedExit->brokenFlag)
                if (! $self->selectedExit->brokenFlag) {

                    $item_toggleBrokenExit->set_sensitive(FALSE);
                }

                $subSubMenu_setBroken->append(Gtk3::SeparatorMenuItem->new());  # Separator

                my $item_restoreBrokenExit = Gtk3::MenuItem->new('_Restore unbroken exit');
                $item_restoreBrokenExit->signal_connect('activate' => sub {

                    $self->restoreBrokenExitCallback();
                });
                $subSubMenu_setBroken->append($item_restoreBrokenExit);

            my $item_setBroken = Gtk3::MenuItem->new('Set _broken');
            $item_setBroken->set_submenu($subSubMenu_setBroken);
            $subMenu_setExitType->append($item_setBroken);

                # 'Set one-way' sub-submenu
                my $subSubMenu_setOneWay = Gtk3::Menu->new();

                my $item_markOneWayExit = Gtk3::MenuItem->new('_Mark exit as one-way');
                $item_markOneWayExit->signal_connect('activate' => sub {

                    $self->markOneWayExitCallback();
                });
                $subSubMenu_setOneWay->append($item_markOneWayExit);

                $subSubMenu_setOneWay->append(Gtk3::SeparatorMenuItem->new());  # Separator

                my $item_restoreUncertainExit = Gtk3::MenuItem->new('Restore _uncertain exit');
                $item_restoreUncertainExit->signal_connect('activate' => sub {

                    $self->restoreOneWayExitCallback(FALSE);
                });
                $subSubMenu_setOneWay->append($item_restoreUncertainExit);

                my $item_restoreTwoWayExit = Gtk3::MenuItem->new('Restore _two-way exit');
                $item_restoreTwoWayExit->signal_connect('activate' => sub {

                    $self->restoreOneWayExitCallback(TRUE);
                });
                $subSubMenu_setOneWay->append($item_restoreTwoWayExit);

                $subSubMenu_setOneWay->append(Gtk3::SeparatorMenuItem->new());  # Separator

                my $item_setIncomingDir = Gtk3::MenuItem->new('Set incoming _direction...');
                $item_setIncomingDir->signal_connect('activate' => sub {

                    $self->setIncomingDirCallback();
                });
                $subSubMenu_setOneWay->append($item_setIncomingDir);
                # (Also requires either a $self->selectedExit which is a one-way exit)
                if (! $self->selectedExit->oneWayFlag) {

                    $item_setIncomingDir->set_sensitive(FALSE);
                }

            my $item_setOneWay = Gtk3::MenuItem->new('Set _one-way');
            $item_setOneWay->set_submenu($subSubMenu_setOneWay);
            $subMenu_setExitType->append($item_setOneWay);

                # 'Set retracing' sub-submenu
                my $subSubMenu_setRetracing = Gtk3::Menu->new();

                my $item_markRetracingExit = Gtk3::MenuItem->new('_Mark exit as retracing');
                $item_markRetracingExit->signal_connect('activate' => sub {

                    $self->markRetracingExitCallback();
                });
                $subSubMenu_setRetracing->append($item_markRetracingExit);

                $subSubMenu_setRetracing->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_restoreRetracingExit = Gtk3::MenuItem->new('_Restore incomplete exit');
                $item_restoreRetracingExit->signal_connect('activate' => sub {

                    $self->restoreRetracingExitCallback();
                });
                $subSubMenu_setRetracing->append($item_restoreRetracingExit);

            my $item_setRetracing = Gtk3::MenuItem->new('Set _retracing');
            $item_setRetracing->set_submenu($subSubMenu_setRetracing);
            $subMenu_setExitType->append($item_setRetracing);

                # 'Set random' sub-submenu
                my $subSubMenu_setRandomExit = Gtk3::Menu->new();

                my $item_markRandomRegion = Gtk3::MenuItem->new(
                    'Set random destination in same _region',
                );
                $item_markRandomRegion->signal_connect('activate' => sub {

                    $self->markRandomExitCallback('same_region');
                });
                $subSubMenu_setRandomExit->append($item_markRandomRegion);

                my $item_markRandomAnywhere = Gtk3::MenuItem->new(
                    'Set random destination _anywhere',
                );
                $item_markRandomAnywhere->signal_connect('activate' => sub {

                    $self->markRandomExitCallback('any_region');
                });
                $subSubMenu_setRandomExit->append($item_markRandomAnywhere);

                my $item_randomTempRegion = Gtk3::MenuItem->new(
                    '_Create destination in temporary region',
                );
                $item_randomTempRegion->signal_connect('activate' => sub {

                    $self->markRandomExitCallback('temp_region');
                });
                $subSubMenu_setRandomExit->append($item_randomTempRegion);

                my $item_markRandomList = Gtk3::MenuItem->new('_Use list of random destinations');
                $item_markRandomList->signal_connect('activate' => sub {

                    $self->markRandomExitCallback('room_list');
                });
                $subSubMenu_setRandomExit->append($item_markRandomList);

                $subSubMenu_setRandomExit->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_restoreRandomExit = Gtk3::MenuItem->new('Restore _incomplete exit');
                $item_restoreRandomExit->signal_connect('activate' => sub {

                    $self->restoreRandomExitCallback();
                });
                $subSubMenu_setRandomExit->append($item_restoreRandomExit);

            my $item_setRandomExit = Gtk3::MenuItem->new('Set r_andom');
            $item_setRandomExit->set_submenu($subSubMenu_setRandomExit);
            $subMenu_setExitType->append($item_setRandomExit);

                # 'Set super' sub-submenu
                my $subSubMenu_setSuperExit = Gtk3::Menu->new();

                my $item_markSuper = Gtk3::MenuItem->new('Mark exit as _super-region exit');
                $item_markSuper->signal_connect('activate' => sub {

                    $self->markSuperExitCallback(FALSE);
                });
                $subSubMenu_setSuperExit->append($item_markSuper);

                my $item_markSuperExcl = Gtk3::MenuItem->new(
                    'Mark exit as _exclusive super-region exit',
                );
                $item_markSuperExcl->signal_connect('activate' => sub {

                    $self->markSuperExitCallback(TRUE);
                });
                $subSubMenu_setSuperExit->append($item_markSuperExcl);

                $subSubMenu_setSuperExit->append(Gtk3::SeparatorMenuItem->new());    # Separator

                my $item_markNotSuper = Gtk3::MenuItem->new('Mark exit as _normal region exit');
                $item_markNotSuper->signal_connect('activate' => sub {

                    $self->restoreSuperExitCallback();
                });
                $subSubMenu_setSuperExit->append($item_markNotSuper);

            my $item_setSuperExit = Gtk3::MenuItem->new('Set _super');
            $item_setSuperExit->set_submenu($subSubMenu_setSuperExit);
            $subMenu_setExitType->append($item_setSuperExit);
            # (Also requires $self->selectedExit->regionFlag)
            if (! $self->selectedExit->regionFlag) {

                $item_setSuperExit->set_sensitive(FALSE);
            }

            $subMenu_setExitType->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_setExitTwin = Gtk3::MenuItem->new('Set exit _twin...');
            $item_setExitTwin->signal_connect('activate' => sub {

                $self->setExitTwinCallback();
            });
            $subMenu_setExitType->append($item_setExitTwin);
            # (Also requires either a $self->selectedExit which is either a one-way exit or an
            #   uncertain exit)
            if (
                ! $self->selectedExit->oneWayFlag
                || ! (
                    $self->selectedExit->destRoom
                    && ! $self->selectedExit->twinExit
                    && ! $self->selectedExit->retraceFlag
                    && $self->selectedExit->randomType eq 'none'
                )
            ) {
                $item_setExitTwin->set_sensitive(FALSE);
            }

        my $item_setExitType = Gtk3::MenuItem->new('Set _exit type');
        $item_setExitType->set_submenu($subMenu_setExitType);
        $menu_exits->append($item_setExitType);

            # 'Exit tags' submenu
            my $subMenu_exitTags = Gtk3::Menu->new();

            my $item_editTag = Gtk3::MenuItem->new('_Edit exit tag');
            $item_editTag->signal_connect('activate' => sub {

                $self->editExitTagCallback();
            });
            $subMenu_exitTags->append($item_editTag);

            my $item_toggleExitTag = Gtk3::MenuItem->new('_Toggle exit tag');
            $item_toggleExitTag->signal_connect('activate' => sub {

                $self->toggleExitTagCallback();
            });
            $subMenu_exitTags->append($item_toggleExitTag);

            $subMenu_exitTags->append(Gtk3::SeparatorMenuItem->new());    # Separator

            my $item_resetPosition = Gtk3::MenuItem->new('_Reset text position');
            $item_resetPosition->signal_connect('activate' => sub {

                $self->resetExitOffsetsCallback();
            });
            $subMenu_exitTags->append($item_resetPosition);

        my $item_exitTags = Gtk3::MenuItem->new('Exit _tags');
        $item_exitTags->set_submenu($subMenu_exitTags);
        $menu_exits->append($item_exitTags);
        # (Also requires either a $self->selectedExit which is a region exit)
        if (! $self->selectedExit->regionFlag) {

            $item_exitTags->set_sensitive(FALSE);
        }

        $menu_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_editExit = Gtk3::ImageMenuItem->new('Edit e_xit...');
        my $img_editExit = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_editExit->set_image($img_editExit);
        $item_editExit->signal_connect('activate' => sub {

            $self->editExitCallback();
        });
        $menu_exits->append($item_editExit);

        $menu_exits->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_deleteExit = Gtk3::ImageMenuItem->new('_Delete exit');
        my $img_deleteExit = Gtk3::Image->new_from_stock('gtk-add', 'menu');
        $item_deleteExit->set_image($img_deleteExit);
        $item_deleteExit->signal_connect('activate' => sub {

            $self->deleteExitCallback();
        });
        $menu_exits->append($item_deleteExit);

        # Setup complete
        $menu_exits->show_all();

        return $menu_exits;
    }

    sub enableExitTagsPopupMenu {

        # Called by $self->canvasObjEventHandler
        # Creates a popup-menu for the selected exit tag
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->enableExitTagsPopupMenu',
                @_,
            );
        }

        # Set up the popup menu
        my $menu_tags = Gtk3::Menu->new();
        if (! $menu_tags) {

            return undef;
        }

        # (Everything here assumes $self->currentRegionmap and $self->selectedExitTag)

        my $item_editTag = Gtk3::MenuItem->new('_Edit exit tag');
        $item_editTag->signal_connect('activate' => sub {

            $self->editExitTagCallback();
        });
        $menu_tags->append($item_editTag);

        my $item_cancelTag = Gtk3::MenuItem->new('_Cancel exit tag');
        $item_cancelTag->signal_connect('activate' => sub {

            $self->toggleExitTagCallback();
        });
        $menu_tags->append($item_cancelTag);

        $menu_tags->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_viewDestination = Gtk3::MenuItem->new('_View destination');
        $item_viewDestination->signal_connect('activate' => sub {

            $self->viewExitDestination();
        });
        $menu_tags->append($item_viewDestination);

        $menu_tags->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_resetPosition = Gtk3::MenuItem->new('_Reset position');
        $item_resetPosition->signal_connect('activate' => sub {

            $self->resetExitOffsetsCallback();
        });
        $menu_tags->append($item_resetPosition);

        # Setup complete
        $menu_tags->show_all();

        return $menu_tags;
    }

    sub enableLabelsPopupMenu {

        # Called by $self->canvasObjEventHandler
        # Creates a popup-menu for the selected label
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Menu created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableLabelsPopupMenu', @_);
        }

        # Set up the popup menu
        my $menu_labels = Gtk3::Menu->new();
        if (! $menu_labels) {

            return undef;
        }

        # (Everything here assumes $self->currentRegionmap and $self->selectedLabel)

        my $item_setLabel = Gtk3::ImageMenuItem->new('_Set label...');
        my $img_setLabel = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_setLabel->set_image($img_setLabel);
        $item_setLabel->signal_connect('activate' => sub {

            $self->setLabelCallback(FALSE)
        });
        $menu_labels->append($item_setLabel);

        my $item_customiseLabel = Gtk3::ImageMenuItem->new('_Customise label...');
        my $img_customiseLabel = Gtk3::Image->new_from_stock('gtk-edit', 'menu');
        $item_customiseLabel->set_image($img_customiseLabel);
        $item_customiseLabel->signal_connect('activate' => sub {

            $self->setLabelCallback(TRUE);
        });
        $menu_labels->append($item_customiseLabel);

        $menu_labels->append(Gtk3::SeparatorMenuItem->new());  # Separator

            # 'Set label style' submenu
            my $subMenu_setStyle = Gtk3::Menu->new();

            foreach my $style (
                sort {lc($a) cmp lc($b)} ($self->worldModelObj->ivKeys('mapLabelStyleHash'))
            ) {
                my $item_thisStyle = Gtk3::MenuItem->new($style);
                $item_thisStyle->signal_connect('activate' => sub {

                    $self->setLabelDirectCallback($style);
                });
                $subMenu_setStyle->append($item_thisStyle);
            }

        my $item_setStyle = Gtk3::MenuItem->new('S_et label style');
        $item_setStyle->set_submenu($subMenu_setStyle);
        $menu_labels->append($item_setStyle);
        # (Also requires at least one label style)
        if (! $self->worldModelObj->mapLabelStyleHash) {

            $item_setStyle->set_sensitive(FALSE);
        }

        $menu_labels->append(Gtk3::SeparatorMenuItem->new());  # Separator

        my $item_deleteLabel = Gtk3::ImageMenuItem->new('_Delete label');
        my $img_deleteLabel = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_deleteLabel->set_image($img_deleteLabel);
        $item_deleteLabel->signal_connect('activate' => sub {

            if ($self->selectedLabel) {

                $self->worldModelObj->deleteLabels(
                    TRUE,           # Update Automapper windows now
                    $self->selectedLabel,
                );
            }
        });
        $menu_labels->append($item_deleteLabel);

        my $item_quickDelete = Gtk3::ImageMenuItem->new('_Quick label deletion...');
        my $img_quickDelete = Gtk3::Image->new_from_stock('gtk-delete', 'menu');
        $item_quickDelete->set_image($img_quickDelete);
        $item_quickDelete->signal_connect('activate' => sub {

            $self->session->pseudoCmd('quicklabeldelete', $self->pseudoCmdMode);
        });
        $menu_labels->append($item_quickDelete);

        # Setup complete
        $menu_labels->show_all();

        return $menu_labels;
    }

    # Toolbar widget methods

    sub enableToolbar {

        # Called by $self->drawWidgets
        # Sets up the Automapper window's Gtk3::Toolbar widget(s)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if one of the widgets can't be created
        #   Otherwise returns a list of Gtk3::Toolbar widgets created

        my ($self, $check) = @_;

        # Local variables
        my (
            $flag,
            @emptyList, @setList, @widgetList,
            %checkHash,
        );

        # Check for improper arguments
        if (defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->enableToolbar', @_);
            return @emptyList;
        }

        # Import the list of button sets from the world model
        # Remove 'default' (which shouldn't be there) and also remove any duplicates or unrecognised
        #   sets
        foreach my $set ($self->worldModelObj->buttonSetList) {

            if (
                $set ne $self->constToolbarDefaultSet
                && ! exists $checkHash{$set}
                && $self->ivExists('buttonSetHash', $set)
            ) {
                push (@setList, $set);
                # Watch out for duplicates
                $checkHash{$set} = undef;
            }
        }

        # Draw the original (first) toolbar. The TRUE argument means that the add/switcher buttons
        #   should be drawn
        my $origToolbar = $self->drawToolbar($self->toolbarOriginalSet, TRUE);

        if (! $origToolbar) {

            # Give up on the first error
            return @emptyList;

        } else {

            push (@widgetList, $origToolbar);
        }

        # Draw a toolbar for each button set in turn, updating IVs as we go
        foreach my $set (@setList) {

            my $otherToolbar = $self->drawToolbar($set);

            if (! $otherToolbar) {

                # Give up on the first error (@widgetList might be an empty list)
                return @widgetList;

            } else {

                push (@widgetList, $otherToolbar);
            }
        }

        # On success, update the world model's list of button sets (having removed anything that
        #   shouldn't be there
        $self->worldModelObj->set_buttonSetList(@setList);

        # If all button sets are visible, the 'add'/'switch' buttons in the default set are
        #   desensitised
        OUTER: foreach my $key ($self->ivKeys('buttonSetHash')) {

            if (! $self->ivShow('buttonSetHash', $key)) {

                $flag = TRUE;
                last OUTER;
            }
        }

        if (! $flag) {

            $self->toolbarAddButton->set_sensitive(FALSE);
            $self->toolbarSwitchButton->set_sensitive(FALSE);
        }

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        return @widgetList;
    }

    sub drawToolbar {

        # Called by $self->enableToolbar
        # Creates a new toolbar using the button set specified by the calling function, and updates
        #   IVs accordingly
        #
        # Expected arguments
        #   $set        - The button set to use (one of the items in $self->constButtonSetList)
        #
        # Optional arguments
        #   $origFlag   - If TRUE, this is the original (first) toolbar; FALSE (or 'undef') for any
        #                   subsequent toolbar
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Toolbar created

        my ($self, $set, $origFlag, $check) = @_;

        # Local variables
        my (
            $text, $text2, $text3,
            @buttonList,
        );

        # Check for improper arguments
        if (! defined $set || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawToolbar', @_);
        }

        # Check the button set actually exists
        if (! $self->ivExists('buttonSetHash', $set)) {

            return undef;
        }

        # Create the toolbar widget
        my $toolbar = Gtk3::Toolbar->new();
        if (! $toolbar) {

            return undef;
        }

        # Store the widget and update associated IVs
        $self->ivAdd('buttonSetHash', $set, TRUE);
        $self->ivPush('toolbarList', $toolbar);
        $self->ivAdd('toolbarHash', $toolbar, $set);

        # Use large icons, and allow the menu to shrink when there's not enough space for it
        $toolbar->set_icon_size('large-toolbar');
        $toolbar->set_show_arrow(TRUE);

        if ($axmud::CLIENT->toolbarLabelFlag) {

            # Otherwise, these values continue to be 'undef', which is what Gtk3::ToolButton is
            #   expecting
            $text = 'Switch button sets';
            $text2 = 'Add button set';
            $text3 = 'Remove button set';
        }

        # Add buttons that are displayed, regardless of which button set is visible
        if ($origFlag) {

            # Draw an add button, which adds new toolbars (unless all button sets are visible)
            my $toolButton = Gtk3::ToolButton->new(
                Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_add.png'),
                $text2,
            );
            $toolButton->signal_connect('clicked' => sub {

                # Switch to the next set of toolbar buttons
                $self->addToolbar();
            });
            $toolButton->set_tooltip_text('Add button set');
            $toolbar->insert($toolButton, -1);

            # Draw a switcher button, which cycles through button sets that aren't visible in other
            #   toolbars
            my $toolButton2 = Gtk3::ToolButton->new(
                Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_switch.png'),
                $text,
            );
            $toolButton2->signal_connect('clicked' => sub {

                # Switch to the next set of toolbar buttons
                $self->switchToolbarButtons();
            });
            $toolButton2->set_tooltip_text('Switch button sets');
            $toolbar->insert($toolButton2, -1);

            # Update IVs
            $self->ivPoke('toolbarOriginalSet', $set);
            $self->ivPoke('toolbarAddButton', $toolButton);
            $self->ivPoke('toolbarSwitchButton', $toolButton2);

        } else {

            # Draw a remove button, which adds new toolbars (unless all button sets are visible)
            my $toolButton = Gtk3::ToolButton->new(
                Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_remove.png'),
                $text3,
            );
            $toolButton->signal_connect('clicked' => sub {

                # Switch to the next set of toolbar buttons
                $self->removeToolbar($toolbar);
            });

            $toolButton->set_tooltip_text('Remove button set');
            $toolbar->insert($toolButton, -1);
        }

        # Immediately to the right of those buttons is a separator
        my $separator2 = Gtk3::SeparatorToolItem->new();
        $toolbar->insert($separator2, -1);

        # After the separator, we draw the specified button set. This function decides which
        #   specific function to call, and returns the result
        @buttonList = $self->chooseButtonSet($toolbar, $set);

        # Add the buttons/separators to the toolbar
        foreach my $button (@buttonList) {

            my $label;

            # (Separators don't have labels, so we need to check for that)
            if (! $axmud::CLIENT->toolbarLabelFlag && $button->isa('Gtk3::ToolButton')) {

                $button->set_label(undef);
            }

            $toolbar->insert($button, -1);
        }

        # Update IVs
        if ($origFlag) {

            $self->ivPoke('toolbarButtonList', @buttonList);
        }

        # Setup complete
        return $toolbar;
    }

    sub switchToolbarButtons {

        # Called by a ->signal_connect in $self->addToolbar whenever the user clicks the original
        #   toolbar's switcher button
        # Removes the existing button set (preserving the switcher and add buttons, and the
        #   separator that follows them), and then draws a new button set
        # NB This function is only called for the original (first) toolbar, not for any additional
        #   toolbars
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if there's an error
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $toolbar, $currentSet, $foundFlag, $nextSet,
            @setList, @beforeList, @afterList, @buttonList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->switchToolbarButtons', @_);
        }

        # Get the original toolbar (always the first one in this list)
        $toolbar = $self->ivFirst('toolbarList');
        if (! $toolbar) {

            return undef;
        }

        # Decide which button set to show next
        $currentSet = $self->ivShow('toolbarHash', $toolbar);
        if (! $currentSet) {

            return undef;
        }

        # Compile a list of all button sets but the current one, starting with those which appear
        #   after the current one, then those that appear before it
        #   e.g. (A B current D E F) > (D E F A B)
        @setList = $self->constButtonSetList;
        for (my $count = 0; $count < scalar @setList; $count++) {

            if ($setList[$count] eq $currentSet) {

                $foundFlag = TRUE;

            } elsif ($foundFlag) {

                push (@afterList, $setList[$count]);

            } else {

                push (@beforeList, $setList[$count]);
            }
        }

        # Go through that list, from the beginning, and use the first button set that's not already
        #   visible
        OUTER: foreach my $set (@afterList, @beforeList) {

            if (! $self->ivShow('buttonSetHash', $set)) {

                $nextSet = $set;
                last OUTER;
            }
        }

        if (! $nextSet) {

            # All button sets are visible; cannot switch set
            return undef;
        }

        # Remove the existing button set (preserving the switcher and add buttons, and the separator
        #   that follows them)
        foreach my $widget ($self->toolbarButtonList) {

            $axmud::CLIENT->desktopObj->removeWidget($toolbar, $widget);
        }

        # After the separator, we draw the specified button set. This function decides which
        #   specific function to call, and returns the result
        @buttonList = $self->chooseButtonSet($toolbar, $nextSet);

        # Add the buttons/separators to the toolbar
        foreach my $button (@buttonList) {

            my $label;

            # (Separators don't have labels, so we need to check for that)
            if (! $axmud::CLIENT->toolbarLabelFlag && $button->isa('Gtk3::ToolButton')) {

                $button->set_label(undef);
            }

            $toolbar->insert($button, -1);
        }

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        # Update IVs
        $self->ivAdd('buttonSetHash', $currentSet, FALSE);
        $self->ivAdd('buttonSetHash', $nextSet, TRUE);
        $self->ivAdd('toolbarHash', $toolbar, $nextSet);
        $self->ivPoke('toolbarButtonList', @buttonList);
        $self->ivPoke('toolbarOriginalSet', $nextSet);

        # Not worth calling $self->redrawWidgets, so must do a ->show_all()
        $toolbar->show_all();

        return 1;
    }

    sub addToolbar {

        # Called by a ->signal_connect in $self->drawToolbar whenever the user clicks the original
        #   toolbar's add button
        # Creates a popup menu containing all of the button sets that aren't currently visible, then
        #   imlements the user's choice
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if there's an error
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            @list,
            %hash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addToolbar', @_);
        }

        # Get a list of button sets that aren't already visible
        # NB The 'default' set can only be viewed in the original (first) toolbar, so it's not added
        #   to this list
        foreach my $set ($self->constButtonSetList) {

            my $descrip = $self->ivShow('constButtonDescripHash', $set);

            if ($set ne $self->constToolbarDefaultSet && ! $self->ivShow('buttonSetHash', $set)) {

                push (@list, $descrip);
                $hash{$descrip} = $set;
            }
        }

        if (! @list) {

            # All button sets are visible (this shouldn't happen)
            return undef;
        }

        # Set up the popup menu
        my $popupMenu = Gtk3::Menu->new();
        if (! $popupMenu) {

            return undef;
        }

        # Add a title menu item, which does nothing
        my $title_item = Gtk3::MenuItem->new('Add button set:');
        $title_item->signal_connect('activate' => sub {

            return undef;
        });
        $title_item->set_sensitive(FALSE);
        $popupMenu->append($title_item);

        $popupMenu->append(Gtk3::SeparatorMenuItem->new());     # Separator

        # Fill the popup menu with button sets
        foreach my $descrip (@list) {

            my $menu_item = Gtk3::MenuItem->new($descrip);
            $menu_item->signal_connect('activate' => sub {

                # Add the set to the world model's list of button sets...
                $self->worldModelObj->add_buttonSet($hash{$descrip});
                # ...then redraw the window component containing the toolbar(s)
                $self->redrawWidgets('toolbar');
            });
            $popupMenu->append($menu_item);
        }

        # Also add a 'Cancel' menu item, which does nothing
        $popupMenu->append(Gtk3::SeparatorMenuItem->new());     # Separator

        my $cancel_item = Gtk3::MenuItem->new('Cancel');
        $cancel_item->signal_connect('activate' => sub {

            return undef;
        });
        $popupMenu->append($cancel_item);

        # Display the popup menu
        $popupMenu->popup(
            undef, undef, undef, undef,
            1,                              # Left mouse button
            Gtk3::get_current_event_time(),
        );

        $popupMenu->show_all();

        # Operation complete. Now wait for the user's response
        return 1;
    }

    sub removeToolbar {

        # Called by a ->signal_connect in $self->drawToolbar whenever the user clicks on the remove
        #   button in any toolbar except the original one
        # Removes the specified toolbar and updates IVs
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget to be removed
        #
        # Return values
        #   'undef' on improper arguments or if there's an error
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my (
            $set,
            @modList,
        );

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->removeToolbar', @_);
        }

        # Check the toolbar widget still exists (no reason it shouldn't, but it doesn't hurt to
        #   check)
        if (! $self->ivExists('toolbarHash', $toolbar)) {

            return undef;

        } else {

            # Get the button set that was drawn in this toolbar
            $set = $self->ivShow('toolbarHash', $toolbar);
        }

        # Add the set to the world model's list of button sets...
        $self->worldModelObj->del_buttonSet($set);
        # ...then redraw the window component containing the toolbar(s)
        $self->redrawWidgets('toolbar');

        return 1;
    }

    sub chooseButtonSet {

        # Called by $self->drawToolbar and ->switchToolbarButtons
        # Calls the right function for the specified button set, and returns the result
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #   $set        - The button set to use (one of the items in $self->constButtonSetList)
        #
        # Return values
        #   'undef' on improper arguments or if there's an error
        #   1 otherwise

        my ($self, $toolbar, $set, $check) = @_;

        # Local variables
        my @emptyList;

        # Check for improper arguments
        if (! defined $toolbar || ! defined $set || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->chooseButtonSet', @_);
            return @emptyList;
        }

        if ($set eq 'default') {
            return $self->drawDefaultButtonSet($toolbar);
        } elsif ($set eq 'exits') {
            return $self->drawExitsButtonSet($toolbar);
        } elsif ($set eq 'painting') {
            return $self->drawPaintingButtonSet($toolbar);
        } elsif ($set eq 'quick') {
            return $self->drawQuickButtonSet($toolbar);
        } elsif ($set eq 'background') {
            return $self->drawBackgroundButtonSet($toolbar);
        } elsif ($set eq 'tracking') {
            return $self->drawTrackingButtonSet($toolbar);
        } elsif ($set eq 'misc') {
            return $self->drawMiscButtonSet($toolbar);
        } elsif ($set eq 'flags') {
            return $self->drawFlagsButtonSet($toolbar);
        } elsif ($set eq 'interiors') {
            return $self->drawInteriorsButtonSet($toolbar);
        } else {
            return @emptyList;
        }
    }

    sub drawDefaultButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my @buttonList;

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawDefaultButtonSet', @_);
        }

        # Radio button for 'wait mode'
        my $radioButton_waitMode = Gtk3::RadioToolButton->new(undef);
        if ($self->mode eq 'wait') {

            $radioButton_waitMode->set_active(TRUE);
        }
        $radioButton_waitMode->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_wait.png')
        );
        $radioButton_waitMode->set_label('Wait mode');
        $radioButton_waitMode->set_tooltip_text('Wait mode');
        $radioButton_waitMode->signal_connect('toggled' => sub {

            # (To stop the equivalent menu item from being toggled by the call to ->setMode, make
            #   use of $self->ignoreMenuUpdateFlag)
            if ($radioButton_waitMode->get_active && ! $self->ignoreMenuUpdateFlag) {

                $self->setMode('wait');
            }
        });
        push (@buttonList, $radioButton_waitMode);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_set_wait_mode', $radioButton_waitMode);

        # Radio button for 'follow mode'
        my $radioButton_followMode = Gtk3::RadioToolButton->new_from_widget($radioButton_waitMode);
        if ($self->mode eq 'follow') {

            $radioButton_followMode->set_active(TRUE);
        }
        $radioButton_followMode->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_follow.png')
        );
        $radioButton_followMode->set_label('Follow mode');
        $radioButton_followMode->set_tooltip_text('Follow mode');
        $radioButton_followMode->signal_connect('toggled' => sub {

            if ($radioButton_followMode->get_active && ! $self->ignoreMenuUpdateFlag) {

                $self->setMode('follow');
            }
        });
        push (@buttonList, $radioButton_followMode);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_set_follow_mode', $radioButton_followMode);

        # Radio button for 'update' mode
        my $radioButton_updateMode = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_followMode,
        );
        if ($self->mode eq 'update') {

            $radioButton_updateMode->set_active(TRUE);
        }
        $radioButton_updateMode->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_update.png')
        );
        $radioButton_updateMode->set_label('Update mode');
        $radioButton_updateMode->set_tooltip_text('Update mode');
        $radioButton_updateMode->signal_connect('toggled' => sub {

            if ($radioButton_updateMode->get_active && ! $self->ignoreMenuUpdateFlag) {

                $self->setMode('update');
            }
        });
        push (@buttonList, $radioButton_updateMode);
        # (Requires $self->currentRegionmap, GA::Obj::WorldModel->disableUpdateModeFlag set to
        #   FALSE and a session not in 'connect offline' mode
        $self->ivAdd('menuToolItemHash', 'icon_set_update_mode', $radioButton_updateMode);

#        # Separator
#        my $separator = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator);

        # DEBUG
        # Temporary fix for Gtk problems: on MSWin, don't show separators (this applies to several
        #   functions in the automapper window)
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toolbutton for 'move up level'
        my $toolButton_moveUpLevel = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_move_up.png'),
            'Move up level',
        );
        $toolButton_moveUpLevel->set_tooltip_text('Move up level');
        $toolButton_moveUpLevel->signal_connect('clicked' => sub {

            $self->setCurrentLevel($self->currentRegionmap->currentLevel + 1);

            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $self->restrictWidgets();
        });
        push (@buttonList, $toolButton_moveUpLevel);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_move_up_level', $toolButton_moveUpLevel);

        # Toolbutton for 'move down level'
        my $toolButton_moveDownLevel = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_move_down.png'),
            'Move down level',
        );
        $toolButton_moveDownLevel->set_tooltip_text('Move down level');
        $toolButton_moveDownLevel->signal_connect('clicked' => sub {

            $self->setCurrentLevel($self->currentRegionmap->currentLevel - 1);

            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $self->restrictWidgets();
        });
        push (@buttonList, $toolButton_moveDownLevel);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_move_down_level', $toolButton_moveDownLevel);

#        # Separator
#        my $separator2 = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator2);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toolbutton for 'reset locator'
        my $toolButton_resetLocator = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_reset_locator.png'),
            'Reset Locator task',
        );
        $toolButton_resetLocator->set_tooltip_text('Reset locator task');
        $toolButton_resetLocator->signal_connect('clicked' => sub {

            $self->resetLocatorCallback();
        });
        push (@buttonList, $toolButton_resetLocator);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_reset_locator', $toolButton_resetLocator);

        # Toolbutton for 'set current room'
        my $toolButton_setCurrentRoom = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_set.png'),
            'Set current room',
        );
        $toolButton_setCurrentRoom->set_tooltip_text('Set current room');
        $toolButton_setCurrentRoom->signal_connect('clicked' => sub {

            $self->mapObj->setCurrentRoom($self->selectedRoom);
        });
        push (@buttonList, $toolButton_setCurrentRoom);
        # (Requires $self->currentRegionmap & $self->selectedRoom)
        $self->ivAdd('menuToolItemHash', 'icon_set_current_room', $toolButton_setCurrentRoom);

        # Toolbutton for 'set failed exit'
        my $toolButton_setFailedExit = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_fail_exit.png'),
            'Set failed exit',
        );
        $toolButton_setFailedExit->set_tooltip_text('Set failed exit');
        $toolButton_setFailedExit->signal_connect('clicked' => sub {

            $self->session->pseudoCmd('insertfailexit');
        });
        push (@buttonList, $toolButton_setFailedExit);
        # (Requires $self->currentRegionmap and a current room
        $self->ivAdd('menuToolItemHash', 'icon_fail_exit', $toolButton_setFailedExit);

        # Toggle button for 'drag mode'
        my $toggleButton_dragMode = Gtk3::ToggleToolButton->new();
        $toggleButton_dragMode->set_active($self->dragModeFlag);
        $toggleButton_dragMode->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_drag_mode.png'),
        );
        $toggleButton_dragMode->set_label('Drag mode');
        $toggleButton_dragMode->set_tooltip_text('Drag mode');
        $toggleButton_dragMode->signal_connect('toggled' => sub {

            if ($toggleButton_dragMode->get_active()) {
                $self->ivPoke('dragModeFlag', TRUE);
            } else {
                $self->ivPoke('dragModeFlag', FALSE);
            }

            # Set the equivalent menu item
            if ($self->ivExists('menuToolItemHash', 'drag_mode')) {

                my $menuItem = $self->ivShow('menuToolItemHash', 'drag_mode');
                $menuItem->set_active($self->dragModeFlag);
            }
        });
        push (@buttonList, $toggleButton_dragMode);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_drag_mode', $toggleButton_dragMode);

        # Toolbutton for 'move selected rooms to click'
        my $toolButton_moveClick = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_move_click.png'),
            'Move selected rooms to click',
        );
        $toolButton_moveClick->set_tooltip_text('Move selected rooms to click');
        $toolButton_moveClick->signal_connect('clicked' => sub {

            # Set the free clicking mode: $self->mouseClickEvent will move the objects  when the
            #   user next clicks on an empty part of the map
            $self->set_freeClickMode('move_room');
        });
        push (@buttonList, $toolButton_moveClick);
        # (Requires $self->currentRegionmap and one or more selected rooms)
        $self->ivAdd('menuToolItemHash', 'icon_move_to_click', $toolButton_moveClick);

        # Toolbutton for 'connect to click'
        my $toolButton_connectClick = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_connect_click.png'),
            'Connect selected exit to room',
        );
        $toolButton_connectClick->set_tooltip_text('Connect selected exit to room');
        $toolButton_connectClick->signal_connect('clicked' => sub {

            $self->connectToClickCallback();
        });
        push (@buttonList, $toolButton_connectClick);
        # (Requires $self->currentRegionmap, $self->selectedExit and
        #   $self->selectedExit->drawMode is 'primary', 'temp_unalloc' or 'perm_alloc')
        $self->ivAdd('menuToolItemHash', 'icon_connect_click', $toolButton_connectClick);

        # Toolbutton for 'take screenshot'
        my $toolButton_visibleScreenshot = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_take_screenshot.png'),
            'Take screenshot of visible map',
        );
        $toolButton_visibleScreenshot->set_tooltip_text('Take screenshot of visible map');
        $toolButton_visibleScreenshot->signal_connect('clicked' => sub {

            $self->visibleScreenshotCallback();
        });
        push (@buttonList, $toolButton_visibleScreenshot);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_visible_screenshot', $toolButton_visibleScreenshot);

        return @buttonList;
    }

    sub drawExitsButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my @buttonList;

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawExitsButtonSet', @_);
        }

        # Radio button for 'use region exit settings' mode
        my $radioButton_deferDrawExits = Gtk3::RadioToolButton->new(undef);
        $radioButton_deferDrawExits->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_use_region.png'),
        );
        $radioButton_deferDrawExits->set_label('Use region exit settings');
        $radioButton_deferDrawExits->set_tooltip_text('Use region exit settings');
        $radioButton_deferDrawExits->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag && $radioButton_deferDrawExits->get_active()) {

                $self->worldModelObj->switchMode(
                    'drawExitMode',
                    'ask_regionmap',    # New value of ->drawExitMode
                    TRUE,               # Do call $self->redrawRegions
                    'draw_defer_exits',
                    'icon_draw_defer_exits',
                );
            }
        });
        push (@buttonList, $radioButton_deferDrawExits);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_draw_defer_exits', $radioButton_deferDrawExits);

        # Radio button for 'draw no exits' mode
        my $radioButton_drawNoExits = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_deferDrawExits,
        );
        if ($self->worldModelObj->drawExitMode eq 'no_exit') {

            $radioButton_drawNoExits->set_active(TRUE);
        }
        $radioButton_drawNoExits->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_draw_none.png'),
        );
        $radioButton_drawNoExits->set_label('Draw no exits');
        $radioButton_drawNoExits->set_tooltip_text('Draw no exits');
        $radioButton_drawNoExits->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag && $radioButton_drawNoExits->get_active()) {

                $self->worldModelObj->switchMode(
                    'drawExitMode',
                    'no_exit',          # New value of ->drawExitMode
                    TRUE,               # Do call $self->redrawRegions
                    'draw_no_exits',
                    'icon_draw_no_exits',
                );
            }
        });
        push (@buttonList, $radioButton_drawNoExits);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_draw_no_exits', $radioButton_drawNoExits);

        # Radio button for 'draw simple exits' mode
        my $radioButton_drawSimpleExits = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_drawNoExits,
        );
        if ($self->worldModelObj->drawExitMode eq 'simple_exit') {

            $radioButton_drawSimpleExits->set_active(TRUE);
        }
        $radioButton_drawSimpleExits->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_draw_simple.png'),
        );
        $radioButton_drawSimpleExits->set_label('Draw simple exits');
        $radioButton_drawSimpleExits->set_tooltip_text('Draw simple exits');
        $radioButton_drawSimpleExits->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag && $radioButton_drawSimpleExits->get_active()) {

                $self->worldModelObj->switchMode(
                    'drawExitMode',
                    'simple_exit',      # New value of ->drawExitMode
                    TRUE,               # Do call $self->redrawRegions
                    'draw_simple_exits',
                    'icon_draw_simple_exits',
                );
            }
        });
        push (@buttonList, $radioButton_drawSimpleExits);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_draw_simple_exits', $radioButton_drawSimpleExits);

        # Radio button for 'draw complex exits' mode
        my $radioButton_drawComplexExits = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_drawSimpleExits,
        );
        if ($self->worldModelObj->drawExitMode eq 'complex_exit') {

            $radioButton_drawComplexExits->set_active(TRUE);
        }
        $radioButton_drawComplexExits->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_draw_complex.png'),
        );
        $radioButton_drawComplexExits->set_label('Draw complex exits');
        $radioButton_drawComplexExits->set_tooltip_text('Draw complex exits');
        $radioButton_drawComplexExits->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag && $radioButton_drawComplexExits->get_active()) {

                $self->worldModelObj->switchMode(
                    'drawExitMode',
                    'complex_exit',     # New value of ->drawExitMode
                    TRUE,               # Do call $self->redrawRegions
                    'draw_complex_exits',
                    'icon_draw_complex_exits',
                );
            }
        });
        push (@buttonList, $radioButton_drawComplexExits);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_draw_complex_exits', $radioButton_drawComplexExits);

        # Toggle button for 'obscure unimportant exits'
        my $toggleButton_obscuredExits = Gtk3::ToggleToolButton->new();
        $toggleButton_obscuredExits->set_active($self->worldModelObj->obscuredExitFlag);
        $toggleButton_obscuredExits->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_obscured_exits.png'),
        );
        $toggleButton_obscuredExits->set_label('Obscure unimportant exits');
        $toggleButton_obscuredExits->set_tooltip_text('Obscure unimportant exits');
        $toggleButton_obscuredExits->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag) {

                $self->worldModelObj->toggleFlag(
                    'obscuredExitFlag',
                    $toggleButton_obscuredExits->get_active(),
                    TRUE,      # Do call $self->redrawRegions
                    'obscured_exits',
                    'icon_obscured_exits',
                );
            }
        });
        push (@buttonList, $toggleButton_obscuredExits);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_obscured_exits', $toggleButton_obscuredExits);

        # Toggle button for 'auto-redraw obscured exits'
        my $toggleButton_autoRedraw = Gtk3::ToggleToolButton->new();
        $toggleButton_autoRedraw->set_active($self->worldModelObj->obscuredExitRedrawFlag);
        $toggleButton_autoRedraw->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_auto_redraw.png'),
        );
        $toggleButton_autoRedraw->set_label('Auto-redraw obscured exits');
        $toggleButton_autoRedraw->set_tooltip_text('Auto-redraw obscured exits');
        $toggleButton_autoRedraw->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag) {

                $self->worldModelObj->toggleFlag(
                    'obscuredExitRedrawFlag',
                    $toggleButton_autoRedraw->get_active(),
                    TRUE,      # Do call $self->redrawRegions
                    'auto_redraw_obscured',
                    'icon_auto_redraw_obscured',
                );
            }
        });
        push (@buttonList, $toggleButton_autoRedraw);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_auto_redraw_obscured', $toggleButton_autoRedraw);

        # Toolbutton for 'obscure exits in radius'
        my $toolButton_obscuredRadius = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_obscured_radius.png'),
            'Obscure exits in radius',
        );
        $toolButton_obscuredRadius->set_tooltip_text('Obscure exits in radius');
        $toolButton_obscuredRadius->signal_connect('clicked' => sub {

            $self->obscuredRadiusCallback();
        });
        push (@buttonList, $toolButton_obscuredRadius);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_obscured_radius', $toolButton_obscuredRadius);

#        # Separator
#        my $separator = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toolbutton for 'horizontal exit length'
        my $toolButton_horizontalLengths = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR
                . '/icons/map/icon_horizontal_lengths.png'),
            'Horizontal exit length',
        );
        $toolButton_horizontalLengths->set_tooltip_text('Horizontal exit length');
        $toolButton_horizontalLengths->signal_connect('clicked' => sub {

            $self->setExitLengthCallback('horizontal');
        });
        push (@buttonList, $toolButton_horizontalLengths);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_horizontal_lengths', $toolButton_horizontalLengths);

        # Toolbutton for 'vertical exit length'
        my $toolButton_verticalLengths = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR
                . '/icons/map/icon_vertical_lengths.png'),
            'Vertical exit length',
        );
        $toolButton_verticalLengths->set_tooltip_text('Vertical exit length');
        $toolButton_verticalLengths->signal_connect('clicked' => sub {

            $self->setExitLengthCallback('vertical');
        });
        push (@buttonList, $toolButton_verticalLengths);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_vertical_lengths', $toolButton_verticalLengths);

#        # Separator
#        my $separator2 = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator2);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toggle button for 'draw exit ornaments'
        my $toggleButton_drawExitOrnaments = Gtk3::ToggleToolButton->new();
        $toggleButton_drawExitOrnaments->set_active($self->worldModelObj->drawOrnamentsFlag);
        $toggleButton_drawExitOrnaments->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_draw_ornaments.png'),
        );
        $toggleButton_drawExitOrnaments->set_label('Draw exit ornaments');
        $toggleButton_drawExitOrnaments->set_tooltip_text('Draw exit ornaments');
        $toggleButton_drawExitOrnaments->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag) {

                $self->worldModelObj->toggleFlag(
                    'drawOrnamentsFlag',
                    $toggleButton_drawExitOrnaments->get_active(),
                    TRUE,      # Do call $self->redrawRegions
                    'draw_ornaments',
                    'icon_draw_ornaments',
                );
            }
        });
        push (@buttonList, $toggleButton_drawExitOrnaments);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_draw_ornaments', $toggleButton_drawExitOrnaments);

        # Toolbutton for 'no ornament'
        my $toolButton_noOrnament = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_no_ornament.png'),
            'Set no ornament',
        );
        $toolButton_noOrnament->set_tooltip_text('Set no ornament');
        $toolButton_noOrnament->signal_connect('clicked' => sub {

            $self->exitOrnamentCallback('none');
        });
        push (@buttonList, $toolButton_noOrnament);
        # (Requires $self->currentRegionmap & either $self->selectedExit or
        #   $self->selectedExitHash)
        $self->ivAdd('menuToolItemHash', 'icon_no_ornament', $toolButton_noOrnament);

#        # Separator
#        my $separator3 = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator3);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toolbutton for 'openable exit'
        my $toolButton_openableExit = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_openable_exit.png'),
            'Set openable exit',
        );
        $toolButton_openableExit->set_tooltip_text('Set openable exit');
        $toolButton_openableExit->signal_connect('clicked' => sub {

            $self->exitOrnamentCallback('open');
        });
        push (@buttonList, $toolButton_openableExit);
        # (Requires $self->currentRegionmap & either $self->selectedExit or
        #   $self->selectedExitHash)
        $self->ivAdd('menuToolItemHash', 'icon_openable_exit', $toolButton_openableExit);

        # Toolbutton for 'lockable exit'
        my $toolButton_lockableExit = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_lockable_exit.png'),
            'Set lockable exit',
        );
        $toolButton_lockableExit->set_tooltip_text('Set lockable exit');
        $toolButton_lockableExit->signal_connect('clicked' => sub {

            $self->exitOrnamentCallback('lock');
        });
        push (@buttonList, $toolButton_lockableExit);
        # (Requires $self->currentRegionmap & either $self->selectedExit or
        #   $self->selectedExitHash)
        $self->ivAdd('menuToolItemHash', 'icon_lockable_exit', $toolButton_lockableExit);

        # Toolbutton for 'pickable exit'
        my $toolButton_pickableExit = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_pickable_exit.png'),
            'Set pickable exit',
        );
        $toolButton_pickableExit->set_tooltip_text('Set pickable exit');
        $toolButton_pickableExit->signal_connect('clicked' => sub {

            $self->exitOrnamentCallback('pick');
        });
        push (@buttonList, $toolButton_pickableExit);
        # (Requires $self->currentRegionmap & either $self->selectedExit or
        #   $self->selectedExitHash)
        $self->ivAdd('menuToolItemHash', 'icon_pickable_exit', $toolButton_pickableExit);

        # Toolbutton for 'breakable exit'
        my $toolButton_breakableExit = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_breakable_exit.png'),
            'Set breakable exit',
        );
        $toolButton_breakableExit->set_tooltip_text('Set breakable exit');
        $toolButton_breakableExit->signal_connect('clicked' => sub {

            $self->exitOrnamentCallback('break');
        });
        push (@buttonList, $toolButton_breakableExit);
        # (Requires $self->currentRegionmap & either $self->selectedExit or
        #   $self->selectedExitHash)
        $self->ivAdd('menuToolItemHash', 'icon_breakable_exit', $toolButton_breakableExit);

        # Toolbutton for 'impassable exit'
        my $toolButton_impassableExit = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_impassable_exit.png'),
            'Set impassable exit',
        );
        $toolButton_impassableExit->set_tooltip_text('Set impassable exit');
        $toolButton_impassableExit->signal_connect('clicked' => sub {

            $self->exitOrnamentCallback('impass');
        });
        push (@buttonList, $toolButton_impassableExit);
        # (Requires $self->currentRegionmap & either $self->selectedExit or
        #   $self->selectedExitHash)
        $self->ivAdd('menuToolItemHash', 'icon_impassable_exit', $toolButton_impassableExit);

        # Toolbutton for 'mystery exit'
        my $toolButton_mysteryExit = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_mystery_exit.png'),
            'Set mystery exit',
        );
        $toolButton_mysteryExit->set_tooltip_text('Set mystery exit');
        $toolButton_mysteryExit->signal_connect('clicked' => sub {

            $self->exitOrnamentCallback('mystery');
        });
        push (@buttonList, $toolButton_mysteryExit);
        # (Requires $self->currentRegionmap & either $self->selectedExit or
        #   $self->selectedExitHash)
        $self->ivAdd('menuToolItemHash', 'icon_mystery_exit', $toolButton_mysteryExit);

        return @buttonList;
    }

    sub drawPaintingButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my (
            @buttonList,
            %oldHash,
        );

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawPaintingButtonSet', @_);
        }

        # This hash must be reset whenever the toolbar is redrawn. Make a temporary copy, so any
        #   colour buttons can remain toggled, if they were toggled before being drawn
        %oldHash = $self->toolbarRoomFlagHash;
        $self->ivEmpty('toolbarRoomFlagHash');

        # Toggle button for 'enable painter'
        my $toggleButton_enablePainter = Gtk3::ToggleToolButton->new();
        $toggleButton_enablePainter->set_active($self->painterFlag);
        $toggleButton_enablePainter->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_enable_painter.png'),
        );
        $toggleButton_enablePainter->set_label('Enable painter');
        $toggleButton_enablePainter->set_tooltip_text('Enable painter');
        $toggleButton_enablePainter->signal_connect('toggled' => sub {

            my $item;

            # Toggle the flag
            if ($toggleButton_enablePainter->get_active()) {
                $self->ivPoke('painterFlag', TRUE);
            } else {
                $self->ivPoke('painterFlag', FALSE);
            }

            # Update the corresponding menu item
            $item = $self->ivShow('menuToolItemHash', 'enable_painter');
            if ($item) {

                $item->set_active($self->painterFlag);
            }
        });
        push (@buttonList, $toggleButton_enablePainter);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_enable_painter', $toggleButton_enablePainter);

        # Toolbutton for 'edit painter'
        my $toolButton_editPainter = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_edit_painter.png'),
            'Edit painter',
        );
        $toolButton_editPainter->set_tooltip_text('Edit painter');
        $toolButton_editPainter->signal_connect('clicked' => sub {

            # Open an 'edit' window for the painter object
            $self->createFreeWin(
                'Games::Axmud::EditWin::Painter',
                $self,
                $self->session,
                'Edit world model painter',
                $self->worldModelObj->painterObj,
                FALSE,          # Not temporary
            );
        });
        push (@buttonList, $toolButton_editPainter);

        # Radio button for 'paint all rooms'
        my $radioButton_paintAllRooms = Gtk3::RadioToolButton->new(undef);
        $radioButton_paintAllRooms->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_paint_all.png'),
        );
        $radioButton_paintAllRooms->set_label('Paint all rooms');
        $radioButton_paintAllRooms->set_tooltip_text('Paint all rooms');
        $radioButton_paintAllRooms->signal_connect('toggled' => sub {

            if ($radioButton_paintAllRooms->get_active()) {

                $self->worldModelObj->set_paintAllRoomsFlag(TRUE);

                # Set the equivalent menu item
                if ($self->ivExists('menuToolItemHash', 'paint_all')) {

                    $self->ivShow('menuToolItemHash', 'paint_all')->set_active(TRUE);
                }
            }
        });
        push (@buttonList, $radioButton_paintAllRooms);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_paint_all', $radioButton_paintAllRooms);

        # Radio button for 'paint only new rooms'
        my $radioButton_paintNewRooms = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_paintAllRooms,
        );
        if (! $self->worldModelObj->paintAllRoomsFlag) {

            $radioButton_paintNewRooms->set_active(TRUE);
        }
        $radioButton_paintNewRooms->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_paint_new.png'),
        );
        $radioButton_paintNewRooms->set_label('Paint only new rooms');
        $radioButton_paintNewRooms->set_tooltip_text('Paint only new rooms');
        $radioButton_paintNewRooms->signal_connect('toggled' => sub {

            if ($radioButton_paintNewRooms->get_active) {

                $self->worldModelObj->set_paintAllRoomsFlag(FALSE);

                # Set the equivalent menu item
                if ($self->ivExists('menuToolItemHash', 'paint_new')) {

                    $self->ivShow('menuToolItemHash', 'paint_new')->set_active(TRUE);
                }
            }
        });
        push (@buttonList, $radioButton_paintNewRooms);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_paint_new', $radioButton_paintNewRooms);

        # Radio button for 'paint normal rooms'
        my $radioButton_paintNormalRooms = Gtk3::RadioToolButton->new(undef);
        $radioButton_paintNormalRooms->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_paint_normal.png'),
        );
        $radioButton_paintNormalRooms->set_label('Paint normal rooms');
        $radioButton_paintNormalRooms->set_tooltip_text('Paint normal rooms');
        $radioButton_paintNormalRooms->signal_connect('toggled' => sub {

            if ($radioButton_paintNormalRooms->get_active()) {

                $self->worldModelObj->painterObj->ivPoke('wildMode', 'normal');

                # Set the equivalent menu item
                if ($self->ivExists('menuToolItemHash', 'paint_normal')) {

                    $self->ivShow('menuToolItemHash', 'paint_normal')->set_active(TRUE);
                }
            }
        });
        push (@buttonList, $radioButton_paintNormalRooms);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_paint_normal', $radioButton_paintNormalRooms);

        # Radio button for 'paint wilderness rooms'
        my $radioButton_paintWildRooms = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_paintNormalRooms,
        );
        if ($self->worldModelObj->painterObj->wildMode eq 'wild') {

            $radioButton_paintWildRooms->set_active(TRUE);
        }
        $radioButton_paintWildRooms->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_paint_wild.png'),
        );
        $radioButton_paintWildRooms->set_label('Paint wilderness rooms');
        $radioButton_paintWildRooms->set_tooltip_text('Paint wilderness rooms');
        $radioButton_paintWildRooms->signal_connect('toggled' => sub {

            if ($radioButton_paintWildRooms->get_active) {

                $self->worldModelObj->painterObj->ivPoke('wildMode', 'wild');

                # Set the equivalent menu item
                if ($self->ivExists('menuToolItemHash', 'paint_wild')) {

                    $self->ivShow('menuToolItemHash', 'paint_wild')->set_active(TRUE);
                }
            }
        });
        push (@buttonList, $radioButton_paintWildRooms);
        # (Requires $self->session->currentWorld->basicMappingFlag to be FALSE)
        $self->ivAdd('menuToolItemHash', 'icon_paint_wild', $radioButton_paintWildRooms);

        # Radio button for 'paint wilderness border rooms'
        my $radioButton_paintBorderRooms = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_paintWildRooms,
        );
        if ($self->worldModelObj->painterObj->wildMode eq 'border') {

            $radioButton_paintBorderRooms->set_active(TRUE);
        }
        $radioButton_paintBorderRooms->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_paint_border.png'),
        );
        $radioButton_paintBorderRooms->set_label('Paint wilderness border rooms');
        $radioButton_paintBorderRooms->set_tooltip_text('Paint wilderness border rooms');
        $radioButton_paintBorderRooms->signal_connect('toggled' => sub {

            if ($radioButton_paintBorderRooms->get_active) {

                $self->worldModelObj->painterObj->ivPoke('wildMode', 'border');

                # Set the equivalent menu item
                if ($self->ivExists('menuToolItemHash', 'paint_border')) {

                    $self->ivShow('menuToolItemHash', 'paint_border')->set_active(TRUE);
                }
            }
        });
        push (@buttonList, $radioButton_paintBorderRooms);
        # (Requires $self->session->currentWorld->basicMappingFlag to be FALSE)
        $self->ivAdd('menuToolItemHash', 'icon_paint_border', $radioButton_paintBorderRooms);

#        # Separator
#        my $separator = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        my $toolButton_addRoomFlag = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_add_room_flag.png'),
            'Add preferred room flag',
        );
        $toolButton_addRoomFlag->set_tooltip_text('Add preferred room flag');
        $toolButton_addRoomFlag->signal_connect('clicked' => sub {

            $self->addRoomFlagButton();
        });
        push (@buttonList, $toolButton_addRoomFlag);

        my $toolButton_removeRoomFlag = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_remove_room_flag.png'),
            'Remove preferred room flag',
        );
        $toolButton_removeRoomFlag->set_tooltip_text('Remove preferred room flag');
        $toolButton_removeRoomFlag->signal_connect('clicked' => sub {

            $self->removeRoomFlagButton();
        });
        push (@buttonList, $toolButton_removeRoomFlag);
        # (Requires non-empty $self->worldModelObj->preferRoomFlagList)
        $self->ivAdd('menuToolItemHash', 'icon_remove_room_flag', $toolButton_removeRoomFlag);

        foreach my $roomFlag ($self->worldModelObj->preferRoomFlagList) {

            my ($roomFlagObj, $colour, $frameColour, $text);

            $roomFlagObj = $self->worldModelObj->ivShow('roomFlagHash', $roomFlag);
            if ($roomFlagObj) {

                $colour = $roomFlagObj->colour;
                $text = $roomFlagObj->name;
            }

            if ($colour) {

                # Convert RGB colours to Gdk RGBA
                $colour =~ s/^#//;
                $colour = ((hex $colour) * 256) + 255;
                $frameColour = ((hex '000000') * 256) + 255;

                # Create a pixbuf, with its own sub-region. Use $colour to fill the sub-region,
                #   leaving the renaming area of the pixbuf as a black frame
                my $pixbuf = Gtk3::Gdk::Pixbuf->new(
                    'GDK_COLORSPACE_RGB',
                    FALSE,
                    # Same values as ->get_bits_per_sample, ->get_width, ->get_height as a
                    #   Gtk3::Gdk::Pixbuf loaded from one of the icon files in ../share/icons/map
                    8,
                    20,
                    20,
                );

                $pixbuf->fill($frameColour);

                # Create the sub-region, drawn in $colour
                my $subPixbuf = $pixbuf->new_subpixbuf(1, 1, 18, 18);
                $subPixbuf->fill($colour);

                my $toolButton = Gtk3::ToggleToolButton->new();
                if (exists $oldHash{$roomFlag}) {

                    $toolButton->set_active(TRUE);
                    # (Toggled buttons must survive the toolbar redraw)
                    $self->ivAdd('toolbarRoomFlagHash', $roomFlag, undef);
                }
                $toolButton->set_icon_widget(
                    Gtk3::Image->new_from_pixbuf($pixbuf),
                );
                $toolButton->set_label($text);
                $toolButton->set_tooltip_text($text);
                $toolButton->signal_connect('toggled' => sub {

                    # Add or remove the room flag from the painter
                    if (! $toolButton->get_active()) {

                        $self->worldModelObj->painterObj->ivDelete('roomFlagHash', $roomFlag);
                        # (Entries in this hash may or may not exist in the painter's hash)
                        $self->ivDelete('toolbarRoomFlagHash', $roomFlag);

                    } else {

                        $self->worldModelObj->painterObj->ivAdd('roomFlagHash', $roomFlag);
                        $self->ivAdd('toolbarRoomFlagHash', $roomFlag, undef);
                    }
                });
                push (@buttonList, $toolButton);
            }
        }

        return @buttonList;
    }

    sub drawQuickButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my (
            $ignoreFlag,
            @buttonList, @colourButtonList,
            %oldHash,
        );

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawPaintingButtonSet', @_);
        }

        # This hash must be reset whenever the toolbar is redrawn. Make a temporary copy, so any
        #   colour buttons can remain toggled, if they were toggled before being drawn
        %oldHash = $self->toolbarRoomFlagHash;
        $self->ivEmpty('toolbarRoomFlagHash');

        # Radio button for 'paint all rooms'
        my $radioButton_quickSingle = Gtk3::RadioToolButton->new(undef);
        $radioButton_quickSingle->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_quick_single.png'),
        );
        $radioButton_quickSingle->set_label('Quick paint then reset');
        $radioButton_quickSingle->set_tooltip_text('Quick paint then reset');
        $radioButton_quickSingle->signal_connect('toggled' => sub {

            if ($radioButton_quickSingle->get_active()) {

                $self->worldModelObj->set_quickPaintMultiFlag(FALSE);

                foreach my $button (@colourButtonList) {

                    $button->set_active(FALSE);
                }
            }
        });
        push (@buttonList, $radioButton_quickSingle);

        # Radio button for 'paint only new rooms'
        my $radioButton_quickMulti = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_quickSingle,
        );
        if ($self->worldModelObj->quickPaintMultiFlag) {

            $radioButton_quickMulti->set_active(TRUE);
        }
        $radioButton_quickMulti->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_quick_multi.png'),
        );
        $radioButton_quickMulti->set_label('Quick paint without resetting');
        $radioButton_quickMulti->set_tooltip_text('Quick paint without resetting');
        $radioButton_quickMulti->signal_connect('toggled' => sub {

            if ($radioButton_quickMulti->get_active) {

                $self->worldModelObj->set_quickPaintMultiFlag(TRUE);
            }
        });
        push (@buttonList, $radioButton_quickMulti);

#        # Separator
#        my $separator = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        my $toolButton_addRoomFlag = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_add_quick_flag.png'),
            'Add preferred room flag',
        );
        $toolButton_addRoomFlag->set_tooltip_text('Add preferred room flag');
        $toolButton_addRoomFlag->signal_connect('clicked' => sub {

            $self->addRoomFlagButton();
        });
        push (@buttonList, $toolButton_addRoomFlag);

        my $toolButton_removeRoomFlag = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_remove_quick_flag.png'),
            'Remove preferred room flag',
        );
        $toolButton_removeRoomFlag->set_tooltip_text('Remove preferred room flag');
        $toolButton_removeRoomFlag->signal_connect('clicked' => sub {

            $self->removeRoomFlagButton();
        });
        push (@buttonList, $toolButton_removeRoomFlag);
        # (Requires non-empty $self->worldModelObj->preferRoomFlagList)
        $self->ivAdd('menuToolItemHash', 'icon_remove_room_flag_2', $toolButton_removeRoomFlag);

        foreach my $roomFlag ($self->worldModelObj->preferRoomFlagList) {

            my ($roomFlagObj, $colour, $frameColour, $text);

            $roomFlagObj = $self->worldModelObj->ivShow('roomFlagHash', $roomFlag);
            if ($roomFlagObj) {

                $colour = $roomFlagObj->colour;
                $text = $roomFlagObj->name;
            }

            if ($colour) {

                # Convert RGB colours to Gdk RGBA
                $colour =~ s/^#//;
                $colour = ((hex $colour) * 256) + 255;
                $frameColour = ((hex '000000') * 256) + 255;

                # Create a pixbuf, with its own sub-region. Use $colour to fill the sub-region,
                #   leaving the renaming area of the pixbuf as a black frame
                my $pixbuf = Gtk3::Gdk::Pixbuf->new(
                    'GDK_COLORSPACE_RGB',
                    FALSE,
                    # Same values as ->get_bits_per_sample, ->get_width, ->get_height as a
                    #   Gtk3::Gdk::Pixbuf loaded from one of the icon files in ../share/icons/map
                    8,
                    20,
                    20,
                );

                $pixbuf->fill($frameColour);

                # Create the sub-region, drawn in $colour
                my $subPixbuf = $pixbuf->new_subpixbuf(1, 1, 18, 18);
                $subPixbuf->fill($colour);

                my $toolButton = Gtk3::ToggleToolButton->new();
                if (exists $oldHash{$roomFlag}) {

                    $toolButton->set_active(TRUE);
                    # (Toggled buttons must survive the toolbar redraw)
                    $self->ivAdd('toolbarRoomFlagHash', $roomFlag, undef);
                }
                $toolButton->set_icon_widget(
                    Gtk3::Image->new_from_pixbuf($pixbuf),
                );
                $toolButton->set_label($text);
                $toolButton->set_tooltip_text($text);
                $toolButton->signal_connect('toggled' => sub {

                    # Add or remove the room flag from hash IV, so that $self->doQuickPaint knows to
                    #   use it (or not to use it)
                    if (! $ignoreFlag) {

                        # If this button has been toggled by the user, other buttons might receive
                        #   the same signal; tell their ->signal_connect to ignore it
                        $ignoreFlag = TRUE;

                        if (! $toolButton->get_active()) {

                            $self->ivUndef('toolbarQuickPaintColour');

                        } else {

                            $self->ivPoke('toolbarQuickPaintColour', $roomFlag);

                            # When this colour button is selected, deselect all the other colour
                            #   buttons
                            # (If the $radioButton_quickSingle button is selected, also deselect
                            #   this button, as the user wants the choice of room flag to reset
                            #   as soon as they click on a room)
                            foreach my $otherButton (@colourButtonList) {

                                if (
                                    $radioButton_quickSingle->get_active()
                                    || $otherButton ne $toolButton
                                ) {
                                    $otherButton->set_active(FALSE);
                                }
                            }
                        }

                        $ignoreFlag = FALSE;
                    }
                });

                push (@buttonList, $toolButton);
                # Only one of the colour buttons should be toggled at a time. Don't use radio
                #   buttons because we want it to be possible for none of the colour buttons to be
                #   selected
                push (@colourButtonList, $toolButton);
            }
        }

        return @buttonList;
    }

    sub drawBackgroundButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my @buttonList;

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->drawBackgroundButtonSet',
                @_,
            );
        }

        # Radio button for 'no background colouring'
        my $radioButton_bgDefault = Gtk3::RadioToolButton->new(undef);
        if ($self->bgColourMode eq 'default') {

            $radioButton_bgDefault->set_active(TRUE);
        }
        $radioButton_bgDefault->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_bg_default.png')
        );
        $radioButton_bgDefault->set_label('No background colouring');
        $radioButton_bgDefault->set_tooltip_text('No background colouring');
        $radioButton_bgDefault->signal_connect('toggled' => sub {

            my $item;

            # Update the IVs
            if ($radioButton_bgDefault->get_active()) {

                $self->ivPoke('bgColourMode', 'default');
                $self->ivUndef('bgRectXPos');
                $self->ivUndef('bgRectYPos');

                # Make sure any free click mode operations, like connecting exits or moving rooms,
                #   are cancelled
                $self->reset_freeClickMode();
            }
        });
        push (@buttonList, $radioButton_bgDefault);

        # Radio button for 'colour single blocks'
        my $radioButton_bgColour = Gtk3::RadioToolButton->new_from_widget($radioButton_bgDefault);
        if ($self->bgColourMode eq 'square_start') {

            $radioButton_bgColour->set_active(TRUE);
        }
        $radioButton_bgColour->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_bg_colour.png')
        );
        $radioButton_bgColour->set_label('Colour single blocks');
        $radioButton_bgColour->set_tooltip_text('Colour single blocks');
        $radioButton_bgColour->signal_connect('toggled' => sub {

            my $item;

            # Update the IV
            if ($radioButton_bgColour->get_active()) {

                $self->ivPoke('bgColourMode', 'square_start');

                # Make sure any free click mode operations, like connecting exits or moving rooms,
                #   are cancelled
                $self->reset_freeClickMode();
            }
        });
        push (@buttonList, $radioButton_bgColour);

        # Radio button for 'colour multiple blocks'
        my $radioButton_bgShape = Gtk3::RadioToolButton->new_from_widget($radioButton_bgDefault);
        if ($self->bgColourMode eq 'rect_start' || $self->bgColourMode eq 'rect_stop') {

            $radioButton_bgShape->set_active(TRUE);
        }
        $radioButton_bgShape->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_bg_shape.png')
        );
        $radioButton_bgShape->set_label('Colour multiple blocks');
        $radioButton_bgShape->set_tooltip_text('Colour multiple blocks');
        $radioButton_bgShape->signal_connect('toggled' => sub {

            my $item;

            # Update the IV
            if ($radioButton_bgShape->get_active()) {

                $self->ivPoke('bgColourMode', 'rect_start');

                # Make sure any free click mode operations, like connecting exits or moving rooms,
                #   are cancelled
                $self->reset_freeClickMode();
            }
        });
        push (@buttonList, $radioButton_bgShape);

        # Toggle button for 'colour on all levels'
        my $toggleButton_colourAllLevel = Gtk3::ToggleToolButton->new();
        $toggleButton_colourAllLevel->set_active($self->bgAllLevelFlag);
        $toggleButton_colourAllLevel->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_colour_all_level.png'),
        );
        $toggleButton_colourAllLevel->set_label('Colour on all levels');
        $toggleButton_colourAllLevel->set_tooltip_text('Colour on all levels');
        $toggleButton_colourAllLevel->signal_connect('toggled' => sub {

            # Update the IV
            if ($toggleButton_colourAllLevel->get_active()) {
                $self->ivPoke('bgAllLevelFlag', TRUE);
            } else {
                $self->ivPoke('bgAllLevelFlag', FALSE);
            }
        });
        push (@buttonList, $toggleButton_colourAllLevel);

#        # Separator
#        my $separator = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toolbutton for 'add background colour'
        my $toolButton_addColour = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_bg_add.png'),
            'Add background colour',
        );
        $toolButton_addColour->set_tooltip_text('Add background colour');
        $toolButton_addColour->signal_connect('clicked' => sub {

            $self->addBGColourButton();
        });
        push (@buttonList, $toolButton_addColour);

        # Toolbutton for 'remove background colour'
        my $toolButton_removeColour = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_bg_remove.png'),
            'Remove background colour',
        );
        $toolButton_removeColour->set_tooltip_text('Remove background colour');
        $toolButton_removeColour->signal_connect('clicked' => sub {

            $self->removeBGColourButton();
        });
        push (@buttonList, $toolButton_removeColour);
        # (Requires non-empty $self->worldModelObj->preferBGColourList)
        $self->ivAdd('menuToolItemHash', 'icon_bg_remove', $toolButton_removeColour);

        # Radiobutton for 'use default colour'
        my $radioButton_useDefault = Gtk3::RadioToolButton->new(undef);
        if (! defined $self->bgColourChoice) {

            $radioButton_useDefault->set_active(TRUE);
        }
        $radioButton_useDefault->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_bg_blank.png'),
        );
        $radioButton_useDefault->set_label('No background colouring');
        $radioButton_useDefault->set_tooltip_text('No background colouring');
        $radioButton_useDefault->signal_connect('toggled' => sub {

            if ($radioButton_useDefault->get_active()) {

                $self->ivUndef('bgColourChoice');
            }
        });
        push (@buttonList, $radioButton_useDefault);

        foreach my $rgb ($self->worldModelObj->preferBGColourList) {

            my ($text, $colour, $frameColour);

            $text = 'Use colour ' . uc($rgb);

            # Convert RGB colours to Gdk RGBA
            $colour = $rgb;
            $colour =~ s/^#//;
            $colour = ((hex $colour) * 256) + 255;
            $frameColour = ((hex '000000') * 256) + 255;

            # Create a pixbuf, with its own sub-region. Use $colour to fill the sub-region, leaving
            #   the renaming area of the pixbuf as a black frame
            my $pixbuf = Gtk3::Gdk::Pixbuf->new(
                'GDK_COLORSPACE_RGB',
                FALSE,
                # Same values as ->get_bits_per_sample, ->get_width, ->get_height as a
                #   Gtk3::Gdk::Pixbuf loaded from one of the icon files in ../share/icons/map
                8,
                20,
                20,
            );

            $pixbuf->fill($frameColour);

            # Create the sub-region, drawn in $colour
            my $subPixbuf = $pixbuf->new_subpixbuf(1, 1, 18, 18);
            $subPixbuf->fill($colour);

            my $radioButton = Gtk3::RadioToolButton->new_from_widget($radioButton_useDefault);
            if (defined $self->bgColourChoice && $self->bgColourChoice eq $rgb) {

                $radioButton->set_active(TRUE);
            }
            $radioButton->set_icon_widget(
                Gtk3::Image->new_from_pixbuf($pixbuf),
            );
            $radioButton->set_label($text);
            $radioButton->set_tooltip_text($text);
            $radioButton->signal_connect('toggled' => sub {

                if ($radioButton->get_active()) {

                    $self->ivPoke('bgColourChoice', $rgb);
                }
            });
            push (@buttonList, $radioButton);
        }

        return @buttonList;
    }

    sub drawTrackingButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my @buttonList;

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawTrackingButtonSet', @_);
        }

        # Toolbutton for 'centre map on current room'
        my $toolButton_centreCurrentRoom = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_centre_current.png'),
            'Centre map on current room',
        );
        $toolButton_centreCurrentRoom->set_tooltip_text('Centre map on current room');
        $toolButton_centreCurrentRoom->signal_connect('clicked' => sub {

            $self->centreMapOverRoom($self->mapObj->currentRoom);
        });
        push (@buttonList, $toolButton_centreCurrentRoom);
        # (Requires $self->currentRegionmap & $self->mapObj->currentRoom)
        $self->ivAdd(
            'menuToolItemHash',
            'icon_centre_map_current_room',
            $toolButton_centreCurrentRoom,
        );

        # Toolbutton for 'centre map on selected room'
        my $toolButton_centreSelectedRoom = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_centre_selected.png'),
            'Centre map on selected room',
        );
        $toolButton_centreSelectedRoom->set_tooltip_text('Centre map on selected room');
        $toolButton_centreSelectedRoom->signal_connect('clicked' => sub {

            $self->centreMapOverRoom($self->selectedRoom);
        });
        push (@buttonList, $toolButton_centreSelectedRoom);
        # (Requires $self->currentRegionmap & $self->selectedRoom)
        $self->ivAdd(
            'menuToolItemHash',
            'icon_centre_map_selected_room',
            $toolButton_centreSelectedRoom,
        );

        # Toolbutton for 'centre map on last known room'
        my $toolButton_centreLastKnownRoom = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_centre_last.png'),
            'Centre map on last known room',
        );
        $toolButton_centreLastKnownRoom->set_tooltip_text('Centre map on last known room');
        $toolButton_centreLastKnownRoom->signal_connect('clicked' => sub {

            $self->centreMapOverRoom($self->mapObj->lastKnownRoom);
        });
        push (@buttonList, $toolButton_centreLastKnownRoom);
        # (Requires $self->currentRegionmap & $self->mapObj->lastknownRoom)
        $self->ivAdd(
            'menuToolItemHash',
            'icon_centre_map_last_known_room',
            $toolButton_centreLastKnownRoom,
        );

        # Toolbutton for 'centre map on middle of grid'
        my $toolButton_centreMiddleGrid = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_centre_middle.png'),
            'Centre map on middle of grid',
        );
        $toolButton_centreMiddleGrid->set_tooltip_text('Centre map on middle of grid');
        $toolButton_centreMiddleGrid->signal_connect('clicked' => sub {

            $self->setMapPosn(0.5, 0.5);
        });
        push (@buttonList, $toolButton_centreMiddleGrid);
        # (Requires $self->currentRegionmap)
        $self->ivAdd(
            'menuToolItemHash',
            'icon_centre_map_middle_grid',
            $toolButton_centreMiddleGrid,
        );

#        # Separator
#        my $separator = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toggle button for 'track current room'
        my $toggleButton_trackCurrentRoom = Gtk3::ToggleToolButton->new();
        $toggleButton_trackCurrentRoom->set_active($self->worldModelObj->trackPosnFlag);
        $toggleButton_trackCurrentRoom->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_track_room.png'),
        );
        $toggleButton_trackCurrentRoom->set_label('Track current room');
        $toggleButton_trackCurrentRoom->set_tooltip_text('Track current room');
        $toggleButton_trackCurrentRoom->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag) {

                $self->worldModelObj->toggleFlag(
                    'trackPosnFlag',
                    $toggleButton_trackCurrentRoom->get_active(),
                    FALSE,      # Don't call $self->redrawRegions
                    'track_current_room',
                    'icon_track_current_room',
                );
            }
        });
        push (@buttonList, $toggleButton_trackCurrentRoom);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_track_current_room', $toggleButton_trackCurrentRoom);

        # Radio button for 'always track position'
        my $radioButton_trackAlways = Gtk3::RadioToolButton->new(undef);
        if (
            $self->worldModelObj->trackingSensitivity != 0.33
            && $self->worldModelObj->trackingSensitivity != 0.66
            && $self->worldModelObj->trackingSensitivity != 1
        ) {
            # Only the sensitivity values 0, 0.33, 0.66 and 1 are curently allowed; act as
            #   though the IV was set to 0
            $radioButton_trackAlways->set_active(TRUE);
        }
        $radioButton_trackAlways->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_track_always.png'),
        );
        $radioButton_trackAlways->set_label('Always track position');
        $radioButton_trackAlways->set_tooltip_text('Always track position');
        $radioButton_trackAlways->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag && $radioButton_trackAlways->get_active()) {

                $self->worldModelObj->setTrackingSensitivity(0);
            }
        });
        push (@buttonList, $radioButton_trackAlways);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_track_always', $radioButton_trackAlways);

        # Radio button for 'track position near centre'
        my $radioButton_trackNearCentre = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_trackAlways,
        );
        if ($self->worldModelObj->trackingSensitivity == 0.33) {

            $radioButton_trackNearCentre->set_active(TRUE);
        }
        $radioButton_trackNearCentre->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_track_centre.png'),
        );
        $radioButton_trackNearCentre->set_label('Track position near centre');
        $radioButton_trackNearCentre->set_tooltip_text('Track position near centre');
        $radioButton_trackNearCentre->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag && $radioButton_trackNearCentre->get_active()) {

                $self->worldModelObj->setTrackingSensitivity(0.33);
            }
        });
        push (@buttonList, $radioButton_trackNearCentre);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_track_near_centre', $radioButton_trackNearCentre);

        # Radio button for 'track near edge'
        my $radioButton_trackNearEdge = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_trackNearCentre,
        );
        if ($self->worldModelObj->trackingSensitivity == 0.66) {

            $radioButton_trackNearEdge->set_active(TRUE);
        }
        $radioButton_trackNearEdge->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_track_edge.png'),
        );
        $radioButton_trackNearEdge->set_label('Track position near edge');
        $radioButton_trackNearEdge->set_tooltip_text('Track position near edge');
        $radioButton_trackNearEdge->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag && $radioButton_trackNearEdge->get_active()) {

                $self->worldModelObj->setTrackingSensitivity(0.66);
            }
        });
        push (@buttonList, $radioButton_trackNearEdge);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_track_near_edge', $radioButton_trackNearEdge);

        # Radio button for 'track if not visible'
        my $radioButton_trackNotVisible = Gtk3::RadioToolButton->new_from_widget(
            $radioButton_trackNearEdge,
        );
        if ($self->worldModelObj->trackingSensitivity == 1) {

            $radioButton_trackNotVisible->set_active(TRUE);
        }
        $radioButton_trackNotVisible->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_track_visible.png'),
        );
        $radioButton_trackNotVisible->set_label('Track if not visible');
        $radioButton_trackNotVisible->set_tooltip_text('Track position if not visible');
        $radioButton_trackNotVisible->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag && $radioButton_trackNotVisible->get_active()) {

                $self->worldModelObj->setTrackingSensitivity(1);
            }
        });
        push (@buttonList, $radioButton_trackNotVisible);
        # (Never desensitised)
        $self->ivAdd('menuToolItemHash', 'icon_track_not_visible', $radioButton_trackNotVisible);

        return @buttonList;
    }

    sub drawMiscButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my @buttonList;

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawMiscButtonSet', @_);
        }

        # Toolbutton for 'increase visits and set current'
        my $toolButton_incVisitsCurrent = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file(
                $axmud::SHARE_DIR . '/icons/map/icon_inc_visits_current.png',
            ),
            'Increase visits and set current',
        );
        $toolButton_incVisitsCurrent->set_tooltip_text('Increase visits and set current');
        $toolButton_incVisitsCurrent->signal_connect('clicked' => sub {

            $self->updateVisitsCallback('increase');
            $self->mapObj->setCurrentRoom($self->selectedRoom);
        });
        push (@buttonList, $toolButton_incVisitsCurrent);
        # (Requires $self->currentRegionmap & $self->selectedRoom)
        $self->ivAdd('menuToolItemHash', 'icon_inc_visits_current', $toolButton_incVisitsCurrent);

        # Toolbutton for 'increase visits by one'
        my $toolButton_incVisits = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_inc_visits.png'),
            'Increase visits by one',
        );
        $toolButton_incVisits->set_tooltip_text('Increase visits by one');
        $toolButton_incVisits->signal_connect('clicked' => sub {

            $self->updateVisitsCallback('increase');
        });
        push (@buttonList, $toolButton_incVisits);
        # (Requires $self->currentRegionmap & either $self->selectedRoom or
        #   $self->selectedRoomHash)
        $self->ivAdd('menuToolItemHash', 'icon_inc_visits', $toolButton_incVisits);

        # Toolbutton for 'decrease visits by one'
        my $toolButton_decVisits = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_dec_visits.png'),
            'Decrease visits by one',
        );
        $toolButton_decVisits->set_tooltip_text('Decrease visits by one');
        $toolButton_decVisits->signal_connect('clicked' => sub {

            $self->updateVisitsCallback('decrease');
        });
        push (@buttonList, $toolButton_decVisits);
        # (Requires $self->currentRegionmap & either $self->selectedRoom or
        #   $self->selectedRoomHash)
        $self->ivAdd('menuToolItemHash', 'icon_dec_visits', $toolButton_decVisits);

        # Toolbutton for 'set visits manually'
        my $toolButton_setVisits = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_set_visits.png'),
            'Set visits manually',
        );
        $toolButton_setVisits->set_tooltip_text('Set visits manually');
        $toolButton_setVisits->signal_connect('clicked' => sub {

            $self->updateVisitsCallback('manual');
        });
        push (@buttonList, $toolButton_setVisits);
        # (Requires $self->currentRegionmap & either $self->selectedRoom or
        #   $self->selectedRoomHash)
        $self->ivAdd('menuToolItemHash', 'icon_set_visits', $toolButton_setVisits);

        # Toolbutton for 'reset visits'
        my $toolButton_resetVisits = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_reset_visits.png'),
            'Reset visits to zero',
        );
        $toolButton_resetVisits->set_tooltip_text('Reset visits to zero');
        $toolButton_resetVisits->signal_connect('clicked' => sub {

            $self->updateVisitsCallback('reset');
        });
        push (@buttonList, $toolButton_resetVisits);
        # (Requires $self->currentRegionmap & either $self->selectedRoom or
        #   $self->selectedRoomHash)
        $self->ivAdd('menuToolItemHash', 'icon_reset_visits', $toolButton_resetVisits);

#        # Separator
#        my $separator = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toggle button for 'graffiti mode'
        my $toggleButton_graffitMode = Gtk3::ToggleToolButton->new();
        $toggleButton_graffitMode->set_active($self->graffitiModeFlag);
        $toggleButton_graffitMode->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_graffiti_mode.png'),
        );
        $toggleButton_graffitMode->set_label('Graffiti mode');
        $toggleButton_graffitMode->set_tooltip_text('Graffiti mode');
        $toggleButton_graffitMode->signal_connect('toggled' => sub {

            if ($toggleButton_graffitMode->get_active()) {
                $self->ivPoke('graffitiModeFlag', TRUE);
            } else {
                $self->ivPoke('graffitiModeFlag', FALSE);
            }

            # Set the equivalent menu item
            if ($self->ivExists('menuToolItemHash', 'graffiti_mode')) {

                my $menuItem = $self->ivShow('menuToolItemHash', 'graffiti_mode');
                $menuItem->set_active($self->graffitiModeFlag);
            }

            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $self->restrictWidgets();
        });
        push (@buttonList, $toggleButton_graffitMode);
        # (Requires $self->currentRegionmap)
        $self->ivAdd('menuToolItemHash', 'icon_graffiti_mode', $toggleButton_graffitMode);

        # Toolbutton for 'toggle graffiti'
        my $toolButton_toggleGraffiti = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_toggle_graffiti.png'),
            'Toggle graffiti',
        );
        $toolButton_toggleGraffiti->set_tooltip_text('Toggle graffiti in selected rooms');
        $toolButton_toggleGraffiti->signal_connect('clicked' => sub {

            $self->toggleGraffitiCallback();
        });
        push (@buttonList, $toolButton_toggleGraffiti);
        # (Requires $self->currentRegionmap, $self->graffitiModeFlag and one or more selected rooms
        $self->ivAdd('menuToolItemHash', 'icon_toggle_graffiti', $toolButton_toggleGraffiti);

#        # Separator
#        my $separator2 = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator2);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Toolbutton for 'edit world model'
        my $toolButton_editWorldModel = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_edit_model.png'),
            'Edit world model preferences',
        );
        $toolButton_editWorldModel->set_tooltip_text('Edit world model');
        $toolButton_editWorldModel->signal_connect('clicked' => sub {

            # Open an 'edit' window for the world model
            $self->createFreeWin(
                'Games::Axmud::EditWin::WorldModel',
                $self,
                $self->session,
                'Edit world model',
                $self->session->worldModelObj,
                FALSE,                          # Not temporary
            );
        });
        push (@buttonList, $toolButton_editWorldModel);

        # Toolbutton for 'search world model'
        my $toolButton_searchWorldModel = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_search_model.png'),
            'Search world model',
        );
        $toolButton_searchWorldModel->set_tooltip_text('Search world model');
        $toolButton_searchWorldModel->signal_connect('clicked' => sub {

            # Open a 'pref' window to conduct the search
            $self->createFreeWin(
                'Games::Axmud::PrefWin::Search',
                $self,
                $self->session,
                'World model search',
            );
        });
        push (@buttonList, $toolButton_searchWorldModel);

        # Toolbutton for 'add words'
        my $toolButton_addQuickWords = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_add_word.png'),
            'Add dictionary words',
        );
        $toolButton_addQuickWords->set_tooltip_text('Add dictionary words');
        $toolButton_addQuickWords->signal_connect('clicked' => sub {

            $self->createFreeWin(
                'Games::Axmud::OtherWin::QuickWord',
                $self,
                $self->session,
                'Quick word adder',
            );
        });
        push (@buttonList, $toolButton_addQuickWords);

        # Toolbutton for 'edit dictionary'
        my $toolButton_editDictionary = Gtk3::ToolButton->new(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_edit_dict.png'),
            'Edit current dictionary',
        );
        $toolButton_editDictionary->set_tooltip_text('Edit current dictionary');
        $toolButton_editDictionary->signal_connect('clicked' => sub {

            # Open an 'edit' window for the current dictionary
            $self->createFreeWin(
                'Games::Axmud::EditWin::Dict',
                $self,
                $self->session,
                'Edit \'' . $self->session->currentDict->name . '\' dictionary',
                $self->session->currentDict,
                FALSE,          # Not temporary
            );
        });
        push (@buttonList, $toolButton_editDictionary);

        return @buttonList;
    }

    sub drawFlagsButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my @buttonList;

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawFlagsButtonSet', @_);
        }

        # Toggle button for 'release all filters'
        my $radioButton_releaseAllFilters = Gtk3::ToggleToolButton->new();
        $radioButton_releaseAllFilters->set_active($self->worldModelObj->allRoomFiltersFlag);
        $radioButton_releaseAllFilters->set_icon_widget(
            Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/icon_all_filters.png'),
        );
        $radioButton_releaseAllFilters->set_label('Release all filters');
        $radioButton_releaseAllFilters->set_tooltip_text('Release all filters');
        $radioButton_releaseAllFilters->signal_connect('toggled' => sub {

            if (! $self->ignoreMenuUpdateFlag) {

                $self->worldModelObj->toggleFlag(
                    'allRoomFiltersFlag',
                    $radioButton_releaseAllFilters->get_active(),
                    TRUE,      # Do call $self->redrawRegions
                    'release_all_filters',
                    'icon_release_all_filters',
                );
            }
        });
        push (@buttonList, $radioButton_releaseAllFilters);
        # (Never desensitised)
        $self->ivAdd(
            'menuToolItemHash',
            'icon_release_all_filters',
            $radioButton_releaseAllFilters,
        );

#        # Separator
#        my $separator = Gtk3::SeparatorToolItem->new();
#        push (@buttonList, $separator);
        if ($^O ne 'MSWin32') {

            my $separator = Gtk3::SeparatorToolItem->new();
            push (@buttonList, $separator);
        }

        # Filter icons
        foreach my $filter ($axmud::CLIENT->constRoomFilterList) {

            # Filter button
            my $toolButton_filter = Gtk3::ToggleToolButton->new();
            $toolButton_filter->set_active(
                $self->worldModelObj->ivShow('roomFilterApplyHash', $filter),
            );
            $toolButton_filter->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag) {

                    $self->worldModelObj->toggleFilter(
                        $filter,
                        $toolButton_filter->get_active(),
                    );
                }
            });

            # If it's one of the standard filters, we can use one of the existing icons;
            #   otherwise, use a spare icon
            my $iconFile = $axmud::SHARE_DIR . '/icons/map/icon_' . $filter . '.png';
            if (! -e $iconFile) {

                $iconFile = $axmud::SHARE_DIR . '/icons/map/icon_spare_filter.png'
            }

            $toolButton_filter->set_icon_widget(
                Gtk3::Image->new_from_file($iconFile)
            );

            $toolButton_filter->set_label('Toggle ' . $filter . ' filter');
            $toolButton_filter->set_tooltip_text('Toggle ' . $filter . ' filter');
            push (@buttonList, $toolButton_filter);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'icon_' . $filter . '_filter', $toolButton_filter);
        }

        return @buttonList;
    }

    sub drawInteriorsButtonSet {

        # Called by $self->chooseButtonSet, which in turn was called by $self->drawToolbar or
        #   ->switchToolbarButtons
        # Draws buttons for this button set, and adds them to the toolbar
        #
        # Expected arguments
        #   $toolbar    - The toolbar widget on which the buttons are drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $toolbar, $check) = @_;

        # Local variables
        my (
            $lastButton,
            @initList, @interiorList, @buttonList,
            %interiorHash, %iconHash,
        );

        # Check for improper arguments
        if (! defined $toolbar || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->drawInteriorsButtonSet', @_);
        }

        @initList = (
            'none',
                'Don\'t draw interior counts',
                'icon_no_counts.png',
            'shadow_count',
                'Draw shadow/unallocated exits',
                'icon_draw_shadow.png',
            'region_count',
                'Draw region/super-region exits',
                'icon_draw_super.png',
            'checked_count',
                'Draw checked/checkable directions',
                'icon_draw_checked.png',
            'room_content',
                'Draw room contents',
                'icon_draw_contents.png',
            'hidden_count',
                'Draw hidden contents',
                'icon_draw_hidden.png',
            'temp_count',
                'Draw temporary contents',
                'icon_draw_temp.png',
            'word_count',
                'Draw recognised words',
                'icon_draw_words.png',
            'room_tag',
                'Draw room tag',
                'icon_draw_room_tag.png',
            'room_flag',
                'Draw room flag text',
                'icon_draw_room_flag.png',
            'visit_count',
                'Draw character visits',
                'icon_draw_visits.png',
            'compare_count',
                'Draw matching rooms',
                'icon_draw_compare.png',
            'profile_count',
                'Draw exclusive profiles',
                'icon_draw_exclusive.png',
            'title_descrip',
                'Draw titles/descriptions',
                'icon_draw_descrips.png',
            'exit_pattern',
                'Draw exit patterns',
                'icon_draw_patterns.png',
            'source_code',
                'Draw room source code',
                'icon_draw_code.png',
            'vnum',
                'Draw world\'s room _vnum',
                'icon_draw_vnum.png',
            'grid_posn',
                'Draw grid position',
                'icon_draw_grid_posn.png',
        );

        do {

            my ($mode, $descrip, $icon);

            $mode = shift @initList;
            $descrip = shift @initList;
            $icon = shift @initList;

            push (@interiorList, $mode);
            $interiorHash{$mode} = $descrip;
            $iconHash{$mode} = $icon;

        } until (! @initList);

        for (my $count = 0; $count < (scalar @interiorList); $count++) {

            my ($icon, $mode);

            $mode = $interiorList[$count];

            # (For $count = 0, $buttonGroup is 'undef')
            my $radioButton;
            if ($mode eq 'none') {
                $radioButton = Gtk3::RadioToolButton->new(undef);
            } else {
                $radioButton = Gtk3::RadioToolButton->new_from_widget($lastButton);
            }

            if ($self->worldModelObj->roomInteriorMode eq $mode) {

                $radioButton->set_active(TRUE);
            }
            $radioButton->set_icon_widget(
                Gtk3::Image->new_from_file($axmud::SHARE_DIR . '/icons/map/' . $iconHash{$mode}),
            );
            $radioButton->set_label($interiorHash{$mode});
            $radioButton->set_tooltip_text($interiorHash{$mode});

            $radioButton->signal_connect('toggled' => sub {

                if (! $self->ignoreMenuUpdateFlag && $radioButton->get_active()) {

                    $self->worldModelObj->switchRoomInteriorMode($mode);
                }
            });
            push (@buttonList, $radioButton);
            # (Never desensitised)
            $self->ivAdd('menuToolItemHash', 'icon_interior_mode_' . $mode, $radioButton);

            $lastButton = $radioButton;

            # (Add a separator after the first toolbar button)
            if ($mode eq 'none') {

#                # Separator
#                my $separator = Gtk3::SeparatorToolItem->new();
#                push (@buttonList, $separator);
                if ($^O ne 'MSWin32') {

                    my $separator = Gtk3::SeparatorToolItem->new();
                    push (@buttonList, $separator);
                }
            }
        }

        return @buttonList;
    }

    sub addRoomFlagButton {

        # Called by a ->signal_connect in $self->drawPaintingButtonSet whenever the user clicks the
        #   'add room flag' button in the 'painting' button set
        # Creates a popup menu containing all room flags, then implements the user's choice
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if there's an error
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my %checkHash;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addRoomFlagButton', @_);
        }

        # Compile a hash of existing preferred room flags (we don't want the user to add the same
        #   room flag twice)
        foreach my $roomFlag ($self->worldModelObj->preferRoomFlagList) {

            $checkHash{$roomFlag} = undef;
        }

        # Set up the popup menu
        my $popupMenu = Gtk3::Menu->new();
        if (! $popupMenu) {

            return undef;
        }

        # Add a title menu item, which does nothing
        my $title_item = Gtk3::MenuItem->new('Add preferred room flag:');
        $title_item->signal_connect('activate' => sub {

            return undef;
        });
        $title_item->set_sensitive(FALSE);
        $popupMenu->append($title_item);

        $popupMenu->append(Gtk3::SeparatorMenuItem->new());     # Separator

        # Fill the popup menu with room flags
        foreach my $filter ($axmud::CLIENT->constRoomFilterList) {

            # A sub-sub menu for $filter
            my $subSubMenu_filter = Gtk3::Menu->new();

            my @nameList = $self->worldModelObj->getRoomFlagsInFilter($filter);
            foreach my $name (@nameList) {

                my $obj = $self->worldModelObj->ivShow('roomFlagHash', $name);
                if ($obj) {

                    my $menuItem = Gtk3::MenuItem->new($obj->descrip);
                    $menuItem->signal_connect('activate' => sub {

                        # Add the room flag to the world model's list of preferred room flags...
                        $self->worldModelObj->add_preferRoomFlag($name);
                        # ...then redraw the window component containing the toolbar(s), toggling
                        #   the button for the new room flag
                        $self->redrawWidgets('toolbar');
                    });
                    $subSubMenu_filter->append($menuItem);
                }
            }

            if (! @nameList) {

                my $menuItem = Gtk3::MenuItem->new('(No flags in this filter)');
                $menuItem->set_sensitive(FALSE);
                $subSubMenu_filter->append($menuItem);
            }

            my $menuItem = Gtk3::MenuItem->new(ucfirst($filter));
            $menuItem->set_submenu($subSubMenu_filter);
            $popupMenu->append($menuItem);
        }

        # Also add a 'Cancel' menu item, which does nothing
        $popupMenu->append(Gtk3::SeparatorMenuItem->new());     # Separator

        my $cancel_item = Gtk3::MenuItem->new('Cancel');
        $cancel_item->signal_connect('activate' => sub {

            return undef;
        });
        $popupMenu->append($cancel_item);

        # Display the popup menu
        $popupMenu->popup(
            undef, undef, undef, undef,
            1,                              # Left mouse button
            Gtk3::get_current_event_time(),
        );

        $popupMenu->show_all();

        # Operation complete. Now wait for the user's response
        return 1;
    }

    sub removeRoomFlagButton {

        # Called by a ->signal_connect in $self->drawPaintingButtonSet whenever the user clicks the
        #   'remove room flag' button in the 'painting' button set
        # Removes the specified room flag from the toolbar and updates IVs
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if there's an error
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->removeRoomFlagButton', @_);
        }

        # Set up the popup menu
        my $popupMenu = Gtk3::Menu->new();
        if (! $popupMenu) {

            return undef;
        }

        # Add a title menu item, which does nothing
        my $title_item = Gtk3::MenuItem->new('Remove preferred room flag:');
        $title_item->signal_connect('activate' => sub {

            return undef;
        });
        $title_item->set_sensitive(FALSE);
        $popupMenu->append($title_item);

        $popupMenu->append(Gtk3::SeparatorMenuItem->new());     # Separator

        # Fill the popup menu with room flags
        foreach my $roomFlag ($self->worldModelObj->preferRoomFlagList) {

            my $menu_item = Gtk3::MenuItem->new($roomFlag);
            $menu_item->signal_connect('activate' => sub {

                # Remove the room flag from the world model's list of preferred room flags...
                $self->worldModelObj->del_preferRoomFlag($roomFlag);
                # ...and from the painter object iself...
                $self->worldModelObj->painterObj->ivDelete('roomFlagHash', $roomFlag);
                # ...then redraw the window component containing the toolbar(s)
                $self->redrawWidgets('toolbar');
            });
            $popupMenu->append($menu_item);
        }

        # Add a 'remove all' menu item
        $popupMenu->append(Gtk3::SeparatorMenuItem->new());     # Separator

        my $remove_all_item = Gtk3::MenuItem->new('Remove all');
        $remove_all_item->signal_connect('activate' => sub {

            my ($total, $choice);

            $total = scalar $self->worldModelObj->preferRoomFlagList;

            # If there's more than one colour, prompt the user for confirmation
            if ($total > 1) {

                $choice = $self->showMsgDialogue(
                    'Remove all room flag buttons',
                    'question',
                    'Are you sure you want to remove all ' . $total . ' room flag buttons?',
                    'yes-no',
                );

            } else {

                $choice = 'yes';
            }

            if (defined $choice && $choice eq 'yes') {

                # Reset the world model's list of preferred room flags
                $self->worldModelObj->reset_preferRoomFlagList();

                # Update the painter object (which might contain room flags not added with these
                #   tools)
                foreach my $roomFlag ($self->worldModelObj->preferRoomFlagList) {

                    $self->worldModelObj->ivDelete('roomFlagHash', $roomFlag);
                }

                # Then redraw the window component containing the toolbar(s)
                $self->redrawWidgets('toolbar');
            }
        });
        $popupMenu->append($remove_all_item);

        # Also add a 'Cancel' menu item, which does nothing
        my $cancel_item = Gtk3::MenuItem->new('Cancel');
        $cancel_item->signal_connect('activate' => sub {

            return undef;
        });
        $popupMenu->append($cancel_item);

        # Display the popup menu
        $popupMenu->popup(
            undef, undef, undef, undef,
            1,                              # Left mouse button
            Gtk3::get_current_event_time(),
        );

        $popupMenu->show_all();

        # Operation complete. Now wait for the user's response
        return 1;
    }

    sub addBGColourButton {

        # Called by a ->signal_connect in $self->drawBackgroundButtonSet whenever the user clicks
        #   the 'add background colour' button in the 'background' button set
        # Prompts the user for a new RGB colour tag, then implements the user's choice
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if there's an error
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $colour;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addBGColourButton', @_);
        }

        $colour = $self->showColourSelectionDialogue('Add preferred background colour');
        if (defined $colour) {

            # Add the room flag to the world model's list of preferred background colours...
            $self->worldModelObj->add_preferBGColour($colour);
            # ...then redraw the window component containing the toolbar(s), selecting the new
            #   colour
            $self->ivPoke('bgColourChoice', $colour);
            $self->redrawWidgets('toolbar');
        }

        return 1;
    }

    sub removeBGColourButton {

        # Called by a ->signal_connect in $self->drawBackgroundButtonSet whenever the user clicks
        #   the 'remove background colour' button in the 'background' button set
        # Removes the specified colour from the toolbar and updates IVs
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if there's an error
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->removeBGColourButton', @_);
        }

        # Set up the popup menu
        my $popupMenu = Gtk3::Menu->new();
        if (! $popupMenu) {

            return undef;
        }

        # Add a title menu item, which does nothing
        my $title_item = Gtk3::MenuItem->new('Remove preferred background colour:');
        $title_item->signal_connect('activate' => sub {

            return undef;
        });
        $title_item->set_sensitive(FALSE);
        $popupMenu->append($title_item);

        $popupMenu->append(Gtk3::SeparatorMenuItem->new());     # Separator

        # Fill the popup menu with colours
        foreach my $colour ($self->worldModelObj->preferBGColourList) {

            my $menu_item = Gtk3::MenuItem->new($colour);
            $menu_item->signal_connect('activate' => sub {

                # Remove the colour from the world model's list of preferred background colours...
                $self->worldModelObj->del_preferBGColour($colour);
                # ...then redraw the window component containing the toolbar(s)
                $self->redrawWidgets('toolbar');
            });
            $popupMenu->append($menu_item);
        }

        # Add a 'remove all' menu item
        $popupMenu->append(Gtk3::SeparatorMenuItem->new());     # Separator

        my $remove_all_item = Gtk3::MenuItem->new('Remove all');
        $remove_all_item->signal_connect('activate' => sub {

            my ($total, $choice);

            $total = scalar $self->worldModelObj->preferBGColourList;

            # If there's more than one colour, prompt the user for confirmation
            if ($total > 1) {

                $choice = $self->showMsgDialogue(
                    'Remove all colour buttons',
                    'question',
                    'Are you sure you want to remove all ' . $total . ' colour buttons?',
                    'yes-no',
                );

            } else {

                $choice = 'yes';
            }

            if (defined $choice && $choice eq 'yes') {

                # Reset the world model's list of preferred background colour...
                $self->worldModelObj->reset_preferBGColourList();
                # ...then redraw the window component containing the toolbar(s)
                $self->redrawWidgets('toolbar');
            }
        });
        $popupMenu->append($remove_all_item);

        # Also add a 'Cancel' menu item, which does nothing
        my $cancel_item = Gtk3::MenuItem->new('Cancel');
        $cancel_item->signal_connect('activate' => sub {

            return undef;
        });
        $popupMenu->append($cancel_item);

        # Display the popup menu
        $popupMenu->popup(
            undef, undef, undef, undef,
            1,                              # Left mouse button
            Gtk3::get_current_event_time(),
        );

        $popupMenu->show_all();

        # Operation complete. Now wait for the user's response
        return 1;
    }

    # Treeview widget methods

    sub enableTreeView {

        # Called by $self->drawWidgets
        # Sets up the Automapper window's treeview widget
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::ScrolledWindow containing the Gtk3::TreeView created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableTreeView', @_);
        }

        # Create the treeview
        my $objectModel = Gtk3::TreeStore->new( ['Glib::String'] );
        my $treeView = Gtk3::TreeView->new($objectModel);
        if (! $objectModel || ! $treeView) {

            return undef;
        }

        # No interactive searches required
        $treeView->set_enable_search(FALSE);

        # Append a single column to the treeview
        $treeView->append_column(
            Gtk3::TreeViewColumn->new_with_attributes(
                'Regions',
                Gtk3::CellRendererText->new,
                markup => 0,
            )
        );

        # Make the treeview scrollable
        my $treeViewScroller = Gtk3::ScrolledWindow->new;
        $treeViewScroller->add($treeView);
        $treeViewScroller->set_policy(qw/automatic automatic/);

        # Make the branches of the list tree clickable, so the rows can be expanded and collapsed
        $treeView->signal_connect('row_activated' => sub {

            my ($treeView, $path, $column) = @_;

            $self->treeViewRowActivatedCallback();
        });

        $treeView->get_selection->set_mode('browse');
        $treeView->get_selection->signal_connect('changed' => sub {

            my ($selection) = @_;

            $self->treeViewRowChangedCallback($selection);
        });

        # Respond when the user expands/collapses rows
        $treeView->signal_connect('row_expanded' => sub {

            my ($widget, $iter, $path) = @_;

            $self->treeViewRowExpandedCallback($iter);
        });
        $treeView->signal_connect('row_collapsed' => sub {

            my ($widget, $iter, $path) = @_;

            $self->treeViewRowCollapsedCallback($iter);
        });

        # Store the widgets
        $self->ivPoke('treeView', $treeView);
        $self->ivPoke('treeViewScroller', $treeViewScroller);
        $self->ivPoke('treeViewModel', $objectModel);

        # Fill the tree with a list of regions
        $self->resetTreeView();

        # Setup complete
        return $treeViewScroller;
    }

    sub resetTreeView {

        # Called by $self->winEnable and various other functions
        # Fills the object tree on the left of the Automapper window, listing all the regions in
        #   the current world model
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $expandRegion   - If specified, this function makes sure the object tree is expanded to
        #                       make the specified region visible. $expandRegion is the region's
        #                       name
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $expandRegion, $check) = @_;

        # Local variables
        my (
            $model, $count, $firstRegionObj,
            @initList, @otherList, @tempList, @combList, @childList,
            %pointerHash, %regionHash, %markupHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetTreeView', @_);
        }

        # Fill a model of the tree, not the tree itself
        $model = $self->treeView->get_model();
        $model->clear();

        # Import the list of regions
        @initList = $self->worldModelObj->ivValues('regionModelHash');

        # Remove a region which is supposed to be at the top of the list, and any temporary regions
        #   which should be at the bottom of it
        foreach my $regionObj (@initList) {

            if (
                defined $self->worldModelObj->firstRegion
                && $self->worldModelObj->firstRegion eq $regionObj->name
            ) {
                $firstRegionObj = $regionObj;
                $markupHash{$regionObj} = $regionObj->name;

            } elsif ($regionObj->tempRegionFlag) {

                push (@tempList, $regionObj);
                $markupHash{$regionObj} = "<i>" . $regionObj->name . "</i>";

            } else {

                push (@otherList, $regionObj);
                $markupHash{$regionObj} = $regionObj->name;
            }
        }

        # Sort the regions in their lists
        # NB If the flag is set to TRUE, the regions are shown in reverse alphabetical order
        if ($self->worldModelObj->reverseRegionListFlag) {

            # Reverse order
            @otherList = sort {lc($b->name) cmp lc($a->name)} (@otherList);
            @tempList = sort {lc($b->name) cmp lc($a->name)} (@tempList);

        } else {

            # Normal order
            @otherList = sort {lc($a->name) cmp lc($b->name)} (@otherList);
            @tempList = sort {lc($a->name) cmp lc($b->name)} (@tempList);
        }

        # Restore the combined, ordered list
        @combList = (@otherList, @tempList);
        if ($firstRegionObj) {

            unshift (@combList, $firstRegionObj);
        }

        # Import the hash which records the rows that have been expanded (and not then collapsed),
        #   before emptying it, ready for re-filling
        %regionHash = $self->treeViewRegionHash;
        $self->ivEmpty('treeViewRegionHash');

        # We need to add parent regions to the treeview before we add any child regions. Go through
        #   the list, removing regions that have no parent, and adding them to the treeview
        foreach my $regionObj (@combList) {

            my $pointer;

            # Each row containing a region is, by default, not expanded
            $self->ivAdd('treeViewRegionHash', $regionObj->name, 0);

            if ($regionObj->parent) {

                # This is a child region; add it to the treeview later
                push (@childList, $regionObj);

            } else {

                # Add this region to the treeview now
                $pointer = $model->append(undef);
                $model->set( $pointer, [0], [$markupHash{$regionObj}] );

                # Store $pointer in a hash, so that if this region has any child regions, they can
                #   be added directly below in the treeview
                $pointerHash{$regionObj->name} = $pointer;
            }
        }

        # Now, if there are any child regions, add them to the treeview just below their parent
        #   regions. Do this operation recursively until there are no regions left
        do {

            my (
                @grandChildList,
                %newPointerHash,
            );

            $count = 0;

            foreach my $regionObj (@childList) {

                my ($parentObj, $pointer, $childPointer);

                $parentObj = $self->worldModelObj->ivShow('modelHash', $regionObj->parent);
                if (! exists $pointerHash{$parentObj->name}) {

                    # This region's parent hasn't been added to the treeview yet; add it later
                    push (@grandChildList, $regionObj);

                } else {

                    $count++;

                    # Add this region to the treeview, just below its parent
                    $pointer = $pointerHash{$parentObj->name};
                    $childPointer = $model->append($pointer);
                    $model->set( $childPointer, [0], [$markupHash{$regionObj}] );

                    # Store $childPointer in a hash, so that if this region has any child regions,
                    #   they can be added directly below in the treeview
                    # (Don't add it to %pointerHash until the end of this loop iteration, otherwise
                    #   some regions won't appear in alphabetical order in the treeview)
                    $newPointerHash{$regionObj->name} = $childPointer;
                }
            }

            # All regions that were added in this loop must be moved from %newPointerHash to
            #   %pointerHash
            foreach my $key (keys %newPointerHash) {

                $pointerHash{$key} = $newPointerHash{$key};
            }

            %newPointerHash = ();

            # If there is anything in @grandChildList, they must be processed on the next iteration
            @childList = @grandChildList;

        } until (! @childList || ! $count);

        # If @childList still contains any regions, their parent(s) are either not regions (this
        #   should never happen), or the regions don't exist any more (ditto)
        # Display them at the end of the treeview
        foreach my $regionObj (@childList) {

            my $pointer;

            # Add this region to the treeview now
            $pointer = $model->append(undef);
            $model->set( $pointer, [0], [$regionObj->name] );
        }

        # Now expand any of the rows that were expanded before the call to this function
        if (%regionHash) {

            foreach my $regionName (keys %regionHash) {

                my $path;

                if (
                    $regionHash{$regionName}
                    && $self->ivExists('treeViewRegionHash', $regionName)
                ) {
                    # This row must be expanded
                    $path = $model->get_path($pointerHash{$regionName});
                    $self->treeView->expand_row($path, FALSE);
                    # Mark it as expanded
                    $self->ivAdd('treeViewRegionHash', $regionName, TRUE);
                }
            }
        }

        # Store the hash of pointers ($self->treeViewSelectLine needs them)
        $self->ivPoke('treeViewPointerHash', %pointerHash);

        # If a specific region was specified as $expandRegion, it must be visible (we must expand
        #   all its parents, if not already expanded)
        if ($expandRegion) {

            $self->expandTreeView($model, $expandRegion);
        }

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        # Operation complete
        return 1;
    }

    sub expandTreeView {

        # Called by $self->resetTreeView and by this function, recursively
        # Expands rows in the tree model, to make sure that a certain region is visible.
        #
        # Expected arguments
        #   $model          - The treeview model (a Gtk3::TreeModel object)
        #   $expandRegion   - The name of a region that should be visible. This function expands
        #                       the row belonging to $expandRegion's parent (if any), then calls
        #                       this function recursively, to expand the row for the parent's
        #                       parent (if any)
        #
        # Return values
        #   'undef' on improper arguments or if no further expansions are required
        #   1 otherwise

        my ($self, $model, $expandRegion, $check) = @_;

        # Local variables
        my ($expandObj, $parentObj, $pointer, $path);

        # Check for improper arguments
        if (! defined $model || ! defined $expandRegion || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandTreeView', @_);
        }

        # Find the corresponding world model object
        $expandObj = $self->findRegionObj($expandRegion);
        if (! $expandObj || ! $expandObj->parent) {

            # No further expansions required
            return undef;
        }

        # Get the parent object's name
        $parentObj = $self->worldModelObj->ivShow('modelHash', $expandObj->parent);

        # Expand the parent's row (if it's not already expanded)
        if (
            $self->ivExists('treeViewRegionHash', $parentObj->name)
            && ! $self->ivShow('treeViewRegionHash', $parentObj->name)
        ) {
            # This row must be expanded
            $pointer = $self->ivShow('treeViewPointerHash', $parentObj->name);
            $path = $model->get_path($pointer);
            $self->treeView->expand_row($path, TRUE);
            # Mark it as expanded
            $self->ivAdd('treeViewRegionHash', $parentObj->name, TRUE);
        }

        # Call this function recursively, to expand the parent's parent (if it has one)
        $self->expandTreeView($model, $parentObj->name);

        return 1;
    }

    sub treeViewRowActivatedCallback {

        # Treeview's 'row_activated' callback - called when the user double-clicks on one of the
        #   treeview's cells
        # Called from an anonymous sub in $self->enableTreeView
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $regionName;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->treeViewRowActivatedCallback',
                @_,
            );
        }

        # Don't do anything if the canvas is currently invisible
        if ($self->worldModelObj->showCanvasFlag) {

            # Get the selected region
            $regionName = $self->treeViewSelectedLine;

            if ($regionName) {

                if ($self->worldModelObj->ivExists('regionmapHash', $regionName)) {

                    # Make it the selected region, and draw it on the map
                    $self->setCurrentRegion($regionName);

                } else {

                    # Remove any markup to get the actual region name
                    $regionName =~ s/^\<i\>//;
                    $regionName =~ s/\<\/i\>$//;

                    if ($self->worldModelObj->ivExists('regionmapHash', $regionName)) {

                        # Make it the selected region, and draw it on the map
                        $self->setCurrentRegion($regionName);
                    }
                }
            }
        }

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        return 1;
    }

    sub treeViewRowChangedCallback {

        # Treeview's 'changed' callback - called when the user single-clicks on one of the
        #   treeview's cells
        # Called from an anonymous sub in $self->enableTreeView
        #
        # Expected arguments
        #   $selection  - A Gtk3::Selection
        #
        # Return values
        #   'undef' on improper arguments or if the selection is not recognised
        #   1 otherwise

        my ($self, $selection, $check) = @_;

        # Local variables
        my ($model, $iter, $region);

        # Check for improper arguments
        if (! defined $selection || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->treeViewRowChangedCallback',
                @_,
            );
        }

        ($model, $iter) = $selection->get_selected();
        if (! $iter) {

            return undef;

        } else {

            # Get the region on the selected line
            $region = $model->get($iter, 0);
            # Store it, so that other methods can access the region on the selected line
            $self->ivPoke('treeViewSelectedLine', $region);
            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $self->restrictWidgets();

            return 1;
        }
    }

    sub treeViewRowExpandedCallback {

        # Treeview's 'row_expanded' callback - called when the user expands one of the treeview's
        #   rows to reveal a region's child regions
        # Called from an anonymous sub in $self->enableTreeView
        #
        # Expected arguments
        #   $iter       - A Gtk3::TreeIter
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $iter, $check) = @_;

        # Local variables
        my $region;

        # Check for improper arguments
        if (! defined $iter || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->treeViewRowExpandedCallback',
                @_,
            );
        }

        # Get the region in the expanded row
        $region = $self->treeViewModel->get($iter, 0);

        # Mark the row as expanded
        $self->ivAdd('treeViewRegionHash', $region, TRUE);

        return 1;
    }

    sub treeViewRowCollapsedCallback {

        # Treeview's 'row_collapsed' callback - called when the user collapses one of the treeview's
        #   rows to hide a region's child regions
        # Called from an anonymous sub in $self->enableTreeView
        #
        # Expected arguments
        #   $iter       - A Gtk3::TreeIter
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $iter, $check) = @_;

        # Local variables
        my $region;

        # Check for improper arguments
        if (! defined $iter || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->treeViewRowCollapsedCallback',
                @_,
            );
        }

        # Get the region in the collapsed row
        $region = $self->treeViewModel->get($iter, 0);

        # Mark the row as collapsed
        $self->ivAdd('treeViewRegionHash', $region, FALSE);

        return 1;
    }

    sub treeViewSelectLine {

        # Called by $self->setCurrentRegion when the current region is set (or unset)
        # Makes sure that, if there's a new current region, it is the one highlighted in the
        #   treeview's list
        #
        # Expected arguments
        #   $region     - The name of the highlighted region (matches $self->currentRegionmap->name)
        #                   - if not specified, there is no current region
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $region, $check) = @_;

        # Local variables
        my ($pointer, $path);

        # Check for improper arguments
        if (! defined $region || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->treeViewSelectLine', @_);
        }

        # Highlight the region named $region
        $pointer = $self->ivShow('treeViewPointerHash', $region);
        $path = $self->treeViewModel->get_path($pointer);

        # If the new region has a parent, we need to expand the parent so that the new region is
        #   visible, once highlighted
        if ($path->up()) {

            $self->treeView->expand_to_path($path);
            # Reset the path to the region we want to highlight
            $path = $self->treeViewModel->get_path($pointer);
        }

        # Highlight the region
        $self->treeView->set_cursor($path, undef, 0);

        return 1;
    }

    # Canvas widget methods

    sub enableCanvas {

        # Called by $self->drawWidgets and ->redrawWidgets (only)
        # Sets up canvas widgets
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the widget can't be created
        #   Otherwise returns the Gtk3::Frame containing the Gtk3::Canvas created

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableCanvas', @_);
        }

        # Create a frame
        my $canvasFrame = Gtk3::Frame->new(undef);
        $canvasFrame->set_border_width(3);

        # Create a scrolled window
        my $canvasScroller = Gtk3::ScrolledWindow->new();
        my $canvasHAdjustment = $canvasScroller->get_hadjustment();
        my $canvasVAdjustment = $canvasScroller->get_vadjustment();
        $canvasScroller->set_border_width(3);
        # Set the scrolling policy
        $canvasScroller->set_policy('always','always');

        # Add the scrolled window to the frame
        $canvasFrame->add($canvasScroller);

        # The only way to scroll the map to the correct position, is to store the scrolled window's
        #   size allocation whenever it is set
        $canvasScroller->signal_connect('size-allocate' => sub {

            my ($widget, $hashRef) = @_;

            $self->ivPoke('canvasScrollerWidth', $$hashRef{width});
            $self->ivPoke('canvasScrollerHeight', $$hashRef{height});
        });

        # Store the remaining widgets
        $self->ivPoke('canvasFrame', $canvasFrame);
        $self->ivPoke('canvasScroller', $canvasScroller);
        $self->ivPoke('canvasHAdjustment', $canvasHAdjustment);
        $self->ivPoke('canvasVAdjustment', $canvasVAdjustment);

        # Set up tooltips
        $self->enableTooltips();
        # Draw the empty background map (default is white)
        $self->resetMap();

        # Setup complete
        return $canvasFrame;
    }

    sub enableTooltips {

        # Called by $self->enableCanvas (only)
        # Sets up tooltips
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->enableTooltips', @_);
        }

        # Create a Gtk3::Window to act as a tooltip, being visible (or not) as appropriate
        my $tooltipLabel = Gtk3::Label->new();
        my $tooltipWin = Gtk3::Window->new('popup');
        $tooltipWin->set_decorated(FALSE);
        $tooltipWin->set_position('mouse');
        $tooltipWin->set_border_width(2);
        $tooltipWin->modify_fg('normal', [Gtk3::Gdk::Color::parse('black')]->[1]);
        $tooltipWin->modify_bg('normal', [Gtk3::Gdk::Color::parse('yellow')]->[1]);
        $tooltipWin->add($tooltipLabel);

        # Update IVs
        $self->ivPoke('canvasTooltipObj', undef);
        $self->ivPoke('canvasTooltipObjType', undef);
        $self->ivPoke('canvasTooltipFlag', FALSE);

        # Setup complete
        return 1;
    }

    sub setMapPosn {

        # Can be called by anything
        # Scroll the canvas to the desired position, revealing a portion of the map
        #
        # Expected arguments
        #   $xPos   - Value between 0 (far left) and 1 (far right)
        #   $yPos   - Value between 0 (far top) and 1 (far bottom)
        #
        # Return values
        #   'undef' on improper arguments or if there is no current regionmap
        #   1 otherwise

        my ($self, $xPos, $yPos, $check) = @_;

        # Local variables
        my ($canvasWidget, $xBlocks, $yBlocks, $xPixels, $yPixels, $scrollX, $scrollY);

        # Check for improper arguments
        if (! defined $xPos || ! defined $yPos || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setMapPosn', @_);
        }

        # Do nothing if there is no current regionmap
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Get the canvas widget to be scrolled
        $canvasWidget = $self->currentParchment->ivShow(
            'canvasWidgetHash',
            $self->currentRegionmap->currentLevel,
        );

        # The code in this function, which uses GooCanvas2::Canvas->scroll_to, produces a slightly
        #   different value to the code in $self->getMapPosn, which uses scrollbar positions
        # When moving up and down through map levels, this causes the scroll position to drift from
        #   its original position
        # The only way to deal with this is to adjust $xPos and $yPos so that they represent the
        #   middle of a gridblock. In that way, the first change of level might adjust the map's
        #   scroll position (slightly), but subsequent changes preserve the exact same scroll
        #   position
        $xBlocks = int(
            ($xPos * $self->currentRegionmap->mapWidthPixels)
            / $self->currentRegionmap->blockWidthPixels
        );

        $yBlocks = int(
            ($yPos * $self->currentRegionmap->mapHeightPixels)
            / $self->currentRegionmap->blockHeightPixels
        );

        $xPixels = ($xBlocks * $self->currentRegionmap->blockWidthPixels)
            + int($self->currentRegionmap->blockWidthPixels / 2) + 1;
        $yPixels = ($yBlocks * $self->currentRegionmap->blockHeightPixels)
            + int ($self->currentRegionmap->blockHeightPixels / 2) + 1;

        $xPos = $xPixels / $self->currentRegionmap->mapWidthPixels;
        $yPos = $yPixels / $self->currentRegionmap->mapHeightPixels;

        # Previously, the map's position was set by moving the scrollbars directly. Under GooCanvas2
        #   that no longer works, so we'll use the ->scroll_to() function instead
        # Previously, a map that was smaller than the available area was positioned in the centre
        #   of the available area. Under GooCanvas2 that's no longer possible, so a small map is now
        #   positioned in the top-left corner
        $scrollX = int (
            ($xPos * $self->currentRegionmap->mapWidthPixels)
            - (($self->canvasScrollerWidth / $canvasWidget->get_scale()) / 2)
        );

        if ($scrollX < 0) {

            $scrollX = 0;
        }

        $scrollY = int (
            ($yPos * $self->currentRegionmap->mapHeightPixels)
            - (($self->canvasScrollerHeight / $canvasWidget->get_scale()) / 2)
        );

        if ($scrollY < 0) {

            $scrollY = 0;
        }

        $canvasWidget->scroll_to($scrollX, $scrollY);

        return 1;
    }

    sub getMapPosn {

        # Can be called by anything
        # Gets the position and size of canvas scrollbars, expressed as values between 0 and 1
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return arguments
        #   An empty list on improper arguments or if there is no current regionmap
        #   Otherwise, a list in the form ($xOffset, $yOffset, $xPos, $yPos, $width, $height):
        #       $xOffset    - Position of the horizontal scrollbar. 0 - as far left as possible,
        #                       1 - as far right as possible
        #       $yOffset    - Position of the vertical scrollbar. 0 - as far up as possible,
        #                       1 - as down right as possible
        #       $xPos       - Position of the left end of the visible portion of the map. 0 - left
        #                       edge is visible, 0.5 - middle of the map is visible on the left
        #                       border, etc
        #       $yPos       - Position of the top end of the visible portion of the map. 0 - top
        #                       edge is visible, 0.5 - middle of the map is visible on the top
        #                       border, etc
        #       $width      - Width of the currently visible portion of the map. 1 - total width of
        #                       map is visible; 0.5 - 50% of the total width of the map is visible,
        #                       0.1 - 10% of the total width of the map is visible (etc)
        #       $height     - Height of the currently visible portion of the map. 1 - total height
        #                       of map is visible; 0.5 - 50% of the total height of the map is
        #                       visible, 0.1 - 10% of the total height of the map is visible (etc)

        my ($self, $check) = @_;

        # Local variables
        my (
            $xOffset, $yOffset, $xPos, $yPos, $width, $height,
            @emptyList,
        );

        # Check for improper arguments
        if (defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->getMapPosn', @_);
            return @emptyList;
        }

        # Do nothing if there is no current regionmap
        if (! $self->currentRegionmap) {

            return @emptyList;
        }

        # Get the position of the horizontal scrollbar (a value between 0 and 1)
        if ($self->canvasHAdjustment->get_upper() == $self->canvasHAdjustment->get_page_size()) {

            $xOffset = 0;

        } else {

            $xOffset = $self->canvasHAdjustment->get_value()
                / (
                    $self->canvasHAdjustment->get_upper()
                    - $self->canvasHAdjustment->get_page_size()
                );
        }

        # Get the position of the vertical scrollbar (a value between 0 and 1)
        if ($self->canvasVAdjustment->get_upper() == $self->canvasVAdjustment->get_page_size()) {

            $yOffset = 0;

        } else {

            $yOffset = $self->canvasVAdjustment->get_value()
                / (
                    $self->canvasVAdjustment->get_upper()
                    - $self->canvasVAdjustment->get_page_size()
                );
        }

        # Get the position of the left end of the visible portion of the map (a value between
        #   0 and 1)
        $xPos = $self->canvasHAdjustment->get_value() / $self->canvasHAdjustment->get_upper();
        # Get the position of the top end of the visible portion of the map (a value between
        #   0 and 1)
        $yPos = $self->canvasVAdjustment->get_value() / $self->canvasVAdjustment->get_upper();

        # Get the size of the horizontal scrollbar (a value between 0 and 1)
        $width = $self->canvasHAdjustment->get_page_size() / $self->canvasHAdjustment->get_upper();
        # Get the size of the horizontal scrollbar (a value between 0 and 1)
        $height = $self->canvasVAdjustment->get_page_size() / $self->canvasVAdjustment->get_upper();

        return ($xOffset, $yOffset, $xPos, $yPos, $width, $height);
    }

    sub getMapPosnInBlocks {

        # Can be called by anything (e.g. by ->trackPosn)
        # Converts the output of $self->getMapPosn (a list of six values, all in the range 0-1)
        #   into the position and size of the visible map, measured in gridblocks, ignoring any
        #   partial gridblocks along the four edges of the visible map
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return arguments
        #   An empty list on improper arguments, or if there is no current regionmap
        #   Otherwise, a list in the form ($xPosBlocks, $yPosBlocks, $widthBlocks, $heightBlocks):
        #       $xPosBlocks, $yPosBlocks
        #           - The grid coordinates of top-left corner of the visible portion of the map
        #       $widthBlocks, $heightBlocks
        #           - The size of the visible map, in gridblocks

        my ($self, $check) = @_;

        # Local variables
        my (
            $xOffsetRatio, $yOffsetRatio, $xPosRatio, $yPosRatio, $widthRatio, $heightRatio,
            $xRawBlocks, $xPosBlocks, $xDiff, $yRawBlocks, $yPosBlocks, $yDiff, $widthBlocks,
            $heightBlocks,
            @emptyList,
        );

        # Check for improper arguments
        if (defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->getMapPosnInBlocks', @_);
            return @emptyList;
        }

        # Do nothing if there is no current regionmap
        if (! $self->currentRegionmap) {

            return @emptyList;
        }

        # Get the size and position of the visible map. The return values of $self->getMapPosn are
        #   all values in the range 0-1
        # (We don't need $xOffsetRatio or $yOffsetRatio)
        ($xOffsetRatio, $yOffsetRatio, $xPosRatio, $yPosRatio, $widthRatio, $heightRatio)
            = $self->getMapPosn();

        # Convert these values into gridblocks. The code gets values that ignore partial gridblocks
        #   along all four edges of the visible area
        $xRawBlocks = $xPosRatio * $self->currentRegionmap->gridWidthBlocks;
        $xPosBlocks = POSIX::ceil($xRawBlocks);
        $xDiff = $xPosBlocks - $xRawBlocks;

        $yRawBlocks = $yPosRatio * $self->currentRegionmap->gridHeightBlocks;
        $yPosBlocks = POSIX::ceil($yRawBlocks);
        $yDiff = $yPosBlocks - $yRawBlocks;

        $widthBlocks = int(
            ($widthRatio * $self->currentRegionmap->gridWidthBlocks) - $xDiff,
        );

        $heightBlocks = int(
            ($heightRatio * $self->currentRegionmap->gridHeightBlocks) - $yDiff,
        );

        return ($xPosBlocks, $yPosBlocks, $widthBlocks, $heightBlocks);
    }

    sub centreMapOverRoom {

        # Can be called by anything
        # Centres the map over a specified room, as far as possible (if the room is near map's
        #   edges, the map will be centred as close as possible to the room)
        # If the specified room isn't in the current region, a new current region is set
        # Alternatively, instead of specifying a room, the calling function can specify a gridblock;
        #   the map is centred over that gridblock, even if it doesn't contain a room
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $roomObj    - The room over which to centre the map. If specified, the remaining
        #                   arguments are ignored
        #   $xPosBlocks, $yPosBlocks
        #               - A gridblock on the map, in the current region, on the current level
        #                   (ignored if $roomObj is specified)
        #
        # Return values
        #   'undef' on improper arguments or if no arguments are specified at all
        #   1 otherwise

        my ($self, $roomObj, $xPosBlocks, $yPosBlocks, $check) = @_;

        # Local variables
        my ($regionObj, $blockCentreXPosPixels, $blockCentreYPosPixels);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->centreMapOverRoom', @_);
        }

        # If a room was specified...
        if ($roomObj) {

            # Check that the specified room is in the current region
            if ($roomObj->parent && $self->currentRegionmap) {

                if ($roomObj->parent != $self->currentRegionmap->number) {

                    $regionObj = $self->worldModelObj->ivShow('modelHash', $roomObj->parent);
                    # Change the current region to the one containing the specified room
                    $self->setCurrentRegion($regionObj->name);
                }

                # Set the right level
                if ($self->currentRegionmap->currentLevel != $roomObj->zPosBlocks) {

                    $self->setCurrentLevel($roomObj->zPosBlocks);
                }
            }

            $xPosBlocks = $roomObj->xPosBlocks;
            $yPosBlocks = $roomObj->yPosBlocks;

        } elsif (! defined $xPosBlocks || ! defined $yPosBlocks) {

            # Can't do anything without arguments
            return undef;
        }

        # Convert that position into canvas coordinates, and centre the map at that position
        ($blockCentreXPosPixels, $blockCentreYPosPixels) = $self->getBlockCentre(
            $roomObj->xPosBlocks,
            $roomObj->yPosBlocks,
        );

        $self->setMapPosn(
            ($blockCentreXPosPixels / $self->currentRegionmap->mapWidthPixels),
            ($blockCentreYPosPixels / $self->currentRegionmap->mapHeightPixels),
        );

        return 1;
    }

    sub doZoom {

        # Called by $self->worldModelObj->setMagnification
        # Zooms the map in or out, depending on the new value of
        #   $self->currentRegionmap->magnification
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if no arguments are specified at all
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            @redrawList,
            %newHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->doZoom', @_);
        }

        # Set the visible map's size. Each GooCanvas2::Canvas automatically takes care of its
        #   position, so that the same part of the map is visible in the window
        foreach my $canvasWidget ($self->currentParchment->ivValues('canvasWidgetHash')) {

            $canvasWidget->set_scale($self->currentRegionmap->magnification);
        }

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        return 1;
    }

    # Menu bar/toolbar widget sensitisers

    sub restrictWidgets {

        # Many menu bar and toolbar items can be sensitised, or desensitised, depending on
        #   conditions
        # This function can be called by anything, any time one of those conditions changes, so that
        #   every menu bar/toolbar item can be sensitised or desensitised correctly
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $regionObj,
            @list, @sensitiseList, @desensitiseList, @magList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->restrictWidgets', @_);
        }

        # Modified v1.0.150 - anything that requires the current regionmap, also requires
        #   the character to be logged in (with a handful of exceptions)
        # Modified v1.0.363 - we now allow zooming and a few other things from the 'View' menu
        #   when the character isn't logged in

        # Menu items that require a current regionmap AND a logged in character
        @list = (
            'select', 'unselect_all',
            'selected_objs',
            'set_follow_mode', 'icon_set_follow_mode',
            'screenshots', 'icon_visible_screenshot',
            'drag_mode', 'icon_drag_mode',
            'graffiti_mode', 'icon_graffiti_mode',
            'edit_region',
            'edit_regionmap',
            'current_region',
            'redraw_region',
            'recalculate_paths',
            'exit_tags',
            'exit_options',
            'empty_region',
            'delete_region',
            'add_room',
            'add_label_at_click',
            'add_label_at_block',
            'room_text',
            'other_room_features',
            'select_label',
            'report_region',
            'report_visits_2',
            'report_guilds_2',
            'report_flags_2',
            'report_flags_4',
            'report_rooms_2',
            'report_exits_2',
            'report_checked_2',
            'reset_locator', 'icon_reset_locator',
        );

        if ($self->currentRegionmap && $self->session->loginFlag) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap BUT NOT a logged in character
        @list = (
            'zoom_sub',
            'level_sub',
            'centre_map_middle_grid', 'icon_centre_map_middle_grid',
            'centre_map_sub',
            'move_up_level', 'icon_move_up_level',
            'move_down_level', 'icon_move_down_level',
            'this_region_scheme',
            'exit_lengths', 'icon_horizontal_lengths', 'icon_vertical_lengths',
        );

        if ($self->currentRegionmap) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, GA::Obj::WorldModel->disableUpdateModeFlag
        #   set to FALSE and a session not in 'connect offline' mode
        @list = (
            'set_update_mode', 'icon_set_update_mode',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ! $self->worldModelObj->disableUpdateModeFlag
            && $self->session->status ne 'offline'
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap for a region that doesn't have a parent region
        @list = (
            'move_region_top',
        );

        if ($self->currentRegionmap) {

            $regionObj
                = $self->worldModelObj->ivShow('regionModelHash', $self->currentRegionmap->number);
        }

        if ($regionObj && ! $regionObj->parent) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a current room
        @list = (
            'centre_map_current_room', 'icon_centre_map_current_room',
            'add_room_contents',
            'add_hidden_object',
            'add_search_result',
            'unset_current_room',
            'update_locator',
            'repaint_current',
            'execute_scripts',
            'add_failed_room',
            'add_involuntary_exit',
            'add_repulse_exit',
            'add_special_depart',
            'add_unspecified_pattern',
            'icon_fail_exit',
        );

        if ($self->currentRegionmap && $self->session->loginFlag && $self->mapObj->currentRoom) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected room
        @list = (
            'centre_map_selected_room', 'icon_centre_map_selected_room',
            'set_current_room', 'icon_set_current_room',
            'select_exit',
            'increase_set_current',
            'edit_room',
            'set_file_path',
            'add_contents_string',
            'add_hidden_string',
            'add_exclusive_prof',
            'icon_inc_visits_current',
        );

        if ($self->currentRegionmap && $self->session->loginFlag && $self->selectedRoom) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and either a single selected room or a single
        #   selected room tag
        @list = (
            'set_room_tag',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->selectedRoom || $self->selectedRoomTag)
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, a current room and a single selected room
        #   (the current room and selected room shouldn't be the same)
        @list = (
            'path_finding_highlight',
            'path_finding_edit',
            'path_finding_go',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->mapObj->currentRoom
            && $self->selectedRoom
            && $self->mapObj->currentRoom ne $self->selectedRoom
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and either a current room or a single selected
        #   room
        @list = (
            'add_to_model',
        );
        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->mapObj->currentRoom || $self->selectedRoom)
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected room with one or more
        #   checked directions
        @list = (
            'remove_checked', 'remove_checked_all',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedRoom
            && $self->selectedRoom->checkedDirHash
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected room with
        #   ->sourceCodePath set, but ->virtualAreaPath not set
        @list = (
            'view_source_code',
            'edit_source_code',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedRoom
            && $self->selectedRoom->sourceCodePath
            && ! $self->selectedRoom->virtualAreaPath
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected room with
        #   ->virtualAreaPath set
        @list = (
            'view_virtual_area',
            'edit_virtual_area',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedRoom
            && $self->selectedRoom->virtualAreaPath
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected room whose ->wildMode
        #   is not set to 'wild' (the value 'border' is ok, though)
        @list = (
            'add_normal_exit', 'add_hidden_exit',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedRoom
            && $self->selectedRoom->wildMode ne 'wild'
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, one or more selected rooms and
        #   $self->graffitiModeFlag set to TRUE
        @list = (
            'toggle_graffiti', 'icon_toggle_graffiti',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->selectedRoom || $self->selectedRoomHash)
            && $self->graffitiModeFlag
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and one or more selected rooms
        @list = (
            'move_rooms_dir', 'move_rooms_click',
            'icon_move_to_click',
            'toggle_room_flag_sub',
            'reset_positions',
            'room_exclusivity', 'room_exclusivity_sub',
            'set_exits',
            'add_multiple_exits',
            'wilderness_normal',
            'update_visits',
            'delete_room',
            'repaint_selected',
            'set_virtual_area',
            'reset_virtual_area',
            'toggle_exclusivity',
            'clear_exclusive_profs',
            'connect_adjacent',
            'icon_inc_visits', 'icon_dec_visits', 'icon_set_visits', 'icon_reset_visits',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->selectedRoom || $self->selectedRoomHash)
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and either one or more selected rooms or one
        #   or more selected room guilds (or a mixture of both)
        @list = (
            'set_room_guild',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && (
                $self->selectedRoom || $self->selectedRoomHash || $self->selectedRoomGuild
                || $self->selectedRoomGuildHash
            )
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and EITHER one or more selected rooms OR a
        #   current room
        @list = (
            'identify_room',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->selectedRoom || $self->selectedRoomHash || $self->mapObj->currentRoom)
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, one or more selected rooms and at least two
        #   regions in the world model
        @list = (
            'transfer_to_region',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->selectedRoom || $self->selectedRoomHash)
            && $self->worldModelObj->ivPairs('regionmapHash') > 1
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, a current room and the automapper object
        #   being set up to perform a merge operation
        @list = (
            'move_merge_rooms',
        );

        if (
            $self->currentRegionmap
            && $self->mapObj->currentRoom
            && $self->mapObj->currentMatchFlag
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and EITHER one or more selected rooms OR a
        #   current room and the automapper being set up to perform a merge)
        @list = (
            'move_rooms_labels',
        );

        if (
            $self->currentRegionmap
            && (
                $self->selectedRoom
                || $self->selectedRoomHash
                || ($self->mapObj->currentRoom && $self->mapObj->currentMatchFlag)
            )
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and an empty
        #   $self->currentRegionmap->gridRoomHash
        @list = (
            'add_first_room',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ! $self->currentRegionmap->gridRoomHash
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Meny items that require a current regionmap and a non-empty
        #   $self->currentRegionmap->gridRoomHash
        @list = (
            'recalculate_in_region',
            'locate_room_in_current',
        );
        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->currentRegionmap->gridRoomHash
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected exit
        @list = (
            'set_exit_dir',
            'edit_exit',
            'disconnect_exit',
            'delete_exit',
            'set_exit_type',
        );

        if ($self->currentRegionmap && $self->session->loginFlag && $self->selectedExit) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and one or more selected exits
        @list = (
            'set_ornament_sub',
            'icon_no_ornament', 'icon_openable_exit', 'icon_lockable_exit',
            'icon_pickable_exit', 'icon_breakable_exit', 'icon_impassable_exit',
            'icon_mystery_exit',
            'identify_exit',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->selectedExit || $self->selectedExitHash)
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, a single selected exit and
        #   $self->selectedExit->drawMode is 'temp_alloc' or 'temp_unalloc'
        @list = (
            'allocate_map_dir',
            'allocate_shadow',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedExit
            && (
                $self->selectedExit->drawMode eq 'temp_alloc'
                || $self->selectedExit->drawMode eq 'temp_unalloc'
            )
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, a single selected exit and
        #   $self->selectedExit->drawMode is 'primary' or 'perm_alloc'
        @list = (
            'change_direction',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedExit
            && (
                $self->selectedExit->drawMode eq 'primary'
                || $self->selectedExit->drawMode eq 'perm_alloc'
            )
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, a single selected exit and
        #   $self->selectedExit->drawMode is 'primary', 'temp_unalloc' or 'perm_alloc'
        @list = (
            'connect_to_click',
            'set_assisted_move',
            'icon_connect_click',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedExit
            && $self->selectedExit->drawMode ne 'temp_alloc'
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected exit which is a broken
        #   exit
        @list = (
            'toggle_bent_exit',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedExit
            && $self->selectedExit->brokenFlag
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected exit which is a region
        #   exit
        @list = (
            'set_super_sub',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedExit
            && $self->selectedExit->regionFlag
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and either a single selected exit which is a
        #   region exit, or a single selected exit tag
        @list = (
            'toggle_exit_tag',
            'edit_tag_text',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && (
                ($self->selectedExit && $self->selectedExit->regionFlag)
                || $self->selectedExitTag
            )
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and one or more selected exits or selected
        #   exit tags
        @list = (
            'reset_exit_tags',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && (
                $self->selectedExit || $self->selectedExitHash
                || $self->selectedExitTag || $self->selectedExitTagHash
            )
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected exit which is a
        #   super-region exit
        @list = (
            'recalculate_from_exit',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedExit
            && $self->selectedExit->superFlag
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected exit which is an
        #   uncertain exit or a one-way exit
        @list = (
            'set_exit_twin',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedExit
            && (
                $self->selectedExit->oneWayFlag
                || (
                    $self->selectedExit->destRoom
                    && ! $self->selectedExit->twinExit
                    && ! $self->selectedExit->retraceFlag
                    && $self->selectedExit->randomType eq 'none'
                )
            )
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected exit which is a one-way
        #   exit
        @list = (
            'set_incoming_dir',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->selectedExit
            && $self->selectedExit->oneWayFlag
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a single selected label
        @list = (
            'set_label',
            'customise_label',
        );

        if ($self->currentRegionmap && $self->session->loginFlag && $self->selectedLabel) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and one or more selected labels
        @list = (
            'delete_label',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->selectedLabel || $self->selectedLabelHash)
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a selected region (in the treeview)
        @list = (
            'identify_region',
        );

        if ($self->treeViewSelectedLine) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, and $self->currentRegionmap->magnification
        #   to be within a certain range of values
        @magList = $self->constMagnifyList;

        @list = (
            'zoom_out',
        );

        # (Don't try to zoom out, if already zoomed out to the maximum extent)
        if (
            $self->currentRegionmap
            && $self->currentRegionmap->magnification > $magList[0]
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        @list = (
            'zoom_in',
        );

        # (Don't try to zoom in, if already zoomed in to the maximum extent)
        if (
            $self->currentRegionmap
            && $self->currentRegionmap->magnification < $magList[-1]
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and $self->worldModelObj->drawExitMode is
        #   'ask_regionmap'
        @list = (
            'draw_region_exits',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && $self->worldModelObj->drawExitMode eq 'ask_regionmap'
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current character profile
        @list = (
            'report_visits_3',
        );

        if ($self->session->currentChar) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a current character profile
        @list = (
            'report_visits_4',
        );

        if ($self->currentRegionmap && $self->session->loginFlag && $self->session->currentChar) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current guild profile
        @list = (
            'report_guilds_3',
        );

        if ($self->session->currentGuild) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap whose ->gridColourBlockHash and/or
        #   ->gridColourObjHash is not empty)
        @list = (
            'empty_bg_colours',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && (
                $self->currentRegionmap->gridColourBlockHash
                || $self->currentRegionmap->gridColourObjHash
            )
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and a current guild profile
        @list = (
            'report_guilds_4',
        );

        if ($self->currentRegionmap && $self->session->loginFlag && $self->session->currentGuild) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require assisted moves to be turned on
        @list = (
            'allow_protected_moves',
            'allow_super_protected_moves',
        );

        if ($self->worldModelObj->assistedMovesFlag) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require protected moves to be turned off
        @list = (
            'allow_crafty_moves',
        );

        if (! $self->worldModelObj->protectedMovesFlag) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require basic mapping mode to be turned off
        @list = (
            'paint_wild', 'icon_paint_wild',
            'paint_border', 'icon_paint_border',
        );

        if (! $self->session->currentWorld->basicMappingFlag) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap, one or more selected rooms and basic mapping
        #   mode to be turned off
        @list = (
            'wilderness_wild', 'wilderness_border',
        );

        if (
            $self->currentRegionmap
            && $self->session->loginFlag
            && ($self->selectedRoom || $self->selectedRoomHash)
            && ! $self->session->currentWorld->basicMappingFlag
        ) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a non-empty list of preferred room flags
        @list = (
            'icon_remove_room_flag', 'icon_remove_room_flag_2',
        );

        if ($self->worldModelObj->preferRoomFlagList) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a non-empty list of preferred background colours
        @list = (
            'icon_bg_remove',
        );

        if ($self->worldModelObj->preferBGColourList) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap and at least one non-default colour scheme
        @list = (
            'attach_region_scheme',
        );

        if ($self->currentRegionmap && $self->worldModelObj->ivPairs('regionSchemeHash') > 1) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require a current regionmap with a non-default region scheme attached
        @list = (
            'detach_region_scheme',
        );

        if ($self->currentRegionmap && defined $self->currentRegionmap->regionScheme) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Menu items that require at least one map label style
        @list = (
            'edit_style',
        );

        if ($self->worldModelObj->mapLabelStyleHash) {
            push (@sensitiseList, @list);
        } else {
            push (@desensitiseList, @list);
        }

        # Sensitise and desensitise menu items and toolbar buttons, as required
        $self->sensitiseWidgets(@sensitiseList);
        $self->desensitiseWidgets(@desensitiseList);

        return 1;
    }

    sub sensitiseWidgets {

        # Called by anything. Frequently called by $self->restrictWidgets
        # Given a list of Gtk3 widgets (all of them menu/toolbar items), sets them as sensitive
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   @widgetList - A list of widgets - keys in the hash $self->menuToolItemHash
        #                 (e.g. 'move_up_level')
        #
        # Return values
        #   1

        my ($self, @widgetList) = @_;

        # (No improper arguments to check)

        foreach my $widgetName (@widgetList) {

            my $widget = $self->ivShow('menuToolItemHash', $widgetName);
            if ($widget) {

                $widget->set_sensitive(TRUE);
            }
        }

        return 1;
    }

    sub desensitiseWidgets {

        # Called by anything. Frequently called by $self->restrictWidgets
        # Given a list of Gtk3 widgets (all of them menu/toolbar items), sets them as insensitive
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   @widgetList - A list of widgets - keys in the hash $self->menuToolItemHash
        #                 (e.g. 'move_up_level')
        #
        # Return values
        #   1

        my ($self, @widgetList) = @_;

        # (No improper arguments to check)

        foreach my $widgetName (@widgetList) {

            my $widget = $self->ivShow('menuToolItemHash', $widgetName);
            if ($widget) {

                $widget->set_sensitive(FALSE);
            }
        }

        return 1;
    }

    sub setActiveItem {

        # Can be called by anything, but mostly called by functions in GA::Obj::WorldModel that want
        #   to set a menu bar or toolbar item as active (or not)
        #
        # Expected arguments
        #   $widgetName - The widget's name, a key in the hash $self->menuToolItemHash
        #
        # Optional arguments
        #   $flag       - Any TRUE value to set the menu bar/toolbar item as active, any FALSE value
        #                   (including 'undef') to set the item as not active
        #
        # Return values
        #   'undef' on improper arguments or if $widgetName doesn't appear in
        #       $self->menuToolItemHash
        #   1 otherwise

        my ($self, $widgetName, $flag, $check) = @_;

        # Local variables
        my $widget;

        # Check for improper arguments
        if (! defined $widgetName || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setActiveItem', @_);
        }

        $widget = $self->ivShow('menuToolItemHash', $widgetName);
        if (! defined $widget) {

            return undef;

        } else {

            if (! $flag) {
                $widget->set_active(FALSE);
            } else {
                $widget->set_active(TRUE);
            }

            return 1;
        }
    }

    sub restrictUpdateMode {

        # Called by $self->setMode
        # Sensitises or desensitises the menu and toolbar buttons that allow the user to switch to
        #   update mode, depending on the value of GA::Obj::WorldModel->disableUpdateModeFlag and
        #   GA::Session->status
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($radioMenuItem, $toolbarButton);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->restrictUpdateMode', @_);
        }

        # Mark the radio/toolbar buttons for 'update mode' as sensitive, or not
        $radioMenuItem = $self->ivShow('menuToolItemHash', 'set_update_mode');
        $toolbarButton = $self->ivShow('menuToolItemHash', 'icon_set_update_mode');

        if ($self->worldModelObj->disableUpdateModeFlag || $self->session->status eq 'offline') {

            if ($radioMenuItem) {

                $radioMenuItem->set_sensitive(FALSE);
            }

            if ($toolbarButton) {

                $toolbarButton->set_sensitive(FALSE);
            }

        } else {

            if ($radioMenuItem) {

                $radioMenuItem->set_sensitive(TRUE);
            }

            if ($toolbarButton) {

                $toolbarButton->set_sensitive(TRUE);
            }
        }

        return 1;
    }

    # Pause windows handlers

    sub showPauseWin {

        # Can be called by anything
        # Makes the pause window visible (a 'dialogue' window used only by this automapper)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->showPauseWin', @_);
        }

        if (! $axmud::CLIENT->busyWin) {

            # Show the window widget
            $self->showBusyWin(
                $axmud::SHARE_DIR . '/icons/system/mapper.png',
                'Working...',
            );
        }

        return 1;
    }

    sub hidePauseWin {

        # Can be called by anything
        # Makes the pause window invisible
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->hidePauseWin', @_);
        }

        if ($axmud::CLIENT->busyWin) {

            $self->closeDialogueWin($axmud::CLIENT->busyWin);
        }

        return 1;
    }

    # Canvas callbacks

    sub setupCanvasEvent {

        # Called by $self->resetMap() to create an anonymous function to intercept signals from the
        #   map background, filter out the signals we don't want, and pass the signals we do want to
        #   an event handler
        # Because the background is at the 'bottom', the anonymous function is only called when the
        #   user is clicking on an empty part of the map
        # Also called by $self->drawRoomEcho when the user clicks on a room echo, which we should
        #   treat as if it were a click on the map background
        #
        # Expected arguments
        #   $canvasObj     - The GooCanvas2::CanvasRect which is the map's background (or the
        #                       GooCanvas2::CanvasRect which is a room echo, which should be treated
        #                       as part of the map background)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $canvasObj, $check) = @_;

        # Check for improper arguments
        if (! defined $canvasObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupCanvasEvent', @_);
        }

        $canvasObj->signal_connect('button_press_event' => sub {

            my ($item, $target, $event) = @_;

            # If the tooltips are visible, hide them
            $self->hideTooltips();

            # All clicks on the canvas itself are handled by this function
            $self->canvasEventHandler($canvasObj, $event);
        });

        $canvasObj->signal_connect('button_release_event' => sub {

            my ($item, $target, $event) = @_;

            # If the tooltips are visible, hide them
            $self->hideTooltips();

            # All clicks on the canvas itself are handled by this function
            $self->canvasEventHandler($canvasObj, $event);
        });

        $canvasObj->signal_connect('motion_notify_event' => sub {

            my ($item, $target, $event) = @_;

            if ($self->selectBoxFlag && $event->state =~ m/button1-mask/) {

                # Continue the selection box operation by re-drawing the canvas object at its new
                #   position
                $self->continueSelectBox($event);
            }
        });

        # Setup complete
        return 1;
    }

    sub setupCanvasObjEvent {

        # Called by various functions to create an anonymous function to intercept signals from
        #   canvas objects above the map background, filter out the signals we don't want, and pass
        #   the signals we do want to an event handler
        # Because canvas objects are 'above' the background, the anonymous function (and not the
        #   one in $self->setupCanvasEvent) is called when the user clicks on a coloured block,
        #   rectangle, room, room tag, room guild, exit or label directly
        #
        # Expected arguments
        #   $type       - What type of canvas object this is - 'room', 'room_tag', 'room_guild',
        #                   'exit', 'exit_tag', 'label', 'square' or 'rect'
        #   $canvasObj  - The canvas object on which the user has clicked (i.e.
        #                   GooCanvas2::CanvasRect, GooCanvas2::CanvasPath,
        #                   GooCanvas2::CanvasEllipse or GooCanvas2::CanvasText)
        #
        # Optional arguments
        #   $modelObj   - The GA::ModelObj::Room, GA::Obj::Exit or GA::Obj::MapLabel which is
        #                   represented by this canvas object. 'undef' for coloured blocks or
        #                   rectangles, which can't be clicked
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $type, $canvasObj, $modelObj, $check) = @_;

        # Check for improper arguments
        if (! defined $type || ! defined $canvasObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupCanvasObjEvent', @_);
        }

        $canvasObj->signal_connect('button_press_event' => sub {

            my ($item, $target, $event) = @_;

            # Coloured blocks/rectangles can't be clicked; treat a click on one of these canvas
            #   objects as if it was a click on the map background
            if (! $modelObj) {

                $self->canvasEventHandler($canvasObj, $event);

            # For left-clicks, if the Alt-Gr key is pressed down (or if we're in drag mode), it's a
            #   drag operation
            } elsif (
                $event->button == 1 && ($event->state =~ m/mod5-mask/ || $self->dragModeFlag)
            ) {
                # Respond to the start of a drag operation
                $self->startDrag(
                    $type,
                    $canvasObj,
                    $modelObj,
                    $event,
                    $event->x_root,
                    $event->y_root,
                );

            # All other clicks on a canvas object are handled by the event handler
            } elsif ($event->type eq 'button-press' || $event->type eq '2button-press') {

                $self->canvasObjEventHandler($type, $canvasObj, $modelObj, $event);
            }
        });

        $canvasObj->signal_connect('button_release_event' => sub {

            my ($item, $target, $event) = @_;

            if (! $modelObj) {

                $self->canvasEventHandler($canvasObj, $event);

            } elsif (
                $self->dragFlag
                && (
                    $canvasObj eq $self->dragCanvasObj
                    # When dragging labels with a box, there are two canvas objects
                    || $self->ivFind('dragCanvasObjList', $canvasObj)
                )
            ) {
                # Respond to the end of a drag operation
                $self->stopDrag($event, $event->x_root, $event->y_root);
            }
        });

        $canvasObj->signal_connect('motion_notify_event' => sub {

            my ($item, $target, $event) = @_;

            if (! $modelObj) {

                if ($self->selectBoxFlag && $event->state =~ m/button1-mask/) {

                    # Continue the selection box operation by re-drawing the canvas object at its
                    #   new position
                    $self->continueSelectBox($event);

                } else {

                    $self->canvasEventHandler($canvasObj, $event);
                }

            # Process mouse events - when the mouse moves over a canvas object, or leaves it -
            #   in order to display tooltips, etc
            } elsif (
                $self->dragFlag
                && (
                    $canvasObj eq $self->dragCanvasObj
                    # When dragging labels with a box, there are two canvas objects
                    || $self->ivFind('dragCanvasObjList', $canvasObj)
                )
                && $event->state =~ m/button1-mask/
            ) {
                # Continue the drag operation by re-drawing the object(s) at their new position
                $self->continueDrag($event, $event->x_root, $event->y_root);
            }
        });

        $canvasObj->signal_connect('enter_notify_event' => sub {

            my ($item, $target, $event) = @_;

            if ($modelObj && $self->worldModelObj->showTooltipsFlag && ! $self->canvasTooltipObj) {

                # Show the tooltips window
                $self->showTooltips($type, $canvasObj, $modelObj);
            }
        });

        $canvasObj->signal_connect('leave_notify_event' => sub {

            my ($item, $target, $event) = @_;

            if (
                $modelObj
                && $self->canvasTooltipFlag
                && $self->canvasTooltipObj eq $canvasObj
                && $self->canvasTooltipObjType eq $type
            ) {
                # Hide the tooltips window
                $self->hideTooltips();
            }
        });

        # Setup complete
        return 1;
    }

    sub canvasEventHandler {

        # Handles events on the map background (i.e. clicking on an empty part of the background
        #   which doesn't contain a room, room tag, room guild, exit, exit tag, label, or checked
        #   direction)
        # The calling function, an anonymous sub defined in $self->setupCanvasEvent, filters out the
        #   signals we don't want
        # At the moment, the signals let through the filter are:
        #   button_press, 2button_press, 3button_press, button_release
        #
        # Expected arguments
        #   $canvasObj  - The canvas object which intercepted an event signal
        #   $event      - The Gtk3::Gdk::Event that caused the signal
        #
        # Return values
        #   'undef' on improper arguments, if there is no region map or if the signal $event is one
        #       that this function doesn't handle
        #   1 otherwise

        my ($self, $canvasObj, $event, $check) = @_;

        # Local variables
        my (
            $clickXPosPixels, $clickYPosPixels, $clickType, $button, $shiftFlag, $ctrlFlag,
            $clickXPosBlocks, $clickYPosBlocks, $newRoomObj, $roomNum, $roomObj, $exitObj, $listRef,
            $result, $twinExitObj, $result2, $popupMenu,
        );

        # Check for improper arguments
        if (! defined $canvasObj || ! defined $event || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->canvasEventHandler', @_);
        }

        # Don't do anything if there is no current regionmap
        if (! $self->currentRegionmap) {

            return undef;
        }

        # In case the previous click on the canvas was a right-click on an exit, we no longer need
        #   the coordinates of the click
        $self->ivUndef('exitClickXPosn');
        $self->ivUndef('exitClickYPosn');

        # Get the coordinates on the map of the clicked pixel. If the map is magnified we might get
        #   fractional values, so we need to use int()
        ($clickXPosPixels, $clickYPosPixels) = (int($event->x), int($event->y));

        # For mouse button clicks, get the click type and whether or not the SHIFT and/or CTRL keys
        #   were held down
        ($clickType, $button, $shiftFlag, $ctrlFlag) = $self->checkMouseClick($event);
        if (! $clickType) {

            # Not an event in which we're interested
            return undef;
        }

        # Work out which gridblock is underneath the mouse click
        ($clickXPosBlocks, $clickYPosBlocks) = $self->findGridBlock(
            $clickXPosPixels,
            $clickYPosPixels,
            $self->currentRegionmap,
        );

        # If $self->freeClickMode and/or $self->bgColourMode aren't set to 'default', left-clicking
        #   on empty space causes something unusual to happen
        if (
            $clickType eq 'single'
            && $button eq 'left'
            && ($self->freeClickMode ne 'default' || $self->bgColourMode ne 'default')
        ) {
            # Free click mode 'add_room' - 'Add room at click' menu option
            # (NB If this code is altered, the equivalent code in ->enableCanvasPopupMenu must also
            #   be altered)
            if ($self->freeClickMode eq 'add_room') {

                # Only add one new room
                $self->reset_freeClickMode();

                $newRoomObj = $self->mapObj->createNewRoom(
                    $self->currentRegionmap,
                    $clickXPosBlocks,
                    $clickYPosBlocks,
                    $self->currentRegionmap->currentLevel,
                );

                # When using the 'Add room at block' menu item, the new room is selected to make it
                #   easier to see where it was drawn
                # To make things consistent, select this new room, too
                $self->setSelectedObj(
                    [$newRoomObj, 'room'],
                    FALSE,              # Select this object; unselect all other objects
                );

            # Free click mode 'connect_exit' - 'Connect exit to click' menu option
            # Free click mode 'merge_room' - 'Merge/move rooms' menu option
            } elsif (
                $self->freeClickMode eq 'connect_exit'
                || $self->freeClickMode eq 'merge_room'
            ) {
                # If the user has selected the either of these menu option, $self->freeClickMode has
                #   been set and we're waiting for a click on a room; since this part of the grid is
                #   not occupied by a room, we can cancel it now
                $self->reset_freeClickMode();

            # Free click mode 'add_label' - 'Add label at click' menu option
            } elsif ($self->freeClickMode eq 'add_label') {

                $self->addLabelAtClickCallback($clickXPosPixels, $clickYPosPixels);

                # Only add one new label
                $self->reset_freeClickMode();

            # Free click mode 'move_room' - 'Move selected rooms to click' menu option
            } elsif ($self->freeClickMode eq 'move_room') {

                $self->moveRoomsToClick($clickXPosBlocks, $clickYPosBlocks);

                # Only do it once
                $self->reset_freeClickMode();

            # Background colour mode 'square_start' (no menu option)
            } elsif ($self->freeClickMode eq 'default' && $self->bgColourMode eq 'square_start') {

                $self->setColouredSquare($clickXPosBlocks, $clickYPosBlocks);

            # Background colour mode 'rect_start' (no menu option)
            } elsif ($self->freeClickMode eq 'default' && $self->bgColourMode eq 'rect_start') {

                # Store the coordinates of the click, and wait for the second click
                $self->ivPoke('bgColourMode', 'rect_stop');
                $self->ivPoke('bgRectXPos', $clickXPosBlocks);
                $self->ivPoke('bgRectYPos', $clickYPosBlocks);

            # Background colour mode 'rect_stop' (no menu option)
            } elsif ($self->freeClickMode eq 'default' && $self->bgColourMode eq 'rect_stop') {

                $self->setColouredRect($clickXPosBlocks, $clickYPosBlocks);
            }

            # Non-default operation complete
            return 1;
        }

        # Otherwise, see if there's a room inside the gridblock that was clicked (if there is, we
        #   will be able to detect clicks near exits)
        $roomNum = $self->currentRegionmap->fetchRoom(
            $clickXPosBlocks,
            $clickYPosBlocks,
            $self->currentRegionmap->currentLevel,
        );

        if (defined $roomNum) {

            $roomObj = $self->worldModelObj->ivShow('modelHash', $roomNum);
        }

        if ($roomObj && $clickType eq 'single' && $self->currentRegionmap->gridExitHash) {

            # Usually, when we click on the map on an empty pixel, all selected objects are
            #   unselected
            # However, because exits are often drawn only 1 pixel wide, they're quite difficult to
            #   click on. This section checks whether the mouse click occured close enough to an
            #   exit
            # A left-click near an exit causes the exit to be selected/unselected. A right-click
            #   selects the exit (unselecting everything else) and opens a popup menu for that exit.
            #   If the click isn't close enough to an exit, the user is deemed to have clicked in
            #   open space
            # (NB If no exits have been drawn, don't bother checking)

            # Now we check if they clicked near an exit, or in open space
            $exitObj = $self->findClickedExit(
                $clickXPosPixels,
                $clickYPosPixels,
                $roomObj,
                $self->currentRegionmap,
            );

            if ($exitObj) {

                if ($button eq 'left' && $event->state =~ m/mod5-mask/) {

                    # This is a drag operation on the nearby exit
                    $listRef = $self->currentParchment->getDrawnExit($exitObj);
                    if (defined $listRef) {

                        $self->startDrag(
                            'exit',
                            $$listRef[0],        # The exit's canvas object
                            $exitObj,
                            $event,
                            $clickXPosPixels,
                            $clickYPosPixels,
                        );
                    }

                } elsif ($button eq 'left') {

                    # If this exit (and/or its twin) is a selected exit, unselect them
                    $result = $self->unselectObj($exitObj);
                    if ($exitObj->twinExit) {

                        $twinExitObj
                            = $self->worldModelObj->ivShow('exitModelHash', $exitObj->twinExit);

                        if ($twinExitObj) {

                            $result2 = $self->unselectObj($twinExitObj);
                        }
                    }

                    if (! $result && ! $result2) {

                        # The exit wasn't already selected, so select it
                        $self->setSelectedObj(
                            [$exitObj, 'exit'],
                            # Retain other selected objects if CTRL key held down
                            $ctrlFlag,
                        );
                    }

                } elsif ($button eq 'right') {

                    # Select the exit, unselecting all other selected objects
                   $self->setSelectedObj(
                        [$exitObj, 'exit'],
                        FALSE,          # Select this object; unselect all other objects
                    );

                    # Create the popup menu
                    if ($self->selectedExit) {

                        $popupMenu = $self->enableExitsPopupMenu();
                        if ($popupMenu) {

                            $popupMenu->popup(
                                undef, undef, undef, undef,
                                $event->button,
                                $event->time,
                            );
                        }
                    }
                }

                return 1;
            }
        }

        # Otherwise, the user clicked in open space

        # If it was a right-click, open a popup menu
        if ($clickType eq 'single' && $button eq 'right') {

            $popupMenu = $self->enableCanvasPopupMenu(
                $clickXPosPixels,
                $clickYPosPixels,
                $clickXPosBlocks,
                $clickYPosBlocks,
            );

            if ($popupMenu) {

                $popupMenu->popup(
                    undef, undef, undef, undef,
                    $event->button,
                    $event->time,
                );
            }

        # If it was a left-click, it's potentially a selection box operation
        } elsif ($clickType eq 'single' && $button eq 'left') {

            # The selection box isn't actually drawn until the user moves their mouse. If they
            #   release the button instead, at that point we unselect all selected objects
            $self->startSelectBox($clickXPosPixels, $clickYPosPixels);

        # If it's a mouse button release, handle the end of any selection box operation
        } elsif ($clickType eq 'release' && $button eq 'left' && $self->selectBoxFlag) {

            $self->stopSelectBox($event, $clickXPosPixels, $clickYPosPixels);

        # Otherwise, if it's a button click (not a button release), just unselect all selected
        #   objects
        } elsif ($clickType ne 'release') {

            $self->setSelectedObj();
        }

        return 1;
    }

    sub canvasObjEventHandler {

        # Handles events on canvas object (i.e. clicking on a room, room tag, room guild, exit,
        #   exit tag or label). Note that clicks on canvas objects for checked directions are
        #   ignored; they are not handled by this function nor by $self->canvasEventHandler
        # The calling function, an anonymous sub defined in $self->setupCanvasObjEvent, filters out
        #   the signals we don't want
        # At the moment, the signals let through the filter are:
        #   button_press, 2button_press
        #
        # Expected arguments
        #   $objType    - 'room', 'room_tag', 'room_guild', 'exit', 'exit_tag' or 'label'
        #   $canvasObj  - The canvas object which intercepted an event signal
        #   $modelObj   - The GA::ModelObj::Room, GA::Obj::Exit or GA::Obj::MapLabel which is
        #                   represented by this canvas object
        #   $event      - The Gtk3::Gdk::Event that caused the signal
        #
        # Return values
        #   'undef' on improper arguments or if the signal $event is one that this function doesn't
        #       handle
        #   1 otherwise

        my ($self, $objType, $canvasObj, $modelObj, $event, $check) = @_;

        # Local variables
        my (
            $clickType, $button, $shiftFlag, $ctrlFlag, $selectFlag, $clickTime, $otherRoomObj,
            $startX, $stopX, $startY, $stopY, $result, $twinExitObj, $result2, $popupMenu,
        );

        # Check for improper arguments
        if (
            ! defined $objType || ! defined $canvasObj || ! defined $modelObj || ! defined $event
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->canvasObjEventHandler', @_);
        }

        # In case the previous click on the canvas was a right-click on an exit, we no longer need
        #   the coordinates of the click
        $self->ivUndef('exitClickXPosn');
        $self->ivUndef('exitClickYPosn');

        # If $self->freeClickMode has been set to 'add_room' or 'add_label' by the 'Add room at
        #   click' or 'Add label at click' menu options, since this part of the grid is already
        #   occupied, we can go back to normal
        if ($self->freeClickMode eq 'add_room' || $self->freeClickMode eq 'add_label') {

            $self->reset_freeClickMode();
        }

        # For mouse button clicks, get the click type and whether or not the SHIFT and/or CTRL keys
        #   were held down
        ($clickType, $button, $shiftFlag, $ctrlFlag) = $self->checkMouseClick($event);
        if (! $clickType) {

            # Not an event in which we're interested
            return undef;
        }

        # Various parts of the function check that these hashes contain at least one item between
        #   them
        if (
            $self->selectedRoomHash || $self->selectedRoomTagHash || $self->selectedRoomGuildHash
            || $self->selectedExitHash || $self->selectedExitTagHash || $self->selectedLabelHash
        ) {
            $selectFlag = TRUE;
        }

        # For capturing double-clicks on rooms, we need to compare the times at which each click is
        #   received
        $clickTime = $axmud::CLIENT->getTime();

        # Process single left clicks
        if ($clickType eq 'single' && $button eq 'left') {

            # Process a left-clicked room differently, if ->freeClickMode has been set to
            #   'connect_exit' by the 'Connect to click' menu option (ignoring the SHIFT/CTRL keys)
            if ($self->freeClickMode eq 'connect_exit' && $objType eq 'room') {

                # Occasionally get an error, when there's no selected exit. $self->freeClickMode
                #   should get reset, but not in these situations
                if (! $self->selectedExit) {

                    $self->reset_freeClickMode();

                } else {

                    # Get the selected exit's parent room, and the room's parent region
                    $otherRoomObj
                        = $self->worldModelObj->ivShow('modelHash', $self->selectedExit->parent);

                    # If that room and the clicked room are in the same region...
                    if ($otherRoomObj && $modelObj->parent == $otherRoomObj->parent) {

                        # The two rooms are in the same region, so it's (possibly) a broken exit
                        $self->connectExitToRoom($modelObj, 'broken');

                    } else {

                        # The two rooms are in different regions, so it's a region exit
                        $self->connectExitToRoom($modelObj, 'region');
                    }

                    # Only do it once
                    $self->reset_freeClickMode();
                }

            # Process a left-clicked room differently, if ->freeClickMode has been set to
            #   'move_room' by the 'Move selected rooms to click' menu option (ignoring the
            #   SHIFT/CTRL keys)
            } elsif ($self->freeClickMode eq 'move_room' && $objType eq 'room') {

                $self->moveRoomsToExit($modelObj);

                # Only do it once
                $self->reset_freeClickMode();

            # Process a left-clicked room differently, if ->toolbarQuickPaintColour is set (ignoring
            #   the SHIFT/CTRL keys)
            } elsif ($self->toolbarQuickPaintColour && $objType eq 'room') {

                $self->doQuickPaint($modelObj);

            # Process a left-clicked room differently, if ->freeClickMode has been set to
            #   'merge_room' by the 'Merge/move rooms' menu option (ignoring the SHIFT/CTRL keys)
            } elsif ($self->freeClickMode eq 'merge_room' && $objType eq 'room') {

                $self->doMerge($self->mapObj->currentRoom, $modelObj);

                # Only do it once
                $self->reset_freeClickMode();

            # Process left-clicked rooms (ignoring the CTRL key, but checking for the SHIFT key)
            } elsif (
                $objType eq 'room'
                && $shiftFlag
                && ($self->selectedRoom || $self->selectedRoomHash)
            ) {
                # Find the coordinates of opposite corners (top-left and bottom-right) of the area
                #   of the grid which contains currently selected rooms
                ($startX, $startY, $stopX, $stopY) = $self->findSelectedRoomArea();

                # If there are no selected rooms on this level...
                if (! defined $startX) {

                    # Select this room, only
                    $startX = $modelObj->xPosBlocks;
                    $startY = $modelObj->yPosBlocks;
                    $stopX = $modelObj->xPosBlocks;
                    $stopY = $modelObj->yPosBlocks;

                # Otherwise, if the clicked room is selected...
                } elsif ($self->checkRoomIsSelected($modelObj)) {

                    # If the clicked room is at the top-left of the area containing selected rooms,
                    #   select only this room, and unselect all the others
                    if ($modelObj->xPosBlocks == $startX && $modelObj->yPosBlocks == $startY) {

                        $stopX = $startX;
                        $stopY = $startY;

                    # Otherwise, the clicked room is the new bottom-right of the selected area
                    } else {

                        $stopX = $modelObj->xPosBlocks;
                        $stopY = $modelObj->yPosBlocks;
                    }

                # ...but if the clicked room isn't selected...
                } else {

                    # If the clicked room's x-coordinate is to the left of the area's starting
                    #   x-coordinate, change the area's starting x co-ordinate
                    if ($modelObj->xPosBlocks < $startX) {

                        $startX = $modelObj->xPosBlocks;

                    # Likewise for the other three corners
                    } elsif ($modelObj->xPosBlocks > $stopX) {

                        $stopX = $modelObj->xPosBlocks;
                    }

                    if ($modelObj->yPosBlocks < $startY) {

                        $startY = $modelObj->yPosBlocks;

                    } elsif ($modelObj->yPosBlocks > $stopY) {

                        $stopY = $modelObj->yPosBlocks;
                    }
                }

                # Select all rooms in the (modified) area, and unselect all rooms outside it (along
                #   with any selected exits, room tags and labels)
                $self->selectRoomsInArea($startX, $startY, $stopX, $stopY);

            # Process double-clicked rooms
            } elsif (
                $objType eq 'room'
                && $button eq 'left'
                && ! $shiftFlag
                && ! $ctrlFlag
                && $self->worldModelObj->quickPathFindFlag
                && $self->mapObj->currentRoom
                && $self->leftClickTime
                && ($self->leftClickTime + $self->leftClickWaitTime) > $clickTime
                && $self->leftClickObj eq $modelObj
            ) {
                # Double-click detected. Reset IVs
                $self->ivUndef('leftClickTime');
                $self->ivUndef('leftClickObj');

                # Don't do anything if the user clicked on the current room and the automapper
                #   object isn't set up to perform a merge
                if ($modelObj eq $self->mapObj->currentRoom) {

                    if ($self->mapObj->currentMatchFlag) {

                        $self->doMerge($modelObj);
                    }

                } else {

                    # Ensure the double-clicked room is the only one selected...
                    $self->setSelectedObj(
                        [$modelObj, 'room'],
                        FALSE,          # Select this object; unselect all other objects
                    );

                    # ...so the callback function knows which room is the destination room
                    return $self->processPathCallback('send_char');
                }

            # Process left-clicked room tags (ignoring the SHIFT key, but checking for the CTRL key)
            } elsif ($objType eq 'room_tag') {

                # If a group of things are already selected, unselect them all and select the object
                #   that was clicked
                if (! $ctrlFlag && $selectFlag) {

                    # Select this room tag, unselecting all other objects
                    $self->setSelectedObj(
                        [$modelObj, 'room_tag'],
                        # Retain other selected objects if CTRL key held down
                        $ctrlFlag,
                    );

                } else {

                    # If this object is already a selected object, unselect it
                    if (! $self->unselectObj($modelObj, 'room_tag')) {

                        # The room tag wasn't already selected, so select it
                        $self->setSelectedObj(
                            [$modelObj, 'room_tag'],
                            # Retain other selected objects if CTRL key held down
                            $ctrlFlag,
                        );
                    }
                }

            # Process left-clicked room guilds (ignoring the SHIFT key, but checking for the CTRL
            #   key)
            } elsif ($objType eq 'room_guild') {

                # If a group of things are already selected, unselect them all and select the object
                #   that was clicked
                if (! $ctrlFlag && $selectFlag) {

                    # Select this room guild, unselecting all other objects
                    $self->setSelectedObj(
                        [$modelObj, 'room_guild'],
                        # Retain other selected objects if CTRL key held down
                        $ctrlFlag,
                    );

                } else {

                    # If this object is already a selected object, unselect it
                    if (! $self->unselectObj($modelObj, 'room_guild')) {

                        # The room guild wasn't already selected, so select it
                        $self->setSelectedObj(
                            [$modelObj, 'room_guild'],
                            # Retain other selected objects if CTRL key held down
                            $ctrlFlag,
                        );
                    }
                }

            # Process left-clicked exits (ignoring the SHIFT key, but checking for the CTRL key)
            } elsif ($objType eq 'exit') {

                # For twin exits - which share a canvas object - use the exit whose parent room is
                #   closest to the click
                $modelObj = $self->chooseClickedExit($modelObj, int($event->x), int($event->y));

                # If a group of things are already selected, unselect them all and select the object
                #   that was clicked
                if (! $ctrlFlag && $selectFlag) {

                    # Select this exit, unselecting all other objects
                    $self->setSelectedObj(
                        [$modelObj, 'exit'],
                        # Retain other selected objects if CTRL key held down
                        $ctrlFlag,
                    );

                } else {

                    # If this exit (and/or its twin) is a selected exit, unselect them
                    $result = $self->unselectObj($modelObj);
                    if ($modelObj->twinExit) {

                        $twinExitObj
                            = $self->worldModelObj->ivShow('exitModelHash', $modelObj->twinExit);

                        if ($twinExitObj) {

                            $result2 = $self->unselectObj($twinExitObj);
                        }
                    }

                    if (! $result && ! $result2) {

                        # The exit wasn't already selected, so select it
                        $self->setSelectedObj(
                            [$modelObj, 'exit'],
                            # Retain other selected objects if CTRL key held down
                            $ctrlFlag,
                        );
                    }
                }

            # Process left-clicked exit tags (ignoring the SHIFT key, but checking for the CTRL key)
            } elsif ($objType eq 'exit_tag') {

                # If a group of things are already selected, unselect them all and select the object
                #   that was clicked
                if (! $ctrlFlag && $selectFlag) {

                    # Select this exit tag, unselecting all other objects
                    $self->setSelectedObj(
                        [$modelObj, 'exit_tag'],
                        # Retain other selected objects if CTRL key held down
                        $ctrlFlag,
                    );

                } else {

                    # If this object is already a selected object, unselect it
                    if (! $self->unselectObj($modelObj, 'exit_tag')) {

                        # The exit tag wasn't already selected, so select it
                        $self->setSelectedObj(
                            [$modelObj, 'exit_tag'],
                            # Retain other selected objects if CTRL key held down
                            $ctrlFlag,
                        );
                    }
                }

            # Process other kinds of left-click
            } else {

                if ($objType eq 'room' && ! $shiftFlag && ! $ctrlFlag) {

                    # Single click detected; it might be the start of a double-click
                    $self->ivPoke('leftClickTime', $clickTime);
                    $self->ivPoke('leftClickObj', $modelObj);
                }

                # If a group of things are already selected, unselect them all and select the object
                #   that was clicked
                if (! $ctrlFlag && $selectFlag) {

                    # Select this room/label, unselecting all other objects
                    $self->setSelectedObj(
                        [$modelObj, $objType],
                        # Retain other selected objects if CTRL key held down
                        $ctrlFlag,
                    );

                } else {

                    # If this object is already a selected object, unselect it
                    if (! $self->unselectObj($modelObj)) {

                        # The room or label wasn't already selected, so select it
                        $self->setSelectedObj(
                            [$modelObj, $objType],
                            # Retain other selected objects if CTRL key held down
                            $ctrlFlag,
                        );
                    }
                }
            }

        # Process right-clicks
        } elsif ($clickType eq 'single' && $button eq 'right') {

            if ($objType eq 'exit') {

                # For twin exits - which share a canvas object - use the exit whose parent room is
                #   closest to the click
                $modelObj = $self->chooseClickedExit($modelObj, int($event->x), int($event->y));
            }

            # If a group of things are already selected, unselect them all and select the object
            #   that was clicked
            if ($selectFlag) {

                # Select this room/label, unselecting all other objects
                $self->setSelectedObj(
                    [$modelObj, $objType],
                    # Retain other selected objects if CTRL key held down
                    $ctrlFlag,
                );

            } else {

                # If this object isn't already selected, select it (but don't unselect something
                #   as we would for a left-click)
                if ($objType eq 'room_tag') {

                    $self->setSelectedObj(
                        [$modelObj, 'room_tag'],
                        FALSE,          # Select this object; unselect all other objects
                    );

                } elsif ($objType eq 'room_guild') {

                    $self->setSelectedObj(
                        [$modelObj, 'room_guild'],
                        FALSE,          # Select this object; unselect all other objects
                    );

                } elsif ($objType eq 'exit_tag') {

                    $self->setSelectedObj(
                        [$modelObj, 'exit_tag'],
                        FALSE,          # Select this object; unselect all other objects
                    );

                } else {

                    $self->setSelectedObj(
                        [$modelObj, $objType],
                        FALSE,          # Select this object; unselect all other objects
                    );
                }
            }

            # Create the popup menu
            if ($objType eq 'room' && $self->selectedRoom) {
                $popupMenu = $self->enableRoomsPopupMenu();
            } elsif ($objType eq 'room_tag' && $self->selectedRoomTag) {
                $popupMenu = $self->enableRoomTagsPopupMenu();
            } elsif ($objType eq 'room_guild' && $self->selectedRoomGuild) {
                $popupMenu = $self->enableRoomGuildsPopupMenu();
            } elsif ($objType eq 'exit_tag' && $self->selectedExitTag) {
                $popupMenu = $self->enableExitTagsPopupMenu();
            } elsif ($objType eq 'exit' && $self->selectedExit) {

                # Store the position of the right-click, in case the user wants to add a bend from
                #   the popup menu
                $self->ivPoke('exitClickXPosn', int($event->x));
                $self->ivPoke('exitClickYPosn', int($event->y));
                # Now we can open the poup menu
                $popupMenu = $self->enableExitsPopupMenu();

            } elsif ($objType eq 'label' && $self->selectedLabel) {

                $popupMenu = $self->enableLabelsPopupMenu();
            }

            if ($popupMenu) {

                $popupMenu->popup(undef, undef, undef, undef, $event->button, $event->time);
            }
        }

        return 1;
    }

    sub deleteCanvasObj {

        # Called by numerous functions
        #
        # When a region object, room object, room tag, room guild, exit, exit tag or label is being
        #   drawn, redrawn or deleted from the world model, this function must be called
        # The function checks whether the model object is currently drawn on a map as one or more
        #   canvas objects and, if it is, destroys the canvas objects
        #
        # This function also handles coloured blocks and rectangles on the background map, details
        #   of which are stored in the regionmap object (GA::Obj::Regionmap), not the world model
        # The function checks whether a canvas object for the coloured block/rectangle is currently
        #   displayed on the map as a canvas object and, if so, destroys the canvas object
        #
        # Expected arguments
        #   $type       - Set to 'region', 'room', 'room_tag', 'room_guild', 'exit', 'exit_tag' or
        #                   'label' for world model objects, 'checked_dir' for checked directions
        #                   and 'square', 'rect' for coloured blocks/rectangles
        #   $modelObj   - The GA::ModelObj::Region, GA::ModelObj::Room, GA::Obj::Exit or
        #                   GA::Obj::MapLabel being drawn /redrawn / deleted
        #               - For checked directions, the GA::ModelObj::Room in which the checked
        #                   direction is stored
        #               - For coloured squares, it's not a blessed reference, but a coordinate in
        #                   the form 'x_y' (to delete canvas objects on all levels), or 'x_y_z' (to
        #                   delete the canvas object on one level)
        #               - For coloured rectangles, it's not a blessed reference, but a key in the
        #                   form 'object-number' (to delete canvas objects on all levels), or
        #                   'object-number_level' (to delete the canvas object on one level)
        #
        # Optional arguments
        #   $regionmapObj, $parchmentObj
        #               - The regionmap and parchment object for $modelObj. If not set, this
        #                   function fetches them. Both must be specified if $type is 'square' or
        #                   'rect')
        #   $deleteFlag - Set to TRUE if the object is being deleted from the world model, FALSE
        #                   (or 'undef') if not. Never TRUE for coloured blocks/rectangles which
        #                   are not stored in the world model
        #
        # Return values
        #   'undef' on improper arguments, if there's an error or if there are no canvas objects to
        #       destroy
        #   1 otherwise

        my ($self, $type, $modelObj, $regionmapObj, $parchmentObj, $deleteFlag, $check) = @_;

        # Local variables
        my (
            $roomObj,
            @redrawList,
            %redrawHash,
        );

        # Check for improper arguments
        if (! defined $type || ! defined $modelObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->deleteCanvasObj', @_);
        }

        # Fetch the regionmap and parchment object, if not specified
        if (! $regionmapObj) {

            if ($type eq 'region') {

                $regionmapObj = $self->worldModelObj->ivShow('regionmapHash', $modelObj->name);

            } elsif (
                $type eq 'room' || $type eq 'room_tag' || $type eq 'room_guild'
                || $type eq 'checked_dir'
            ) {
                $regionmapObj = $self->findRegionmap($modelObj->parent);

            } elsif ($type eq 'exit' || $type eq 'exit_tag') {

                $roomObj = $self->worldModelObj->ivShow('modelHash', $modelObj->parent);
                $regionmapObj = $self->findRegionmap($roomObj->parent);

            } elsif ($type eq 'label') {

                $regionmapObj = $self->worldModelObj->ivShow('regionmapHash', $modelObj->region);

            } else {

                # $type is 'square' or 'rect', for which $regionmapObj should have been be specified
                return undef;
            }
        }

        if ($regionmapObj && ! $parchmentObj) {

            $parchmentObj = $self->ivShow('parchmentHash', $regionmapObj->name);
        }

        if (! $parchmentObj) {

            # No parchment object for this region exists, so there are no canvas objects to destroy
            return undef;
        }

        # Handle a region deletion
        if ($type eq 'region' && $deleteFlag) {

            # Reset the treeview, so that the deleted region is no longer visible in it
            $self->resetTreeView();

            if ($self->currentRegionmap && $self->currentRegionmap eq $regionmapObj) {

                # The currently displayed region was the one deleted. Draw an empty map
                $self->resetMap();

            } else {

                # Redraw all rooms containing region exits (which automatically redraws the exits)
                foreach my $otherRegionmap ($self->worldModelObj->ivValues('regionmapHash')) {

                    # The same room can have more than one region exit; add affected rooms to a hash
                    #   to eliminate duplicates
                    foreach my $number ($otherRegionmap->ivKeys('regionExitHash')) {

                        my $exitObj = $self->worldModelObj->ivShow('exitModelHash', $number);
                        if ($exitObj) {

                            $redrawHash{$exitObj->parent} = undef;
                        }
                    }
                }

                # Having eliminated duplicates, compile the list of rooms to redraw
                foreach my $number (keys %redrawHash) {

                    my $thisRoomObj = $self->worldModelObj->ivShow('modelHash', $number);
                    if ($thisRoomObj) {

                           push (@redrawList, 'room', $thisRoomObj);
                    }
                }

                # Redraw the affected rooms
                $self->markObjs(@redrawList);
                $self->doDraw();

                # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
                $self->restrictWidgets();
            }

        # Handle a room draw/redraw/deletion
        } elsif ($type eq 'room') {

            if ($deleteFlag) {

                # Unselect the room, if selected
                $self->unselectObj(
                    $modelObj,
                    undef,              # A room, not a room tag or room guild
                    TRUE,               # No re-draw
                );

                # Also unselect the room tag and/or room guild, if either is selected
                if (defined $modelObj->roomTag) {

                    $self->unselectObj(
                        $modelObj,
                        'room_tag',
                        TRUE,           # No re-draw
                    );
                }

                if (defined $modelObj->roomGuild) {

                    $self->unselectObj(
                        $modelObj,
                        'room_guild',
                        TRUE,           # No re-draw
                    );
                }
            }

            if (! defined $modelObj->xPosBlocks) {

                # No canvas objects to destroy
                return undef;
            }

            # (The TRUE argument means to destroy canvas objects for the room, and also for any
            #   room echoes/room tags/room guilds/room text/checked directions)
            $parchmentObj->deleteDrawnRoom($modelObj, TRUE);

        # Handle a room tag deletion
        } elsif ($type eq 'room_tag') {

            if ($deleteFlag) {

                # Unselect the room tag, if selected
                $self->unselectObj(
                    $modelObj,
                    'room_tag',
                    TRUE,           # No re-draw
                );
            }

            if (! defined $modelObj->xPosBlocks) {

                # No canvas objects to destroy
                return undef;
            }

            $parchmentObj->deleteDrawnRoomTag($modelObj);

        # Handle a room guild deletion
        } elsif ($type eq 'room_guild') {

            if ($deleteFlag) {

                # Unselect the room guild, if selected
                $self->unselectObj(
                    $modelObj,
                    'room_guild',
                    TRUE,           # No re-draw
                );
            }

            if (! defined $modelObj->xPosBlocks) {

                # No canvas objects to destroy
                return undef;
            }

            $parchmentObj->deleteDrawnRoomGuild($modelObj);

        # Handle an exit deletion
        } elsif ($type eq 'exit') {

            # Unselect the exit, if selected
            if ($deleteFlag) {

                $self->unselectObj(
                    $modelObj,
                    undef,              # An exit, not an exit tag
                    TRUE,               # No re-draw
                );

                # Also unselect the exit tag, if it is selected
                if (defined $modelObj->exitTag) {

                    $self->unselectObj(
                        $modelObj,
                        'exit_tag',
                        TRUE,           # No re-draw
                    );
                }
            }

            # (The TRUE argument means to destroy canvas objects for the exit, and also for any
            #   exit tags/ornaments)
            $parchmentObj->deleteDrawnExit($modelObj, undef, TRUE);

        # Handle an exit tag deletion
        } elsif ($type eq 'exit_tag') {

            if ($deleteFlag) {

                # Unselect the exit tag, if selected
                $self->unselectObj(
                    $modelObj,
                    'exit_tag',
                    TRUE,               # No re-draw
                );
            }

            $parchmentObj->deleteDrawnExitTag($modelObj);

        # Handle a label deletion
        } elsif ($type eq 'label') {

            if ($deleteFlag) {

                # Unselect the label, if selected
                $self->unselectObj(
                    $modelObj,
                    undef,              # A label, not a room tag or room guild
                    TRUE,               # No re-draw
                );
            }

            $parchmentObj->deleteDrawnLabel($modelObj);

        # Handle a checked direction deletion
        } elsif ($type eq 'checked_dir') {

            # (Checked directions can't be selected)

            if (! defined $modelObj->xPosBlocks) {

                # No canvas objects to destroy
                return undef;
            }

            $parchmentObj->deleteDrawnCheckedDir($modelObj);

        # Handle a coloured block deletion
        } elsif ($type eq 'square') {

            # (Coloured squares can't be selected)

            # For coloured squares, $modelObj is not a blessed reference, but a coordinate in the
            #   form 'x_y' (to delete canvas objects on all levels), or 'x_y_z' (to delete the
            #   canvas object on one level)
            $parchmentObj->deleteColouredSquare($modelObj)

        # Handle a coloured rectangle deletion
        } elsif ($type eq 'rect') {

            # (Coloured rectangles can't be selected)

            # For coloured rectangles, $modelObj is not a blessed reference, but a key in the form
            #   'object-number' (to delete canvas objects on all levels), or 'object-number_level'
            #   (to delete the canvas object on one level)
            $parchmentObj->deleteColouredRect($modelObj)

        } else {

            # Unrecognised object type
            return undef;
        }

        return 1;
    }

    sub startDrag {

        # Called by $self->setupCanvasObjEvent and ->canvasEventHandler at the start of a drag
        #   operation
        # Grabs the clicked canvas object and sets up IVs
        #
        # Expected arguments
        #   $type           - What type of canvas object this is - 'room', 'room_tag', 'room_guild',
        #                       'exit', 'exit_tag' or 'label'
        #   $canvasObj      - The canvas object on which the user clicked
        #   $modelObj       - The GA::ModelObj::Room, GA::Obj::Exit or GA::Obj::MapLabel which
        #                       corresponds to the canvas object $canvasObj
        #   $event          - The mouse click event (a Gtk::Gdk::Event)
        #   $xPos, $yPos    - The coordinates of the click on the canvas
        #
        # Return values
        #   'undef' on improper arguments or if a dragging operation has already started
        #   1 otherwise

        my ($self, $type, $canvasObj, $modelObj, $event, $xPos, $yPos, $check) = @_;

        # Local variables
        my (
            $exitMode, $obscuredFlag, $ornamentsFlag,
            @canvasObjList, @fakeRoomList,
        );

        # Check for improper arguments
        if (
            ! defined $type || ! defined $canvasObj || ! defined $modelObj || ! defined $event
            || ! defined $xPos || ! defined $yPos || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->startDrag', @_);
        }

        # Double-clicking on a canvas object can cause this function to be called twice; the second
        #   time, don't do anything
        if ($self->dragFlag) {

            return undef;
        }

        # If the tooltips are visible, hide them
        $self->hideTooltips();

        # Gtk3 can return fractional values for $xPos, $yPos. We definitely want only integers
        $xPos = int($xPos);
        $yPos = int($yPos);

        # For dragged rooms/exits, we need an $exitMode value the same as would be used during a
        #   draw cycle
        if ($self->worldModelObj->drawExitMode eq 'ask_regionmap') {
            $exitMode = $self->currentRegionmap->drawExitMode;
        } else {
            $exitMode = $self->worldModelObj->drawExitMode;
        }

        # We also need values for $obscuredFlag and $ornamentsFlag, the same as would be used during
        #   a draw cycle
        if ($self->worldModelObj->drawExitMode eq 'ask_regionmap') {
            $obscuredFlag = $self->currentRegionmap->obscuredExitFlag;
        } else {
            $obscuredFlag = $self->worldModelObj->obscuredExitFlag;
        }

        if ($self->worldModelObj->drawExitMode eq 'ask_regionmap') {
            $ornamentsFlag = $self->currentRegionmap->drawOrnamentsFlag;
        } else {
            $ornamentsFlag = $self->worldModelObj->drawOrnamentsFlag;
        }

        if ($type eq 'room') {

            # If the room(s) have been drawn with an emphasised border, we need to re-draw them
            #   without emphasis - otherwise the extra square will be left behind on the canvas
            #   while the room is being dragged around

            # Check that the room is selected. If not, we need to select it (which unselects any
            #   other selected rooms/labels)
            if (
                ($self->selectedRoom && $self->selectedRoom ne $modelObj)
                || (
                    $self->selectedRoomHash
                    && ! $self->ivExists('selectedRoomHash', $modelObj->number)
                )
                || (! $self->selectedRoom && ! $self->selectedRoomHash)
            ) {
                $self->setSelectedObj([$modelObj, 'room']);
            }

            # Temporarily set a few drawing cycle IVs, which allows the drawing functions to work as
            #   if we were in a drawing cycle (i.e. a call to $self->doDraw). They are reset by
            #   $self->stopDrag
            $self->ivPoke('drawRegionmap', $self->currentRegionmap);
            $self->ivPoke('drawParchment', $self->currentParchment);
            $self->ivPoke(
                'drawScheme',
                $self->worldModelObj->getRegionScheme($self->currentRegionmap),
            );

            $self->prepareDraw($exitMode);

            # If multiple rooms/labels are selected, they are all dragged alongside $canvasObj (as
            #   long as they're in the same region as $roomObj)
            foreach my $roomObj ($self->compileSelectedRooms) {

                my ($listRef, $thisCanvasObj, $fakeRoomObj);

                # If the rooms are in the same region...
                if ($roomObj->parent == $modelObj->parent) {

                    # Redraw the room without extra markings like interior text or an emphasised
                    #   border
                    # (NB Calling $self->drawRoom here, instead of calling ->markObjs then ->doDraw
                    #   as usual, is allowed. Because of the TRUE argument, ->drawRoom is expecting
                    #   a call from this function)
                    $self->drawRoom($roomObj, $exitMode, $obscuredFlag, $ornamentsFlag, TRUE);

                    $listRef = $self->currentParchment->getDrawnRoom($roomObj);
                    $thisCanvasObj = $$listRef[0];
                    push (@canvasObjList, $thisCanvasObj);

                    if ($roomObj eq $modelObj) {

                        # This canvas object replaces the one that is actually being dragged/grabbed
                        $canvasObj = $thisCanvasObj;
                    }

                    # Draw a fake room at the same position, so that $modelObj's exits don't look
                    #   odd
                    # (NB $self->drawFakeRoomBox is only called by this function, never by ->doDraw)
                    $fakeRoomObj = $self->drawFakeRoomBox($roomObj);
                    if ($fakeRoomObj) {

                        push (@fakeRoomList, $fakeRoomObj);

                        # Don't let the fake room object be obscured by room echos
                        $fakeRoomObj->raise();
                    }

                    # Raise the canvas object above others so that, while we're dragging it around,
                    #   it doesn't disappear under exits (and so on)
                    $thisCanvasObj->raise();
                }
            }

            foreach my $labelObj ($self->compileSelectedLabels) {

                my ($listRef, $thisCanvasObj, $thisCanvasObj2);

                # Drag both the label and its box (if it has one), as long as it's in the same
                #   region
                if ($labelObj->region eq $self->currentRegionmap->name) {

                    $listRef = $self->currentParchment->getDrawnLabel($labelObj);
                    ($thisCanvasObj, $thisCanvasObj2) = @$listRef;

                    if ($thisCanvasObj2) {

                        push (@canvasObjList, $thisCanvasObj2);
                        $thisCanvasObj2->raise();
                    }

                    push (@canvasObjList, $thisCanvasObj);
                    $thisCanvasObj->raise();
                }
            }

        } elsif ($type eq 'exit') {

            my ($twinExitObj, $bendNum, $bendIndex, $twinBendNum, $twinBendIndex);

            if ($modelObj->twinExit) {

                $twinExitObj = $self->worldModelObj->ivShow('exitModelHash', $modelObj->twinExit);
            }

            # Temporarily set a few drawing cycle IVs, which allows the drawing functions to work as
            #   if we were in a drawing cycle (i.e. a call to $self->doDraw). They are reset by
            #   $self->stopDrag
            $self->ivPoke('drawRegionmap', $self->currentRegionmap);
            $self->ivPoke('drawParchment', $self->currentParchment);
            $self->ivPoke(
                'drawScheme',
                $self->worldModelObj->getRegionScheme($self->currentRegionmap),
            );

            $self->prepareDraw($exitMode);

            # See if the click was near a bend
            $bendNum = $self->findExitBend($modelObj, $xPos, $yPos);
            if (defined $bendNum) {

                # Set IVs to monitor the bend's position, relative to its position right now
                $self->ivPoke('dragBendNum', $bendNum);
                # (The first bend, $bendNum = 0, occupies the first two items in ->bendOffsetList)
                $bendIndex = $bendNum * 2;

                $self->ivPoke('dragBendInitXPos', $modelObj->ivIndex('bendOffsetList', $bendIndex));
                $self->ivPoke(
                    'dragBendInitYPos',
                    $modelObj->ivIndex('bendOffsetList', ($bendIndex + 1)),
                );

                $self->ivPoke('dragExitDrawMode', $exitMode);
                $self->ivPoke('dragExitOrnamentsFlag', $ornamentsFlag);

                # If there's a twin exit, set IVs for the corresponding bend in the twin
                if ($twinExitObj) {

                    $twinBendNum = ((scalar $twinExitObj->bendOffsetList / 2) - $bendNum - 1);
                    $self->ivPoke('dragBendTwinNum', $twinBendNum);
                    # (The 2nd bend, $bendNum = 1, occupies the 2nd two items in ->bendOffsetList)
                    $twinBendIndex = $twinBendNum * 2;
                    $self->ivPoke(
                        'dragBendTwinInitXPos',
                        $twinExitObj->ivIndex('bendOffsetList', $twinBendIndex),
                    );

                    $self->ivPoke(
                        'dragBendTwinInitYPos',
                        $twinExitObj->ivIndex('bendOffsetList', ($twinBendIndex + 1)),
                    );
                }

            } else {

                # Destroy the existing canvas object
                $self->deleteCanvasObj(
                    'exit',
                    $modelObj,
                    $self->currentRegionmap,
                    $self->currentParchment,
                );

                # The canvas objects for the exit may have been drawn associated with $exitObj, or
                #   with its twin exit (if any); make sure those canvas objects are destroyed, too
                #   (except for normal broken exits, region exits, impassable exits and mystery
                #   exits)
                if (
                    $twinExitObj
                    && (
                        (! $twinExitObj->brokenFlag || $twinExitObj->bentFlag)
                        && ! $twinExitObj->regionFlag
                        && $twinExitObj->exitOrnament ne 'impass'
                        && $twinExitObj->exitOrnament ne 'mystery'
                    )
                ) {
                    $self->deleteCanvasObj(
                        'exit',
                        $twinExitObj,
                        $self->currentRegionmap,
                        $self->currentParchment,
                    );
                }

                # Draw a draggable exit, starting from $exitObj's normal start position, and ending
                #   at the position of the mouse click
                $canvasObj = $self->drawDraggableExit($modelObj, $xPos, $yPos);
            }

            push (@canvasObjList, $canvasObj);

        } elsif ($type eq 'label') {

            my $listRef;

            # If the label has a box, the user might have clicked on either the label or the box.
            #   In either case, both objects need to be dragged
            $listRef = $self->currentParchment->getDrawnLabel($modelObj);

            foreach my $thisCanvasObj (reverse @$listRef) {

                push (@canvasObjList, $thisCanvasObj);
                $thisCanvasObj->raise();         # Text last, raised above box
            }

            # The canvas object to grab is always the label text, not the box (as $self->stopDrag
            #   works out the distance using the label text)
            $canvasObj = $$listRef[0];

        } else {

            # For room tags, room guilds and exit tags, just raise the canvas object above others
            push (@canvasObjList, $canvasObj);
            $canvasObj->raise();
        }

        # Grab the dragged canvas object
        $canvasObj->get_canvas->pointer_grab(
            $canvasObj,
            [qw/pointer-motion-mask button-release-mask/],
            Gtk3::Gdk::Cursor->new('fleur'),
            $event->time,
        );

        # Mark the drag as started, and update IVs (the IVs for bent exits have already been set)
        $self->ivPoke('dragFlag', TRUE);
        $self->ivPoke('dragCanvasObj', $canvasObj);
        $self->ivPoke('dragCanvasObjList', @canvasObjList);
        $self->ivPoke('dragModelObj', $modelObj);
        $self->ivPoke('dragModelObjType', $type);
        $self->ivPoke('dragInitXPos', $xPos);
        $self->ivPoke('dragInitYPos', $yPos);
        $self->ivPoke('dragCurrentXPos', $xPos);
        $self->ivPoke('dragCurrentYPos', $yPos);
        $self->ivPoke('dragFakeRoomList', @fakeRoomList);

        return 1;
    }

    sub continueDrag {

        # Called by $self->setupCanvasObjEvent in the middle of a drag operation
        # Redraws canvas object(s) on the canvas and updates IVs
        #
        # Expected arguments
        #   $event          - The mouse click event (a Gtk::Gdk::Event)
        #   $xPos, $yPos    - The coordinates of the mouse above the canvas
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $event, $xPos, $yPos, $check) = @_;

        # Local variables
        my ($moveX, $moveY, $twinExitObj, $listRef, $canvasObj);

        # Check for improper arguments
        if (! defined $event || ! defined $xPos || ! defined $yPos || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->continueDrag', @_);
        }

        # Don't do anything if an earlier call to this function hasn't been completed (happens quite
        #   a lot if the user is dragging objects around rapidly, and it messes up the correct value
        #   of $self->dragCurrentXPos and ->dragCurrentYPos)
        if ($self->dragContinueFlag) {

            return undef;

        } else {

            $self->ivPoke('dragContinueFlag', TRUE);
        }

        # Gtk3 can return fractional values for $xPos, $yPos. We definitely want only integers
        $xPos = int($xPos);
        $yPos = int($yPos);

        # For everything except exits, move the canvas object(s)
        if ($self->dragModelObjType ne 'exit') {

            $moveX = $xPos - $self->dragCurrentXPos;
            $moveY = $yPos - $self->dragCurrentYPos;

            foreach my $canvasObj ($self->dragCanvasObjList) {

                $canvasObj->translate($moveX, $moveY);
            }

        } else {

            # Ungrab the exit's canvas object
            $self->dragCanvasObj->get_canvas->pointer_ungrab($self->dragCanvasObj, $event->time);

            # If dragging an exit bend...
            if (defined $self->dragBendNum) {

                # Update the exit's list of bend positions
                $self->worldModelObj->adjustExitBend(
                    $self->dragModelObj,
                    $self->dragBendNum,
                    $self->dragBendInitXPos + ($xPos - $self->dragInitXPos),
                    $self->dragBendInitYPos + ($yPos - $self->dragInitYPos),
                );

                # Adjust the corresponding bend in the twin exit, if there is one
                if (defined $self->dragBendTwinNum) {

                    $twinExitObj = $self->worldModelObj->ivShow(
                        'exitModelHash',
                        $self->dragModelObj->twinExit,
                    );

                    $self->worldModelObj->adjustExitBend(
                        $twinExitObj,
                        $self->dragBendTwinNum,
                        $self->dragBendTwinInitXPos + ($xPos - $self->dragInitXPos),
                        $self->dragBendTwinInitYPos + ($yPos - $self->dragInitYPos),
                    );
                }

                # Destroy the bending exit's existing canvas objects
                $self->deleteCanvasObj(
                    'exit',
                    $self->dragModelObj,
                    $self->currentRegionmap,
                    $self->currentParchment,
                );

                # Redraw the bending exit, with the dragged bend in its new position
                $self->drawBentExit(
                    $self->worldModelObj->ivShow('modelHash', $self->dragModelObj->parent),
                    $self->dragModelObj,
                    $self->dragExitDrawMode,
                    $self->dragExitOrnamentsFlag,
                    $twinExitObj,
                );

                if ($twinExitObj) {

                    $self->deleteCanvasObj(
                        'exit',
                        $twinExitObj,
                        $self->currentRegionmap,
                        $self->currentParchment,
                    );
                }

                # Get the new canvas object to grab. Since the bending exit consists of several
                #   canvas objects, use the first one
                $listRef = $self->currentParchment->getDrawnExit($self->dragModelObj);
                $canvasObj = $$listRef[0];

            # If dragging a draggable exit...
            } else {

                # Destroy the old canvas object
                $self->dragCanvasObj->remove();

                # Replace it with a new one draggable exit at the current mouse position
                $canvasObj = $self->drawDraggableExit($self->dragModelObj, $xPos, $yPos);
            }

            # Grab the new canvas object
            $canvasObj->get_canvas->pointer_grab(
                $canvasObj,
                [qw/pointer-motion-mask button-release-mask/],
                Gtk3::Gdk::Cursor->new('fleur'),
                $event->time,
            );

            $self->ivPoke('dragCanvasObj', $canvasObj);
        }

        # Update IVs
        $self->ivPoke('dragCurrentXPos', $xPos);
        $self->ivPoke('dragCurrentYPos', $yPos);
        $self->ivPoke('dragContinueFlag', FALSE);

        return 1;
    }

    sub stopDrag {

        # Called by $self->setupCanvasObjEvent at the end of a drag operation
        #
        # Expected arguments
        #   $event          - The mouse click event (a Gtk::Gdk::Event)
        #   $xPos, $yPos    - The coordinates of the click on the canvas
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $event, $xPos, $yPos, $check) = @_;

        # Local variables
        my @drawList;

        # Check for improper arguments
        if (! defined $event || ! defined $xPos || ! defined $yPos || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stopDrag', @_);
        }

        # Gtk3 can return fractional values for $xPos, $yPos. We definitely want only integers
        $xPos = int($xPos);
        $yPos = int($yPos);

        # Ungrab the grabbed canvas object
        $self->dragCanvasObj->get_canvas->pointer_ungrab($self->dragCanvasObj, $event->time);

        # Mark the drag operation as finished
        $self->ivPoke('dragFlag', FALSE);

        # If rooms have been dragged, there may be some entries in ->drawCycleExitHash. Empty it,
        #   allowing the exits in all affected rooms to be redrawn
        $self->ivEmpty('drawCycleExitHash');

        # Respond to the end of the drag operation
        if ($self->dragModelObjType eq 'room') {

            my (
                $newXPos, $newYPos, $adjustXPos, $adjustYPos, $occupyRoomNum, $occupyRoomObj,
                $failFlag,
                %roomHash, %labelHash,
            );

            # Destroy any fake room canvas objects
            foreach my $fakeRoomObj ($self->dragFakeRoomList) {

                $fakeRoomObj->remove();
            }

            # Destroy the moving room/label canvas objects, as we're not redrawing the whole region
            foreach my $moveRoomObj ($self->dragCanvasObjList) {

                $moveRoomObj->remove();
            }

            # Calculate the grid coordinates of the dragged room's new gridblock
            $newXPos = int ($self->dragCurrentXPos / $self->currentRegionmap->blockWidthPixels);
            $newYPos = int ($self->dragCurrentYPos / $self->currentRegionmap->blockHeightPixels);
            # Calculate the distance travelled (in blocks)
            $adjustXPos = $newXPos
                - int ($self->dragInitXPos / $self->currentRegionmap->blockWidthPixels);
            $adjustYPos = $newYPos
                - int ($self->dragInitYPos / $self->currentRegionmap->blockHeightPixels);
            # Fetch the room occupying that gridblock (if any)
            $occupyRoomNum = $self->currentRegionmap->fetchRoom(
                $newXPos,
                $newYPos,
                $self->currentRegionmap->currentLevel,
            );

            if ($occupyRoomNum) {

                $occupyRoomObj = $self->worldModelObj->ivShow('modelHash', $occupyRoomNum);
            }

            # If the grabbed room hasn't been dragged to a new gridblock, don't do anything
            if (! $adjustXPos && ! $adjustYPos) {

                # Just redraw the dragged room(s)/label(s) at their original position
                foreach my $roomObj ($self->compileSelectedRooms) {

                    push (@drawList, 'room', $roomObj);
                }

                foreach my $labelObj ($self->compileSelectedLabels) {

                    push (@drawList, 'label', $labelObj);
                }

                $self->markObjs(@drawList);
                $self->doDraw();

            # If the dragged room is the current room, and the automapper object is set up to merge
            #   rooms, and one of its matching rooms is the one occupying the dragged room's new
            #   gridblock
            } elsif (
                $self->mapObj->currentRoom
                && $self->mapObj->currentRoom eq $self->dragModelObj
                && $occupyRoomObj
                && defined $self->mapObj->ivFind('currentMatchList', $occupyRoomObj)
            ) {
                # The TRUE argument means 'don't prompt for confirmation', as the map looks very
                #   odd at the moment with only room boxes, exists, room interior info and so on
                #   visible
                if (! $self->doMerge($self->dragModelObj, $occupyRoomObj, TRUE)) {

                    # Merge/move operation failed, so redraw the selected rooms/labels at their
                    #   original positions
                    foreach my $roomObj ($self->compileSelectedRooms) {

                        push (@drawList, 'room', $roomObj);
                    }

                    foreach my $labelObj ($self->compileSelectedLabels) {

                        push (@drawList, 'label', $labelObj);
                    }

                    $self->markObjs(@drawList);
                    $self->doDraw();
                }

            # If a single room and no labels are selected...
            } elsif ($self->selectedRoom && ! $self->selectedLabel && ! $self->selectedLabelHash) {

                # Check that the new gridblock isn't occupied
                if ($occupyRoomNum) {

                    # The room has been dragged to an occupied gridblock. Don't move the room, just
                    #   redraw it at its original position
                    $self->markObjs('room', $self->dragModelObj);
                    $self->doDraw();

                } else {

                    # Move the (selected) room to its new location
                    $self->moveSelectedObjs(
                        ($newXPos - $self->dragModelObj->xPosBlocks),
                        ($newYPos - $self->dragModelObj->yPosBlocks),
                        0,      # Room doesn't change level
                    );

                    # Select the room
                    $self->setSelectedObj(
                        [$self->dragModelObj, 'room'],
                        FALSE,                      # Select this object; unselect all other objects
                    );
                }

            # Multiple rooms and/or labels are selected; move all of them the same distance
            } else {

                # Start by creating combined hashes, merging two IVs into one hash
                %roomHash = $self->selectedRoomHash;
                if ($self->selectedRoom) {

                    $roomHash{$self->seletedRoom->number} = $self->selectedRoom;
                }

                %labelHash = $self->selectedLabelHash;
                if ($self->selectedLabel) {

                    $labelHash{$self->selectedLabel->id} = $self->selectedLabel;
                }

                # Check every room to make sure it's been dragged to a new, unoccupied gridblock
                OUTER: foreach my $roomObj (values %roomHash) {

                    my $existRoomNum = $self->currentRegionmap->fetchRoom(
                        $roomObj->xPosBlocks + $adjustXPos,
                        $roomObj->yPosBlocks + $adjustYPos,
                        $self->currentRegionmap->currentLevel,
                    );

                    if (defined $existRoomNum && ! exists $roomHash{$existRoomNum}) {

                        # At least one of the gridblocks is occupied by a room that's not going
                        #   to be moved along with all the others
                        $failFlag = TRUE;
                        last OUTER;
                    }
                }

                if ($failFlag) {

                    # One or more of the selected rooms have been dragged to a gridblock occupied by
                    #   a room that isn't one of those being moved
                    # Don't move anything, just redraw the selected rooms/labels at their original
                    #   positions
                    foreach my $roomObj ($self->compileSelectedRooms) {

                        push (@drawList, 'room', $roomObj);
                    }

                    foreach my $labelObj ($self->compileSelectedLabels) {

                        push (@drawList, 'label', $labelObj);
                    }

                    $self->markObjs(@drawList);
                    $self->doDraw();

                } else {

                    # Move all selected rooms to their new locations
                    $self->worldModelObj->moveRoomsLabels(
                        $self->session,
                        TRUE,                       # Update Automapper windows now
                        $self->currentRegionmap,    # Move from this region...
                        $self->currentRegionmap,    # ...to this one...
                        $adjustXPos,                # ...using this vector
                        $adjustYPos,
                        0,                          # No vertical displacement
                        \%roomHash,
                        \%labelHash,
                    );
                }
            }

        } elsif ($self->dragModelObjType eq 'exit') {

            my $destRoomObj;

            # Don't need to do anything at the end of a drag operation, if we're dragging an exit
            #   bend
            if (! defined $self->dragBendNum) {

                # Destroy the draggable exit
                $self->dragCanvasObj->remove();

                # Work out whether the end of the draggable exit (the coordinates are $xPos, $yPos)
                #   was over a room that wasn't the exit's parent room or its existing destination
                #   room
                $destRoomObj = $self->findMouseOverRoom($xPos, $yPos, $self->dragModelObj);
                if (! $destRoomObj) {

                    # No connection to make. Redraw the original exit, at its original size and
                    #   position
                    $self->markObjs('exit', $self->dragModelObj);
                    $self->doDraw();

                } else {

                    # Connect the exit to the room (it's in the same region as the exit's parent
                    #   room, so it's potentially a broken exit, and definitely not a region exit)
                    $self->connectExitToRoom($destRoomObj, 'broken', $self->dragModelObj);
                }
            }

        } else {

            my ($newXPos, $newYPos);

            # Work out the difference between the new position and the original position
            $newXPos = int ($self->dragCurrentXPos - $self->dragInitXPos);
            $newYPos = int ($self->dragCurrentYPos - $self->dragInitYPos);

            # Move the object and instruct the world model to update its Automapper windows
            $self->worldModelObj->moveOtherObjs(
                TRUE,       # Update Automapper windows immediately
                $self->dragModelObjType,
                $self->dragModelObj,
                $newXPos,
                $newYPos,
            );
        }

        # Reset other IVs (->dragFlag was set above)
        $self->ivUndef('dragCanvasObj');
        $self->ivEmpty('dragCanvasObjList');
        $self->ivUndef('dragModelObj');
        $self->ivUndef('dragModelObjType');
        $self->ivUndef('dragInitXPos');
        $self->ivUndef('dragInitYPos');
        $self->ivUndef('dragCurrentXPos');
        $self->ivUndef('dragCurrentYPos');
        $self->ivEmpty('dragFakeRoomList');
        $self->ivUndef('dragBendNum');
        $self->ivUndef('dragBendInitXPos');
        $self->ivUndef('dragBendInitYPos');
        $self->ivUndef('dragBendTwinNum');
        $self->ivUndef('dragBendTwinInitXPos');
        $self->ivUndef('dragBendTwinInitYPos');
        $self->ivUndef('dragExitDrawMode');
        $self->ivUndef('dragExitOrnamentsFlag');

        # Also reset the drawing cycle IVs set by $self->startDrag
        $self->tidyUpDraw();

        return 1;
    }

    sub startSelectBox {

        # Called by $self->canvasEventHandler
        # When the user holds down their left mouse button on an empty area of the map, and then
        #   moves their mouse, we draw a selection box. When the user releases the mouse button,
        #   all rooms and labels inside the box are selected
        # When this function is called, the user has merely left-clicked the empty area of the map.
        #   This function initialises IVs. We don't actually draw the selection box until the user
        #   moves their mouse while holding down the left mouse button
        #
        # Expected arguments
        #   $xPos, $yPos    - The coordinates of the click on the canvas
        #
        # Return values
        #   'undef' on improper arguments or if a selection box operation has already started
        #   1 otherwise

        my ($self, $xPos, $yPos, $check) = @_;

        # Check for improper arguments
        if (! defined $xPos || ! defined $yPos || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->startSelectBox', @_);
        }

        # Double-clicking on a canvas object can cause this function to be called twice; the second
        #   time, don't do anything
        if ($self->selectBoxFlag) {

            return undef;
        }

        # If the tooltips are visible, hide them
        $self->hideTooltips();

        # Initialise IVs
        $self->ivPoke('selectBoxFlag', TRUE);
        $self->ivUndef('selectBoxCanvasObj');
        $self->ivPoke('selectBoxInitXPos', $xPos);
        $self->ivPoke('selectBoxInitYPos', $yPos);
        $self->ivPoke('selectBoxCurrentXPos', $xPos);
        $self->ivPoke('selectBoxCurrentYPos', $yPos);

        # Temporarily set a few drawing cycle IVs, which allows the drawing functions to work as if
        #   we were in a drawing cycle (i.e. a call to $self->doDraw). They are reset by
        #   $self->stopSelectBox
        $self->ivPoke(
            'drawScheme',
            $self->worldModelObj->getRegionScheme($self->currentRegionmap),
        );

        return 1;
    }

    sub continueSelectBox {

        # Called by $self->setupCanvasEvent
        # Draws (or redraws) the selection box, after an earlier call to ->startSelectBox started
        #   the operation
        #
        # Expected arguments
        #   $event      - The Gtk3::Gdk::Event that caused the signal
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $event, $check) = @_;

        # Local variables
        my ($canvasWidget, $x1, $y1, $x2, $y2, $canvasObj);

        # Check for improper arguments
        if (! defined $event || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->continueSelectBox', @_);
        }

        # Get the canvas widget for the current level
        $canvasWidget = $self->currentParchment->ivShow(
            'canvasWidgetHash',
            $self->currentRegionmap->currentLevel,
        );

        # Deleting the existing canvas object (if one has already been drawn)
        if ($self->selectBoxCanvasObj) {

            # No need to call $self->deleteCanvasObj - the canvas object we're drawing doesn't
            #   represent anything in the world model
            $self->selectBoxCanvasObj->remove();
        }

        # Set the selection box's coordinates
        $x1 = $self->selectBoxInitXPos;
        $y1 = $self->selectBoxInitYPos;
        ($x2, $y2) = (int($event->x), int($event->y));

        # If the user has somehow moved the mouse to the exact original location, don't draw a new
        #   canvas object at all (but the operation continues until the user releases the mouse
        #   button)
        if ($x1 == $x2 && $y1 == $y2) {

            $self->ivUndef('selectBoxCanvasObj');

        } else {

            # Swap values so that ($x1, $y1) represents the top-left corner of the selection box,
            #   and ($x2, $y2) represents the bottom-right corner
            if ($x1 > $x2) {

                ($x1, $x2) = ($x2, $x1);
            }

            if ($y1 > $y2) {

                ($y1, $y2) = ($y2, $y1);
            }

            # Draw the new canvas object
            $canvasObj = GooCanvas2::CanvasRect->new(
                'parent' => $canvasWidget->get_root_item(),
                'x' => $x1,
                'y' => $y1,
                'width' => $x2 - $x1 + 1,
                'height' => $y2 - $y1 + 1,
#                'line-width' => 2,
                'stroke-color' => $self->drawScheme->selectBoxColour,
#                'fill-color' => $self->drawScheme->selectBoxColour,
            );

            # Move it above everything else
            $canvasObj->raise();

            $self->ivPoke('selectBoxCanvasObj', $canvasObj);
        }

        # Update remaining IVs
        $self->ivPoke('selectBoxCurrentXPos', int($event->x));
        $self->ivPoke('selectBoxCurrentYPos', int($event->y));

        return 1;
    }

    sub stopSelectBox {

        # Called by $self->canvasEventHandler
        # Terminates the selection box operation. Destroys the canvas object, updates IVs and calls
        #   $self->selectAllInBox to handle selecting any objects within the selection box
        #
        # Expected arguments
        #   $event          - The Gtk3::Gdk::Event that caused the signal
        #   $xPos, $yPos    - The coordinates of the release-click on the canvas
        #
        # Return values
        #   'undef' on improper arguments or if no selection box has been drawn yet
        #   1 otherwise

        my ($self, $event, $xPos, $yPos, $check) = @_;

        # Local variables
        my ($x1, $y1, $x2, $y2);

        # Check for improper arguments
        if (! defined $event || ! defined $xPos || ! defined $yPos || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->stopSelectBox', @_);
        }

        # If the user hasn't actually moved their mouse while the left mouse button was held down,
        #   then no selection box was drawn. Just unselect all selected objects
        if (! $self->selectBoxCanvasObj) {

            $self->setSelectedObj();

        } else {

            $self->selectBoxCanvasObj->remove();

            $x1 = $self->selectBoxInitXPos;
            $y1 = $self->selectBoxInitYPos;
            $x2 = $self->selectBoxCurrentXPos;
            $y2 = $self->selectBoxCurrentYPos;

            # Again, swap values so that ($x1, $y1) represents the top-left corner of the selection
            #   box, and ($x2, $y2) represents the bottom-right corner
            if ($x1 > $x2) {

                ($x1, $x2) = ($x2, $x1);
            }

            if ($y1 > $y2) {

                ($y1, $y2) = ($y2, $y1);
            }

            # Select everything within that zone
            $self->selectAllInBox($event, $x1, $y1, $x2, $y2);
        }

        # In either case, must reset IVs
        $self->ivPoke('selectBoxFlag', FALSE);
        $self->ivUndef('selectBoxCanvasObj');
        $self->ivUndef('selectBoxInitXPos');
        $self->ivUndef('selectBoxInitYPos');
        $self->ivUndef('selectBoxCurrentXPos');
        $self->ivUndef('selectBoxCurrentYPos');

        $self->ivUndef('drawScheme');

        return 1;
    }

    sub chooseClickedExit {

        # Called by $self->startDrag when the user starts to drag an exit, and by
        #   ->canvasObjEventHandler clicks an unselected exit
        # Selects which exit to use - the exit whose canvas object was clicked, or its twin exit -
        #   and returns the exit
        #
        # Expected arguments
        #   $exitObj        - The exit object whose canvas object was clicked
        #   $xPos, $yPos    - The coordinates of the mouse click on the canvas object
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise, returns either $exitObj or its twin exit object

        my ($self, $exitObj, $xPos, $yPos, $check) = @_;

        # Local variables
        my ($twinExitObj, $distance, $twinDistance);

        # Check for improper arguments
        if (! defined $exitObj || ! defined $xPos || ! defined $yPos || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->chooseClickedExit', @_);
        }

        if ($exitObj->twinExit) {

            # If the exit is a region exit, or a normal (not bent) broken exit, then there is no
            #   doubt that the clicked exit (not its twin) is the one to drag/select
            if (
                $exitObj->regionFlag
                || ($exitObj->brokenFlag && ! $exitObj->bentFlag)
            ) {
                return $exitObj;
            }

            # User clicked on a two-way exit, so we need to decide which exit to drag/select
            $twinExitObj = $self->worldModelObj->ivShow('exitModelHash', $exitObj->twinExit);

            # Find the distance, in pixels, between the click and the centre of the exit's parent
            #   room
            $distance = $self->findDistanceToRoom($exitObj, $xPos, $yPos);
            # Find the distance, in pixels, between the click and the centre of the twin exit's
            #   parent room
            $twinDistance = $self->findDistanceToRoom($twinExitObj, $xPos, $yPos);

            # If the distance to $exitObj's parent room is shorter, then use the twin exit as the
            #   clicked exit (otherwise, use $exitObj as the clicked exit)
            if ($distance < $twinDistance) {

                return $twinExitObj;
            }
        }

        # Otherwise, use $exitObj
        return $exitObj;
    }

    sub findDistanceToRoom {

        # Called by $self->chooseClickedExit
        # When the user clicks on an exit, finds the distance between the exit and the centre of the
        #   parent room (in pixels)
        #
        # Expected arguments
        #   $exitObj        - The exit object whose canvas object was clicked
        #   $xPos, $yPos    - The coordinates of the mouse click on the canvas object
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $exitObj, $xPos, $yPos, $check) = @_;

        # Local variables
        my ($roomObj, $roomXPos, $roomYPos, $lengthX, $lengthY);

        # Check for improper arguments
        if (! defined $exitObj || ! defined $xPos || ! defined $yPos || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->findDistanceToRoom', @_);
        }

        $roomObj = $self->worldModelObj->ivShow('modelHash', $exitObj->parent);
        $roomXPos = ($roomObj->xPosBlocks * $self->currentRegionmap->blockWidthPixels)
                            + int($self->currentRegionmap->blockWidthPixels / 2);
        $roomYPos = ($roomObj->yPosBlocks * $self->currentRegionmap->blockHeightPixels)
                            + int($self->currentRegionmap->blockHeightPixels / 2);

        $lengthX = abs($roomXPos - $xPos);
        $lengthY = abs($roomYPos - $yPos);
        return (sqrt(($lengthX ** 2) + ($lengthY ** 2)));
    }

    sub findSelectedRoomArea {

        # Called by $self->canvasObjEventHandler
        # Finds the smallest area on the current level of the grid which contains all the selected
        #   rooms on this level
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   An empty list on improper arguments, or if there are no selected rooms on the current
        #       level
        #   Otherwise, returns a list containing two pairs of grid coordinates - the top-left and
        #       bottom-right gridblock of the currently selected area (which might be the same
        #       block, if there's only one selected room). The list is in the form
        #           ($startX, $startY, $stopX, $stopY)

        my ($self, $check) = @_;

        # Local variables
        my (
            $startX, $startY, $stopX, $stopY, $count,
            @emptyList,
        );

        # Check for improper arguments
        if (defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->findSelectedRoomArea', @_);
            return @emptyList;
        }

        if ($self->selectedRoom) {

            # There is only one selected room. Is it in the current region's current level?
            if (
                $self->selectedRoom->parent == $self->currentRegionmap->number
                && $self->selectedRoom->zPosBlocks == $self->currentRegionmap->currentLevel
            ) {
                # It's on the current level
                $startX = $self->selectedRoom->xPosBlocks;
                $startY = $self->selectedRoom->yPosBlocks;
                $stopX = $self->selectedRoom->xPosBlocks;
                $stopY = $self->selectedRoom->yPosBlocks;

            } else {

                return @emptyList;
            }

        } elsif ($self->selectedRoomHash) {

            # Check every room in ->selectedRoomHash, and expand the borders of the selected area as
            #   we go
            $count = 0;
            foreach my $roomObj ($self->ivValues('selectedRoomHash')) {

                if (
                    $roomObj->parent == $self->currentRegionmap->number
                    && $roomObj->zPosBlocks == $self->currentRegionmap->currentLevel
                ) {
                    $count++;

                    if ($count == 1) {

                        # This is the first room processed
                        $startX = $roomObj->xPosBlocks;
                        $startY = $roomObj->yPosBlocks;
                        $stopX = $roomObj->xPosBlocks;
                        $stopY = $roomObj->yPosBlocks;

                    } else {

                        if ($roomObj->xPosBlocks < $startX) {
                            $startX = $roomObj->xPosBlocks;
                        } elsif ($roomObj->xPosBlocks > $stopX) {
                            $stopX = $roomObj->xPosBlocks;
                        }

                        if ($roomObj->yPosBlocks < $startY) {
                            $startY = $roomObj->yPosBlocks;
                        } elsif ($roomObj->yPosBlocks > $stopY) {
                            $stopY = $roomObj->yPosBlocks;
                        }
                    }
                }
            }

            if (! $count) {

                # No selected rooms in the current region's current level
                return @emptyList;
            }

        } else {

            # No selected rooms at all
            return @emptyList;
        }

        # Return the coordinates of opposite corners of the area
        return ($startX, $startY, $stopX, $stopY);
    }

    sub checkRoomIsSelected {

        # Called by $self->canvasObjEventHandler
        # Checks, as quickly as possible, whether a room is selected, or not
        #
        # Expected arguments
        #   $roomObj    - The room to check
        #
        # Return values
        #   'undef' on improper arguments or if the room is not selected
        #   1 if the room is selected

        my ($self, $roomObj, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkRoomIsSelected', @_);
        }

        if ($self->selectedRoom && $self->selectedRoom eq $roomObj) {

            return 1;

        } elsif ( ! $self->selectedRoomHash) {

            return undef;

        } else {

            if ($self->ivExists('selectedRoomHash', $roomObj->number)) {
                return 1;
            } else {
                return undef;
            }
        }
    }

    sub selectRoomsInArea {

        # Called by $self->canvasObjEventHandler
        # Selects all the rooms on the current level of the current regionmap, within a specified
        #   area, and unselects all other rooms (on all levels)
        # (Also unselects any selected room tags, room guilds, exits, exit tags or labels)
        #
        # Expected arguments
        #   $startX, $startY, $stopX, $stopY
        #       - Grid coordinates of the top-left and bottom-right of the area, in which all rooms
        #           should be selected

        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $startX, $startY, $stopX, $stopY, $check) = @_;

        # Local variables
        my (
            $count, $lastRoomObj,
            @selectedRoomlist, @selectedRoomTagList, @selectedRoomGuildList, @selectedExitList,
            @selectedExitTagList, @selectedLabelList, @redrawList,
            %currentHash, %newHash, %selectedRoomHash,
        );

        # Check for improper arguments
        if (
            ! defined $startX || ! defined $startY || ! defined $stopX || ! defined $stopY
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->selectRoomsInArea', @_);
        }

        # The regionmap's ->gridRoomHash contains all the rooms in the current region. Import the
        #   hash
        %currentHash = $self->currentRegionmap->gridRoomHash;

        # Compile a new hash, in the same format as $self->selectedRoomHash, containing only those
        #   rooms in the selected area (and on the current level)
        $count = 0;
        foreach my $position (keys %currentHash) {

            my ($number, $roomObj);

            $number = $currentHash{$position};
            $roomObj = $self->worldModelObj->ivShow('modelHash', $number);

            if (
                $roomObj
                && $roomObj->zPosBlocks == $self->currentRegionmap->currentLevel
                && $roomObj->xPosBlocks >= $startX
                && $roomObj->xPosBlocks <= $stopX
                && $roomObj->yPosBlocks >= $startY
                && $roomObj->yPosBlocks <= $stopY
            ) {
                # Mark this room for selection
                $newHash{$number} = $roomObj;
                $count++;
                $lastRoomObj = $roomObj;
            }
        }

        # Compile a list of currently selected rooms, exits, room tags, room guilds and labels
        @selectedRoomlist = $self->compileSelectedRooms();
        @selectedRoomTagList = $self->compileSelectedRoomTags();
        @selectedRoomGuildList = $self->compileSelectedRoomGuilds();
        @selectedExitList = $self->compileSelectedExits();
        @selectedExitTagList = $self->compileSelectedExitTags();
        @selectedLabelList = $self->compileSelectedLabels();

        # Transfer the list of currently selected rooms into a hash, so that we can compare them
        #   with %newHash
        foreach my $ref (@selectedRoomlist) {

            $selectedRoomHash{$ref} = $ref;
        }

        # Check that the same room doesn't exist in %newHash and %selectedRoomHash. If so, delete
        #   the entry in %selectedRoomHash
        foreach my $obj (values %newHash) {

            if (exists $selectedRoomHash{$obj}) {

                delete $selectedRoomHash{$obj};
            }
        }

        # Set the IVs that contain all selected objects
        if ($count == 1) {

            $self->ivPoke('selectedRoom', $lastRoomObj);
            $self->ivEmpty('selectedRoomHash');

        } else {

            $self->ivUndef('selectedRoom');
            $self->ivPoke('selectedRoomHash', %newHash);
        }

        # Make sure there are no room tags, room guilds, exits, exit tags or labels selected
        $self->ivUndef('selectedRoomTag');
        $self->ivEmpty('selectedRoomTagHash');
        $self->ivUndef('selectedRoomGuild');
        $self->ivEmpty('selectedRoomGuildHash');
        $self->ivUndef('selectedExit');
        $self->ivEmpty('selectedExitHash');
        $self->ivUndef('selectedExitTag');
        $self->ivEmpty('selectedExitTagHash');
        $self->ivUndef('selectedLabel');
        $self->ivEmpty('selectedLabelHash');

        # Finally, re-draw all objects that have either been selected or unselected. Compile a list
        #   to send to ->markObjs, in the form (type, object, type, object, ...)
        foreach my $obj (values %newHash) {

            push (@redrawList, 'room', $obj);
        }

        foreach my $obj (values %selectedRoomHash) {

            push (@redrawList, 'room', $obj);
        }

        foreach my $obj (@selectedRoomTagList) {

            push (@redrawList, 'room_tag', $obj);
        }

        foreach my $obj (@selectedRoomGuildList) {

            push (@redrawList, 'room_guild', $obj);
        }

        foreach my $obj (@selectedExitList) {

            push (@redrawList, 'exit', $obj);
        }

        foreach my $obj (@selectedExitTagList) {

            push (@redrawList, 'exit_tag', $obj);
        }

        foreach my $obj (@selectedLabelList) {

            push (@redrawList, 'label', $obj);
        }

        # Actually redraw the affected objects
        $self->markObjs(@redrawList);
        $self->doDraw();

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        # Operation complete
        return 1;
    }

    sub selectAllInBox {

        # Called by $self->stopSelectBox
        # After the user has specified an area of the map, get a list of rooms and/or labels within
        #   that area, and select them
        #
        # Expected arguments
        #   $event          - The Gtk3::Gdk::Event that caused the signal
        #   $x1, $y1        - Canvas coordinates of the top-left corner of the selection box
        #   $x2, $y2        - Coordinates of the bottom-right corner
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $event, $x1, $y1, $x2, $y2, $check) = @_;

        # Local variables
        my (
            $level, $coord, $xBlocks1, $yBlocks1, $xBlocks2, $yBlocks2, $exitMode, $borderX1,
            $borderY1, $borderX2, $borderY2,
            @selectList,
            %roomHash,
        );

        # Check for improper arguments
        if (
            ! defined $event || ! defined $x1 || ! defined $y1 || ! defined $x2 || ! defined $y2
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->selectAllInBox', @_);
        }

        # Import the world model's room hash (for speed)
        %roomHash = $self->worldModelObj->roomModelHash;
        # Import the current level (for speed)
        $level = $self->currentRegionmap->currentLevel;

        # Convert canvas (pixel) coordinates to gridblock coordinates
        ($xBlocks1, $yBlocks1) = $self->findGridBlock($x1, $y1, $self->currentRegionmap);
        ($xBlocks2, $yBlocks2) = $self->findGridBlock($x2, $y2, $self->currentRegionmap);

        # Get the position of a room's border drawn within its gridblock
        if ($self->worldModelObj->drawExitMode eq 'ask_regionmap') {
            $exitMode = $self->currentRegionmap->drawExitMode;
        } else {
            $exitMode = $self->worldModelObj->drawExitMode;
        }

        # The coordinates of the pixel at the top-left corner of the room box
        if ($exitMode eq 'no_exit') {

            # Draw exit mode 'no_exit': The room takes up the whole gridblock
            $borderX1 = 0;
            $borderY1 = 0;
            $borderX2 = $self->currentRegionmap->blockWidthPixels - 1;
            $borderY2 = $self->currentRegionmap->blockHeightPixels - 1;

        } else {

            # Draw exit modes 'simple_exit'/'complex_exit': The room takes up the central part of
            #   the gridblock
            $borderX1 = int(
                (
                    $self->currentRegionmap->blockWidthPixels
                        - $self->currentRegionmap->roomWidthPixels
                ) / 2
            );

            $borderY1 = int(
                (
                    $self->currentRegionmap->blockHeightPixels
                        - $self->currentRegionmap->roomHeightPixels
                ) / 2
            );

            $borderX2 = $borderX1 + $self->currentRegionmap->roomWidthPixels - 1;
            $borderY2 = $borderX1 + $self->currentRegionmap->roomHeightPixels - 1;
        }

        if ($xBlocks1 == $xBlocks2 && $yBlocks1 == $yBlocks2) {

            # Special case - if it's only one block, then don't check every room in the region
            $coord = $xBlocks1 . '_' . $yBlocks1 . '_' . $level;
            if ($self->currentRegionmap->ivExists('gridRoomHash', $coord)) {

                push (
                    @selectList,
                    $self->worldModelObj->ivShow(
                        'modelHash',
                        $self->currentRegionmap->ivShow('gridRoomHash', $coord),
                    ),
                    'room',
                );
            }

        } else {

            # Find of rooms that are wholly or partially within the selection box
            foreach my $roomNum ($self->currentRegionmap->ivValues('gridRoomHash')) {

                my ($roomObj, $roomXBlocks, $roomYBlocks);

                $roomObj = $roomHash{$roomNum};

                if ($roomObj && $roomObj->zPosBlocks == $level) {

                    $roomXBlocks = $roomObj->xPosBlocks * $self->currentRegionmap->blockWidthPixels;
                    $roomYBlocks
                        = $roomObj->yPosBlocks * $self->currentRegionmap->blockHeightPixels;

                    if (
                        ($roomXBlocks + $borderX1) <= $x2
                        && ($roomXBlocks + $borderX2) >= $x1
                        && ($roomYBlocks + $borderY1) <= $y2
                        && ($roomYBlocks + $borderY2) >= $y1
                    ) {
                        push (@selectList, $roomObj, 'room');
                    }
                }
            }

            # Find a list of labels whose start position is within the selection box
            foreach my $labelObj ($self->currentRegionmap->ivValues('gridLabelHash')) {

                my ($listRef, $canvasObj, $boundsObj, $labelX1, $labelY1, $labelX2, $labelY2);

                if ($labelObj->level == $level) {

                    $listRef = $self->currentParchment->getDrawnLabel($labelObj);
                    if (defined $listRef) {

                        # If the label has a box, use the boundaries of the box; otherwise use the
                        #   boundaries of the label text
                        if ($labelObj->boxFlag && defined $$listRef[1]) {
                            $canvasObj = $$listRef[1];
                        } else {
                            $canvasObj = $$listRef[0];
                        }

                        $boundsObj = $canvasObj->get_bounds();
                        if (
                            $boundsObj->x1 <= $x2
                            && $boundsObj->x2 >= $x1
                            && $boundsObj->y1 <= $y2
                            && $boundsObj->y2 >= $y1
                        ) {
                            push (@selectList, $labelObj, 'label');
                        }
                    }
                }
            }
        }

        # Search complete. If the CTRL keys are held down, add the rooms/labels to any objects which
        #   are already selected; otherwise select just these objects
        if (! ($event->state =~ m/control-mask/)) {

            $self->setSelectedObj();
        }

        $self->setSelectedObj(\@selectList, TRUE);

        return 1;
    }

    sub doQuickPaint {

        # Called by $self->canvasEventHandler when a left-click is detected on any room while
        #   $self->toolbarQuickPaintColour is set
        # Toggles room flags in the clicked room. If it's a selected room, toggles room flags in
        #   all selected rooms
        #
        # Expected arguments
        #   $clickRoomObj   - The room that was left-clicked by the user
        #
        # Return values
        #   'undef' on improper arguments or if none of the buttons in the quick painting toolbar
        #       are selected
        #   1 otherwise

        my ($self, $clickRoomObj, $check) = @_;

        # Local variables
        my @roomList;

        # Check for improper arguments
        if (! defined $clickRoomObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->doQuickPaint', @_);
        }

        # Do nothing if none of the colour buttons in the quick painting toolbar are selected
        if (! $self->toolbarQuickPaintColour) {

            return undef;
        }

        # If the room is a selected room, room flags should be toggled in all rooms
        if (
            ($self->selectedRoom && $self->selectedRoom eq $clickRoomObj)
            || $self->ivExists('selectedRoomHash', $clickRoomObj->number)
        ) {
            push (@roomList, $self->compileSelectedRooms());

        } else {

            # Just toggle room flags in the clicked room
            push (@roomList, $clickRoomObj);
        }

        # Toggle the room flag in those rooms
        $self->worldModelObj->toggleRoomFlags(
            $self->session,
            TRUE,                               # Update automapper windows
            $self->toolbarQuickPaintColour,
            @roomList,
        );

        if (! $self->worldModelObj->quickPaintMultiFlag) {

            # User wants the choice of room flag to reset after clicking on a room
            $self->ivUndef('toolbarQuickPaintColour');
        }

        return 1;
    }

    sub doMerge {

        # Called by $self->canvasEventHandler when a double-click is detected on the current room
        #   and the automapper object is set up to perform a merge (i.e.
        #   GA::Map::Obj->currentMatchFlag is TRUE)
        # Also called by $self->enableRoomsColumnm ->enableRoomsPopupMenu and
        #   ->canvasObjEventHandler
        #
        # Prepares a call to GA::Obj::WorldModel->mergeMap, then makes the call
        #
        # Expected arguments
        #   $currentRoomObj - The room that is definitely to be merged with another room (at the
        #                       moment, it's always the automapper object's current room)
        #
        # Optional arguments
        #   $targetRoomObj  - When called by $self->canvasObjEventHandler, because
        #                       GA::Obj::Map->currentMatchList specifies several rooms that match
        #                       the current room, then this variable is the room that was clicked
        #   $noConfirmFlag  - TRUE if the confirmation dialogue window should not be shown; FALSE
        #                       (or 'undef') if it should be shown as usual
        #
        # Return values
        #   'undef' on improper arguments, if the user declines to perform the operation after a
        #       prompt or if the merge operation fails
        #   1 otherwise

        my ($self, $currentRoomObj, $targetRoomObj, $noConfirmFlag, $check) = @_;

        # Local variables
        my (
            $autoRescueFlag, $regionmapObj, $response,
            @selectList, @otherRoomList, @labelList,
        );

        # Check for improper arguments
        if (! defined $currentRoomObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->doMerge', @_);
        }

        if ($self->mapObj->ivNumber('currentMatchList') > 1 && ! $targetRoomObj) {

            # GA::Obj::Map->currentMatchList specifies multiple rooms which match the current
            #   room (which are all selected), so allow the user to click on one of those rooms,
            #   after which this function is called again by $self->canvasObjEventHandler
            $self->set_freeClickMode('merge_room');

            # After a double-click on the current room, none of the matching rooms will be selected,
            #   so select them again
            if (! $self->selectedRoom && ! $self->selectedRoomHash) {

                foreach my $roomObj ($self->mapObj->currentMatchList) {

                    push (@selectList, $roomObj, 'room');
                }

                $self->setSelectedObj(\@selectList, TRUE);
            }

        } else {

            # If $targetRoomObj was set by the calling function, then GA::Obj::Map->currentMatchList
            #   specifies multiple rooms which match the current room, and the user has clicked one
            #   of them
            if (! $targetRoomObj) {

                # Otherwise, GA::Obj::Map->currentMatchList specifies a single room which matches
                #   the current room
                $targetRoomObj = $self->mapObj->ivIndex('currentMatchList', 0);
            }

            if (
                $self->mapObj->rescueTempRegionObj
                && $self->mapObj->rescueTempRegionObj->number == $currentRoomObj->parent
            ) {
                # In Auto-rescue mode, attempt to merge or move all of the rooms in the temporary
                #   region
                $autoRescueFlag = TRUE;

                $regionmapObj = $self->worldModelObj->ivShow(
                    'regionmapHash',
                    $self->mapObj->rescueTempRegionObj->name,
                );

                foreach my $roomNum ($regionmapObj->ivValues('gridRoomHash')) {

                    # @otherRoomList shouldn't contain $currentRoomObj or $targetRoomObj
                    if (
                        $roomNum != $currentRoomObj->number
                        && $roomNum != $targetRoomObj->number
                    ) {
                        push (@otherRoomList, $self->worldModelObj->ivShow('modelHash', $roomNum));
                    }
                }

            } else {

                # Otherwise, attempt to merge or move the $currentRoomObj and any other selected
                #   rooms that are in the same region
                foreach my $roomObj ($self->compileSelectedRooms()) {

                    if (
                        $roomObj ne $targetRoomObj
                        && $roomObj ne $currentRoomObj
                        && $roomObj->parent eq $currentRoomObj->parent
                        && ! defined $self->mapObj->ivFind('currentMatchList', $roomObj)
                    ) {
                        push (@otherRoomList, $roomObj);
                    }
                }

                # Because we're not in auto-rescue mode, prompt the user before merging/moving more
                #   than one room
                if (@otherRoomList && ! $noConfirmFlag) {

                    $response = $self->showMsgDialogue(
                        'Merge/move rooms',
                        'question',
                        'Are you sure you want to merge/move ' . ((scalar @otherRoomList) + 1)
                        . ' rooms?',
                        'yes-no',
                    );

                    if (! defined $response || $response ne 'yes') {

                        return undef;
                    }
                }

                # Get the regionmap for the code just below
                $regionmapObj = $self->findRegionmap($currentRoomObj->parent);
            }

            # Any selected labels in the same region as $currentRoomObj should also be moved
            foreach my $labelObj ($self->compileSelectedLabels()) {

                if ($labelObj->region eq $regionmapObj->name) {

                    push (@labelList, $labelObj);
                }
            }

            # Merge the room(s)
            if (
                ! $self->worldModelObj->mergeMap(
                    $self->session,
                    $targetRoomObj,
                    $currentRoomObj,
                    \@otherRoomList,
                    \@labelList,
                )
            ) {
                # Merge operation failed
                $self->showMsgDialogue(
                    'Merge/move rooms',
                    'error',
                    'Merge operation failed',
                    'ok',
                );

                if ($autoRescueFlag) {

                    # Auto-rescue mode was already activated, but rooms can't be moved from the
                    #   temporary region back to the previous region
                    # Treat the temporary region as an ordinary temporary region from now on
                    $self->mapObj->reset_rescueRegion();
                }

                return undef;

            } elsif ($autoRescueFlag) {

                # Merge operation succeeded, and auto-rescue mode was already activated, so we can
                #   discard the temporary region
                $self->worldModelObj->deleteRegions(
                    $self->session,
                    TRUE,                                   # Update automapper windows now
                    $self->mapObj->rescueTempRegionObj,
                );

                # The GA::Obj::Map IVs should have been reset, but there's no harm in checking
                $self->mapObj->reset_rescueRegion();
            }
        }

        return 1;
    }

    sub checkMouseClick {

        # Called by $self->canvasEventHandler and ->canvasObjEventHandler
        # After an event caused by a mouse click, checks the event to find out whether it was a
        #   single/double/triple click, which button was used (left or right), and whether the SHIFT
        #   and/or CTRL keys were held down during the click
        #
        # Expected arguments
        #   $event  - The Gtk3::Gdk::Event caused by the mouse click
        #
        # Return values
        #   An empty list on improper arguments, or if $event wasn't cause by a single, double or
        #       triple-mouse click, or by the user releasing the mouse button but by something else
        #       - mouse motion, perhaps
        #   Otherwise, returns a list in the form ($clickType, $button, $shiftFlag, $ctrlFlag):
        #       $clickType  - 'single', 'double' or 'triple' or 'release'
        #       $button     - 'left' or 'right'
        #       $shiftFlag  - Set to TRUE if the SHIFT key was held down during the click, set to
        #                       FALSE otherwise
        #       $ctrlFlag   - Set to TRUE if the CTRL key was held down during the click, set to
        #                       FALSE otherwise

        my ($self, $event, $check) = @_;

        # Local variables
        my (
            $clickType, $button, $shiftFlag, $ctrlFlag,
            @emptyList,
        );

        # Check for improper arguments
        if (! defined $event || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->checkMouseClick', @_);
            return @emptyList;
        }

        # Set the type of click
        if ($event->type eq 'button-press') {

            $clickType = 'single';

        } elsif ($event->type eq '2button-press') {

            $clickType = 'double';

        } elsif ($event->type eq '3button-press') {

            $clickType = 'triple';

        } elsif ($event->type eq 'button-release') {

            $clickType = 'release';

        } else {

            # Not an event we're interested in
            return @emptyList;
        }

        # Set the button
        if ($event->button == 1) {

            $button = 'left';

        } elsif ($event->button == 3) {

            $button = 'right';

        } else {

            # Not an event we're interested in
            return @emptyList;
        }

        # Check whether the SHIFT and/or CTRL keys were held down, when the mouse was clicked
        if ($event->state =~ m/shift-mask/) {
            $shiftFlag = TRUE;
        } else {
            $shiftFlag = FALSE;
        }

        if ($event->state =~ m/control-mask/) {
            $ctrlFlag = TRUE;
        } else {
            $ctrlFlag = FALSE;
        }

        return ($clickType, $button, $shiftFlag, $ctrlFlag);
    }

    sub showTooltips {

        # Called by $self->setupCanvasObjEvent
        # Shows tooltips (assumes the GA::Obj::WorldModel->showTooltipsFlag is TRUE)
        #
        # Expected arguments
        #   $type       - What type of canvas object caused the mouse event - 'room', 'room_tag',
        #                   'room_guild', 'exit', 'exit_tag' or 'label'
        #   $canvasObj  - The canvas object itself
        #   $modelObj   - The GA::ModelObj::Room, GA::Obj::Exit or GA::Obj::MapLabel which
        #                   corresponds to the canvas object $canvasObj
        #
        # Return values
        #   'undef' on improper arguments or if the Automapper window isn't ready and active
        #   1 otherwise

        my ($self, $type, $canvasObj, $modelObj, $check) = @_;

        # Local variables
        my ($xPos, $yPos, $label);

        # Check for improper arguments
        if (! defined $type || ! defined $canvasObj || ! defined $modelObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->showTooltips', @_);
        }

        # Don't show tooltips if the Automapper window isn't ready and active
        if (! $self->canvasFrame || ! $self->winWidget->is_active()) {

            return undef;
        }

        # Get the label to draw
        $label = $self->setTooltips($type, $modelObj);
        if ($label) {

            $self->canvasFrame->set_tooltip_text($label);

            $self->ivPoke('canvasTooltipObj', $canvasObj);
            $self->ivPoke('canvasTooltipObjType', $type);
            $self->ivPoke('canvasTooltipFlag', TRUE);
        }

        return 1;
    }

    sub hideTooltips {

        # Called by $self->setupCanvasObjEvent and several other functions
        # Hides tooltips, if visible
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->hideTooltips', @_);
        }

        # Hide tooltips, if visible
        if ($self->canvasFrame && $self->canvasTooltipObj) {

            $self->canvasFrame->set_tooltip_text('');

            $self->ivUndef('canvasTooltipObj');
            $self->ivUndef('canvasTooltipObjType');
            $self->ivPoke('canvasTooltipFlag', FALSE);
        }

        return 1;
    }

    sub setTooltips {

        # Called by $self->showTooltips
        # Compiles the text to show in the tooltips window, and returns it
        #
        # Expected arguments
        #   $type       - What type of canvas object caused the mouse event - 'room', 'room_tag',
        #                   'room_guild', 'exit', 'exit_tag' or 'label'
        #   $modelObj   - The GA::ModelObj::Room, GA::Obj::Exit or GA::Obj::MapLabel which
        #                   corresponds to the canvas object $canvasObj
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns the text to display in the tooltips window

        my ($self, $type, $modelObj, $check) = @_;

        # Local variables
        my (
            $label, $vNum, $name, $area, $worldX, $worldY, $worldZ, $text, $flag, $standardDir,
            $abbrevDir, $parentRoomObj, $destRoomObj, $twinExitObj, $xPos, $yPos, $modText,
        );

        # Check for improper arguments
        if (! defined $type || ! defined $modelObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setTooltips', @_);
        }

        if ($type eq 'room') {

            $label = "Room #" . $modelObj->number;
            if ($modelObj->roomTag) {

                $label .= " \'" . $modelObj->roomTag . "\'";
            }

            # Show the room's coordinates on the map
            $label .= " (" . $modelObj->xPosBlocks . ", " . $modelObj->yPosBlocks;
            $label .= ", " . $modelObj->zPosBlocks . ")";

            # Show the world's room vnum, etc (if known)
            if ($modelObj->protocolRoomHash) {

                $label .= "\nWorld:";

                $vNum = $modelObj->ivShow('protocolRoomHash', 'vNum');
                if (defined $vNum) {

                    $label .=  "#" . $vNum;
                }

                $name = $modelObj->ivShow('protocolRoomHash', 'name');
                if (defined $name) {

                    $label .= " " . $name;
                }

                $area = $modelObj->ivShow('protocolRoomHash', 'area');
                if (defined $area) {

                    $label .= " " . $area;
                }

                $worldZ = $modelObj->ivShow('protocolRoomHash', 'zpos');
                if (! defined $worldZ) {

                    # (Guard against X being defined, but Y/Z not being defined, etc
                    $worldZ = "?";
                }

                $worldY = $modelObj->ivShow('protocolRoomHash', 'ypos');
                if (! defined $worldY) {

                    $worldY = "?";
                }

                $worldX = $modelObj->ivShow('protocolRoomHash', 'xpos');
                if (defined $worldX) {

                    $label .= " $worldX-$worldY-$worldY";
                }
            }

            # Add the room title (if there is one)
            if ($modelObj->titleList) {

                # Using the first item in the list, use the whole title
                $label .= "\n(T) " . $modelObj->ivFirst('titleList');
                $flag = TRUE;
            }

            # Add a (verbose) description (if there is one)
            if ($modelObj->descripHash) {

                # Use the description matching the current light status, if it exists
                if (
                    $self->worldModelObj->lightStatus
                    && $modelObj->ivExists('descripHash', $self->worldModelObj->lightStatus)
                ) {
                    $text = $modelObj->ivShow('descripHash', $self->worldModelObj->lightStatus);

                } else {

                    # Cycle through light statuses, looking for a matching verbose description
                    OUTER: foreach my $status ($self->worldModelObj->lightStatusList) {

                        if ($modelObj->ivExists('descripHash', $status)) {

                            $text = $modelObj->ivShow('descripHash', $status);
                            last OUTER;
                        }
                    }
                }

                if ($text) {

                    # Split the text into two lines of no more than 40 characters. The TRUE
                    #   arguments tells the function to append an ellipsis, if any text is
                    #   removed
                    $text = $axmud::CLIENT->splitText($text, 2, 40, TRUE);
                    $label .= "\n(D) " . $text;

                    $flag = TRUE;
                }
            }

            # Add the room's source code path (if set)
            if ($modelObj->sourceCodePath) {

                $label .= "\n(S) " . $modelObj->sourceCodePath;
                $flag = TRUE;
            }

            # If there is no title or (verbose) description available, show an explanatory message
            if (! $flag) {

                $label .= "\n(No description available)";
            }

            # Show room notes (if set)
            if ($self->worldModelObj->showNotesFlag && $modelObj->notesList) {

                $text = $axmud::CLIENT->trimWhitespace(join(' ', $modelObj->notesList), TRUE);

                # Split the text into five lines of no more than 40 characters, appending an
                #   ellipsis
                $text = $axmud::CLIENT->splitText($text, 5, 40, TRUE);
                $label .= "\n(N) " . $text;
            }

        } elsif ($type eq 'room_tag') {

            $label = "Room tag \'" . $modelObj->roomTag . "\'";
            $label .= "\n  Room #" . $modelObj->number;

            # Show the room's coordinates on the map
            $label .= " (" . $modelObj->xPosBlocks . ", " . $modelObj->yPosBlocks;
            $label .= ", " . $modelObj->zPosBlocks . ")";

        } elsif ($type eq 'room_guild') {

            $label = "Room guild \'" . $modelObj->roomGuild . "\'";
            $label .= "\n  Room #" . $modelObj->number;

            # Show the room's coordinates on the map
            $label .= " (" . $modelObj->xPosBlocks . ", " . $modelObj->yPosBlocks;
            $label .= ", " . $modelObj->zPosBlocks . ")";

        } elsif ($type eq 'exit') {

            $label = "Exit #" . $modelObj->number . " \'" . $modelObj->dir . "\'";

            # Get the standard form of the exit's direction so we can compare it with the exit's
            #   map direction, ->mapDir
            $standardDir = $self->session->currentDict->ivShow('combRevDirHash', $modelObj->dir);
            if (
                $standardDir
                && $modelObj->mapDir
                && $modelObj->mapDir ne $standardDir
            ) {
                # Convert the allocated map direction to its abbreviated form
                $abbrevDir = $self->session->currentDict->ivShow(
                    'primaryAbbrevHash',
                    $modelObj->mapDir,
                );

                if (! $abbrevDir) {

                    # We're forced to use the unabbreviated form
                    $abbrevDir = $modelObj->mapDir;
                }

                $label .= " (> " . $abbrevDir . ")";
            }

            $parentRoomObj = $self->worldModelObj->ivShow('modelHash', $modelObj->parent);
            $label .= "\n  Parent room #" . $parentRoomObj->number;

            if ($modelObj->destRoom) {

                $destRoomObj = $self->worldModelObj->ivShow('modelHash', $modelObj->destRoom);
                $label .= "\n  Destination room #" . $destRoomObj->number;
            }

            if ($modelObj->twinExit) {

                $twinExitObj = $self->worldModelObj->ivShow('exitModelHash', $modelObj->twinExit);
                $label .= "\n  Twin exit #" . $twinExitObj->number . " \'" . $twinExitObj->dir
                            . "\'";
            }

            if (defined $modelObj->altDir) {

                $label .= "\n  Alternative directions:\n    " . $modelObj->altDir;
            }

            if ($modelObj->exitInfo) {

                $label .= "\n  Info: " . $modelObj->exitInfo;
            }

        } elsif ($type eq 'exit_tag') {

            $label = "Exit tag \'" . $modelObj->exitTag . "\'";
            $label .= "\n  Exit #" . $modelObj->number . " \'" . $modelObj->dir . "\'";

        } elsif ($type eq 'label') {

            $label = "Label #" . $modelObj->number;
            # Convert the label's coordinates in pixels to gridblocks
            $xPos = int($modelObj->xPosPixels / $self->currentRegionmap->blockWidthPixels);
            $yPos = int($modelObj->yPosPixels / $self->currentRegionmap->blockHeightPixels);
            $label .= " (" . $xPos . ", " . $yPos . ", " . $modelObj->level . ")";

            if (! defined $modelObj->style) {
                $label .= "\nStyle: <custom>";
            } else {
                $label .= "\nStyle: \'" . $modelObj->style . "\'";
            }

            # (The text can include a lot of empty space and newline characters, so strip all of
            #   that)
            $modText = $modelObj->name;
            $modText =~ s/^[\s\n]*//;
            $modText =~ s/[\s\n]*$//;
            $modText =~ s/[\s\n]+/ /g;

            $label .= "\nText: \'" . $modText . "\'";

        } else {

            # Failsafe: empty string
            $label = "";
        }

        return $label;
    }

    # Menu 'File' column callbacks

    sub importModelCallback {

        # Called by $self->enableFileColumn
        # Imports a world model file specified by the user and (if successful) loads it into memory
        #   (a combination of ';importfiles' and ';load -m')
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->importModelCallback', @_);
        }

        # (No standard callback checks for this function)

        # Watch out for file operation failures
        $axmud::CLIENT->set_fileFailFlag(FALSE);
        # Allow a world model, associated with a world profile with a different name, to be imported
        #   into the current world's file structures (but only if the archive file contains only a
        #   world model)
        $self->session->set_transferWorldModelFlag(TRUE);

        # Import a file, specified by the user
        if (
            $self->session->pseudoCmd('importfiles')
            && ! $axmud::CLIENT->fileFailFlag
        ) {
            # The world model data has been incorporated into Axmud's data files, but not loaded
            #   into memory. Load it into memory now
            if (
                $self->session->pseudoCmd('load -m')
                && ! $axmud::CLIENT->fileFailFlag
            ) {
                # Make sure the world model object has the right parent world set, after the file
                #   import
                $self->session->worldModelObj->{_parentWorld} = $self->session->currentWorld->name;
                # Save the world model, to make sure the file has the right parent world set, too
                $self->session->pseudoCmd('save -f -m');
            }
        }

        # Reset the flag
        $self->session->set_transferWorldModelFlag(FALSE);

        return 1;
    }

    sub exportModelCallback {

        # Called by $self->enableFileColumn
        # Saves the current world model and (if successful) exports the 'worldmodel' file to a
        #   folder specified by the user (a combination of ';save -m' and ';exportfiles -m'
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($fileObj, $choice);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exportModelCallback', @_);
        }

        # (No standard callback checks for this function)

        # If the world model data in memory is unsaved, prompt whether to save it first
        $fileObj = $self->session->ivShow('sessionFileObjHash', 'worldmodel');
        if ($fileObj && $fileObj->modifyFlag) {

            # Watch out for file operation failures
            $axmud::CLIENT->set_fileFailFlag(FALSE);

            # Prompt the user
            $choice = $self->showMsgDialogue(
                'Unsaved world model',
                'question',
                'The world model in memory is not saved. Do you want to save it before exporting?'
                . ' (If you choose \'No\', the previously saved world model file will be exported'
                . ' instead)',
                'yes-no',
            );

            if ($choice eq 'yes') {

                # Save the world model
                $self->session->pseudoCmd('save -m', 'win_error');

                if ($axmud::CLIENT->fileFailFlag) {

                    # Something went wrong; don't attempt to export anything
                    return 1;
                }
            }
        }

        # Export the world model data file
        $self->session->pseudoCmd(
            'exportfiles -m ' . $self->session->currentWorld->name,
            'win_error',
        );

        return 1;
    }

    # Menu 'Edit' column callbacks

    sub selectInRegionCallback {

        # Called by $self->enableEditColumn
        # Selects rooms, exits, room tags, room guilds or labels (or everything) in the current
        #   region
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $type   - Set to 'room', 'exit', 'room_tag', 'room_guild', or 'label'. If not defined,
        #               selects everything
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $type, $check) = @_;

        # Local variables
        my (
            $count,
            @roomList, @exitList, @roomTagList, @roomGuildList, @labelList,
            %roomHash, %exitHash, %roomTagHash, %roomGuildHash, %labelHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->selectInRegionCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Make sure there are no rooms, exits, room tags or labels selected
        $self->ivUndef('selectedRoom');
        $self->ivEmpty('selectedRoomHash');
        $self->ivUndef('selectedExit');
        $self->ivEmpty('selectedExitHash');
        $self->ivUndef('selectedRoomTag');
        $self->ivEmpty('selectedRoomTagHash');
        $self->ivUndef('selectedRoomGuild');
        $self->ivEmpty('selectedRoomGuildHash');
        $self->ivUndef('selectedLabel');
        $self->ivEmpty('selectedLabelHash');

        # Select all rooms, exits, room tags, room guilds and/or labels
        if (! defined $type || $type eq 'room') {

            # $self->currentRegionmap->gridRoomHash contains all the rooms in the regionmap
            # Get a list of world model numbers for each room
            @roomList = $self->currentRegionmap->ivValues('gridRoomHash');
        }

        if (! defined $type || $type eq 'exit') {

            # ->gridExitHash contains all the drawn exits
            # Get a list of exit model numbers for each exit
            @exitList = $self->currentRegionmap->ivKeys('gridExitHash');
        }

        if (! defined $type || $type eq 'room_tag') {

            # ->gridRoomTagHash contains all the rooms with room tags
            # Get a list of world model numbers for the rooms containing room tag
            @roomTagList = $self->currentRegionmap->ivValues('gridRoomTagHash');
        }

        if (! defined $type || $type eq 'room_guild') {

            # ->gridRoomGuildHash contains all the rooms with room guilds
            # Get a list of world model numbers for the roomw containing room guilds
            @roomGuildList = $self->currentRegionmap->ivValues('gridRoomGuildHash');
        }

        if (! defined $type || $type eq 'label') {

            # ->gridLabelHash contains all the labels
            # Get a list of blessed references to GA::Obj::MapLabel objects
            @labelList = $self->currentRegionmap->ivValues('gridLabelHash');
        }

        # The IVs that store selected objects behave differently when there is one selected object
        #   and when there is more than one. Count how many selected objects we have
        $count = (scalar @roomList) + (scalar @exitList) + (scalar @roomTagList)
                    + (scalar @roomGuildList) + (scalar @labelList);

        # Select a single object...
        if ($count == 1) {

            if (@roomList) {

                # Select the blessed reference of a GA::ModelObj::Room
                $self->ivPoke(
                    'selectedRoom',
                    $self->worldModelObj->ivShow('modelHash', $roomList[0]),
                );

            } elsif (@exitList) {

                # Select the blessed reference of a GA::Obj::Exit
                $self->ivPoke(
                    'selectedExit',
                    $self->worldModelObj->ivShow('exitModelHash', $exitList[0]),
                );

            } elsif (@roomTagList) {

                # Select the blessed reference of the GA::ModelObj::Room which contains the room
                #   tag
                $self->ivPoke(
                    'selectedRoomTag',
                    $self->worldModelObj->ivShow('modelHash', $roomTagList[0]),
                );

            } elsif (@roomGuildList) {

                # Select the blessed reference of the GA::ModelObj::Room which contains the room
                #   guild
                $self->ivPoke(
                    'selectedRoomGuild',
                    $self->worldModelObj->ivShow('modelHash', $roomGuildList[0]),
                );

            } elsif (@labelList) {

                # Select the blessed reference of a GA::Obj::MapLabel
                $self->ivPoke('selectedLabel', $labelList[0]);
            }

        # ...or select multiple objects
        } else {

            # (For speed, update local variable hashes, before storing the whole hash(es) in IVs
            foreach my $number (@roomList) {

                $roomHash{$number} = $self->worldModelObj->ivShow('modelHash', $number);
            }

            foreach my $number (@exitList) {

                $exitHash{$number} = $self->worldModelObj->ivShow('exitModelHash', $number);
            }

            foreach my $number (@roomTagList) {

                $roomTagHash{$number} = $self->worldModelObj->ivShow('modelHash', $number);
            }

            foreach my $number (@roomGuildList) {

                $roomGuildHash{$number} = $self->worldModelObj->ivShow('modelHash', $number);
            }

            foreach my $obj (@labelList) {

                $labelHash{$obj->id} = $obj;
            }

            # Update the IVs
            $self->ivPoke('selectedRoomHash', %roomHash);
            $self->ivPoke('selectedExitHash', %exitHash);
            $self->ivPoke('selectedRoomTagHash', %roomTagHash);
            $self->ivPoke('selectedRoomGuildHash', %roomGuildHash);
            $self->ivPoke('selectedLabelHash', %labelHash);
        }

        # Redraw the current region
        $self->redrawRegions();

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        return 1;
    }

    sub selectInMapCallback {

        # Called by $self->enableEditColumn
        # Selects rooms, exits, room tags, room guilds or labels (or everything) in all regions
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $type   - Set to 'room', 'exit', 'room_tag', 'room_guild', or 'label'. If not defined,
        #               selects everything
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $type, $check) = @_;

        # Local variables
        my (
            $count,
            @roomList, @exitList, @roomTagList, @roomGuildList, @labelList,
            %roomHash, %exitHash, %roomTagHash, %roomGuildHash, %labelHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->selectInMapCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Make sure there are no rooms, exits, room tags or labels selected
        $self->ivUndef('selectedRoom');
        $self->ivEmpty('selectedRoomHash');
        $self->ivUndef('selectedExit');
        $self->ivEmpty('selectedExitHash');
        $self->ivUndef('selectedRoomTag');
        $self->ivEmpty('selectedRoomTagHash');
        $self->ivUndef('selectedRoomGuild');
        $self->ivEmpty('selectedRoomGuildHash');
        $self->ivUndef('selectedLabel');
        $self->ivEmpty('selectedLabelHash');

        # Select all rooms, exits, room tags, room guilds and/or labels
        if (! defined $type || $type eq 'room') {

            @roomList = $self->worldModelObj->ivValues('modelHash');
        }

        if (! defined $type || $type eq 'exit') {

            @exitList = $self->worldModelObj->ivValues('exitModelHash');
        }

        foreach my $regionmapObj ($self->worldModelObj->ivValues('regionmapHash')) {

            if (! defined $type || $type eq 'room_tag') {

                # ->gridRoomTagHash contains all the rooms with room tags
                # Get a list of world model numbers for the rooms containing room tag
                push (@roomTagList, $regionmapObj->ivValues('gridRoomTagHash'));
            }

            if (! defined $type || $type eq 'room_guild') {

                # ->gridRoomGuildHash contains all the rooms with room guilds
                # Get a list of world model numbers for the roomw containing room guilds
                push (@roomGuildList, $regionmapObj->ivValues('gridRoomGuildHash'));
            }

            if (! defined $type || $type eq 'label') {

                # ->gridLabelHash contains all the labels
                # Get a list of blessed references to GA::Obj::MapLabel objects
                push (@labelList, $regionmapObj->ivValues('gridLabelHash'));
            }
        }

        # The IVs that store selected objects behave differently when there is one selected object
        #   and when there is more than one. Count how many selected objects we have
        $count = (scalar @roomList) + (scalar @exitList) + (scalar @roomTagList)
                    + (scalar @roomGuildList) + (scalar @labelList);

        # Select a single object...
        if ($count == 1) {

            if (@roomList) {

                # Select the blessed reference of a GA::ModelObj::Room
                $self->ivPoke(
                    'selectedRoom',
                    $self->worldModelObj->ivShow('modelHash', $roomList[0]),
                );

            } elsif (@exitList) {

                # Select the blessed reference of a GA::Obj::Exit
                $self->ivPoke(
                    'selectedExit',
                    $self->worldModelObj->ivShow('exitModelHash', $exitList[0]),
                );

            } elsif (@roomTagList) {

                # Select the blessed reference of the GA::ModelObj::Room which contains the room
                #   tag
                $self->ivPoke(
                    'selectedRoomTag',
                    $self->worldModelObj->ivShow('modelHash', $roomTagList[0]),
                );

            } elsif (@roomGuildList) {

                # Select the blessed reference of the GA::ModelObj::Room which contains the room
                #   guild
                $self->ivPoke(
                    'selectedRoomGuild',
                    $self->worldModelObj->ivShow('modelHash', $roomGuildList[0]),
                );

            } elsif (@labelList) {

                # Select the blessed reference of a GA::Obj::MapLabel
                $self->ivPoke('selectedLabel', $labelList[0]);
            }

        # ...or select multiple objects
        } else {

            # (For speed, update local variable hashes, before storing the whole hash(es) in IVs
            foreach my $number (@roomList) {

                $roomHash{$number} = $self->worldModelObj->ivShow('modelHash', $number);
            }

            foreach my $number (@exitList) {

                $exitHash{$number} = $self->worldModelObj->ivShow('exitModelHash', $number);
            }

            foreach my $number (@roomTagList) {

                $roomTagHash{$number} = $self->worldModelObj->ivShow('modelHash', $number);
            }

            foreach my $number (@roomGuildList) {

                $roomGuildHash{$number} = $self->worldModelObj->ivShow('modelHash', $number);
            }

            foreach my $obj (@labelList) {

                $labelHash{$obj->id} = $obj;
            }

            # Update the IVs
            $self->ivPoke('selectedRoomHash', %roomHash);
            $self->ivPoke('selectedExitHash', %exitHash);
            $self->ivPoke('selectedRoomTagHash', %roomTagHash);
            $self->ivPoke('selectedRoomGuildHash', %roomGuildHash);
            $self->ivPoke('selectedLabelHash', %labelHash);
        }

        # Redraw the current region's current level now, and mark all other levels in the same
        #   region (as well as any other regions for which a parchment exists) as needing to be
        #   drawn
        $self->drawAllRegions();

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        return 1;
    }

    sub selectRoomCallback {

        # Called by $self->enableEditColumn
        # Selects certain rooms
        #
        # Expected arguments
        #   $type   - Set to 'no_title', 'no_descrip', 'no_title_descrip', 'title_descrip',
        #               'no_visit_char', 'no_visit_all', 'visit_char', 'visit_all', 'checkable'
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if no matching
        #       rooms are found
        #   1 otherwise

        my ($self, $type, $check) = @_;

        # Local variables
        my (
            $title, $msg,
            @roomList, @selectList,
            %dirHash,
        );

        # Check for improper arguments
        if (! defined $type || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->selectRoomCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Import a list of all rooms in the current region (for convenience)
        foreach my $roomNum ($self->currentRegionmap->ivValues('gridRoomHash')) {

            my $roomObj = $self->worldModelObj->ivShow('roomModelHash', $roomNum);
            if ($roomObj) {

                push (@roomList, $roomObj);
            }
        }

        # Compile a list of matching rooms
        if ($type eq 'no_title') {

            $title = 'Select rooms with no titles';

            foreach my $roomObj (@roomList) {

                if ($roomObj && ! $roomObj->titleList) {

                    push (@selectList, $roomObj, 'room');
                }
            }

        } elsif ($type eq 'no_descrip') {

            $title = 'Select rooms with no descriptions';

            foreach my $roomObj (@roomList) {

                if (! $roomObj->descripHash) {

                    push (@selectList, $roomObj, 'room');
                }
            }

        } elsif ($type eq 'no_title_descrip') {

            $title = 'Select rooms with no titles or descriptions';

            foreach my $roomObj (@roomList) {

                if (! $roomObj->titleList && ! $roomObj->descripHash) {

                    push (@selectList, $roomObj, 'room');
                }
            }

        } elsif ($type eq 'title_descrip') {

            $title = 'Select rooms with titles and descriptions';

            foreach my $roomObj (@roomList) {

                if ($roomObj->titleList && $roomObj->descripHash) {

                    push (@selectList, $roomObj, 'room');
                }
            }

        } elsif ($type eq 'no_visit_char' && $self->session->currentChar) {

            $title = 'Select unvisited rooms';

            foreach my $roomObj (@roomList) {

                if (! $roomObj->ivShow('visitHash', $self->session->currentChar->name)) {

                    push (@selectList, $roomObj, 'room');
                }
            }

        } elsif ($type eq 'no_visit_all') {

            $title = 'Select unvisited rooms';

            foreach my $roomObj (@roomList) {

                if (! $roomObj->visitHash) {

                    push (@selectList, $roomObj, 'room');
                }
            }

        } elsif ($type eq 'visit_char' && $self->session->currentChar) {

            $title = 'Select visited rooms';

            foreach my $roomObj (@roomList) {

                if ($roomObj->ivShow('visitHash', $self->session->currentChar->name)) {

                    push (@selectList, $roomObj, 'room');
                }
            }

        } elsif ($type eq 'visit_all') {

            $title = 'Select visited rooms';

            foreach my $roomObj (@roomList) {

                if ($roomObj->visitHash) {

                    push (@selectList, $roomObj, 'room');
                }
            }

        } elsif ($type eq 'checkable') {

            $title = 'Select rooms with checkable directions';

            # Get a hash of custom primary directions which can be checked in each room
            %dirHash = $self->worldModelObj->getCheckableDirs($self->session);

            foreach my $roomObj (@roomList) {

                my %checkHash = %dirHash;

                foreach my $dir ($roomObj->ivKeys('checkedDirHash')) {

                    delete $checkHash{$dir};
                }

                foreach my $dir ($roomObj->sortedExitList) {

                    delete $checkHash{$dir};
                }

                if (%checkHash) {

                    push (@selectList, $roomObj, 'room');
                }
            }
        }

        # Show a confirmation in both cases. Even if matching rooms are found, they might not be
        #   visible
        if (! @selectList) {

            $self->showMsgDialogue(
                $title,
                'error',
                'No matching rooms found in this region',
                'ok',
            );

            return undef;

        } else {

            # Make sure nothing is selected
            $self->setSelectedObj();

            # Select matching rooms
            $self->setSelectedObj(\@selectList, TRUE);

            if (@selectList == 2) {
                $msg = '1 matching room found in this region';
            } else {
                $msg = ((scalar @selectList) / 2) . ' matching rooms found in this region';
            }

            $self->showMsgDialogue(
                $title,
                'info',
                $msg,
                'ok',
            );

            return 1;
        }
    }

    sub selectExitTypeCallback {

        # Called by $self->enableEditColumn
        # Scours the current map, looking for unallocated, unallocatable, uncertain or incomplete
        #   exits (or all four of them together)
        # Once found, selects both the exits and the parent rooms
        # Finally, displays a 'dialogue' window showing how many were found
        #
        # Expected arguments
        #   $type   - What to search for. Must be either 'in_rooms' 'unallocated', 'unallocatable,
        #               'uncertain', 'incomplete', 'all_above', 'impass', 'mystery', 'region' or
        #               'super'
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $type, $check) = @_;

        # Local variables
        my (
            $obj, $number, $title, $text,
            @exitNumList, @exitObjList,
            %roomHash, %selectExitHash, %selectRoomHash,
        );

        # Check for improper arguments
        if (
            ! defined $type
            || (
                $type ne 'in_rooms' && $type ne 'uncertain' && $type ne 'incomplete'
                && $type ne 'unallocated' && $type ne 'unallocatable' && $type ne 'all_above'
                && $type ne 'impass' && $type ne 'mystery' && $type ne 'region' && $type ne 'super'
            ) || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->selectExitTypeCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Compile a list of all exit objects drawn in this map
        @exitNumList = $self->currentRegionmap->ivKeys('gridExitHash');
        foreach my $exitNum (@exitNumList) {

            push (@exitObjList, $self->worldModelObj->ivShow('exitModelHash', $exitNum));
        }

        if ($type eq 'in_rooms') {

            # Import a list of all the selected rooms, and copy them into a hash
            if ($self->selectedRoom) {
                $roomHash{$self->selectedRoom->number} = $self->selectedRoom;
            } else {
                %roomHash = $self->selectedRoomHash;
            }

            # Check each selected room in turn, marking all of its exits for selection
            foreach my $roomObj (values %roomHash) {

                foreach my $exitNum ($roomObj->ivValues('exitNumHash')) {

                    my $exitObj = $self->worldModelObj->ivShow('exitModelHash', $exitNum);
                    if ($exitObj) {

                        $selectExitHash{$exitNum} = $exitObj;
                    }
                }
            }

        } else {

            # Check each exit in turn. If it's one of the exits for which we're looking, mark it for
            #   selection
            OUTER: foreach my $exitObj (@exitObjList) {

                if (
                    (
                        ($type eq 'uncertain' || $type eq 'all_above')
                        && $exitObj->destRoom
                        && (! $exitObj->twinExit)
                        && (! $exitObj->oneWayFlag)
                        && (! $exitObj->retraceFlag)
                    ) || (
                        ($type eq 'incomplete' || $type eq 'all_above')
                        && (! $exitObj->destRoom && $exitObj->randomType eq 'none')
                    ) || (
                        ($type eq 'unallocated' || $type eq 'all_above')
                        && (
                            $exitObj->drawMode eq 'temp_alloc'
                            || $exitObj->drawMode eq 'temp_unalloc'
                        )
                    ) || (
                        $type eq 'unallocatable' && $exitObj->drawMode eq 'temp_unalloc'
                    ) || (
                        $type eq 'impass' && $exitObj->exitOrnament eq 'impass'
                    ) || (
                        $type eq 'mystery' && $exitObj->exitOrnament eq 'mystery'
                    ) || (
                        $type eq 'region' && $exitObj->regionFlag
                    ) || (
                        $type eq 'super' && $exitObj->superFlag
                    )
                ) {
                    $selectExitHash{$exitObj->number} = $exitObj;
                    $selectRoomHash{$exitObj->parent}
                        = $self->worldModelObj->ivShow('modelHash', $exitObj->parent);
                }
            }
        }

        # If anything was marked for selection...
        if (%selectExitHash) {

            # Since we're going to redraw everything on the map, we'll sidestep the normal call to
            #   $self->setSelectedObj, and set IVs directly

            # Make sure there are no rooms, exits, room tags or labels selected
            $self->ivUndef('selectedRoom');
            $self->ivEmpty('selectedRoomHash');
            $self->ivUndef('selectedExit');
            $self->ivEmpty('selectedExitHash');
            $self->ivUndef('selectedRoomTag');
            $self->ivEmpty('selectedRoomTagHash');
            $self->ivUndef('selectedLabel');
            $self->ivEmpty('selectedLabelHash');

            # Select rooms and exits. (There must be at least one of each, if there are any, so we
            #   don't ever set ->selectedRoom or ->selectedExit)
            if (scalar (keys %selectRoomHash) > 1) {

                $self->ivPoke('selectedRoomHash', %selectRoomHash);

            } elsif (%selectRoomHash) {

                ($number) = keys %selectRoomHash;
                $obj = $self->worldModelObj->ivShow('modelHash', $number);
                $self->ivAdd('selectedRoomHash', $number, $obj);
            }

            if (scalar (keys %selectExitHash) > 1) {

                $self->ivPoke('selectedExitHash', %selectExitHash);

            } elsif (%selectExitHash) {

                ($number) = keys %selectExitHash;
                $obj = $self->worldModelObj->ivShow('exitModelHash', $number);
                $self->ivAdd('selectedExitHash', $number, $obj);
            }

            # Redraw the current region's current level now, and mark all other levels in the same
            #   region (as well as any other regions for which a parchment exists) as needing to be
            #   drawn
            $self->drawAllRegions();

            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $self->restrictWidgets();

            # Show a confirmation of how many uncertain/incomplete exits were found
            if ($type eq 'in_rooms') {

                $title = 'Selected rooms\' exits';

                # (This will be a shorter string, so keep it on one line)
                if (scalar (keys %selectExitHash) == 1) {
                    $text = 'Found 1 exit in ';
                } else {
                    $text = 'Found ' . scalar (keys %selectExitHash) . ' exits in ';
                }

                if (scalar (keys %roomHash) == 1) {
                    $text .= '1 selected room';
                } else {
                    $text .= scalar (keys %roomHash) . ' selected rooms';
                }

                $text .= ' in this region';

            } else {

                if ($type eq 'all_above') {

                    $title = 'Select exits';

                    # (This will be a longer string, so spread it across two lines)
                    if (scalar (keys %selectExitHash) == 1) {

                        $text = "Found 1 unallocated/uncertain/incomplete exit\n";

                    } else {

                        $text = "Found " . scalar (keys %selectExitHash) . " unallocated/uncertain/"
                                . "incomplete exits\n";
                    }

                } else {

                    $title = 'Select ' . $type . ' exits';

                    # (This will be a shorter string, so keep it on one line)
                    if (scalar (keys %selectExitHash) == 1) {
                        $text = "Found 1 $type exit ";
                    } else {
                        $text = "Found " . scalar (keys %selectExitHash) . " $type exits ";
                    }
                }

                if (scalar (keys %selectRoomHash) == 1) {
                    $text .= 'in 1 room';
                } else {
                    $text .= 'spread across ' . scalar (keys %selectRoomHash) . ' rooms';
                }

                $text .= ' in this region';
            }

            $self->showMsgDialogue(
                $title,
                'info',
                $text,
                'ok',
                undef,
                TRUE,           # Preserve newline characters in $text
            );

        } else {

            # Show a confirmation that there are no uncertain/incomplete exits in this map
            if ($type eq 'all_above') {

                $title = 'Select exits';
                $text = 'There are no more unallocated, uncertain or incomplete exits in this'
                            . ' region';

            } else {

                $title = 'Select ' . $type . ' exits';
                $text = 'There are no more ' . $type . ' exits in this region';
            }

            $self->showMsgDialogue(
                $title,
                'info',
                $text,
                'ok',
            );
        }

        return 1;
    }

    sub findRoomCallback {

        # Called by $self->enableEditColumn
        # Prompts the user to enter the world model number of a room, and then selects the room
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the user declines to specify a room number
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($msg, $choice, $obj, $num, $regionObj);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->findRoomCallback', @_);
        }

        # (No standard callback checks for this function)

        # Check that the world model isn't empty
        if (! $self->worldModelObj->modelHash) {

            return $self->showMsgDialogue(
                'Find room',
                'error',
                'The world model is currently empty',
                'ok',
            );
        }

        # Prompt the user for a room number
        $msg = 'Enter a room number';
        if ($self->worldModelObj->modelObjCount > 1) {
            $msg .= ' (range 1-' . $self->worldModelObj->modelObjCount . '), or enter a room tag';
        } else {
            $msg .= ' or enter a room tag';
        }

        $choice = $self->showEntryDialogue(
            'Find room',
            $msg,
        );

        if (! defined $choice) {

            return undef;
        }

        # Is $choice a model number?
        if ($axmud::CLIENT->intCheck($choice, 1)) {

            # Does the corresponding world model object exist?
            if (! $self->worldModelObj->ivExists('modelHash', $choice)) {

                return $self->showMsgDialogue(
                    'Find room',
                    'error',
                    'There is no world model object #' . $choice,
                    'ok',
                );

            } else {

                $obj = $self->worldModelObj->ivShow('modelHash', $choice);
            }

        } else {

            # Find the room tag
            $num = $self->worldModelObj->ivShow('roomTagHash', lc($choice));
            if (defined $num) {

                $obj = $self->worldModelObj->ivShow('modelHash', $num);

            } else {

                return $self->showMsgDialogue(
                    'Find room',
                    'error',
                    'There is no room tagged \'' . $choice . '\'',
                    'ok',
                );
            }
        }

        if ($obj->category ne 'room') {

            if ($obj->category eq 'armour') {

                $msg = 'The world model object #' . $obj->number . ' isn\'t a room (but an '
                        . $obj->category . ')';

            } else {

                $msg = 'The world model object #' . $obj->number . ' isn\'t a room (but a '
                    . $obj->category . ')';
            }

            return $self->showMsgDialogue(
                'Find room',
                'error',
                $msg,
                'ok',
            );
        }

        if (! defined $obj->xPosBlocks) {

            # Room not in a regionmap - very unlikely, but we'll display a message anyway
            return $self->showMsgDialogue(
                'Find room',
                'error',
                'The world model object #' . $obj->number . ' exists, but isn\'t on the map',
                'ok',
            );
        }

        # If there isn't a current regionmap, show the one containing the room
        $regionObj = $self->worldModelObj->ivShow('modelHash', $obj->parent);
        if (! $self->currentRegionmap && $regionObj) {

            $self->setCurrentRegion($regionObj->name);
        }

        # If there is (now) a current regionmap, select the room (even if it's not in the same
        #   region)
        if ($self->currentRegionmap) {

            $self->setSelectedObj(
                [$obj, 'room'],
                FALSE,          # Select this object; unselect all other objects
            );

            # Centre the map on the room
            $self->centreMapOverRoom($self->selectedRoom);
        }

        # Prepare a message to display
        $msg = "World model room #" . $obj->number . "\n\n";

        $regionObj = $self->worldModelObj->ivShow('modelHash', $obj->parent);
        if ($regionObj) {
            $msg .= "Region: '" . $regionObj->name . "'\n";
        } else {
            $msg .= "Region: <none>\n";
        }

        $msg .= "X-pos: " . $obj->xPosBlocks . "\n";
        $msg .= "Y-pos: " . $obj->yPosBlocks . "\n";
        $msg .= "Level: " . $obj->zPosBlocks;

        # Display info about the room
        return $self->showMsgDialogue(
            'Find room',
            'info',
            $msg,
            'ok',
            undef,
            TRUE,           # Preserve newline characters in $msg
        );
    }

    sub findExitCallback {

        # Called by $self->enableEditColumn
        # Prompts the user to enter the exit model number of an exit, and then selects the exit
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the user declines to specify an room number
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($msg, $number, $exitObj, $roomObj, $regionObj);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->findExitCallback', @_);
        }

        # (No standard callback checks for this function)

        # Check that the exit model isn't empty
        if (! $self->worldModelObj->exitModelHash) {

            return $self->showMsgDialogue(
                'Find exit',
                'error',
                'The exit model is currently empty',
                'ok',
            );
        }

        # Prompt the user for an exit number
        $msg = 'Enter the exit number';
        if ($self->worldModelObj->exitObjCount > 1) {

            $msg .= ' (range 1-' . $self->worldModelObj->exitObjCount . ')';
        }

        $number = $self->showEntryDialogue(
            'Find exit',
            $msg,
        );

        # We need a positive integer
        if (! $axmud::CLIENT->intCheck($number, 1)) {

            # Do nothing
            return undef;
        }

        # Does the corresponding exit model object exist?
        if (! $self->worldModelObj->ivExists('exitModelHash', $number)) {

            return $self->showMsgDialogue(
                'Find exit',
                'error',
                'There is no exit model object #' . $number,
                'ok',
            );

        } else {

            # Get the blessed reference of the exit object and its parent room object
            $exitObj = $self->worldModelObj->ivShow('exitModelHash', $number);
            $roomObj = $self->worldModelObj->ivShow('modelHash', $exitObj->parent);
        }

        if (! defined $roomObj->xPosBlocks) {

            # Parent room not in a regionmap - rather unlikely, but we'll display a message anyway
            return $self->showMsgDialogue(
                'Find exit',
                'error',
                'The exit\'s parent room (#' . $roomObj->number . ') exists, but isn\'t on the map',
                'ok',
            );
        }

        # If there isn't a current regionmap, show the one containing the exit
        $regionObj = $self->worldModelObj->ivShow('modelHash', $roomObj->parent);
        if (! $self->currentRegionmap && $regionObj) {

            $self->setCurrentRegion($regionObj->name);
        }

        # If there is (now) a current regionmap, select the exit (even if it's not in that region)
        if ($self->currentRegionmap) {

            $self->setSelectedObj(
                [$exitObj, 'exit'],
                FALSE,          # Select this object; unselect all other objects
            );

            # Centre the map on the parent room
            $self->centreMapOverRoom($roomObj);
        }

        # Prepare a message to display
        $msg = "Exit model object #" . $number . "\n\n";
        $msg .= "Dir: " . $exitObj->dir . "\n";
        if ($exitObj->mapDir) {
            $msg .= "Map dir: " . $exitObj->mapDir . "\n";
        } else {
            $msg .= "Map dir: unallocatable\n";
        }

        $msg .= "Parent room: #" . $roomObj->number . "\n";

        if ($regionObj) {
            $msg .= "Region: '" . $regionObj->name . "'\n";
        } else {
            $msg .= "Region: <none>\n";
        }

        $msg .= "X-pos: " . $roomObj->xPosBlocks . "\n";
        $msg .= "Y-pos: " . $roomObj->yPosBlocks . "\n";
        $msg .= "Level: " . $roomObj->zPosBlocks;

        # Display info about the exit
        return $self->showMsgDialogue(
            'Find exit',
            'info',
            $msg,
            'ok',
            undef,
            TRUE,           # Preserve newline characters in $msg
        );
    }

    sub resetRoomDataCallback {

        # Called by $self->enableEditColumn
        # Resets data in one or more rooms
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the user
        #       declines to continue the operation or if they specify no rooms
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $choice, $choice2, $response,
            @list, @comboList, @list2, @comboList2, @roomList,
            %comboHash, %comboHash2,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetRoomDataCallback', @_);
        }

        # (No standard callback check)

        # Prepare combobox lists
        @list = (
            'Room titles'               => 'title',
            'Verbose descriptions'      => 'descrip',
            'Room tags'                 => 'room_tag',
            'Room guilds'               => 'room_guild',
            'Room flags'                => 'room_flag',
            'Room commands'             => 'room_cmd',
            'Unspecified room patterns' => 'unspecified',
            'Exit/depature patterns'    => 'exit_depart',
            'Checked directions'        => 'checked_dir',
            'Axbasic scripts'           => 'script',
            'Character visits'          => 'char_visit',
            'Exclusive profiles'        => 'exclusive',
            'Analysed nouns/adjectives' => 'noun_adj',
            'Search results'            => 'search',
            'Remote data (MSDP/MXP)'    => 'remote',
            'Source code path'          => 'path',
            'All of the above'          => 'all_data',
        );

        do {

            my ($descrip, $arg);

            $descrip = shift @list;
            $arg = shift @list;

            push (@comboList, $descrip);
            $comboHash{$descrip} = $arg;

        } until (! @list);


        if ($self->currentRegionmap) {

            if ($self->mapObj->currentRoom) {

                push (@list2, 'Current room', 'current');
            }

            if ($self->selectedRoom || $self->selectedRoomHash) {

                push (@list2, 'Selected rooms', 'selected');
            }

            push (@list2, 'Rooms in this region', 'region');
        }

        push (@list2, 'Rooms in all regions', 'all_rooms');

        do {

            my ($descrip, $arg);

            $descrip = shift @list2;
            $arg = shift @list2;

            push (@comboList2, $descrip);
            $comboHash2{$descrip} = $arg;

        } until (! @list2);

        # Prompt the user to specify which data in which rooms is to be reset
        ($choice, $choice2) = $self->showDoubleComboDialogue(
            'Reset room data',
            'Choose what kind of data to reset',
            'Choose which rooms to reset',
            \@comboList,
            \@comboList2,
        );

        if (! defined $choice) {

            return undef;
        }

        # Convert the description 'Rooms in all regions' to the argument 'all_rooms', etc
        $choice = $comboHash{$choice};
        $choice2 = $comboHash2{$choice2};

        # If more than one room is to be reset, get a confirmation before doing anything
        if ($choice2 eq 'current') {

            if ($self->mapObj->currentRoom) {

                push (@roomList, $self->mapObj->currentRoom);
            }

        } elsif ($choice2 eq 'selected') {

            if ($self->selectedRoom) {
                push (@roomList, $self->selectedRoom);
            } else {
                push (@roomList, $self->ivValues('selectedRoomHash'));
            }

        } elsif ($choice2 eq 'region') {

            foreach my $roomNum ($self->currentRegionmap->ivValues('gridRoomHash')) {

                push (@roomList, $self->worldModelObj->ivShow('modelHash', $roomNum));
            }

        } elsif ($choice2 eq 'all_rooms') {

            push (@roomList, $self->worldModelObj->ivValues('roomModelHash'));
        }

        if (! @roomList) {

            $self->showMsgDialogue(
                'Reset room data',
                'error',
                'No matching rooms found',
                'ok',
            );

            return undef;

        } elsif (@roomList > 1) {

            $response = $self->showMsgDialogue(
                'Reset room data',
                'question',
                'This operation will reset data in ' . (scalar @roomList) . ' rooms. Are you sure'
                . ' you want to proceed?',
                'yes-no',
            );

            if (! defined $response || $response ne 'yes') {

                return undef;
            }
        }

        # Tell the world model to reset the specified data in the specified rooms
        if (
            ! $self->worldModelObj->resetRoomData(
                TRUE,               # Update automapper windows now
                $choice,
                @roomList,
            )
        ) {
            $self->showMsgDialogue(
                'Reset room data',
                'error',
                'Operation failed (internal error)',
                'ok',
            );

        } else {

            $self->showMsgDialogue(
                'Reset room data',
                'info',
                'Operation complete',
                'ok',
            );
        }

        return 1;
    }

    sub resetVisitsCallback {

        # Called by $self->enableEditColumn
        # Prompts the user to ask the character(s) and region(s) in which character visit counts
        #   should be reset
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the user chooses
        #       'cancel' in the 'dialogue' window or if no characters/regions are found
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $currentCharString, $allCharString, $unCharString, $thisRegionString, $allRegionString,
            $charChoice, $regionChoice, $unCharFlag, $roomCount, $deleteCount,
            @charNameList, @charStringList, @regionStringList, @charList, @regionList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetVisitsCallback', @_);
        }

        # (No standard callback check)

        # Prepare a list of character strings for a combobox
        foreach my $profObj ($self->session->ivValues('profHash')) {

            if (
                $profObj->category eq 'char'
                && (
                    ! $self->session->currentChar
                    || $self->session->currentChar ne $profObj
                )
            ) {
                push (@charNameList, $profObj->name);
            }
        }

        if ($self->session->currentChar) {

            $currentCharString = 'Current character (' . $self->session->currentChar->name . ')';
            push (@charStringList, $currentCharString);
        }

        $allCharString = 'All character profiles';
        $unCharString = 'All characters without profiles';
        push (@charStringList, $allCharString, $unCharString, @charNameList);

        # Prepare region strings for a second combobox
        if ($self->currentRegionmap) {

            $thisRegionString = 'Current region (' . $self->currentRegionmap->name . ')';
            push (@regionStringList, $thisRegionString);
        }

        $allRegionString = 'All regions';
        push (@regionStringList, $allRegionString);

        # Prompt the user to specify which characters/regions to reset
        ($charChoice, $regionChoice) = $self->showDoubleComboDialogue(
            'Reset character visits',
            'Select character(s)',
            'Select a region(s)',
            \@charStringList,
            \@regionStringList,
        );

        if (! defined $charChoice) {

            return undef;
        }

        # Compile a list of specified character(s)
        if (defined $currentCharString && $charChoice eq $currentCharString) {

            # Use the current character
            push (@charList, $self->session->currentChar->name);

        } elsif ($charChoice eq $allCharString) {

            # Use all character profiles - including the current character (if there is one), which
            #   isn't in @charNameList
            push (@charList, @charNameList);
            if ($self->session->currentChar) {

                push (@charList, $self->session->currentChar->name);
            }

        } else {

            # Use the specified character
            push (@charList, $charChoice);
        }

        # Compile a list of specified region(s)
        if (defined $thisRegionString && $regionChoice eq $thisRegionString) {

            push (@regionList, $self->currentRegionmap);

        } elsif ($regionChoice eq $allRegionString) {

            push (@regionList, $self->worldModelObj->ivValues('regionmapHash'));
        }

        # Set a handy flag if we're dealing with non-profile characters
        if ($charChoice eq $unCharString) {

            $unCharFlag = TRUE;
        }

        # Check that some characters and regions are specified
        if ((! $unCharFlag && ! @charList) || ! @regionList) {

            $self->showMsgDialogue(
                'Reset character visits',
                'error',
                'No characters and/or regions found',
                'ok',
            );

            return undef;
        }

        $roomCount = 0;
        $deleteCount = 0;

        # Deal with non-profile characters
        if ($unCharFlag) {

            foreach my $regionmapObj (@regionList) {

                foreach my $roomNum ($regionmapObj->ivValues('gridRoomHash')) {

                    my $roomObj = $self->worldModelObj->ivShow('modelHash', $roomNum);
                    $roomCount++;

                    foreach my $char ($roomObj->ivKeys('visitHash')) {

                        my $profObj = $self->session->ivShow('profHash', $char);

                        if (! $profObj || $profObj->category ne 'char') {

                            # The character which visited this room no longer exists as a character
                            #   profile
                            $self->worldModelObj->resetVisitCount(
                                TRUE,       # Update Automapper windows now
                                $roomObj,
                                $char,
                            );

                            $deleteCount++;
                        }
                    }
                }
            }

        # Deal with profile characters
        } else {

            foreach my $regionmapObj (@regionList) {

                foreach my $roomNum ($regionmapObj->ivValues('gridRoomHash')) {

                    my $roomObj = $self->worldModelObj->ivShow('modelHash', $roomNum);
                    $roomCount++;

                    foreach my $char (@charList) {

                        if ($roomObj->ivExists('visitHash', $char)) {

                            # Remove this character's visits from the room
                            $self->worldModelObj->resetVisitCount(
                                TRUE,       # Update Automapper windows now
                                $roomObj,
                                $char,
                            );

                            $deleteCount++;
                        }
                    }
                }
            }
        }

        # Show confirmation
        return $self->showMsgDialogue(
            'Reset character visits',
            'info',
            'Operation complete (rooms: ' . $roomCount . ', records deleted: ' . $deleteCount . ')',
            'ok',
        );
    }

    # Menu 'View' column callbacks

    sub changeCharDrawnCallback {

        # Called by $self->enableViewColumn
        # In GA::Obj::WorldModel->roomInteriorMode 'visit_count', changes which character's visits
        #   are drawn
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $currentString, $choice, $redrawFlag, $choiceObj,
            @profList, @sortedList, @comboList,
            %comboHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->changeCharDrawnCallback',
                @_,
            );
        }

        # (No standard callback checks for this function)

        # Get a sorted list of character profiles, not including the current character (if any)
        foreach my $profObj ($self->session->ivValues('profHash')) {

            if (
                $profObj->category eq 'char'
                && (! $self->session->currentChar || $self->session->currentChar ne $profObj)
            ) {
                push (@profList, $profObj);
            }
        }

        @sortedList = sort {lc($a->name) cmp lc($b->name)} (@profList);

        # Prepare a list to show in a combo box. At the same time, compile a hash in the form:
        #   $hash{combo_box_string} = blessed_reference_of_equivalent_profile
        foreach my $profObj (@sortedList) {

            push (@comboList, $profObj->name);
            $comboHash{$profObj->name} = $profObj;
        }

        # Add the current character (if there is one) to top of the combo
        if ($self->session->currentChar) {

            $currentString = '<Use current character>';
            unshift (@comboList, $currentString);
        }

        # Don't prompt for a character, if there are none available
        if (! @comboList) {

            return $self->showMsgDialogue(
                'Select character',
                'error',
                'There are no character profiles available',
                'ok',
            );
        }

        # Prompt the user for a character
        $choice = $self->showComboDialogue(
            'Select character',
            'Select which character\'s visits to draw',
            \@comboList,
        );

        if ($choice) {

            if ($choice eq $currentString) {

                if (defined $self->showChar) {

                    $redrawFlag = TRUE;
                }

                # Use the current character profile (this IV uses the value 'undef' to mean the
                #   current character)
                $self->ivUndef('showChar');

            } else {

                $choiceObj = $comboHash{$choice};

                if (! defined $self->showChar || $self->showChar ne $choiceObj->name) {

                    $redrawFlag = TRUE;
                }

                # Use the specified character profile
                $self->ivPoke('showChar', $choiceObj->name);
            }

            # If we are drawing room interiors in 'visit_count' mode, redraw maps to show character
            #   visits for the selected character (but not if the character hasn't changed)
            if ($redrawFlag&& $self->worldModelObj->roomInteriorMode eq 'visit_count') {

                $self->drawAllRegions();
            }
        }

        return 1;
    }

    sub zoomCallback {

        # Called by $self->enableViewColumn
        # Zooms in or out on the map, for the current region only
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $zoom   - Set to 'in' or 'out' for zoom in/zoom out, or set to a number corresponding to
        #               the new value of GA::Obj::Regionmap->magnification (e.g. 2, 1, 0.5; if set
        #               to 'undef', the user is prompted for the magnification)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if there is no
        #       current regionmap, if the standard  magnification list is empty or if the user
        #       declines to specify a magnification, when prompted
        #   1 otherwise

        my ($self, $zoom, $check) = @_;

        # Local variables
        my (
            $index, $match, $currentMag, $newMag,
            @magList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->zoomCallback', @_);
        }

        # Standard callback check. Import the standard magnification list
        @magList = $self->constMagnifyList;
        # Perform the check
        if (
            ! $self->currentRegionmap
            || (
                defined $zoom
                && (
                    ($zoom eq 'out' && $self->currentRegionmap->magnification <= $magList[0])
                    || ($zoom eq 'in' && $self->currentRegionmap->magnification >= $magList[-1])
                )
            )
        ) {
            return undef;
        }

        # If the tooltips are visible, hide them
        $self->hideTooltips();

        # Don't do anything if there is no current regionmap (possible when the user is using the
        #   mouse scroll button) or when the magnification list is empty (no reason why it should
        #   be)
        if (! $self->currentRegionmap || ! $self->constMagnifyList) {

            return undef;
        }

        # Import the current regionmap's current magnification
        $currentMag = $self->currentRegionmap->magnification;

        if (defined $zoom && ($zoom eq 'in' || $zoom eq 'out')) {

            # The map's current magnification is stored in GA::Obj::Regionmap->magnification. The
            #   default value is 1
            # $self->constMagnifyList contains a standard list of magnifications in ascending order,
            #   e.g. (0.5, 1, 2)
            # If GA::Obj::Regionmap->magnification is in the standard list, then we use the previous
            #   (or next) value in the list
            # Otherwise, we find the previous (or next) value in the list as it would be, if
            #   GA::Obj::Regionmap->magnification were in it
            #
            # Try to find GA::Obj::Regionmap->magnification in the standard list, remembering the
            #   index at which it was found
            $index = -1;
            OUTER: foreach my $item ($self->constMagnifyList) {

                $index++;
                if ($magList[$index] == $currentMag) {

                    $match = $index;
                    last OUTER;
                }
            }

            if (! defined $match) {

                # GA::Obj::Regionmap->magnification isn't a standard value. Insert it into the
                #   list as long as it's not smaller than the smallest value or bigger than the
                #   biggest value
                # Try inserting it at the beginning...
                if ($currentMag < $magList[0]) {

                    # Use index 0
                    $match = 0;

                # Or at the end...
                } elsif ($currentMag > $magList[-1]) {

                    # Use last index
                    $match = (scalar @magList) - 1;

                # Or somewhere in the middle...
                } else {

                    OUTER: for ($index = 0; $index < ((scalar @magList) - 1); $index++) {

                        if (
                            $currentMag > $magList[$index]
                            && $currentMag < $magList[($index + 1)]
                        ) {
                            splice (@magList, ($index + 1), 0, $currentMag);

                            $match = $index + 1;
                            last OUTER;
                        }
                    }
                }
            }

            # This error message should be impossible...
            if (! defined $match) {

                return $self->sesion->writeError(
                    'Error dealing with map magnifications',
                    $self->_objClass . '->zoomCallback',
                );
            }

            # Now, zoom out (or in), if possible
            if ($zoom eq 'out') {

                if ($match > 0) {

                    $match--;
                }

            } elsif ($zoom eq 'in') {

                if ($match < ((scalar @magList) - 1)) {

                    $match++;
                }
            }

            # Set the new magnification
            $newMag = $magList[$match];

        } else {

            if (! defined $zoom) {

                # Prompt the user for a zoom factor
                $zoom = $self->showEntryDialogue(
                    'Enter zoom factor',
                    'Enter an integer (e.g. 33 for 33% zoom)',
                );

                # User pressed 'cancel' button
                if (! defined $zoom) {

                    return undef;

                # The calling function has supplied a zoom factor. Make sure it's valid
                } elsif (! $axmud::CLIENT->floatCheck($zoom, 0) || $zoom == 0) {

                    return $self->showMsgDialogue(
                        'Zoom',
                        'error',
                        'Illegal magnification \'' . $zoom . '\'% - must be an integer (e.g. 100,'
                        . ' 50, 200)',
                        'ok',
                    );
                }

                # Convert the zoom factor from a percentage to a number that can be stored in
                #   GA::Obj::Regionmap->magnification (e.g. convert 133.33% to 1.33)
#                $newMag = sprintf('%.2f', ($zoom / 100));
                $newMag = Math::Round::nearest(0.01, ($zoom / 100));

            } else {

                # $zoom is already set to the magnification
                $newMag = $zoom;
            }

            # Make sure the magnification is within limits
            if ($newMag < $magList[0] || $newMag > $magList[-1]) {

                return $self->showMsgDialogue(
                    'Zoom',
                    'error',
                    'Illegal magnification \'' . $zoom . '\' - use a number in the range '
                    . int($magList[0] * 100) . '-' . int($magList[-1] * 100). '%',
                    'ok',
                );
            }
        }

        # Set the new magnification; the called function updates every Automapper window using the
        #   current worldmodel
        $self->worldModelObj->setMagnification($self, $newMag);

        return 1;
    }

    sub changeLevelCallback {

        # Called by $self->enableViewColumn
        # Prompts the user for a new level in the current regionmap, then sets it as the currently-
        #   displayed level
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $level;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->changeLevelCallback', @_);
        }

        # (No standard callback checks for this function)

        # Prompt the user for a new level
        $level = $self->showEntryDialogue(
            'Change level',
            'Enter the new level number (\'ground\' level is 0)',
        );

        if (defined $level) {

            # Check that $level is a valid integer (positive, negative or 0)
            if (! ($level =~ m/^-?\d+$/)) {

                return $self->showMsgDialogue(
                    'Change level',
                    'error',
                    'Invalid level \'' . $level . '\' - you must use an integer',
                    'ok',
                );
            }

            # Set the new current level, which redraws the map
            $self->setCurrentLevel($level);
        }

        return 1;
    }

    # Menu 'Mode' column callbacks

    sub verboseCharsCallback {

        # Called by $self->enableModeColumn
        # Sets the number of characters at the beginning of a verbose description that are checked
        #   to match a world model room with the Locator's current room
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $number;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->verboseCharsCallback', @_);
        }

        # (No standard callback checks for this function)

        # Prompt for a new number of verbose characters to match
        $number = $self->showEntryDialogue(
            'Match verbose description',
            'Enter number of initial characters to match (0 = match whole description)',
            undef,
            $self->worldModelObj->matchDescripCharCount,
        );

        if ($axmud::CLIENT->intCheck($number, 0)) {

            $self->worldModelObj->set_matchDescripCharCount($number);
        }

        return 1;
    }

    sub repaintSelectedRoomsCallback {

        # Called by $self->enableModeColumn
        # 'Repaints' the selected room(s) by copying the values of certain IVs stored in the world
        #   model's painter object (a non-model GA::ModelObj::Room) to each selected room
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@roomList, @redrawList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->repaintSelectedRoomsCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap || (! $self->selectedRoom && ! $self->selectedRoomHash)) {

            return undef;
        }

        # Get a list of selected rooms
        @roomList = $self->compileSelectedRooms();

        foreach my $roomObj (@roomList) {

            # Repaint each selected room
            $self->paintRoom(
                $roomObj,
                FALSE,      # Don't update Automapper windows yet
            );

            push (@redrawList, 'room', $roomObj);
        }

        # Redraw all the selected rooms, so the repainting is visible
        $self->worldModelObj->updateMaps(@redrawList);

        return 1;
    }

    sub autoCompareMaxCallback {

        # Called by $self->enablemodecolumn
        # Sets the maximum number of room comparisons when auto-comparing the Locator task's current
        #   room with rooms in the world model
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $number;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->autoCompareMaxCallback', @_);
        }

        # (No standard callback checks for this function)

        # Prompt for a new maximum
        $number = $self->showEntryDialogue(
            'Set limit on room comparisons',
            'When comparing the Locator task\'s current room against rooms in the model, set the'
            . ' maximum number of rooms to compare (0 - no limit)',
            undef,
            $self->worldModelObj->autoCompareMax,
        );

        if ($axmud::CLIENT->intCheck($number, 0)) {

            $self->worldModelObj->set_autoCompareMax($number);
        }

        return 1;
    }

    sub autoSlideMaxCallback {

        # Called by $self->enablemodecolumn
        # Sets the maximum distance for auto-slide operations
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $number;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->autoSlideMaxCallback', @_);
        }

        # (No standard callback checks for this function)

        # Prompt for a new maximum
        $number = $self->showEntryDialogue(
            'Set limit on slide distance',
            'When sliding a new room into an unoccupied gridblock, set the maximum slide distance'
            . ' (minimum value: 1)',
            undef,
            $self->worldModelObj->autoSlideMax,
        );

        if ($axmud::CLIENT->intCheck($number, 1)) {

            $self->worldModelObj->set_autoSlideMax($number);
        }

        return 1;
    }

    # Menu 'Regions' column callbacks

    sub newRegionCallback {

        # Called by $self->enableRegionsColumn
        # Adds a new region to the world model
        #
        # Expected arguments
        #   $tempFlag   - If set to TRUE, the new region is a temporary region (that should be
        #                   deleted, the next time the world model is loaded from file)
        #
        # Return values
        #   'undef' on improper arguments, if the new model object can't be created or if the user
        #       cancels the operation
        #   1 otherwise

        my ($self, $tempFlag, $check) = @_;

        # Local variables
        my (
            $successFlag, $name, $parentName, $width, $height, $parentNumber, $regionObj, $title,
            $regionmapObj,
        );

        # Check for improper arguments
        if (! defined $tempFlag || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->newRegionCallback', @_);
        }

        # (No standard callback checks for this function)

        # Prompt the user for a region name, parent region name and map size
        ($successFlag, $name, $parentName, $width, $height) = $self->promptNewRegion($tempFlag);
        if (! $successFlag) {

            # User cancelled the operation
            return undef;

        } else {

            # Check the name is not already in use
            if (defined $name && $self->worldModelObj->ivExists('regionmapHash', $name)) {

                if ($tempFlag) {
                    $title = 'New temporary region';
                } else {
                    $title = 'New region';
                }

                $self->showMsgDialogue(
                    $title,
                    'error',
                    'There is already a region called \'' . $name . '\'',
                    'ok',
                );

                return undef;
            }

            # If a parent was specified, find its world model number
            if (defined $parentName) {

                $parentNumber = $self->findRegionNum($parentName);
            }

            # Create the region object
            $regionObj = $self->worldModelObj->addRegion(
                $self->session,
                TRUE,               # Update Automapper windows now
                $name,              # May be an empty string
                $parentNumber,
                $tempFlag,
            );

            if (! $regionObj) {

                # Operation failed
                $self->showMsgDialogue(
                    'New region',
                    'error',
                    'Could not create the new region',
                    'ok',
                );

                return undef;

            } else {

                # Set the new region's size
                $regionmapObj = $self->worldModelObj->ivShow('regionmapHash', $regionObj->name);
                $regionmapObj->ivPoke('gridWidthBlocks', $width);
                $regionmapObj->ivPoke('gridHeightBlocks', $height);
                $regionmapObj->ivPoke('mapWidthPixels', $width * $regionmapObj->blockWidthPixels);
                $regionmapObj->ivPoke(
                    'mapHeightPixels',
                    $height * $regionmapObj->blockHeightPixels,
                );

                # Make it the selected region, and draw it on the map
                return $self->setCurrentRegion($regionObj->name);
            }
        }
    }

    sub renameRegionCallback {

        # Called by $self->enableRegionsColumn
        # Renames a world model region (and its tied GA::Obj::Regionmap)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if a region with
        #       the specified name already exists
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $name;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->renameRegionCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Prompt the user for a new region name
        $name = $self->showEntryDialogue(
            'Change region name',
            'Enter a new name for the \'' . $self->currentRegionmap->name . '\' region (max 32'
            . ' chars)',
            32,
        );

        if ($name) {

            # Check the name is not already in use
            if ($self->worldModelObj->ivExists('regionmapHash', $name)) {

                $self->showMsgDialogue(
                    'Change region name',
                    'error',
                    'There is already a region called \'' . $name . '\'',
                    'ok',
                );

                return undef;

            } else {

                # Rename the region
                $self->worldModelObj->renameRegion($self->currentRegionmap, $name);
            }
        }

        return 1;
    }

    sub changeRegionParentCallback {

        # Called by $self->enableRegionsColumn
        # Changes a region's parent region
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the parent
        #       region can't be set
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $modelNum, $noParentString, $parent, $parentNum,
            @list, @sortedList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->changeRegionParentCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Import the world model number of the current region
        $modelNum = $self->currentRegionmap->number;

        # Get a sorted list of references to world model regions
        @list = sort {lc($a->name) cmp lc($b->name)}
                    ($self->worldModelObj->ivValues('regionModelHash'));

        # Convert this list into region names, and remove the current region
        foreach my $regionObj (@list) {

            if ($regionObj->number ne $modelNum) {

                push (@sortedList, $regionObj->name);
            }
        }

        # Put an option for 'no parent region' at the top of the list
        $noParentString = '<no parent region>';
        unshift(@sortedList, $noParentString);

        # Prompt the user for a new parent region
        $parent = $self->showComboDialogue(
            'Change parent region',
            'Select the new parent region for \'' . $self->currentRegionmap->name . '\'',
            \@sortedList,
        );

        if ($parent) {

            if ($parent eq $noParentString) {

                # Set the region to have no parent
                if (!
                    $self->worldModelObj->setParent(
                        FALSE,      # No update
                        $modelNum,
                    )
                ) {
                    return undef;
                }

            } else {

                $parentNum = $self->findRegionNum($parent);

                # Set the new parent region
                if (
                    ! $self->worldModelObj->setParent(
                        FALSE,          # No update
                        $modelNum,
                        $parentNum,
                    )
                ) {
                    return undef;
                }
            }

            # Redraw the list of regions in the treeview. By using the current region as an
            #   argument, we make sure that it is visible in the treeview, by expanding the tree
            #   model as necessary
            $self->resetTreeView($self->currentRegionmap->name);
            # Make sure the current region is highlighted
            $self->treeViewSelectLine($self->currentRegionmap->name);
        }

        return 1;
    }

    sub addRegionSchemeCallback {

        # Called by $self->enableRegionsColumn
        # Attach a new region scheme
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the region
        #       scheme can't be added
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $choice;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->attachRegionSchemeCallback',
                @_,
            );
        }

        # (No standard callback check)

        # Prompt the user for the name of the new colour scheme
        $choice = $self->showEntryDialogue(
            'Add region colour scheme',
            'Enter a name for the new colour scheme (max 16 chars)',
            16,              # Maximum characters
        );

        if (defined $choice) {

            if ($self->worldModelObj->ivExists('regionSchemeHash', $choice)) {

                $self->showMsgDialogue(
                    'Add region colour scheme',
                    'error',
                    'There is already a region colour scheme called \'' . $choice . '\'',
                    'ok',
                );

            } else {

                $self->worldModelObj->addRegionScheme($self->session, $choice);

                $self->showMsgDialogue(
                    'Add region colour scheme',
                    'info',
                    'Added the region colour scheme \'' . $choice . '\'',
                    'ok',
                );

                return 1;
            }
        }

        return undef;
    }

    sub doRegionSchemeCallback {

        # Called by $self->enableRegionsColumn
        # Edits, renames or deletes a region scheme
        #
        # Expected arguments
        #   $type           - 'edit' to edit a region scheme, 'rename' to rename it, or 'delete' to
        #                       delete it
        # Optional arguments
        #   $regionmapObj   - If specified, manipulate the region scheme attached to this regionmap.
        #                       Otherwise, prompt the user for a region scheme
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the region
        #       scheme can't be manipulated
        #   1 otherwise

        my ($self, $type, $regionmapObj, $check) = @_;

        # Local variables
        my (
            $choice, $choice2,
            @list, @sortedList,
        );

        # Check for improper arguments
        if (
            ! defined $type || ($type ne 'edit' && $type ne 'rename' && $type ne 'delete')
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->doRegionSchemeCallback',
                @_,
            );
        }

        # (No standard callback check)

        if (! $regionmapObj) {

            # Prompt the user for an existing colour scheme. Remove 'default' if renaming/deleting
            foreach my $name ($self->worldModelObj->ivKeys('regionSchemeHash')) {

                if ($type eq 'edit' || $name ne 'default') {

                    push (@list, $name);
                }
            }

            @sortedList = sort {lc($a) cmp lc($b)} (@list);

            $choice = $self->showComboDialogue(
                ucfirst($type) . ' region colour scheme',
                'Select the colour scheme to ' . $type,
                \@sortedList,
            );

        } else {

            if (! defined $regionmapObj->regionScheme) {
                $choice = 'default';
            } else {
                $choice = $regionmapObj->regionScheme;
            }
        }

        if (defined $choice) {

            if ($type eq 'edit') {

                # Open up an 'edit' window to edit the object
                $self->createFreeWin(
                    'Games::Axmud::EditWin::RegionScheme',
                    $self,
                    $self->session,
                    'Edit region colour scheme \'' . $choice . '\'',
                    $self->worldModelObj->ivShow('regionSchemeHash', $choice),
                    FALSE,                          # Not temporary
                );

                return 1;

            } elsif ($type eq 'rename') {

                # Prompt the user for the new name
                $choice2 = $self->showEntryDialogue(
                    'Rename region colour scheme',
                    'Enter a new name for the colour scheme (max 16 chars)',
                    16,              # Maximum characters
                );

                if (
                    defined $choice2
                    && $self->worldModelObj->renameRegionScheme($self->session, $choice, $choice2)
                ) {
                    $self->showMsgDialogue(
                        ucfirst($type) . ' region colour scheme',
                        'info',
                        'Renamed \'' . $choice . '\' to \'' . $choice2 . '\'',
                        'ok',
                    );

                    return 1;
                }

            } else {

                # Delete the region scheme, and redraw regions in affected automapper windows
                $self->worldModelObj->deleteRegionScheme(TRUE, $choice);

                $self->showMsgDialogue(
                    ucfirst($type) . ' region colour scheme',
                    'info',
                    'Deleted \'' . $choice . '\'',
                    'ok',
                );
            }
        }

        return undef;
    }

    sub attachRegionSchemeCallback {

        # Called by $self->enableRegionsColumn
        # Attach a region scheme to the current regionmap
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the region
        #       scheme can't be attached
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $choice,
            @list, @sortedList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->attachRegionSchemeCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap || $self->worldModelObj->ivPairs('regionSchemeHash') < 2) {

            return undef;
        }

        # Prompt the user for a colour scheme to attach. Don't show the currently attached colour
        #   scheme (if any)
        foreach my $schemeObj ($self->worldModelObj->ivValues('regionSchemeHash')) {

            if (
                (
                    ! defined $self->currentRegionmap->regionScheme
                    && $schemeObj ne $self->worldModelObj->defaultSchemeObj
                ) || (
                    defined $self->currentRegionmap->regionScheme
                    && $self->currentRegionmap->regionScheme ne $schemeObj->name
                )
            ) {
                push (@list, $schemeObj->name);
            }
        }

        @sortedList = sort {lc($a) cmp lc($b)} (@list);

        $choice = $self->showComboDialogue(
            'Attach region colour scheme',
            'Select the colour scheme to attach to \'' . $self->currentRegionmap->name . '\'',
            \@sortedList,
        );

        if ($choice) {

            $self->worldModelObj->attachRegionScheme(
                TRUE,       # Update automapper windows
                $choice,
                $self->currentRegionmap->name,
            );
        }

        return 1;
    }

    sub detachRegionSchemeCallback {

        # Called by $self->enableRegionsColumn
        # Detaches a region scheme from the current regionmap
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the region
        #       scheme can't be detached
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->detachRegionSchemeCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap || ! defined $self->currentRegionmap->regionScheme) {

            return undef;
        }

        # Detach the region scheme
        $self->worldModelObj->detachRegionScheme(
            TRUE,       # Update automapper windows
            $self->currentRegionmap->name,
        );

        return 1;
    }

    sub convertRegionExitCallback {

        # Called by $self->enableRegionsColumn
        # Converts all region exits in the region into super-region exits (or deconverts all super-
        #   region exits into normal region exits)
        #
        # Expected arguments
        #   $convertFlag    - TRUE if converting region exits to super-region exits, FALSE if
        #                       deconverting super-region exits into region exits
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if there are no
        #       exits to convert (or deconvert)
        #   1 otherwise

        my ($self, $convertFlag, $check) = @_;

        # Local variables
        my (
            $title, $msg,
            @list, @twinList,
        );

        # Check for improper arguments
        if (! defined $convertFlag || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->changeRegionParentCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        if ($convertFlag) {

            $title = 'Convert region exits';

            # Get a list of (normal) super region exits
            foreach my $exitNum ($self->currentRegionmap->ivKeys('regionExitHash')) {

                my ($exitObj, $twinExitObj);

                $exitObj = $self->worldModelObj->ivShow('exitModelHash', $exitNum);

                if ($exitObj->regionFlag && ! $exitObj->superFlag) {

                    push (@list, $exitObj);
                    # Also convert the twin exit, if there is one
                    if ($exitObj->twinExit) {

                        $twinExitObj = $self->worldModelObj->ivShow(
                            'exitModelHash',
                            $exitObj->twinExit,
                        );

                        if ($twinExitObj->regionFlag && ! $twinExitObj->superFlag) {

                            push (@twinList, $twinExitObj);
                        }
                    }
                }
            }

            if (! @list) {

                # Display the 'dialogue' window
                $self->showMsgDialogue(
                    $title,
                    'error',
                    'There are no (normal) region exits in this region',
                    'ok',
                );

                return undef;
            }

            # Convert the exits
            foreach my $exitObj (@list, @twinList) {

                # Mark the exit as a super-region exit, and instruct the world model to update its
                #   automapper windows
                $self->worldModelObj->setSuperRegionExit(
                    $self->session,
                    TRUE,               # Update Automapper windows now
                    $exitObj,
                    FALSE,              # Not an exclusive super-region exit
                );
            }

            # Let the world model process any necessary changes
            $self->worldModelObj->updateRegionPaths($self->session);

            # Prepare the confirmation to display. Show the number of exits in the current region
            #   converted, not the total number (including all of their twin exits)
            if ((scalar @list) == 1) {
                $msg = 'Converted 1 region exit into a super-region exit';
            } else {
                $msg = 'Converted ' . (scalar @list) . ' region exits into super-region exits';
            }

        } else {

            $title = 'Deconvert super-region exits';

            # Get a list of super region exits
            foreach my $exitNum ($self->currentRegionmap->ivKeys('regionExitHash')) {

                my ($exitObj, $twinExitObj);

                $exitObj = $self->worldModelObj->ivShow('exitModelHash', $exitNum);

                if ($exitObj->regionFlag && $exitObj->superFlag) {

                    push (@list, $exitObj);
                    # Also deconvert the twin exit, if there is one
                    if ($exitObj->twinExit) {

                        $twinExitObj = $self->worldModelObj->ivShow(
                            'exitModelHash',
                            $exitObj->twinExit,
                        );

                        if ($twinExitObj->regionFlag && $twinExitObj->superFlag) {

                            push (@twinList, $twinExitObj);
                        }
                    }
                }
            }

            if (! @list) {

                # Display the 'dialogue' window
                $self->showMsgDialogue(
                    $title,
                    'error',
                    'There are no super-region exits in this region',
                    'ok',
                );

                return undef;
            }

            # Deconvert the exits
            foreach my $exitObj (@list, @twinList) {

                # Mark the exit as a normal region exit, and instruct the world model to update its
                #   automapper windows
                $self->worldModelObj->restoreSuperRegionExit(
                    TRUE,               # Update Automapper windows now
                    $exitObj,
                );
            }

            # Let the world model process any necessary changes
            $self->worldModelObj->updateRegionPaths($self->session);

            # Prepare the confirmation to display. Show the number of exits in the current region
            #   converted, not the total number (including all of their twin exits)
            if ((scalar @list) == 1) {

                $msg = 'Deonverted 1 super-region exit into a normal region exit';

            } else {

                $msg = 'Deconverted ' . (scalar @list)
                            . ' super-region exits into normal region exits';
            }
        }

        # Show a confirmation
        $self->showMsgDialogue(
            $title,
            'info',
            $msg,
            'ok',
        );

        return 1;
    }

    sub identifyRegionCallback {

        # Called by $self->enableRegionsColumn
        # Identifies the currently highlighted region (in the treeview)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($text, $regionObj, $parentObj);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->identifyRegionCallback', @_);
        }

        # Standard callback check
        if (! $self->treeViewSelectedLine) {

            return undef;
        }

        # Prepare the text to display
        $text = 'Currently highlighted region: \'' . $self->treeViewSelectedLine . '\'';

        $regionObj = $self->findRegionObj($self->treeViewSelectedLine);
        $text .= ' (#' . $regionObj->number . ')';

        if ($regionObj->parent) {

            $parentObj = $self->worldModelObj->ivShow('modelHash', $regionObj->parent);
            $text .= "\nParent region: \'" . $parentObj->name . '\' (#' . $parentObj->number . ')';
        }

        # Display the 'dialogue' window
        $self->showMsgDialogue(
            'Highlighted region',
            'info',
            $text,
            'ok',
            undef,
            TRUE,           # Preserve newline characters in $text
        );

        return 1;
    }

    sub editRegionCallback {

        # Called by $self->enableRegionsColumn
        # Opens a GA::EditWin::ModelObj::Region for the current region
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($number, $obj);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->editRegionCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Find the current regionmap's equivalent world model object
        $number = $self->currentRegionmap->number;
        if ($number) {

            $obj = $self->worldModelObj->ivShow('modelHash', $number);

            # Open up an 'edit' window to edit the object
            $self->createFreeWin(
                'Games::Axmud::EditWin::ModelObj::Region',
                $self,
                $self->session,
                'Edit ' . $obj->category . ' model object #' . $obj->number,
                $obj,
                FALSE,                          # Not temporary
            );
        }

        return 1;
    }

    sub regionScreenshotCallback {

        # Called by $self->enableRegionsColumn
        # Takes a screenshot of a portion of the regionmap (or the whole regionmap), at the
        #   currently displayed level, and saves it in the ../screenshots directory
        #
        # Expected arguments
        #   $type       - 'visible' for the visible portion of the canvas, 'occupied' for the
        #                   occupied portion of the canvas, and 'whole' for the whole canvas
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       declines to take the screenshot after a warning
        #   1 otherwise

        my ($self, $type, $check) = @_;

        # Local variables
        my (
            $xOffset, $yOffset, $xPos, $yPos, $left, $top, $right, $bottom, $width, $height, $msg,
            $result, $file, $path, $count,
        );

        # Check for improper arguments
        if (! defined $type || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->regionScreenshotCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # The menu column is presumably still open - which will get in the way of the screenshot.
        #   Give it a chance to close
        $axmud::CLIENT->desktopObj->updateWidgets(
            $self->_objClass . '->regionScreenshotCallback',
        );

        if ($type eq 'visible') {

            # Find the position of the top-left corner of the visible canvas
            ($xOffset, $yOffset, $xPos, $yPos) = $self->getMapPosn();
            $left = int($self->currentRegionmap->mapWidthPixels * $xPos);
            $top = int($self->currentRegionmap->mapHeightPixels * $yPos);
            # Import the size of the visible canvas
            $width = $self->canvasScrollerWidth;
            $height = $self->canvasScrollerHeight;

        } elsif ($type eq 'occupied') {

            # Find the extent of the occupied map, in pixels
            ($left, $right, $top, $bottom) = $self->findOccupiedMap();
            $width = $right - $left + 1;
            $height = $bottom - $top + 1;

        } else {

            # Import the size of the whole canvas
            $width = $self->currentRegionmap->mapWidthPixels;
            $height = $self->currentRegionmap->mapHeightPixels;
        }

        # For very large screenshots, display a warning before starting the operation
        if ($width * $height > 100_000_000) {

            $msg = 'This operation will produce a very large image (' . $width . 'x' . $height
                        . ' pixels). ' . 'Are you sure you want to continue?';

            $result = $self->showMsgDialogue(
                'Screenshot',
                'warning',
                $msg,
                'yes-no',
            );

            if ($result ne 'yes') {

                return undef;
            }
        }

        # For large-ish screenshots, show the pause window
        if ($width * $height > 5_000_000) {

            $self->showPauseWin();
        }

        # Take the screenshot
        my $surface = Cairo::ImageSurface->create('rgb24', $width, $height);
        my $cr = Cairo::Context->create($surface);
        $cr->rectangle(0, 0, $width, $height);
        $cr->set_source_rgb(1, 1, 1);
        $cr->fill();

        if ($type eq 'visible' || $type eq 'occupied') {

            $cr->translate(-$left, -$top);
        }

        $self->canvas->render($cr, undef, 1);

        my $loader = Gtk3::Gdk::PixbufLoader->new();
        $surface->write_to_png_stream (
            sub {
                my ($loader, $buffer) = @_;
                $loader->write([map ord, split //, $buffer]);
                return TRUE;
            },
            $loader,
        );
        $loader->close();

        my $pixbuf = $loader->get_pixbuf();

        $file = $self->currentRegionmap->name . '_level_' . $self->currentRegionmap->currentLevel;
        $path = $axmud::DATA_DIR . '/screenshots/' . $file . '.png';

        # If the file $path already exists, add a postscript to create a filepath that doesn't yet
        #   exist
        if (-e $path) {

            $count = 0;

            do {

                $count++;

                my $newFile = $file . '_(' . $count . ')';
                $path = $axmud::DATA_DIR . '/screenshots/' . $newFile . '.png';

            } until (! -e $path);
        }

        # Save the file as a .jpeg
        $pixbuf->save($path, 'png');

        # Make the pause window invisible
        $self->hidePauseWin();

        # Display a confirmation dialogue
        $self->showMsgDialogue(
            'Screenshot',
            'info',
            'Screenshot saved to ' . $path,
            'ok',
        );

        return 1;
    }

    sub removeRoomFlagsCallback {

        # Called by $self->enableRegionsColumn
        # Prompts the user to select a room flag to be removed from every room in the current region
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the user doesn't
        #       select a room flag, if the region has no rooms or if every room in the region has no
        #       room flags
        #   1 otherwise

        my ($self, $type, $check) = @_;

        # Local variables
        my (
            $choice, $msg, $count,
            @list, @sortedList, @nameList,
            %flagHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->removeRoomFlagsCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Check there are some rooms in the current region
        if (! $self->currentRegionmap->gridRoomHash) {

            $self->showMsgDialogue(
                'Remove room flags',
                'error',
                'There are no rooms in the current region',
                'ok',
            );

            return undef;
        }

        # Go through every room in the region, compiling a list of room flags actually in use
        foreach my $roomNum ($self->currentRegionmap->ivValues('gridRoomHash')) {

            my $roomObj = $self->worldModelObj->ivShow('modelHash', $roomNum);

            foreach my $flag ($roomObj->ivKeys('roomFlagHash')) {

                # Compile a hash containing one entry for each room flag used (regardless of whether
                #   it's used in one room or multiple rooms)
                $flagHash{$flag} = undef;
            }
        }

        if (! %flagHash) {

            $self->showMsgDialogue(
                'Remove room flags',
                'error',
                'No rooms in the current region are using room flags',
                'ok',
            );

            return undef;
        }

        # Get a list of the room flags in use, sorted by priority
        foreach my $roomFlag (keys %flagHash) {

            my $roomFlagObj = $self->worldModelObj->ivShow('roomFlagHash', $roomFlag);
            if ($roomFlagObj) {

                push (@list, $roomFlagObj);
            }
        }

        @sortedList = sort {$a->priority <=> $b->priority} (@list);
        foreach my $roomFlagObj (@sortedList) {

            push (@nameList, $roomFlagObj->name);
        }

        # Prompt the user to select one of the room flags
        $choice = $self->showComboDialogue(
            'Remove room flags',
            'Select which room flag should be removed from every room in this region',
            \@nameList,
        );

        if (! $choice) {

            return undef;

        } else {

            # Remove the room flag from each room in turn
            $count = $self->worldModelObj->removeRoomFlagInRegion($self->currentRegionmap, $choice);

            # Display a confirmation message
            $msg = 'Room flag \'' . $choice . '\' removed from ';
            if ($count == 1) {
                $msg .= '1 room in this region',
            } else {
                $msg .= $count . ' rooms in this region',
            }

            $self->showMsgDialogue(
                'Remove room flags',
                'info',
                $msg,
                'ok',
            );

            return 1;
        }
    }

    sub preDrawSizeCallback {

        # Called by $self->enableRegionsColumn
        # Prompts the user to set the minimum size for regions that should be pre-drawn when the
        #   automapper window opens
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the user declines to modify the current value
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $choice;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->preDrawSizeCallback', @_);
        }

        # (No standard callback check)

        # Prompt the user
        $choice = $self->showEntryDialogue(
            'Pre-drawn regions',
            'Set the minimum size (in rooms) of any regions that should be pre-drawn when the'
            . ' Automapper window opens (or use 0 to pre-draw all regions)',
            undef,              # No maximum characters
            $self->worldModelObj->preDrawMinRooms,
        );

        if (defined $choice && $axmud::CLIENT->intCheck($choice, 0)) {

            $self->worldModelObj->set_preDrawMinRooms($choice);
        }

        return 1;
    }

    sub preDrawRetainCallback {

        # Called by $self->enableRegionsColumn
        # Prompts the user to set the minimum size for drawn parchments that should be retained in
        #   memory when a new current region is set
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the user declines to modify the current value
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $choice;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->preDrawRetainCallback', @_);
        }

        # (No standard callback check)

        # Prompt the user
        $choice = $self->showEntryDialogue(
            'Retain drawn regions',
            'Set the minimum size (in rooms) of any regions that should be retained in memory when'
            . ' they\'re not visible (or use 0 to retain all drawn regions)',
            undef,              # No maximum characters
            $self->worldModelObj->preDrawRetainRooms,
        );

        if (defined $choice && $axmud::CLIENT->intCheck($choice, 0)) {

            $self->worldModelObj->set_preDrawRetainRooms($choice);
        }

        return 1;
    }

    sub preDrawSpeedCallback {

        # Called by $self->enableRegionsColumn
        # Prompts the user to set the (approximate) percentage of available processor time to be
        #   spent on pre-drawing operations
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the user declines to modify the current value
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $choice;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->preDrawSpeedCallback', @_);
        }

        # (No standard callback check)

        # Prompt the user
        $choice = $self->showEntryDialogue(
            'Pre-drawing speed',
            'Set the approximate percentage of processor time that should be spent on pre-drawing'
            . ' maps (use a value in the range 1-100)',
            undef,              # No maximum characters
            $self->worldModelObj->preDrawAllocation,
        );

        if (defined $choice && $axmud::CLIENT->intCheck($choice, 1, 100)) {

            $self->worldModelObj->set_preDrawAllocation($choice);
        }

        return 1;
    }

    sub redrawRegionsCallback {

        # Called by $self->enableRegionsColumn
        # Prompts the user to confirm that all drawn regions should be redrawn, then performs the
        #   operation
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the user declines to modify the current value
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $choice;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->redrawRegionsCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Prompt the user
        $choice = $self->showMsgDialogue(
            'Redraw drawn regions',
            'question',
            'This operation can reduce performance, perhaps for several minutes. Are you sure you'
            . ' want to proceed?',
            'yes-no',
        );

        if (defined $choice && $choice eq 'yes') {

            $self->redrawRegions();
        }
    }

    sub recalculatePathsCallback {

        # Called by $self->enableRegionsColumn
        # Recalculates region paths - paths between each room in the region which has a super-region
        #   exit, and every other room in the region which has a super-region exit (used for quick
        #   pathfinding across different regions)
        #
        # Expected arguments
        #   $type   - Which region to process: 'current' for the current regionmap, 'select' to
        #               prompt the user for a regionmap, 'all' to recalculate paths in all
        #               regionmaps, or 'exit' to recalculate region paths to and from the
        #               selected exit (only)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       declines to specify a region
        #   1 otherwise

        my ($self, $type, $check) = @_;

        # Local variables
        my (
            $choice, $count, $estimate, $msg,
            @nameList, @regionmapList,
        );

        # Check for improper arguments
        if (
            ! defined $type
            || ($type ne 'current' && $type ne 'select' && $type ne 'all' && $type ne 'exit')
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->recalculatePathsCallback',
                @_,
            );
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || ($type eq 'current' && ! $self->currentRegionmap->gridRoomHash)
            || ($type eq 'exit' && ! $self->selectedExit && ! $self->selectedExit->superFlag)
        ) {
            return undef;
        }

        # Recalculate paths in the current region
        if ($type eq 'current') {

            push (@regionmapList, $self->currentRegionmap);

        # Recalculate paths in a region specified by the user
        } elsif ($type eq 'select') {

            # Get a sorted list of references to world model regions
            @nameList = sort {lc($a) cmp lc($b)} ($self->worldModelObj->ivKeys('regionmapHash'));

            # Prompt the user for a region name
            $choice = $self->showComboDialogue(
                'Recalculate region paths',
                'Select the region whose paths should be recalculated',
                \@nameList,
            );

            if (! $choice) {

                return undef;

            } else {

                push (@regionmapList, $self->worldModelObj->ivShow('regionmapHash', $choice));
            }

        # Recalculate paths in all regions
        } elsif ($type eq 'all') {

            # For a large world model, prompt the user for confirmation
            if ($self->worldModelObj->modelActualCount > 3000) {

                $choice = $self->showMsgDialogue(
                    'Recalculate region paths',
                    'question',
                    'The operation to recalculate region paths across all regions may take some'
                    . ' time. Are you sure you want to continue?',
                    'yes-no',
                );

                if (! defined $choice || $choice ne 'yes') {

                    return undef;
                }
            }

            # Compile a list of regionmaps
            @regionmapList = $self->worldModelObj->ivValues('regionmapHash');
        }

        if ($type ne 'exit') {

            # Work out how many region paths there are likely to be
            $estimate = 0;
            foreach my $regionmapObj (@regionmapList) {

                my $exitCount = 0;

                # Count the number of super-region exits
                foreach my $exitNum ($regionmapObj->regionExitHash) {

                    my $exitObj = $self->worldModelObj->ivShow('exitModelHash', $exitNum);

                    if ($exitObj->superFlag) {

                        $exitCount++;
                    }
                }

                # If there are ten super-region exits, each individual exit has nine region paths
                #   joining it to every other super-region exit. We then double the number, because
                #   safe region paths are stored separately. So the estimated number of region
                #   paths is ((n-1) ^ 2 ), all multiplied by 2
                $estimate += (2 * (($exitCount - 1) ** 2));
            }

            # If the estimated number of paths is above the limit set by the world model, make the
            #   pause window visible for the duration of the recalculation
            if ($estimate > $self->worldModelObj->recalculatePauseNum) {

                $self->showPauseWin();
            }

            # Recalculate region paths for each region added to our list
            $count = 0;
            foreach my $regionmapObj (@regionmapList) {

                my $number = $self->worldModelObj->recalculateRegionPaths(
                    $self->session,
                    $regionmapObj,
                );

                if ($number) {

                    $count += $number;
                }
            }

            # Make the pause window invisible
            $self->hidePauseWin();

        } else {

            # Recalculate paths to/from the selected exit.
            $count = $self->worldModelObj->recalculateSpecificPaths(
                $self->session,
                $self->currentRegionmap,
                $self->selectedExit,
            );

            # In case the called function returns 'undef', $count still needs to be an integer
            if (! $count) {

                $count = 0;
            }

            # For the message we're about to compose, @regionmapList must contain the affected
            #   regionmap
            push (@regionmapList, $self->currentRegionmap);
        }

        # Display a popup showing the results
        $msg = 'Recalculation complete: ';

        if (! $count) {
            $msg .= 'no region paths found';
        } elsif ($count == 1) {
            $msg .= '1 region path found';
        } else {
            $msg .= $count . ' region paths found';
        }

        if (@regionmapList == 1) {
            $msg .= ' in 1 region.';
        } else {
            $msg .= ' in ' . scalar @regionmapList . ' regions.';
        }

        $self->showMsgDialogue(
            'Recalculate region paths',
            'info',
            $msg,
            'ok',
        );

        return 1;
    }

    sub locateCurrentRoomCallback {

        # Called by $self->enableRegionsColumn
        # Tries to find the current room by comparing the Locator task's current room with every
        #   room in the current region, in a specified region, or in all regions
        # If there's a single matching room, that room is set as the current room. If the single
        #   matching room is in a different region or level to the current one, the map is redrawn
        # If there are multiple matching rooms, those rooms are selected. If they are all in a
        #   different region or at a different level to the current one, the map is redrawn
        #
        # Expected arguments
        #   $type   - Where to search: 'current' for the current regionmap, 'select' to prompt the
        #               user for a regionmap, or 'all' to search in all regionmaps
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if there is no
        #       current regionmap, or if there is no Locator task (or the task doesn't know the
        #       current location), if the Locator's current room is dark or unspecified or if the
        #       user declines to continue
        #   1 otherwise

        my ($self, $type, $check) = @_;

        # Local variables
        my (
            $taskObj, $msg, $regionName, $regionmapObj, $choice, $matchObj, $regionObj,
            @roomList, @list, @regionList, @selectList, @modList, @newRegionList, @sortedList,
            %regionmapHash,
        );

        # Check for improper arguments
        if (
            ! defined $type || ($type ne 'current' && $type ne 'select' && $type ne 'all')
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->locateCurrentRoomCallback',
                @_,
            );
        }

        # Standard callback check
        if ($type eq 'current' && ! $self->currentRegionmap->gridRoomHash) {
            return undef;
        }

        # Import the Locator task
        $taskObj = $self->session->locatorTask;

        # If there is no Locator task, or if it doesn't know its location, display a warning
        if (! $taskObj || ! $taskObj->roomObj) {

            $msg = 'the Locator task isn\'t ready';

        # Also display a warning if the Locator's current room is dark or unspecified
        } elsif ($taskObj->roomObj->currentlyDarkFlag) {

            $msg = 'it is dark';

        } elsif ($taskObj->roomObj->unspecifiedFlag) {

            $msg = 'it is an unspecified room';
        }

        if ($msg) {

            $self->showMsgDialogue(
                'Locate current room',
                'error',
                'Can\'t locate the current room because ' . $msg,
                'ok',
            );

            return undef;
        }

        # Compile a list of rooms to search

        # Get rooms in the current region
        if ($type eq 'current') {

            # Get a list of rooms in the current region
            @roomList = $self->currentRegionmap->ivValues('gridRoomHash');

        # Get rooms in a region specified by the user
        } elsif ($type eq 'select') {

            # Get a sorted list of references to world model regions
            @list = sort {lc($a->name) cmp lc($b->name)}
                        ($self->worldModelObj->ivValues('regionModelHash'));

            # Convert this list into region names
            foreach my $regionObj (@list) {

                push (@regionList, $regionObj->name);
            }

            # Prompt the user for a region name
            $regionName = $self->showComboDialogue(
                'Select region',
                'Select the region in which to search',
                \@regionList,
            );

            if (! $regionName) {

                return undef;
            }

            # Find the matching regionmap
            $regionmapObj = $self->worldModelObj->ivShow('regionmapHash', $regionName);
            if (! $regionmapObj) {

                return undef;
            }

            # Get a list of rooms in the specified region
            @roomList = $regionmapObj->ivValues('gridRoomHash');

        # Locate rooms in all regions
        } elsif ($type eq 'all') {

            # Get a list of rooms in all regions
            @roomList = $self->worldModelObj->ivKeys('roomModelHash');
        }

        # If a room limit is set, prompt the user for confirmation
        if (
            $self->worldModelObj->locateMaxObjects
            && $self->worldModelObj->locateMaxObjects < @roomList
        ) {
            $choice = $self->showMsgDialogue(
                'Locate current room',
                'question',
                'There are ' . scalar @roomList . ' rooms to search. Do you want to continue?',
                'yes-no',
            );

            if ($choice ne 'yes') {

                return undef;
            }
        }

        # Compare the Locator task's current room with every room in @roomList
        foreach my $roomNum (@roomList) {

            my ($roomObj, $result);

            $roomObj = $self->worldModelObj->ivShow('modelHash', $roomNum);
            ($result) = $self->worldModelObj->compareRooms($self->session, $roomObj);
            if ($result) {

                push (@selectList, $roomObj);
                # Add the parent region to a hash so we can quickly check how many regions
                #   have matching rooms
                $regionmapHash{$roomObj->parent} = undef;
            }
        }

        # No matching rooms found
        if (! @selectList) {

            # Show a confirmation
            if ($type eq 'current') {
                $msg = 'No matching rooms found in the current region';
            } elsif ($type eq 'select') {
                $msg = 'No matching rooms found in the \'' . $regionmapObj->name . '\' region';
            } elsif ($type eq 'all') {
                $msg = 'No matching rooms found in any region';
            }

            $self->showMsgDialogue(
                'Locate current room',
                'error',
                $msg,
                'ok',
            );

        # A single matching room found
        } elsif (@selectList == 1) {

            # To clear a previous location attempt, in which many rooms were selected, unselect any
            #   existing selected objects
            $self->setSelectedObj();

            # Mark the matching room as the automapper's current room. If it's in a different
            #   regionmap (or on a different level), the map is redrawn
            $self->mapObj->setCurrentRoom($selectList[0]);

            # Show a confirmation
            $self->showMsgDialogue(
                'Locate current room',
                'info',
                '1 matching room found; current location set to room #'
                . $self->mapObj->currentRoom->number,
                'ok',
            );

        # Multiple matching rooms were found
        } else {

            # Unselect any existing selected objects
            $self->setSelectedObj();

            # Select all of the matching rooms. $self->setSelectedObj expects a list in the form
            #   (room_object, 'room', room_object, 'room', ...)
            foreach my $roomObj (@selectList) {

                push (@modList, $roomObj, 'room');
            }

            $self->setSelectedObj(
                \@modList,
                TRUE,       # Select multiple objects
            );

            # Get a sorted list of affected regions
            foreach my $number (keys %regionmapHash) {

                my $regionObj = $self->worldModelObj->ivShow('modelHash', $number);

                push (@newRegionList, $regionObj->name);
            }

            @sortedList = sort {lc($a) cmp lc($b)} (@newRegionList);

            # Show a confirmation
            $msg = scalar @selectList . ' matching rooms found in ';

            if ($type eq 'all') {

                if (@sortedList > 1) {

                    $msg .= scalar @sortedList . " regions:\n";

                    # Sort the region names alphabetically
                    foreach my $item (@sortedList) {

                        $msg .= '\'' . $item . '\' ';
                    }

                } else {

                    $msg .= 'the region \'' . $sortedList[0] . '\'';
                }

            } elsif ($type eq 'select') {

                $msg .= 'the region \'' . $regionName . '\'';

            } else {

                $msg .= 'this region';
            }

            $self->showMsgDialogue(
                'Locate current room',
                'info',
                $msg,
                'ok',
            );

            # Check the list of selected rooms, looking for the first one that's in the current
            #   region
            OUTER: foreach my $roomObj (@selectList) {

                if ($roomObj->parent eq $self->currentRegionmap->number) {

                    $matchObj = $roomObj;
                    last OUTER;
                }
            }

            if (! $matchObj) {

                # None of the selected rooms are in the current region. Use the first selected
                #   room...
                $matchObj = $selectList[0];
                # ...and change the current region to show that room
                $regionObj = $self->worldModelObj->ivShow('modelHash', $matchObj->parent);
                $self->setCurrentRegion($regionObj->name);
            }

            # Centre the map over the chosen selected room
            $self->centreMapOverRoom($matchObj);
        }

        return 1;
    }

    sub removeBGColourCallback {

        # Called by $self->enableRegionsColumn
        # Empties an existing region of any coloured blocks and rectangles matching a colour
        #   specified by the user
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $choice,
            @comboList, @squareList, @rectList,
            %colourHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->removeBGColourCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Get a list of all coloured blocks in the region
        @squareList = $self->currentRegionmap->ivKeys('gridColourBlockHash');
        # Get a list of all coloured rectangles in the region
        @rectList = $self->currentRegionmap->ivValues('gridColourObjHash');

        if (@squareList || @rectList) {

            # Compile a hash of RGB colours in use with coloured squares and rectangles, so we can
            #   eliminate duplicates
            foreach my $colour ($self->currentRegionmap->ivValues('gridColourBlockHash')) {

                $colourHash{uc($colour)} = undef;
            }

            foreach my $obj (@rectList) {

                $colourHash{uc($obj->colour)} = undef;
            }

            # Sort into some kind of order
            @comboList = sort {$a cmp $b} (keys %colourHash);

            # Prompt the user
            $choice = $self->showComboDialogue(
                'Remove background colour',
                'Choose a colour to remove from those used on the map background to remove',
                \@comboList,
            );

            if (defined $choice) {

                foreach my $coord (@squareList) {

                    if ($self->currentRegionmap->ivShow('gridColourBlockHash', $coord) eq $choice) {

                        # $coord can be in the form 'x_y_z' or 'x_y'; in either case, we don't
                        #   want the z
                        my ($x, $y) = split (/_/, $coord);

                        $self->currentRegionmap->removeSquare($coord);
                        $self->deleteCanvasObj(
                            'square',
                            $x . '_' . $y,
                            $self->currentRegionmap,
                            $self->currentParchment,
                        );
                    }
                }

                foreach my $obj (@rectList) {

                    if ($obj->colour eq $choice) {

                        $self->currentRegionmap->removeRect($obj);
                        $self->deleteCanvasObj(
                            'rect',
                            $obj->number,
                            $self->currentRegionmap,
                            $self->currentParchment,
                        );
                    }
                }

                # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
                $self->restrictWidgets();
            }
        }

        return 1;
    }

    sub removeBGAllCallback {

        # Called by $self->enableRegionsColumn
        # Empties an existing region of its coloured blocks and rectangles
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $choice,
            @squareList, @rectList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->removeBGAllCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Get a list of all coloured blocks in the region
        @squareList = $self->currentRegionmap->ivKeys('gridColourBlockHash');
        # Get a list of all coloured rectangles in the region
        @rectList = $self->currentRegionmap->ivValues('gridColourObjHash');

        # Prompt the user before removing anything
        if (@squareList || @rectList) {

            $choice = $self->showMsgDialogue(
                'Remove background colours',
                'question',
                'Are you sure you want to remove all background colours in this region?',
                'yes-no',
            );

            if (defined $choice && $choice eq 'yes') {

                foreach my $coord (@squareList) {

                    # $coord can be in the form 'x_y_z' or 'x_y'; in either case, we don't
                    #   want the z
                    my ($x, $y) = split (/_/, $coord);

                    $self->currentRegionmap->removeSquare($coord);
                    $self->deleteCanvasObj(
                        'square',
                        $x . '_' . $y,
                        $self->currentRegionmap,
                        $self->currentParchment,
                    );
                }

                foreach my $obj (@rectList) {

                    $self->currentRegionmap->removeRect($obj);
                    $self->deleteCanvasObj(
                        'rect',
                        $obj->number,
                        $self->currentRegionmap,
                        $self->currentParchment,
                    );
                }

                # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
                $self->restrictWidgets();
            }
        }

        return 1;
    }

    sub emptyRegionCallback {

        # Called by $self->enableRegionsColumn
        # Empties an existing region of its rooms
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the region is
        #       already empty or if the user declines to continue, when prompted
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $regionObj, $msg, $result,
            @roomList, @otherList, @labelList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->emptyRegionCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Get the region object corresponding to the current regionmap
        $regionObj = $self->worldModelObj->ivShow('modelHash', $self->currentRegionmap->number);

        # Get a list of the region's child objects, but don't include any child regions (which won't
        #   be deleted)
        foreach my $childNum ($regionObj->ivKeys('childHash')) {

            my $childObj = $self->worldModelObj->ivShow('modelHash', $childNum);

            if ($childObj->category eq 'room') {
                push (@roomList, $childObj);
            } elsif ($childObj->category ne 'region') {
                push (@otherList, $childObj);
            }
        }

        # Get a list of the regionmap's labels
        @labelList = $self->currentRegionmap->ivValues('gridLabelHash');

        if (! @roomList && ! @otherList && ! @labelList) {

            $self->showMsgDialogue(
                'Empty region',
                'error',
                'The current region doesn\'t contain any rooms, model objects or labels',
                'ok',
            );

            return undef;

        } else {

            # Give the user a chance to change their minds, before emptying the region
            $msg = "Are you sure you want to empty the\n\'" . $regionObj->name
                    . "\'? region? It contains:\n\n   Rooms: " . scalar @roomList
                    . "\n   Other model objects: " . scalar @otherList
                    . "\n   Labels: " . scalar @labelList;


            $result = $self->showMsgDialogue(
                'Empty region',
                'question',
                $msg,
                'yes-no',
                undef,
                TRUE,           # Preserve newline characters in $msg
            );

            if ($result ne 'yes') {

                return undef;

            } else {

                # Show a pause window, if necessary. The call to ->redrawRegions below will turn it
                #   off again
                if ((scalar @roomList) > 200) {

                    # If the tooltips are visible, hide them
                    $self->hideTooltips();
                    # Show the pause window
                    $self->showPauseWin();
                }

                # Empty the region
                $self->worldModelObj->emptyRegion(
                    $self->session,
                    FALSE,
                    $regionObj,
                );

                # Redraw the empty region
                $self->redrawRegions(
                    $self->worldModelObj->ivShow('regionmapHash', $regionObj->name),
                    TRUE,                   # Only redraw this region
                );

                return 1;
            }
        }
    }

    sub deleteRegionCallback {

        # Called by $self->enableRegionsColumn
        # Deletes the current region
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       declines to continue, when prompted
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $regionObj, $msg, $result, $total,
            @roomList, @otherList, @labelList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->deleteRegionCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Get the region object corresponding to the current regionmap
        $regionObj = $self->worldModelObj->ivShow('modelHash', $self->currentRegionmap->number);

        # Get a list of the region's child objects, but don't include any child regions (which won't
        #   be deleted)
        foreach my $childNum ($regionObj->ivKeys('childHash')) {

            my $childObj = $self->worldModelObj->ivShow('modelHash', $childNum);

            if ($childObj->category eq 'room') {
                push (@roomList, $childObj);
            } elsif ($childObj->category ne 'region') {
                push (@otherList, $childObj);
            }
        }

        # Get a list of the regionmap's labels
        @labelList = $self->currentRegionmap->ivValues('gridLabelHash');

        if (@roomList || @otherList || @labelList) {

            # Give the user a chance to change their minds, before emptying the region
            $msg = "Are you sure you want to delete the\n\'" . $regionObj->name
                    . "\' region? It contains:\n\n   Rooms: " . scalar @roomList
                    . "\n   Other model objects: " . scalar @otherList
                    . "\n   Labels: " . scalar @labelList;


            $result = $self->showMsgDialogue(
                'Delete region',
                'question',
                $msg,
                'yes-no',
                undef,
                TRUE,           # Preserve newline characters in $msg
            );

            if ($result ne 'yes') {

                return undef;
            }
        }

        # For large regions, show the pause window
        $total = scalar @roomList + scalar @labelList;
        if ($total > $self->worldModelObj->drawPauseNum) {

            $self->showPauseWin();
        }

        # Delete the region
        $self->worldModelObj->deleteRegions(
            $self->session,
            TRUE,              # Update Automapper windows now
            $regionObj,
        );

        # Make the pause window invisible
        $self->hidePauseWin();

        return 1;
    }

    sub deleteTempRegionsCallback {

        # Called by $self->enableRegionsColumn
        # Deletes all temporary regions
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if there are no temporary regions or if the user declines
        #       to continue, when prompted
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $msg, $result, $total,
            @tempList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->deleteTempRegionsCallback',
                @_,
            );
        }

        # (No standard callback checks for this function)

        # Get a list of temporary region objects
        foreach my $regionObj ($self->worldModelObj->ivValues('regionModelHash')) {

            if ($regionObj->tempRegionFlag) {

                push (@tempList, $regionObj);
            }
        }

        if (! @tempList) {

            $self->showMsgDialogue(
                'Delete temporary regions',
                'error',
                'The world model doesn\'t contain any temporary regions',
                'ok',
            );

            return undef;

        } else {

            # Give the user a chance to change their minds, before emptying the region
            if (@tempList == 1) {

                $msg = 'There is 1 temporary region in the world model. Are you sure you want to'
                            . ' delete it?';
            } else {

                $msg = 'There are ' . scalar @tempList . ' temporary regions in the world model.'
                            . ' Are you sure you want to delete them all?'
            }

            $result = $self->showMsgDialogue(
                'Delete temporary regions',
                'question',
                $msg,
                'yes-no',
            );

            if ($result ne 'yes') {

                return undef;
            }
        }

        # Work out roughly how many rooms and labels will be deleted. If it's a lot, show a pause
        #   window
        $total = 0;
        foreach my $regionObj (@tempList) {

            my $regionmapObj = $self->worldModelObj->ivShow('regionmapHash', $regionObj->name);

            $total += $regionmapObj->ivPairs('gridRoomHash');
            $total += $regionmapObj->ivPairs('gridLabelHash');
        }

        if ($total > $self->worldModelObj->drawPauseNum) {

            $self->showPauseWin();
        }

        # Delete each temporary region in turn
        $self->worldModelObj->deleteTempRegions(
            $self->session,
            TRUE,              # Update Automapper windows now
        );

        # Make the pause window invisible
        $self->hidePauseWin();

        return 1;
    }

    # Menu 'Rooms' column callbacks

    sub resetLocatorCallback {

        # Called by $self->enableRoomsColumn
        # Resets the Locator task, and marks the automapper as lost
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetLocatorCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Reset the Locator task
        $self->session->pseudoCmd('resetlocatortask', $self->pseudoCmdMode);

        # The call to ;resetlocatortask should mark the automapper as lost - but, if it's not, do it
        #   from here
        if ($self->mapObj->currentRoom) {

            return $self->mapObj->setCurrentRoom();
        }

        # Display an explanatory message, if necessary
        if ($self->worldModelObj->explainGetLostFlag) {

            $self->session->writeText('MAP: Lost because of a Locator reset');
        }

        return 1;
    }

    sub setFacingCallback {

        # Called by $self->enableRoomsColumn
        # Sets the direction the character is facing
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $dictObj, $choice,
            @comboList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setFacingCallback', @_);
        }

        # (No standard callback check)

        # Import the current dictionary (for convenience)
        $dictObj = $self->session->currentDict;

        # The permitted facing directions are n/ne/e/se/s/sw/w/nw
        foreach my $dir (qw (north northeast east southeast south southwest west northwest)) {

            # Use custom primary directions in the combo, then convert the user's choice back into
            #   a standard primary direction
            push (@comboList, $dictObj->ivShow('primaryDirHash', $dir));
        }

        # Prompt the user
        $choice = $self->showComboDialogue(
            'Set facing direction',
            'Set the direction the character is facing',
            \@comboList,
        );

        if (defined $choice) {

            $self->session->mapObj->set_facingDir($choice);
        }

        return 1;
    }

    sub resetFacingCallback {

        # Called by $self->enableRoomsColumn
        # Resets the direction the character is facing
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->resetFacingCallback', @_);
        }

        # (No standard callback check)

        # Reset the facing direction
        $self->session->mapObj->set_facingDir();

        # Show a confirmation
        $self->showMsgDialogue(
            'Reset facing direction',
            'info',
            'The direction your character is facing has been reset',
            'ok',
        );

        return 1;
    }

    sub editLocatorRoomCallback {

        # Called by $self->enableRoomsColumn
        # Opens a GA::EditWin::ModelObj::Room for the Locator task's current (non-model) room
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the Locator task doesn't know the current location
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $taskObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->editLocatorRoomCallback',
                @_,
            );
        }

        # (No standard callback checks for this function)

        # Check there's a Locator task which knows the current room
        $taskObj = $self->session->locatorTask;
        if (! $taskObj || ! $taskObj->roomObj) {

            # Show a 'dialogue' window to explain the problem
            $self->showMsgDialogue(
                'View Locator room',
                'error',
                'Either the Locator task isn\'t running or it doesn\'t know the current location',
                'ok',
            );

            return undef;

        } else {

            # Open up an 'edit' window to edit the object
            $self->createFreeWin(
                'Games::Axmud::EditWin::ModelObj::Room',
                $self,
                $self->session,
                'Edit non-model room object',
                $taskObj->roomObj,
                FALSE,                          # Not temporary
            );

            return 1;
        }
    }

    sub processPathCallback {

        # Called by $self->enableRoomsColumn (also called by GA::Cmd::Go->do)
        # Performs the A* algorithm to find a path between the current room and the selected room,
        #   and then does something with it
        #
        # Expected arguments
        #   $mode   - Set to one of the following:
        #               'select_room' - shows the path by selecting every room along the route
        #               'pref_win' - shows the path in a 'pref' window, allowing the user to store
        #                   it as a pre-defined route (using the ';addroute' command)
        #               'send_char' - sends the character to the selected room
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if no path can be
        #       found between the current and selected rooms
        #   1 otherwise

        my ($self, $mode, $check) = @_;

        # Local variables
        my (
            $dictObj, $text, $count, $maxChars, $string, $lastExitObj, $roomListRef, $exitListRef,
            $response,
            @roomList, @exitList, @cmdList, @reverseCmdList, @highlightList, @modList,
        );

        # Check for improper arguments
        if (
            ! defined $mode
            || ($mode ne 'select_room' && $mode ne 'pref_win' && $mode ne 'send_char')
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->processPathCallback', @_);
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || ! $self->mapObj->currentRoom
            || ! $self->selectedRoom
            || $self->mapObj->currentRoom eq $self->selectedRoom
        ) {
            return undef;
        }

        # Import the current dictionary (for speed)
        $dictObj = $self->session->currentDict;

        # Use the universal version of the A* algorithm to find a path between the current and
        #   selected rooms (if they're in the same region, the call is automatically redirected to
        #   ->findPath)
        ($roomListRef, $exitListRef) = $self->worldModelObj->findUniversalPath(
            $self->session,
            $self->mapObj->currentRoom,
            $self->selectedRoom,
            $self->worldModelObj->avoidHazardsFlag,
        );

        if (! defined $roomListRef || ! @$roomListRef) {

            # There is no path between the current and selected room. Notify the user with a popup
            $self->showMsgDialogue(
                'No path found',
                'warning',
                'There is no known path between the current room (#'
                . $self->mapObj->currentRoom->number . ') and the selected room (#'
                . $self->selectedRoom->number . ')',
                'ok',
            );

            return undef;
        }

        # Apply post-processing to the path to remove jagged edges (if allowed)
        if ($self->worldModelObj->postProcessingFlag) {

            ($roomListRef, $exitListRef) = $self->worldModelObj->smoothPath(
                $self->session,
                $roomListRef,
                $exitListRef,
                $self->worldModelObj->avoidHazardsFlag,
            );
        }

        # Convert the list references returned by the called functions into lists
        @roomList = @$roomListRef;
        @exitList = @$exitListRef;

        # Compile a list of commands to get from one end of the route to the other. If assisted
        #   moves are turned on, use them; otherwise, use each exit's nominal direction
        # At the same time, try to compile a list of directions that lead from the end of the
        #   route back to the start
        @cmdList = $self->worldModelObj->convertExitList($self->session, @exitList);
        # Attempt to find the reverse list of directions, if possible (but only bother in
        #   'select_room' mode)
        if ($mode eq 'pref_win') {

            @reverseCmdList = $self->worldModelObj->findPathCmds($self->session, -1, @roomList);
        }

        # 'select_room' - select each room in the path, in order to highlight the route (but don't
        #   select the current room)
        # 'pref_win' - show the route/reverse route in a 'pref' window
        if ($mode eq 'select_room' || $mode eq 'pref_win') {

            foreach my $roomObj (@roomList) {

                if ($roomObj ne $self->mapObj->currentRoom) {

                    push (@highlightList, $roomObj, 'room');
                }
            }

            $self->setSelectedObj(
                \@highlightList,
                TRUE,           # Select multiple objects, including the currently selected room
            );
        }

        # 'pref_win' - show the route/reverse route in a 'pref' window, allowing the user to store
        #   it as a pre-defined route (using the ';addroute' command)
        if ($mode eq 'pref_win') {

            # Open up a path 'pref' window to specify task settings
            $self->createFreeWin(
                'Games::Axmud::PrefWin::Path',
                $self,
                $self->session,
                # Use 'Edit path' rather than 'Path preferences'
                'Edit path',
                # No ->editObj
                undef,
                # The path itself is temporary (although can be stored as a GA::Obj::Route)
                TRUE,
                # Config
                'room_list'     => $roomListRef,
                'exit_list'     => $exitListRef,
                'cmd_list'      => \@cmdList,
                'reverse_list'  => \@reverseCmdList,
            );
        }

        # 'send_char' - Select every room on the path, so that the user can see where the path is,
        #   before moving to the destination room (don't worry about not selecting the current room,
        #   as the character is about to move to a new room anyway)
        if ($mode eq 'send_char') {

            foreach my $roomObj (@roomList) {

                push (@highlightList, $roomObj, 'room');
            }

            $self->setSelectedObj(
                \@highlightList,
                TRUE,           # Select multiple objects, including the currently selected room
            );

            # Offer the user to opportunity to change their mind. Only display one 'dialogue'
            #   window; if the user clicks the 'yes' button, go ahead and move
            if ($self->mode eq 'wait') {

                $response = $self->showMsgDialogue(
                    'Move to room',
                    'question',
                    'The automapper is in \'wait\' mode. Do you really want to move to the'
                    . ' double-clicked room?',
                    'yes-no',
                );

                if ($response ne 'yes') {

                     # Don't move anywhere
                     return 1;
                }

            } elsif ($self->session->locatorTask->moveList) {

                $response = $self->showMsgDialogue(
                    'Move to room',
                    'question',
                    'The Locator task is expecting more room statements; the room displayed'
                    . ' as the automapper\'s current room probably isn\'t the correct one.'
                    . ' Do you really want to move to the double-clicked room?',
                    'yes-no',
                );

                if ($response ne 'yes') {

                    # Don't move anywhere
                    return 1;
                }

            } elsif (
                $self->worldModelObj->pathFindStepLimit
                && $self->worldModelObj->pathFindStepLimit < scalar @cmdList
            ) {
                $response = $self->showMsgDialogue(
                    'Move to room',
                    'warning',
                    'The path contains a large number of steps (' . scalar @cmdList . '). Do you'
                    . ' really want to move to the double-clicked room?',
                    'yes-no',
                );

                if ($response ne 'yes') {

                    # Don't move anywhere
                    return 1;
                }
            }

            # By making a single call to GA::Session->worldCmd, using a command string like
            #   'north;east;north', we avoid the need to redraw the ghost room dozens of hundreds of
            #   times (a slow process)
            # Abbreviate any primary/secondary directions, if possible
            foreach my $cmd (@cmdList) {

                my $abbrevDir = $dictObj->abbrevDir($cmd);

                # (For secondary directions like 'in' with no abbreviation, ->abbrevDir returns
                #   'undef', in which case we should use the original $cmd)
                if (defined $abbrevDir) {
                    push (@modList, $abbrevDir);
                } else {
                    push (@modList, $cmd);
                }
            }

            # Take the route
            $self->session->worldCmd(join($axmud::CLIENT->cmdSep, @modList));

        } else {

            # Unrecognised mode
            return undef;
        }

        return 1;
    }

    sub adjacentModeCallback {

        # Called by $self->enableRoomsColumn (only)
        # Opens a dialogue window to set values of GA::Obj::WorldModel->adjacentMode and
        #   ->adjacentCount
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, or if the user declines to set valid values
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($mode, $count);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->adjacentModeCallback', @_);
        }

        # (No standard callback check)
        ($mode, $count) = $self->promptAdjacentMode();
        if (defined $mode) {

            if ($mode eq 'near') {

                if (! $axmud::CLIENT->intCheck($count, 0)) {

                    $self->showMsgDialogue(
                        'Adjacent regions regions mode',
                        'error',
                        'The number must be a positive integer (or zero)',
                        'ok',
                    );

                    return undef;

                } else {

                    $self->worldModelObj->set_adjacentMode($mode, $count);
                }

            } else {

                $self->worldModelObj->set_adjacentMode($mode);
            }
        }

        return 1;
    }

    sub moveSelectedRoomsCallback {

        # Called by $self->enableEditColumn
        # Prompts the user to select the direction in which to move the selected rooms (and any
        #   selected labels, if there is at least one selected room)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the user
        #       declines to specify a valid distance and direction or if the move operation fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $dictObj, $distance, $choice, $standardDir,
            @shortList, @longList, @dirList, @customList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->moveSelectedRoomsLabelsCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap || (! $self->selectedRoom && ! $self->selectedRoomHash)) {

            return undef;
        }

        # Must reset free click mode, so 'move rooms' and 'move rooms to click' can't be combined
        #   accidentally
        $self->reset_freeClickMode();

        # Import the current dictionary
        $dictObj = $self->session->currentDict;

        # Prepare a list of standard primary directions. Whether we include 'northnortheast', etc,
        #   depends on the current value of $self->worldModelObj->showAllPrimaryFlag
        @shortList = qw(north northeast east southeast south southwest west northwest up down);
        # (For convenience, put the longest directions at the end)
        @longList = qw(
            north northeast east southeast south southwest west northwest up down
            northnortheast eastnortheast eastsoutheast southsoutheast
            southsouthwest westsouthwest westnorthwest northnorthwest
        );

        if ($self->worldModelObj->showAllPrimaryFlag) {
            @dirList = @longList;
        } else {
            @dirList = @shortList;
        }

        # Get a list of (custom) primary directions, in the standard order
        foreach my $key (@dirList) {

            push (@customList, $dictObj->ivShow('primaryDirHash', $key));
        }

        # Prompt the user for a distance and a direction
        ($distance, $choice) = $self->showEntryComboDialogue(
            'Move selected rooms',
            'Enter a distance (in gridblocks)',
            'Select the direction of movement',
            \@customList,
        );

        # If the 'cancel' button was clicked, $distance will be 'undef'. The user might also have
        #   entered the distance 0. In either case, we don't move anything
        if (! $distance) {

            # Operation cancelled
            return undef;

        } else {

            # Check that the distance is a positive integer
            if (! $axmud::CLIENT->intCheck($distance, 1)) {

                # Open a 'dialogue' window to explain the problem
                $self->showMsgDialogue(
                    'Move selected rooms',
                    'error',
                    'The distance must be a positive integer',
                    'ok',
                );

                return undef;
            }

            # $dir is a custom primary direction; convert it into the standard primary direction
            $standardDir = $dictObj->ivShow('combRevDirHash', $choice);

            # Move the selected room(s)
            return $self->moveRoomsInDir($distance, $standardDir);
        }
    }

    sub transferSelectedRoomsCallback {

        # Called by $self->enableEditColumn and ->enableRoomsPopupMenu
        # Transfers the selected rooms (and any selected labels, if there is at least one selected
        #   room) to the same location in a specified region
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $regionName     - The name of the region into which the rooms/labels should be
        #                       transferred. All selected rooms/labels must be in the same region,
        #                       and that region must not be the same as $regionName. If 'undef', the
        #                       user is prompted to select a region
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the move
        #       operation fails
        #   1 otherwise

        my ($self, $regionName, $check) = @_;

        # Local variables
        my @comboList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->transferSelectedRoomsCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap || (! $self->selectedRoom && ! $self->selectedRoomHash)) {

            return undef;
        }

        # Make sure any free click mode operations, like connecting exits or moving rooms, are
        #   cancelled
        $self->reset_freeClickMode();

        # Prompt the user to select a region, if no region was specified by the calling function
        if (! defined $regionName) {

            # Get a sorted list of region names
            @comboList = sort {lc($a) cmp lc($b)} ($self->worldModelObj->ivKeys('regionmapHash'));

            # Prompt the user for a region name
            $regionName = $self->showComboDialogue(
                'Select region',
                'Select the destination region',
                \@comboList,
            );

            if (! defined $regionName) {

                return undef;
            }
        }

        # Move the selected rooms/labels
        return $self->transferRoomsToRegion($regionName);
    }

    sub compareRoomCallback {

        # Called by $self->->enableRoomsPopupMenu
        # Compares the selected room with rooms in the region or the whole world model, and selects
        #   any matching rooms
        #
        # Expected arguments
        #   $wholeFlag  - FALSE to compare rooms in the same region, TRUE to compare rooms in the
        #                   whole world model
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the move
        #       operation fails
        #   1 otherwise

        my ($self, $wholeFlag, $check) = @_;

        # Local variables
        my (
            $wmObj, $selectObj, $regionmapObj, $string,
            @roomList, @matchList, @selectList,
        );

        # Check for improper arguments
        if (! defined $wholeFlag || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->compareRoomCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || ! $self->selectedRoom) {

            return undef;
        }

        # Import the world model object and the selected room (for speed)
        $wmObj = $self->worldModelObj;
        $selectObj = $self->selectedRoom;

        # Get a list of rooms which should be compared with the selected room
        if (! $wholeFlag) {

            $regionmapObj = $self->findRegionmap($self->selectedRoom->parent);
            foreach my $roomNum ($regionmapObj->ivValues('gridRoomHash')) {

                push (@roomList, $wmObj->ivShow('modelHash', $roomNum));
            }

        } else {

            push (@roomList, $wmObj->ivValues('roomModelHash'));
        }

        # Compare rooms in each region, one by one
        foreach my $thisObj (@roomList) {

            my $result;

            if ($thisObj ne $selectObj) {

                ($result) = $wmObj->compareRooms($self->session, $selectObj, $thisObj);
                if ($result) {

                    push (@matchList, $thisObj);
                    push (@selectList, $thisObj, 'room');
                }
            }
        }

        if (! @matchList) {

            # Show a confirmation
            return $self->showMsgDialogue(
                'Compare room',
                'error',
                'No matching rooms found',
                'ok',
            );

        } else {

            # Unselect the currently-selected room...
            $self->setSelectedObj();
            # ...so we can select all matching rooms. The TRUE argument means to select multiple
            #   objects
            $self->setSelectedObj(\@selectList, TRUE);

            if ((scalar @matchList) == 1) {
                $string = '1 room';
            } else {
                $string = (scalar @matchList) . ' rooms';
            }

            # Show a confirmation
            return $self->showMsgDialogue(
                'Compare room',
                'info',
                'Found ' . $string . ' matching room #' . $selectObj->number,
                'ok',
            );
        }
    }

    sub executeScriptsCallback {

        # Called by $self->enableRoomsPopupMenu
        # Executes Axbasic scripts for the current room, as if the character had just arrived
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->executeScriptsCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || ! $self->mapObj->currentRoom) {

            return undef;
        }

        # If there are no Axbasic scripts for the current room, display a warning
        if (! $self->mapObj->currentRoom->arriveScriptList) {

            return $self->showMsgDialogue(
                'Run ' . $axmud::BASIC_NAME . ' scripts',
                'warning',
                'The current room has not been assigned any ' . $axmud::BASIC_NAME . ' scripts',
                'ok',
            );
        }

        # Otherwise, execute the scripts
        foreach my $scriptName ($self->mapObj->currentRoom->arriveScriptList) {

            $self->session->pseudoCmd('runscript ' . $scriptName);
        }

        return 1;
    }

    sub addFirstRoomCallback {

        # Called by $self->enableRoomsColumn. Also called by Axbasic ADDFIRSTROOM function
        # For an empty region, draws a room in the centre of the grid and marks it as the current
        #   room
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the Locator task
        #       isn't running or if it is still expecting room statements or if the new room can't
        #       be created
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($xPosBlocks, $yPosBlocks, $zPosBlocks, $newRoomObj);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addFirstRoomCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || $self->currentRegionmap->gridRoomHash) {

            return undef;
        }

        # If the Locator's ->moveList isn't empty, we won't be able to switch to 'update' mode.
        #   Therefore refuse to add the first room if the list isn't empty (or if the Locator task
        #   isn't running at all)
        if (! $self->session->locatorTask) {

            $self->showMsgDialogue(
                'Add first room',
                'error',
                'Can\'t add a room at the centre of the map - the Locator task is not running',
                'ok',
            );

            return undef;

        } elsif ($self->session->locatorTask->moveList) {

            $self->showMsgDialogue(
                'Add first room',
                'error',
                'Can\'t add a room at the centre of the map - the Locator task is not ready',
                'ok',
            );

            return undef;
        }

        # Find the coordinates of the middle of the grid
        ($xPosBlocks, $yPosBlocks, $zPosBlocks) = $self->currentRegionmap->getGridCentre();

        # Check the location to make sure there's not already a room there
        if ($self->currentRegionmap->fetchRoom($xPosBlocks, $yPosBlocks, $zPosBlocks)) {

            $self->showMsgDialogue(
                'Add first room',
                'error',
                'Can\'t add a room at the centre of the map - the position is already occupied',
                'ok',
            );

            return undef;
        }

        # Free click mode must be reset (nothing special happens when the user clicks on the map)
        $self->reset_freeClickMode();

        # Create a new room object, with this region as its parent, and update the map
        if ($self->session->locatorTask && $self->session->locatorTask->roomObj) {

            # Set the Automapper window's mode to 'update', make the new room the current location
            #   and copy properties from the Locator task's current room (where allowed)
            $newRoomObj = $self->mapObj->createNewRoom(
                $self->currentRegionmap,
                $xPosBlocks,
                $yPosBlocks,
                $zPosBlocks,
                'update',
                TRUE,
                TRUE,
            );

        } else {

            # Locator task doesn't know the current location, so don't make the new room the
            #   current room, and don't change the mode
            $newRoomObj = $self->mapObj->createNewRoom(
                $self->currentRegionmap,
                $xPosBlocks,
                $yPosBlocks,
                $zPosBlocks,
            );
        }

        if (! $newRoomObj) {

            # Could not create the new room (an error message has already been displayed)
            return undef;

        } else {

            # Also update the Locator with the new current room (if there is one)
            $self->mapObj->updateLocator();

            return 1;
        }
    }

    sub addRoomAtBlockCallback {

        # Called by $self->enableRoomsColumn. Also called by the Axbasic ADDROOM function
        # Prompts the user to supply a gridblock (via a 'dialogue' window) and creates a room at
        #   that location. When called by Axbasic, uses the supplied gridblock
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $xPosBlocks, $yPosBlocks, $zPosBlocks
        #       - The coordinates on the gridblock at which to draw the room
        #
        # Return values
        #   'undef' on improper arguments,if the standard callback check fails, if the user cancels
        #       the 'dialogue' window or if the new room can't be created
        #   1 otherwise

        my ($self, $xPosBlocks, $yPosBlocks, $zPosBlocks, $check) = @_;

        # Local variables
        my $roomObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addRoomAtBlockCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Prompt the user for a gridblock, if one was not specified
        if (! defined $xPosBlocks || ! defined $yPosBlocks || ! defined $zPosBlocks) {

            ($xPosBlocks, $yPosBlocks, $zPosBlocks) = $self->promptGridBlock();
            if (! defined $xPosBlocks ) {

                # User clicked the 'cancel' button
                return undef;
            }
        }

        # Check that the specified gridblock actually exists
        if (
            ! $self->currentRegionmap->checkGridBlock(
                $xPosBlocks,
                $yPosBlocks,
                $zPosBlocks,
            )
        ) {
            $self->showMsgDialogue(
                'Add room',
                'error',
                'The gridblock x=' . $xPosBlocks . ', y=' . $yPosBlocks . ', z=' . $zPosBlocks
                . ' is invalid',
                'ok',
            );

            return undef;
        }

        # Check that the gridblock isn't occupied
        if ($self->currentRegionmap->fetchRoom($xPosBlocks, $yPosBlocks, $zPosBlocks)) {

            $self->showMsgDialogue(
                'Add room',
                'error',
                'The gridblock x=' . $xPosBlocks . ', y=' . $yPosBlocks . ', z=' . $zPosBlocks
                . ' is already occupied',
                'ok',
            );

            return undef;
        }

        # Free click mode must be reset (nothing special happens when the user clicks on the map)
        $self->reset_freeClickMode();

        # Create a new room object, with this region as its parent and update the map
        $roomObj = $self->mapObj->createNewRoom(
            $self->currentRegionmap,
            $xPosBlocks,
            $yPosBlocks,
            $zPosBlocks,
        );

        if (! $roomObj) {

            # Could not create the new room (an error message has already been displayed)
            return undef;

        } else {

            # To make it easier to see where the new room was drawn, make it the selected room, and
            #   centre the map on the room
            $self->setSelectedObj(
                [$roomObj, 'room'],
                FALSE,          # Select this object; unselect all other objects
            );

            $self->centreMapOverRoom($roomObj);

            return 1;
        }
    }

    sub addExitCallback {

        # Called by $self->enableRoomsColumn
        # Adds a new exit, prompting the user for its properties
        #
        # Expected arguments
        #   $hiddenFlag - If set to TRUE, a hidden exit should be created (otherwise set to FALSE)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the user clicks
        #       'cancel' on the 'dialogue' window or if the exit can't be added
        #   1 otherwise

        my ($self, $hiddenFlag, $check) = @_;

        # Local variables
        my (
            $title, $dir, $mapDir, $assistedProf, $assistedMove, $result, $exitObj, $redrawFlag,
            $roomObj,
        );

        # Check for improper arguments
        if (! defined $hiddenFlag || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addExitCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || ! $self->selectedRoom) {

            return undef;
        }

        # Prompt the user for properties of the new exit
        if ($hiddenFlag) {
            $title = 'Add hidden exit';
        } else {
            $title = 'Add exit';
        }

        ($dir, $mapDir, $assistedProf, $assistedMove) = $self->promptNewExit(
            $self->selectedRoom,
            $title,
        );

        if (! defined $dir) {

            return undef;
        }

        # Add the exit
        $exitObj = $self->worldModelObj->addExit(
            $self->session,
            FALSE,              # Don't redraw the map yet...
            $self->selectedRoom,
            $dir,
            $mapDir,
        );

        if (! $exitObj) {

            return undef;
        }

        # Add an entry to the exit's assisted moves hash, if one was specified by the user
        if ($assistedProf && $assistedMove) {

            $self->worldModelObj->setAssistedMove($exitObj, $assistedProf, $assistedMove);
        }

        # Mark it as a hidden exit, if necessary
        if ($hiddenFlag) {

            $self->worldModelObj->setHiddenExit(
                FALSE,          # Don't redraw the map yet...
                $exitObj,
                TRUE,           # Exit is now hidden
            );
        }

        # Now, we need to check if the room has any more unallocated exits. If they've temporarily
        #   been assigned the map direction 'undef', we must reallocate them
        OUTER: foreach my $number ($self->selectedRoom->ivValues('exitNumHash')) {

            my $thisExitObj = $self->worldModelObj->ivShow('exitModelHash', $number);

            if (! defined $thisExitObj->mapDir && $thisExitObj->drawMode eq 'primary') {

                # Assign the exit object a new map direction (using one of the sixteen cardinal
                #   directions, but not 'up' and 'down'), if any are available
                $self->worldModelObj->allocateCardinalDir(
                    $self->session,
                    $self->selectedRoom,
                    $thisExitObj,
                );
            }
        }

        # Now, if there are any incoming 1-way exits whose ->mapDir is the opposite of the exit
        #   we've just added, the incoming exit should be marked as an uncertain exit
        $self->worldModelObj->modifyIncomingExits(
            $self->session,
            TRUE,               # Redraw any modified incoming exit
            $self->selectedRoom,
            $exitObj,
        );

        # Remember the (currently selected) room object that must be redrawn in every window
        $roomObj = $self->selectedRoom;
        # Make this exit the selected exit (which redraws it in this window)
        $self->setSelectedObj(
            [$exitObj, 'exit'],
            FALSE,              # Select this object; unselect all other objects
        );

        # Redraw the selected room in every window
        $self->worldModelObj->updateMaps('room', $roomObj);

        return 1;
    }

    sub addMultipleExitsCallback {

        # Called by $self->enableRoomsColumn
        # Prompts the user to select one or more map directions to add to the selected room(s), and
        #   then adds them
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the user doesn't
        #       select any directions or if an attempt to create an exit fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $hiddenFlag,
            @dirList, @drawList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->addMultipleExitsCallback',
                @_,
            );
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (! $self->selectedRoom && ! $self->selectedRoomHash)
        ) {
            return undef;
        }

        # Prompt the user to select some of the selected room's available primary directions
        ($hiddenFlag, @dirList) = $self->promptMultipleExits($self->selectedRoom);
        if (defined $hiddenFlag && @dirList) {

            OUTER: foreach my $roomObj ($self->compileSelectedRooms) {

                INNER: foreach my $customDir (@dirList) {

                    my ($mapDir, $exitObj);

                    # $customDir is a custom primary direction. Get the equivalent standard
                    #   direction
                    $mapDir = $self->session->currentDict->ivShow('combRevDirHash', $customDir);

                    # If the exit doesn't already exist, add it
                    if (! $roomObj->ivExists('exitNumHash', $customDir)) {

                        # Add the normal exit
                        $exitObj = $self->worldModelObj->addExit(
                            $self->session,
                            FALSE,              # Don't redraw the map yet...
                            $roomObj,
                            $customDir,
                            $mapDir,
                        );

                        if (! $exitObj) {

                            $self->showMsgDialogue(
                                'Add multiple exits',
                                'error',
                                'Failed to add one or more exits (internal error)',
                                'ok',
                            );

                            return undef;
                        }

                        # Mark it as a hidden exit, if necessary
                        if ($hiddenFlag) {

                            $self->worldModelObj->setHiddenExit(
                                FALSE,          # Don't redraw the map yet...
                                $exitObj,
                                TRUE,           # Exit is now hidden
                            );
                        }

                        # Mark the room to be redrawn
                        push (@drawList, 'room', $roomObj);

                        # Now, if there are any incoming 1-way exits whose ->mapDir is the opposite
                        #   of the exit we've just added, the incoming exit should be marked as an
                        #   uncertain exit
                        $self->worldModelObj->modifyIncomingExits(
                            $self->session,
                            TRUE,              # Redraw any modified incoming exit
                            $roomObj,
                            $exitObj,
                        );
                    }
                }
            }

            # Redraw the selected room(s) in every window
            $self->worldModelObj->updateMaps(@drawList);

            return 1;

        } else {

            # No exits were selected
            return undef;
        }
    }

    sub addFailedExitCallback {

        # Called by $self->enableRoomsColumn
        # When the character fails to move, and it's not a recognised failed exit pattern, the map
        #   gets messed up
        # This is a convenient way to deal with it. Adds a new failed exit string to the current
        #   world profile or to the specified room, and empties the Locator's move list
        #
        # Expected arguments
        #   $worldFlag   - If set to TRUE, a failed exit pattern is added to the world profile. If
        #                   set to FALSE, the pattern is added to the room
        #
        # Optional arguments
        #   $roomObj    - If $worldFlag is FALSE, the room to which the pattern should be added.
        #                   When called by $self->enableRoomsColumn, it will be the current room;
        #                   when called by ->enableRoomsPopupMenu, it will be the selected room
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       doesn't supply a pattern
        #   1 otherwise

        my ($self, $worldFlag, $roomObj, $check) = @_;

        # Local variables
        my (
            $pattern, $type, $worldObj, $iv, $descrip, $taskObj,
            @comboList,
        );

        # Check for improper arguments
        if (! defined $worldFlag || (! $worldFlag && ! defined $roomObj) || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addFailedExitCallback', @_);
        }

        # Standard callback check
        if (
            $roomObj
            && (
                ! $self->currentRegionmap
                || (! $self->mapObj->currentRoom && ! $self->selectedRoom)
            )
        ) {
            return undef;
        }

        if (! $worldFlag) {

            # Prompt the user for a new failed exit pattern to add to the room
            $pattern = $self->showEntryDialogue(
                'Add failed exit to room',
                'Enter a pattern to match the failed exit',
            );

            if (! $pattern) {

                return undef;

            } else {

                $self->worldModelObj->addExitPattern($roomObj, 'fail', $pattern);
            }

        } else {

            # Import the current world profile
            $worldObj = $self->session->currentWorld;

            # Prompt the user for a new failed exit pattern to add to the world profile
            @comboList = ('Closed door', 'Locked door', 'Other failed exit');
            ($pattern, $type) = $self->showEntryComboDialogue(
                'Add failed exit to world',
                'Enter a pattern to match the failed exit',
                'Which kind of failed exit was it?',
                \@comboList,
            );

            if (! ($pattern && $type)) {

                return undef;

            } else {

                # Check that the pattern isn't already in the list
                if ($type eq 'Closed door') {

                    $iv = 'doorPatternList';
                    $descrip = 'a closed door pattern';

                } elsif ($type eq 'Locked door') {

                    $iv = 'lockedPatternList';
                    $descrip = 'a locked door pattern';

                } else {
                    $iv = 'failExitPatternList';
                    $descrip = 'a failed exit pattern';
                }

                if ($worldObj->ivMatch($iv, $pattern)) {

                    $self->showMsgDialogue(
                        'Add failed exit to world',
                        'error',
                        'The current world profile already has ' . $descrip . ' pattern matching \''
                        . $pattern . '\'',
                        'ok',
                    );

                    return undef;

                } else {

                    # Add the pattern
                    $worldObj->ivPush($iv, $pattern);
                }
            }
        }

        # Import the Locator task
        $taskObj = $self->session->locatorTask;
        if ($taskObj) {

            # Empty the Locator's move list IVs and update its task window
            $taskObj->resetMoveList();
        }

        return 1;
    }

    sub addInvoluntaryExitCallback {

        # Called by $self->enableRoomsColumn
        # This callback adds an involuntary exit pattern to the specified room and empties the
        #   Locator task's move list
        #
        # Expected arguments
        #   $roomObj    - The room to which the pattern should be added. When called by
        #                   $self->enableRoomsColumn, it will be the current room; when called by
        #                   ->enableRoomsPopupMenu, it will be the selected room
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       doesn't supply a pattern
        #   1 otherwise

        my ($self, $roomObj, $check) = @_;

        # Local variables
        my ($pattern, $otherVal, $taskObj);

        # Check for improper arguments
        if (! defined $roomObj || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->addInvoluntaryExitCallback',
                @_,
            );
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (! $self->mapObj->currentRoom && ! $self->selectedRoom)
        ) {
            return undef;
        }

        # Prompt the user for a new involuntary exit pattern to add to the room
        ($pattern, $otherVal) = $self->showDoubleEntryDialogue(
            'Add involuntary exit to room',
            'Enter a pattern to match the involuntary exit',
            '(Optional) add a direction or a destination room',
        );

        if (! defined $pattern || $pattern eq '') {

            return undef;

        } else {

            # Use 'undef' rather than an empty string
            if ($otherVal eq '') {

                $otherVal = undef;
            }

            $self->worldModelObj->addInvoluntaryExit($roomObj, $pattern, $otherVal);

            # Import the Locator task
            $taskObj = $self->session->locatorTask;
            if ($taskObj) {

                # Empty the Locator's move list IVs and update its task window
                $taskObj->resetMoveList();
            }
        }

        return 1;
    }

    sub addRepulseExitCallback {

        # Called by $self->enableRoomsColumn
        # This callback adds a repulse exit pattern to the specified room and empties the Locator
        #   task's move list
        #
        # Expected arguments
        #   $roomObj    - The room to which the pattern should be added. When called by
        #                   $self->enableRoomsColumn, it will be the current room; when called by
        #                   ->enableRoomsPopupMenu, it will be the selected room
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       doesn't supply a pattern
        #   1 otherwise

        my ($self, $roomObj, $check) = @_;

        # Local variables
        my ($pattern, $otherVal, $taskObj);

        # Check for improper arguments
        if (! defined $roomObj || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->addRepulseExitCallback',
                @_,
            );
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (! $self->mapObj->currentRoom && ! $self->selectedRoom)
        ) {
            return undef;
        }

        # Prompt the user for a new repulse exit pattern to add to the room
        ($pattern, $otherVal) = $self->showDoubleEntryDialogue(
            'Add repulse exit to room',
            'Enter a pattern to match the repulse exit',
            '(Optional) add a direction or a destination room',
        );

        if (! defined $pattern || $pattern eq '') {

            return undef;

        } else {

            # Use 'undef' rather than an empty string
            if ($otherVal eq '') {

                $otherVal = undef;
            }

            $self->worldModelObj->addRepulseExit($roomObj, $pattern, $otherVal);

            # Import the Locator task
            $taskObj = $self->session->locatorTask;
            if ($taskObj) {

                # Empty the Locator's move list IVs and update its task window
                $taskObj->resetMoveList();
            }
        }

        return 1;
    }

    sub addSpecialDepartureCallback {

        # Called by $self->enableRoomsColumn
        # When the character moves using an exit which doesn't send a room statement upon arrival
        #   in the new room - usually after some kind of faller - the pattern sent by the world to
        #   confirm arrival (such as 'You land in a big heap!') should be interpreted by the
        #   Locator task as a special kind of room statement
        # This callback adds a special departure pattern to the specified room and empties the
        #   Locator task's move list
        #
        # Expected arguments
        #   $roomObj    - The room to which the pattern should be added. When called by
        #                   $self->enableRoomsColumn, it will be the current room; when called by
        #                   ->enableRoomsPopupMenu, it will be the selected room
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       doesn't supply a pattern
        #   1 otherwise

        my ($self, $roomObj, $check) = @_;

        # Local variables
        my ($pattern, $taskObj);

        # Check for improper arguments
        if (! defined $roomObj || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->addSpecialDepartureCallback',
                @_,
            );
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (! $self->mapObj->currentRoom && ! $self->selectedRoom)
        ) {
            return undef;
        }

        # Prompt the user for a new special departure pattern to add to the room
        $pattern = $self->showEntryDialogue(
            'Add special departure to room',
            'Enter a pattern to match the special departure',
        );

        if (! $pattern) {

            return undef;

        } else {

            $self->worldModelObj->addExitPattern($roomObj, 'special', $pattern);

            # Import the Locator task
            $taskObj = $self->session->locatorTask;
            if ($taskObj) {

                # Empty the Locator's move list IVs and update its task window
                $taskObj->resetMoveList();
            }
        }

        return 1;
    }

    sub addUnspecifiedPatternCallback {

        # Called by $self->enableRoomsColumn
        # GA::Profile::World->unspecifiedRoomPatternList provides a list of patterns that match
        #   a line in 'unspecified' rooms (those that don't use a recognisable room statement;
        #   typically a room whose exit list is completely obscured)
        # Each room has its own list of patterns that match a line in 'unspecified' rooms; this
        #   callback adds a pattern to that list
        #
        # Expected arguments
        #   $roomObj    - The room to which the pattern should be added. When called by
        #                   $self->enableRoomsColumn, it will be the current room; when called by
        #                   ->enableRoomsPopupMenu, it will be the selected room
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       doesn't supply a pattern
        #   1 otherwise

        my ($self, $roomObj, $check) = @_;

        # Local variables
        my ($pattern, $taskObj);

        # Check for improper arguments
        if (! defined $roomObj || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->addUnspecifiedPatternCallback',
                @_,
            );
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (! $self->mapObj->currentRoom && ! $self->selectedRoom)
        ) {
            return undef;
        }

        # Prompt the user for a new unspecified room pattern to add to the room
        $pattern = $self->showEntryDialogue(
            'Add unspecified room pattern',
            'Enter a pattern to match an unspecified room',
        );

        if (! $pattern) {

            return undef;

        } else {

            $self->worldModelObj->addExitPattern($roomObj, 'unspecified', $pattern);

            # Import the Locator task
            $taskObj = $self->session->locatorTask;
            if ($taskObj) {

                # Empty the Locator's move list IVs and update its task window
                $taskObj->resetMoveList();
            }
        }

        return 1;
    }

    sub removeCheckedDirCallback {

        # Called by $self->enableRoomsColumn and ->enableRoomsPopupMenu
        # Removes one or all checked directions from the selected room
        #
        # Expected arguments
        #   $allFlag    - If set to TRUE, all checked directions are removed. If set to FALSE, the
        #                   user is prompted to choose an exit
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       clicks 'cancel' on the 'dialogue' window
        #   1 otherwise

        my ($self, $allFlag, $check) = @_;

        # Local variables
        my (
            $choice,
            @comboList, @sortedList,
        );

        # Check for improper arguments
        if (! defined $allFlag || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->removeCheckedDirCallback',
                @_,
            );
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || ! $self->selectedRoom
            || ! $self->selectedRoom->checkedDirHash
        ) {
            return undef;
        }

        if ($allFlag) {

            # Delete all checked directions
            $self->selectedRoom->ivEmpty('checkedDirHash');

        } else {

            @comboList = sort {lc($a) cmp lc($b)} ($self->selectedRoom->ivKeys('checkedDirHash'));
            @sortedList = $self->session->currentDict->sortExits(@comboList);

            # Prompt the user for a checked direction to remove (even if there's only one)
            $choice = $self->showComboDialogue(
                'Remove checked direction',
                'Select the checked direction to remove',
                \@comboList,
            );

            if (! defined $choice) {

                return undef;

            } else {

                $self->selectedRoom->ivDelete('checkedDirHash', $choice);
            }
        }

        # Redraw the selected room in every window
        $self->worldModelObj->updateMaps('room', $self->selectedRoom);

        return 1;
    }

    sub setWildCallback {

        # Called by $self->enableRoomsColumn and ->enableRoomsPopupMenu
        # Sets the selected room(s)' wilderness mode
        #
        # Expected arguments
        #   $mode   - One of the values for GA::ModelObj::Room->wildMode - 'normal', 'border' or
        #               'wild'
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $mode, $check) = @_;

        # Local variables
        my @drawList;

        # Check for improper arguments
        if (
            ! defined $mode
            || ($mode ne 'normal' && $mode ne 'border' && $mode ne 'wild')
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->setWildCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap && (! $self->selectedRoom && ! $self->selectedRoomHash)) {

            return undef;
        }

        # For each selected room, convert their wilderness mode. The called function handles
        #   redrawing
        $self->worldModelObj->setWildernessRoom(
            $self->session,
            TRUE,                           # Update automapper windows
            $mode,
            $self->compileSelectedRooms(),
        );

        return 1;
    }

    sub selectExitCallback {

        # Called by $self->enableRoomsColumn
        # Prompts the user to select an exit manually
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if there are no
        #       exits to select, or if the user clicks 'cancel' in the 'dialogue' window
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $choice, $selectExitObj,
            @exitList, @comboList,
            %exitHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->selectExitCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || ! $self->selectedRoom) {

            return undef;
        }

        # Get a list of the select room's exits, in the standard order
        @exitList = $self->selectedRoom->sortedExitList;

        # Compile a hash in the form
        #   $hash{'informative_string'} = blessed_reference_to_exit_object
        foreach my $dir (@exitList) {

            my ($exitNum, $exitObj, $string, $customDir);

            # Prepare a string which shows:
            #   The exit's nominal direction and its exit model number
            #   Its temporarily allocated map direction in [square brackets]
            #   Its permanently allocated map direction in <diamond brackets>
            #   An unallocatable exit in {curly brackets}
            $exitNum = $self->selectedRoom->ivShow('exitNumHash', $dir);
            $exitObj = $self->worldModelObj->ivShow('exitModelHash', $exitNum);

            $string = $exitObj->dir . ' #' . $exitObj->number;

            if ($exitObj->mapDir) {

                # Get the equivalent custom direction, so that we can compare it to $dir
                $customDir = $self->session->currentDict->ivShow(
                    'primaryDirHash',
                    $exitObj->mapDir,
                );

                if ($customDir ne $exitObj->dir) {

                    if ($exitObj->drawMode eq 'temp_alloc') {
                        $string .= ' [' . $exitObj->mapDir . ']';
                    } else {
                        $string .= ' <' . $exitObj->mapDir . '>';
                    }
                }

            } elsif ($exitObj->drawMode eq 'temp_unalloc') {

                $string .= ' {unallocatable}';
            }

            # Add an entry to the hash...
            $exitHash{$string} = $exitObj;
            # ...and another in the combo list
            push (@comboList, $string);
        }

        # Don't prompt for an object, if there are none available
        if (! @comboList) {

            return $self->showMsgDialogue(
                'Select exit',
                'error',
                'Can\'t select an exit - this room has no exits',
                'ok',
            );
        }

        # Prompt the user to choose which exit to select
        $choice = $self->showComboDialogue(
            'Select exit',
            'Choose which exit to select',
            \@comboList,
        );

        if (! $choice) {

            return undef;

        } else {

            # Get the corresponding ExitObj
            $selectExitObj = $exitHash{$choice};

            # Select this exit
            $self->setSelectedObj(
                [$selectExitObj, 'exit'],
                FALSE,      # Select this object; unselect all other objects
            );

            return 1;
        }
    }

    sub identifyRoomsCallback {

        # Called by $self->enableRoomsColumn
        # Lists the current room and all the selected rooms in a 'dialogue' window (if more than 10
        #   are selected, we only list the first 10)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $limit, $msg, $roomName,
            @roomList, @sortedList, @reducedList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->identifyRoomsCallback', @_);
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            && (! $self->selectedRoom && ! $self->selectedRoomHash && ! $self->mapObj->currentRoom)
        ) {
            return undef;
        }

        # Compile a list of selected rooms, sorted by world model number
        @roomList = $self->compileSelectedRooms();
        @sortedList = sort {$a->number <=> $b->number} (@roomList);

        # Reduce the size of the list to a maximum of 10
        $limit = 10;
        if (@sortedList > $limit) {
            @reducedList = @sortedList[0..($limit - 1)];
        } else {
            @reducedList = @sortedList;
        }

        # Prepare the message to show in the window
        if ($self->mapObj->currentRoom) {

            $msg = "Current room:\n";
            $msg .= "   #" . $self->mapObj->currentRoom->number . " '";

            # '<unnamed room>' will cause a Pango error, so replace that string
            # GA::ModelObj::Room->name has already been cut down to a maximum of 32 characters. By
            #   checking for a length longer than 31, we can be certain we're not adding an ellipsis
            #   to a room title that was exactly 32 characters long
            $roomName = $self->mapObj->currentRoom->name;
            if ($roomName eq '<unnamed room>') {
                $roomName = '(unnamed room)';
            } elsif (length($roomName) > 31) {
                $roomName = substr($roomName, 0, 29) . '...';
            }

            $msg .= $roomName . "'\n\n";

        } else {

            $msg = '';
        }

        if (@reducedList) {

            if (scalar @sortedList != scalar @reducedList) {

                $msg .= "Selected rooms (first " . $limit . " rooms of " . scalar @sortedList
                        . ")";

            } elsif (scalar @sortedList == 1) {

                $msg .= "Selected rooms (1 room)";

            } else {

                $msg .= "Selected rooms (" . scalar @sortedList . " rooms)";
            }

            foreach my $obj (@reducedList) {

                my $roomName;

                $msg .= "\n   #" . $obj->number . " '";

                $roomName = $obj->name;
                if ($roomName eq '<unnamed room>') {
                    $roomName = '(unnamed room)';
                } elsif (length($roomName) > 31) {
                    $roomName = substr($roomName, 0, 29) . '...';
                }

                $msg .= $roomName . "'";
            }
        }

        # Display a popup to show the results
        $self->showMsgDialogue(
            'Identify rooms',
            'info',
            $msg,
            'ok',
            undef,
            TRUE,           # Preserve newline characters in $msg
        );

        return 1;
    }

    sub updateVisitsCallback {

        # Called by $self->enableRoomsColumn, ->enableRoomsPopupMenu and ->drawMiscButtonSet
        # Adjusts the number of character visits shown in the selected room(s)
        # Normally, the current character's visits are changed. However, if $self->showChar is set,
        #   that character's visits are changed
        #
        # Expected arguments
        #   $mode   - 'increase' to increase the number of visits by one, 'decrease' to decrease the
        #               visits by one, 'manual' to let the user enter a value manually, 'reset' to
        #               reset the number to zero
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the user clicks
        #       the 'cancel' button on a 'dialogue' window or for any other error
        #   1 otherwise

        my ($self, $mode, $check) = @_;

        # Local variables
        my (
            $char, $current, $result, $matchFlag,
            @roomList, @drawList,
        );

        # Check for improper arguments
        if (
            ! defined $mode
            || ($mode ne 'increase' && $mode ne 'decrease' && $mode ne 'manual' && $mode ne 'reset')
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->updateVisitsCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || (! $self->selectedRoom && ! $self->selectedRoomHash)) {

            return undef;
        }

        # Get a list of selected room(s)
        @roomList = $self->compileSelectedRooms();

        # Decide which character to use
        if ($self->showChar) {

            $char = $self->showChar;

        } elsif ($self->session->currentChar) {

            $char = $self->session->currentChar->name;

        } else {

            $self->showMsgDialogue(
                'Update character visits',
                'error',
                'Can\'t update the number of visits - there is no current character set',
                'ok',
            );

            return undef;
        }

        # Update room visits
        if ($mode eq 'increase') {

            # Increase by one
            foreach my $roomObj (@roomList) {

                if ($roomObj->ivExists('visitHash', $char)) {
                    $roomObj->ivIncHash('visitHash', $char);
                } else {
                    $roomObj->ivAdd('visitHash', $char, 1);
                }
            }

        } elsif ($mode eq 'decrease') {

            # Decrease by one
            foreach my $roomObj (@roomList) {

                if ($roomObj->ivExists('visitHash', $char)) {

                    $roomObj->ivDecHash('visitHash', $char);
                    # If the number of visits is down to 0, remove the entry from the hash (so that
                    #   we don't get -1 visits the next time)
                    if (! $roomObj->ivShow('visitHash', $char)) {

                        $roomObj->ivDelete('visitHash', $char);
                    }
                }
            }

        } elsif ($mode eq 'manual') {

            if ($self->selectedRoom) {

                # Set manually (one room only)
                $current = $self->selectedRoom->ivShow('visitHash', $char);
                if (! $current) {

                    # If there's no entry for this character in the room's ->visitHash, make sure
                    #   the 'dialogue' window displays a value of 0
                    $current = 0;
                }

                $result = $self->showEntryDialogue(
                    'Update character visits',
                    'Enter the number of visits to this room #' . $self->selectedRoom->number
                    . ' by \'' . $char . '\'',
                    undef,              # No max number of characters
                    $current,
                );

            } else {

                # Set manually (multiple rooms)
                $result = $self->showEntryDialogue(
                    'Update character visits',
                    'Enter the number of visits for each of these ' . (scalar @roomList) . ' rooms'
                    . ' by \'' . $char . '\'',
                    undef,              # No max number of characters
                );
            }

            if (! defined $result) {

                # User clicked 'cancel' button in the 'dialogue' window
                return undef;

            } elsif (($result =~ /\D/) || $result < 0) {

                $self->showMsgDialogue(
                    'Update character visits',
                    'error',
                    'Invalid value (' . $result . ') - must be an integer, 0 or above',
                    'ok',
                );

                return undef;

            } else {

                foreach my $roomObj (@roomList) {

                    if ($result) {
                        $roomObj->ivAdd('visitHash', $char, $result);
                    } else {
                        $roomObj->ivDelete('visitHash', $char);
                    }
                }
            }

        } else {

            # Reset to zero

            # Before resetting counts in multiple rooms, get a confirmation
            if ($self->selectedRoomHash) {

                $result = $self->showMsgDialogue(
                    'Reset character visits',
                    'question',
                    'Are you sure you want to reset character visits in all ' . (scalar @roomList)
                    . ' rooms?',
                    'yes-no',
                );

                if (! $result || $result eq 'no') {

                    # Don't reset anything
                    return undef;
                }
            }

            # Reset the counts
            foreach my $roomObj (@roomList) {

                if ($self->selectedRoom->ivExists('visitHash', $char)) {

                    $self->selectedRoom->ivDelete('visitHash', $char);
                }
            }
        }

        # Mark the selected room(s) to be re-drawn, in case the room and its character visits are
        #   currently visible
        foreach my $roomObj (@roomList) {

            push (@drawList, 'room', $roomObj);
        }

        $self->worldModelObj->updateMaps(@drawList);

        # Show a confirmation, but only if the selected room(s) are on a different level or in a
        #   different region altogether
        if ($self->selectedRoom) {

            # Get the new number of visits for the single selected room...
            $current = $self->selectedRoom->ivShow('visitHash', $char);
            if (! $current) {

                $current = 0;
            }

            # ...and then show a confirmation (but only if the selected room is on a different
            #   level, or in a different region altogether)
            if (
                $self->selectedRoom->parent != $self->currentRegionmap->number
                || $self->selectedRoom->zPosBlocks != $self->currentRegionmap->currentLevel
            ) {
                $self->showMsgDialogue(
                    'Update character visits',
                    'info',
                    'Visits by \'' . $char . '\' to room #' . $self->selectedRoom->number
                    . ' set to ' . $current,
                    'ok',
                );
            }

        } else {

            # Check every selected room, stopping when we find one on the same level and in the
            #   same region
            OUTER: foreach my $roomObj (@roomList) {

                if (! defined $current) {

                    # (All the selected rooms now have the same number of visits, so $current only
                    #   needs to be set once)
                    $current = $roomObj->ivShow('visitHash', $char);
                    if (! $current) {

                        $current = 0;
                    }
                }

                if (
                    $roomObj->parent == $self->currentRegionmap->number
                    && $roomObj->zPosBlocks == $self->currentRegionmap->currentLevel
                ) {
                    $matchFlag = TRUE;
                    last OUTER;
                }
            }

            if (! $matchFlag) {

                # No selected rooms are actually visible, so show the confirmation
                $self->showMsgDialogue(
                    'Update character visits',
                    'info',
                    'Visits by \'' . $char . '\' in ' . (scalar @roomList) . ' rooms set to '
                    . $current,
                    'ok',
                );
            }
        }

        return 1;
    }

    sub toggleGraffitiCallback {

        # Called by $self->enableRoomsColumn, ->enableRoomsPopupMenu and ->drawMiscButtonSet
        # Toggles graffiti in the selected room(s)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@roomList, @drawList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->toggleGraffitiCallback', @_);
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (! $self->selectedRoom && ! $self->selectedRoomHash)
            || ! $self->graffitiModeFlag
        ) {
            return undef;
        }

        # Get a list of selected room(s)
        @roomList = $self->compileSelectedRooms();
        foreach my $roomObj (@roomList) {

            push (@drawList, 'room', $roomObj);

            if (! $self->ivExists('graffitiHash', $roomObj->number)) {
                $self->ivAdd('graffitiHash', $roomObj->number, undef);
            } else {
                $self->ivDelete('graffitiHash', $roomObj->number);
            }
        }

        # Redraw the room(s) with graffiti on or off
        $self->markObjs(@drawList);
        $self->doDraw();
        # Update room counts in the window's title bar
        $self->setWinTitle();

        return 1;
    }

    sub setFilePathCallback {

        # Called by $self->enableRoomsColumn
        # Sets the file path for the world's source code file (if known) for the selected room
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       clicks the 'cancel' button on the 'dialogue' window
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($filePath, $virtualPath);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setFilePathCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || ! $self->selectedRoom) {

            return undef;
        }

        # Prompt the user for the file path, and (optionally) the virtual area path
        ($filePath, $virtualPath) = $self->promptFilePath($self->selectedRoom);
        if (! defined $filePath) {

            # User clicked 'cancel' button in the 'dialogue' window
            return undef;

        } else {

            # Modify the world model room
            $self->worldModelObj->setRoomSource($self->selectedRoom, $filePath, $virtualPath);
            return 1;
        }
    }

    sub setVirtualAreaCallback {

        # Called by $self->enableRoomsColumn
        # Sets or resets the virtual area path for the selected room(s)
        #
        # Expected arguments
        #   $setFlag    - Set to TRUE if the rooms' ->virtualAreaPath IV should be set; set to FALSE
        #                   if it should be reset (set to 'undef')
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if there are no
        #       rooms that can be modified or if the user clicks the 'cancel' button on the
        #       'dialogue' window
        #   1 otherwise

        my ($self, $setFlag, $check) = @_;

        # Local variables
        my (
            $virtualPath, $msg,
            @roomList, @useList, @ignoreList,
        );

        # Check for improper arguments
        if (! defined $setFlag || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setVirtualAreaCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || (! $self->selectedRoom && ! $self->selectedRoomHash)) {

            return undef;
        }

        # Get a list of selected room(s)
        @roomList = $self->compileSelectedRooms();

        # Check each room to make sure each has a ->sourceCodePath, eliminating those that don't
        #   (but don't bother in reset mode)
        if ($setFlag) {

            foreach my $roomObj (@roomList) {

                if ($roomObj->sourceCodePath) {
                    push (@useList, $roomObj);
                } else {
                    push (@ignoreList, $roomObj);
                }
            }

        } else {

            # When resetting, use all the selected rooms
            @useList = @roomList;
        }

        if (! @useList) {

            $self->showMsgDialogue(
                'Set virtual area',
                'error',
                'Cannot set the virtual area for these rooms (probably because no source code path'
                . ' has been set for them)',
                'ok',
            );

            return undef;
        }

        # Set the virtual area for the selected room(s)
        if ($setFlag) {

            if (@useList == 1) {
                $msg = 'Set the path to the virtual area file for one selected room';
            } else {
                $msg = 'Set the path to the virtual area file for ' . @roomList
                            . ' of the selected rooms';
            }

            # Prompt the user for the virtual area path
            $virtualPath = $self->showEntryDialogue(
                'Set virtual area',
                $msg,
                undef,              # No maximum number of characters
                $self->worldModelObj->lastVirtualAreaPath,
            );

            if (! defined $virtualPath) {

                # User clicked 'cancel' button in the 'dialogue' window
                return undef;

            } else {

                # Set the virtual area paths
                foreach my $roomObj (@useList) {

                    # (Keep the existing value of the room's ->sourceCodePath IV)
                    $self->worldModelObj->setRoomSource(
                        $roomObj,
                        $roomObj->sourceCodePath,
                        $virtualPath,
                    );
                }

                # Display a confirmation
                if (@useList == 1) {
                    $msg = 'one selected room';
                } else {
                    $msg = scalar @useList . ' of the selected rooms';
                }

                $self->showMsgDialogue(
                    'Set virtual area',
                    'info',
                    'Set the virtual area file for ' . $msg . ' to: ' . $virtualPath,
                    'ok',
                );
            }

        # Reset the virtual area for the selected room(s)
        } else {

            # Reset the virtual area paths
            foreach my $roomObj (@roomList) {

                # (Keep the existing value of the room's ->sourceCodePath IV)
                $self->worldModelObj->setRoomSource(
                    $roomObj,
                    $roomObj->sourceCodePath,
                    undef,      # No virtual path
                );
            }

            # Display a confirmation
            if (@useList == 1) {
                $msg = 'the selected room';
            } else {
                $msg = scalar @roomList . ' selected rooms';
            }

            # Display a confirmation
            $self->showMsgDialogue(
                'Reset virtual area',
                'info',
                'The virtual area file for ' . $msg . ' has been reset',
                'ok',
            );
        }

        return 1;
    }

    sub editFileCallback {

        # Called by $self->enableRoomsColumn
        # Opens the mudlib file corresponding to the selected room in Axmud's external text editor
        #   (the one specified by GA::Client->textEditCmd)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   $virtualFlag    - If set to TRUE, we need to edit the file stored in the room object's
        #                       ->virtualAreaPath. If set to FALSE (or 'undef'), we need to edit the
        #                       file stored in $obj->sourceCodePath
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if no external
        #       text editor is specified by the GA::Client
        #   1 otherwise

        my ($self, $virtualFlag, $check) = @_;

        # Local variables
        my ($cmd, $file);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->editFileCallback', @_);
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || ! $self->selectedRoom
            || (
                ! defined $virtualFlag
                && (! $self->selectedRoom->sourceCodePath || $self->selectedRoom->virtualAreaPath)
            ) || (defined $virtualFlag && ! $self->selectedRoom->virtualAreaPath)
        ) {
            return undef;
        }

        # Check that the GA::Client has a text editor command set, and that it is valid
        $cmd = $axmud::CLIENT->textEditCmd;
        if (! $cmd || ! ($cmd =~ m/%s/)) {

            # Show a 'dialogue' window to explain the problem
            $self->showMsgDialogue(
                'Edit source code file',
                'error',
                'Can\'t edit the file: invalid external application command \'' . $cmd . '\'',
                'ok',
            );

            return undef;
        }

        # Set the file to be opened. If the current world model defines a mudlib directory, the
        #   object's ->mudlibPath is relative to that; otherwise it's an absolute path
        if ($self->session->worldModelObj->mudlibPath) {
            $file = $self->session->worldModelObj->mudlibPath;
        } else {
            $file = '';
        }

        if ($virtualFlag) {
            $file .= $self->selectedRoom->virtualAreaPath;
        } else {
            $file .= $self->selectedRoom->sourceCodePath;
        }

        # Add the file extension, if set
        if ($self->session->worldModelObj->mudlibExtension) {

            $file .= $self->session->worldModelObj->mudlibExtension;
        }

        # Check the file exists
        if (! (-e $file)) {

            $self->showMsgDialogue(
                'Edit source code file',
                'error',
                'Can\'t find the file \'' . $file . '\'',
                'ok',
            );

            return undef;
        }

        # Open the file in the external text editor
        $cmd =~ s/%s/$file/;

        system $cmd;

        return 1;
    }

    sub deleteRoomsCallback {

        # Called by $self->enableRoomsColumn
        # If multiple rooms are selected, prompts the user before deleting them (there is no
        #   confirmation prompt if a single room is selected)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the user
        #       changes their mind or if the deletion operation fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $result;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->deleteRoomsCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || (! $self->selectedRoom && ! $self->selectedRoomHash)) {

            return undef;
        }

        # Prompt the user for confirmation before deleting any rooms
        if ($self->selectedRoom) {
            $result = $self->showMsgDialogue(
                'Delete rooms',
                'question',
                'Are you sure you want to delete the selected room?',
                'yes-no',
            );

        } else {

            $result = $self->showMsgDialogue(
                'Delete rooms',
                'question',
                'Are you sure you want to delete ' .  $self->ivPairs('selectedRoomHash')
                . ' rooms?',
                'yes-no',
            );
        }

        if ($result ne 'yes') {

            return undef;

        } else {

            # Delete the selected room(s)
            return $self->worldModelObj->deleteRooms(
                $self->session,
                TRUE,           # Update Automapper windows now
                $self->compileSelectedRooms(),
            );
        }
    }

    sub addContentsCallback {

        # Called by $self->enableRoomsColumn
        # Adds a non-model object (or objects) from the Locator's current room to the world model,
        #   making them children of (and therefore contained in) the current room
        # Alternatively, prompts the user to add a string like 'two hairy orcs and an axe'. Parses
        #   the string into a list of objects, and prompts the user to choose an object from that
        #   list
        #
        # Expected arguments
        #   $parseFlag  - Set to TRUE if the user should be prompted for a sentence to parse. Set to
        #                   FALSE if the list of objects should be taken from the Locator task
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the Locator task
        #       isn't running or doesn't know the current location, if its room's temporary contents
        #       list is empty or if an attempt to parse a string fails
        #   1 otherwise

        my ($self, $parseFlag, $check) = @_;

        # Local variables
        my (
            $taskObj, $roomObj, $string, $allString, $choice,
            @tempList, @useList, @comboList, @addList,
            %comboHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addContentsCallback', @_);
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (
                (! $parseFlag && ! $self->mapObj->currentRoom)
                || ($parseFlag && ! $self->selectedRoom)
            )
        ) {
            return undef;
        }

        if (! $parseFlag) {

            # The list of objects should be taken from the Locator task's current room. Import the
            #   Locator task
            $taskObj = $self->session->locatorTask;
            # Check the Locator task exists and that it knows about the character's current
            #   location
            if (! $taskObj || ! $taskObj->roomObj) {

                # Show a 'dialogue' window to explain the problem
                $self->showMsgDialogue(
                    'Add contents',
                    'error',
                    'Either the Locator task isn\'t running or it doesn\'t know the current'
                    . ' location',
                    'ok',
                );

                return undef;
            }

            # Use the automapper's current room
            $roomObj = $self->mapObj->currentRoom;

            # Import the list of temporary non-model objects from the Locator's current room
            @tempList = $taskObj->roomObj->tempObjList;
            if (! @tempList) {

                $self->showMsgDialogue(
                    'Add contents',
                    'error',
                    'The Locator task\'s current room appears to be empty',
                    'ok',
                );

                return undef;
            }

            # From this list, remove any temporary objects which have already been added to the
            #   automapper room's list of child objects during the current visit to the room
            OUTER: foreach my $tempObj (@tempList) {

                foreach my $childNum ($roomObj->ivKeys('childHash')) {

                    if ($tempObj eq $self->worldModelObj->ivShow('modelHash', $childNum)) {

                        # Don't add it again
                        next OUTER;
                    }
                }

                # $tempObj hasn't been added to the model yet
                push (@useList, $tempObj);
            }

        } else {

            # The user should be prompted for a string to parse. Use the (single) selected room
            $roomObj = $self->selectedRoom;

            # Prompt the user to enter a string to parse
            $string = $self->showEntryDialogue(
                'Add contents',
                'Enter a string to parse (e.g. \'two hairy orcs and an axe\')',
            );

            if (! defined $string) {

                # User clicked 'cancel' or closed the window
                return undef;

            } else {

                # Try to parse the string into a list of objects (parse multiples as separate
                #   objects)
                @useList = $self->worldModelObj->parseObj($self->session, FALSE, $string);
            }
        }

        # Don't prompt for an object, if there are none available
        if (! @useList) {

            return $self->showMsgDialogue(
                'Add contents',
                'error',
                'There are no objects to add',
                'ok',
            );
        }

        # Prepare a list of strings to display in a combobox
        foreach my $obj (@useList) {

            my $line;

            if ($obj->category eq 'portable' || $obj->category eq 'decoration') {
                $line = $obj->name . ' [' . $obj->category . ' - ' . $obj->type . ']';
            } else {
                $line = $obj->name . ' [' . $obj->category . ']';
            }

            push (@comboList, $line);
            $comboHash{$line} = $obj;
        }

        # If there is more than one object that could be added, create something at the top of the
        #   combobox that lets the user add them all
        if (@comboList > 1) {

            $allString = '<add all ' . scalar @comboList . ' objects>';
            unshift (@comboList, $allString);
        }

        # Prompt the user to select an object
        $choice = $self->showComboDialogue(
            'Select object',
            'Choose which object(s) to add to the world model',
            \@comboList,
        );

        if ($choice) {

            if ($allString && $choice eq $allString) {

                # Add all the objects to the model (use @useList, in case @comboList contained
                #   repeating strings, because there's more than one orc, for example, in the room)
                @addList = @useList;

            } else {

                # Add a single object to the model
                push (@addList, $comboHash{$choice});
            }

            # Add the objects to the world model as children of $roomObj
            $self->worldModelObj->addRoomChildren(
                TRUE,                   # Update Automapper windows
                FALSE,                  # Children are not hidden
                $roomObj,
                undef,                  # Children are not hidden
                @addList,
            );
        }

        return 1;
    }

    sub addHiddenObjCallback {

        # Called by $self->enableRoomsColumn
        # Adds a non-model object from the Locator's current room to the world model, making it a
        #   child (and therefore contained in) the current room
        # Alternatively, prompts the user to add a string like 'two hairy orcs and an axe'. Parses
        #   the string into a list of objects, and prompts the user to choose an object from that
        #   list
        #
        # Expected arguments
        #   $parseFlag  - Set to TRUE if the user should be prompted for a sentence to parse. Set to
        #                   FALSE if the list of objects should be taken from the Locator task
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the hidden
        #       object isn't added
        #   1 otherwise

        my ($self, $parseFlag, $check) = @_;

        # Local variables
        my (
            $taskObj, $roomObj, $string, $obtainCmd, $choice,
            @tempList, @useList, @comboList,
            %comboHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->addHiddenObjCallback', @_);
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (
                (! $parseFlag && ! $self->mapObj->currentRoom)
                || ($parseFlag && ! $self->selectedRoom)
            )
        ) {
            return undef;
        }

        if (! $parseFlag) {

            # The list of objects should be taken from the Locator task's current room. Import the
            #   Locator task
            $taskObj = $self->session->locatorTask;
            # Check the Locator task exists and that it knows about the character's current
            #   location
            if (! $taskObj || ! $taskObj->roomObj) {

                # Show a 'dialogue' window to explain the problem
                $self->showMsgDialogue(
                    'Add hidden object',
                    'error',
                    'Either the Locator task isn\'t running or it doesn\'t know the current'
                    . ' location',
                    'ok',
                );

                return undef;
            }

            # Use the automapper's current room
            $roomObj = $self->mapObj->currentRoom;

            # Import the list of temporary non-model objects from the Locator's current room
            @tempList = $taskObj->roomObj->tempObjList;
            if (! @tempList) {

                $self->showMsgDialogue(
                    'Add hidden object',
                    'error',
                    'The Locator task\'s current room appears to be empty',
                    'ok',
                );

                return undef;
            }

            # From this list, remove any temporary objects which have already been added to the
            #   automapper room's list of child objects during the current visit to the room
            OUTER: foreach my $tempObj (@tempList) {

                foreach my $childNum ($roomObj->ivKeys('childHash')) {

                    if ($tempObj eq $self->worldModelObj->ivShow('modelHash', $childNum)) {

                        # Don't add it again
                        next OUTER;
                    }
                }

                # $tempObj hasn't been added to the model yet
                push (@useList, $tempObj);
            }

        } else {

            # The user should be prompted for a string to parse. Use the (single) selected room
            $roomObj = $self->selectedRoom;

            # Prompt the user to enter a string to parse
            $string = $self->showEntryDialogue(
                'Add hidden object',
                'Enter a string to parse (e.g. \'two hairy orcs and an axe\')',
            );

            if (! defined $string) {

                # User clicked 'cancel' or closed the window
                return undef;

            } else {

                # Try to parse the string into a list of objects. The TRUE argument tells the
                #   function to treat 'two hairy orcs' as a single object, with its
                #   ->multiple IV set to 2, so that the same strings don't appear in the combobox
                #   more than once (hopefully)
                @useList = $self->worldModelObj->parseObj($self->session, TRUE, $string);
            }
        }

        # Don't prompt for an object, if there are none available
        if (! @useList) {

            return $self->showMsgDialogue(
                'Add hidden object',
                'error',
                'There are no objects to add',
                'ok',
            );
        }

        # Prepare a list of strings to display in a combobox
        foreach my $obj (@useList) {

            my $line;

            if ($obj->category eq 'portable' || $obj->category eq 'decoration') {
                $line = $obj->name . ' [' . $obj->category . ' - ' . $obj->type . ']';
            } else {
                $line = $obj->name . ' [' . $obj->category . ']';
            }

            push (@comboList, $line);
            $comboHash{$line} = $obj;
        }

        ($obtainCmd, $choice) = $self->showEntryComboDialogue(
            'Select object',
            'Enter the command used to obtain the hidden object',
            'Choose which hidden object to add to the model',
            \@comboList,
        );

        if ($choice) {

            # Add the object to the world model as a (hidden) child of $roomObj
            $self->worldModelObj->addRoomChildren(
                TRUE,                   # Update Automapper windows
                TRUE,                   # Mark child as hidden
                $roomObj,
                $obtainCmd,
                $comboHash{$choice},    # The non-model object to add to the world model
            );
        }

        return 1;
    }

    sub addSearchResultCallback {

        # Called by $self->enableRoomsColumn
        # Adds the results of a 'search' command at the current location (stored in the
        #   room object's ->searchHash IV)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($term, $result);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->addSearchResultCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap || ! $self->mapObj->currentRoom) {

            return undef;
        }

        # Prompt the user for a search term (e.g. 'fireplace') and the result (e.g.
        #   'It's a dirty old fireplace')
        ($term, $result) = $self->showDoubleEntryDialogue(
            'Add search result',
            'Add a search term (e.g. \'fireplace\')',
            'Add the result (e.g. \'It\'s an old fireplace.\')',
        );

        if ($term && $result) {

            # Add the search term and result to the current room's search hash, replacing the entry
            #   for the same search term, if it already exists
            $self->worldModelObj->addSearchTerm($self->mapObj->currentRoom, $term, $result);
        }

        return 1;
    }

    sub setRoomTagCallback {

        # Called by $self->enableRoomsColumn
        # Sets (or resets) the selected room's room tag
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails, if the supplied tag
        #       is invalid or if the user declines to reassign an existing room tag
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($roomObj, $tag, $oldRoomNum, $oldRoomObj, $text, $regionObj, $result);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setRoomTagCallback', @_);
        }

        # Standard callback check
        if (! $self->currentRegionmap || (! $self->selectedRoom && ! $self->selectedRoomTag)) {

            return undef;
        }

        # Decide which room to use. If there's a single selected room; use it. If there's a single
        #   selected room tag, use its parent room
        if ($self->selectedRoom) {

            $roomObj = $self->selectedRoom;

        } elsif ($self->selectedRoomTag) {

            # (The IV stores the blessed reference of the room tag's parent room)
            $roomObj = $self->selectedRoomTag;
        }

        # Prompt the user for a tag
        $tag = $self->showEntryDialogue(
            'Set room tag',
            'Enter the selected room\'s tag (or leave empty to delete a tag)',
            undef,                  # No maximum number of characters
            $roomObj->roomTag,
        );

        if (defined $tag) {

            if (! $tag) {

                # Reset the room's tag. The TRUE argument instructs the world model to update its
                #   Automapper windows
                $self->worldModelObj->resetRoomTag(TRUE, $roomObj);

            } else {

                # Check the tag is valid
                if (length($tag) > 16) {

                    $self->showMsgDialogue(
                        'Set room tag',
                        'error',
                        'Invalid room tag \'' . $tag . '\' - max size 16 characters',
                        'ok',
                    );

                    return undef;

                } elsif ($tag =~ m/@@@/) {

                    $self->showMsgDialogue(
                        'Set room tag',
                        'error',
                        'Invalid room tag \'' . $tag . '\' - tag must not contain \'@@@\'',
                        'ok',
                    );

                    return undef;
                }

                # If the tag already belongs to another room, it gets reassigned to this one
                # If the other room is on the map, but is not currently visible, it won't be obvious
                #   to the user that the tag has been reassigned, rather than created
                # Prompt the user before reassigning a tag from one mapped room to another (but
                #   don't prompt if the old and new room are the same!)
                $oldRoomNum = $self->worldModelObj->checkRoomTag($tag);
                if (defined $oldRoomNum && $oldRoomNum != $roomObj->number) {

                    # Prepare the text to show
                    $oldRoomObj = $self->worldModelObj->ivShow('modelHash', $oldRoomNum);

                    $text = 'The tag \'' . $oldRoomObj->roomTag . '\' is already assigned to room #'
                                . $oldRoomNum;

                    if (
                        $self->currentRegionmap
                        && $self->currentRegionmap->number eq $oldRoomObj->parent
                    ) {
                        $text .= ' in this region. ';

                    } else {

                        $regionObj = $self->worldModelObj->ivShow('modelHash', $oldRoomObj->parent);
                        $text .= ' in the region \'' . $regionObj->name . '\'. ';
                    }

                    $text .= 'Do you want to reassign it?';

                    # Prompt the user
                    $result = $self->showMsgDialogue(
                        'Reassign room tag',
                        'question',
                        $text,
                        'yes-no',
                    );

                    if ($result eq 'no') {

                        return undef;
                    }
                }

                # Set the room's tag
                $self->worldModelObj->setRoomTag(TRUE, $roomObj, $tag);

                # If the Locator task is running, update it
                $self->mapObj->updateLocator();
            }
        }

        return 1;
    }

    sub setRoomGuildCallback {

        # Called by $self->enableRoomsColumn
        # Sets a room's guild (->roomGuild)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $noGuildString, $msg, $choice, $guildName,
            @profList, @sortedList, @comboList, @selectedList, @finalList,
            %comboHash, %itemHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setRoomGuildCallback', @_);
        }

        # Standard callback check
        if (
            ! $self->currentRegionmap
            || (
                ! $self->selectedRoom && ! $self->selectedRoomHash && ! $self->selectedRoomGuild
                && ! $self->selectedRoomGuildHash
            )
        ) {
            return undef;
        }

        # Compile a list of guild profiles and sort alphabetically
        foreach my $profObj ($self->session->ivValues('profHash')) {

            if ($profObj->category eq 'guild') {

                push (@profList, $profObj);
            }
        }

        @sortedList = sort {lc($a->name) cmp lc($b->name)} (@profList);

        # Prepare a list to show in a combo box. At the same time, compile a hash in the form:
        #   $hash{combo_box_string} = blessed_reference_of_corresponding_profile
        foreach my $profObj (@sortedList) {

            push (@comboList, $profObj->name);
            $comboHash{$profObj->name} = $profObj;
        }

        # Put an option to use no guild at the top of the combo list
        $noGuildString = '<room not a guild>';
        unshift (@comboList, $noGuildString);

        if ($self->selectedRoom) {

            $msg = 'selected room';
            if ($self->selectedRoom->roomGuild) {

                $msg .= "\n(currently set to \'" . $self->selectedRoom->roomGuild . "\')";
            }

        } elsif ($self->selectedRoomGuild) {

            $msg = "selected room guild\n(currently set to \'" . $self->selectedRoomGuild->roomGuild
                    . "\')";

        } else {

            $msg = 'selected rooms';
        }

        # Prompt the user for a profile
        $choice = $self->showComboDialogue(
            'Select room guild',
            'Select the guild for the ' . $msg,
            \@comboList,
        );

        if ($choice) {

            # Convert $choice into a guild profile name
            if ($choice eq $noGuildString) {

                $guildName = undef;     # Room has no guild set

            } else {

                $guildName = $comboHash{$choice}->name;
            }

            # Compile a list of selected rooms and selected room guilds
            push (@selectedList, $self->compileSelectedRooms(), $self->compileSelectedRoomGuilds());

            # Combine them into a single list, @finalList, eliminating duplicate rooms
            foreach my $roomObj (@selectedList) {

                if (! exists $itemHash{$roomObj->number}) {

                    push (@finalList, $roomObj);
                    $itemHash{$roomObj->number} = undef;
                }
            }

            # Update the guild for each room
            $self->worldModelObj->setRoomGuild(
                TRUE,           # Update the Automapper windows now
                $guildName,     # Name of a guild profile
                @finalList,
            );
        }

        return 1;
    }

    sub resetRoomOffsetsCallback {

        # Called by $self->enableRoomsColumn
        # Resets the drawn positions (offsets) of the room tags and room guilds for the selected
        #   room(s)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            @roomList, @combinedList,
            %roomHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->resetRoomOffsetsCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap || ! $self->selectedRoom) {

            return undef;
        }

        # Get a list of selected rooms, room tags and room guilds
        push (@roomList,
            $self->compileSelectedRooms(),
            $self->compileSelectedRoomTags(),
            $self->compileSelectedRoomGuilds(),
        );

        # Combine these lists into a single list of affected rooms, eliminating duplicates and any
        #   selected room which doesn't have a room tag or a room guild
        foreach my $roomObj (@roomList) {

            if (
                ! exists $roomHash{$roomObj->number}
                && ($roomObj->roomTag || $roomObj->roomGuild)
            ) {
                push (@combinedList, $roomObj);
                $roomHash{$roomObj->number} = undef;
            }
        }

        # Reset the position of the room tags/room guilds in each affected room (if there are any)
        #   and instruct the world model to update its Automapper windows
        $self->worldModelObj->resetRoomOffsets(
            TRUE,               # Update Automapper windows now
            0,                  # Mode 0 - reset both room tags and room guilds
            @combinedList,
        );

        return 1;
    }

    sub setInteriorOffsetsCallback {

        # Called by $self->enableRoomsColumn
        # Sets the offsets used when a room's grid coordinates are displayed as interior text inside
        #   the room box
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments, if the standard callback check fails or if the user
        #       doesn't supply a pattern
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($xOffset, $yOffset);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->setInteriorOffsetsCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Prompt the user for a new offsets
        ($xOffset, $yOffset) = $self->showDoubleEntryDialogue(
            'Synchronise grid coordinates',
            'Adjust X coordinate (enter an integer)',
            'Adjust Y coordinate (enter an integer)',
        );

        if (
            ! defined $xOffset
            || ! defined $yOffset
            || ! ($axmud::CLIENT->intCheck($xOffset))
            || ! ($axmud::CLIENT->intCheck($yOffset))
        ) {

            return undef;

        } else {

            # Update the world model
            $self->worldModelObj->setInteriorOffsets($xOffset, $yOffset);

            if ($self->worldModelObj->roomInteriorMode eq 'grid_posn') {

                # Redraw the current region
                $self->redrawRegions();

            } else {

                # Remind the user how to make the offset position visible, if they aren't already
                $self->showMsgDialogue(
                    'Synchronise grid coordinates',
                    'info',
                    'To make grid coordinates visible on the map, click \'View > Room interiors >'
                        . ' Draw grid coordinates\'',
                    'ok',
                );
            }

            # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
            $self->restrictWidgets();

            return 1;
        }
    }

    sub resetInteriorOffsetsCallback {

        # Called by $self->enableRoomsColumn
        # Resets the offsets used when a room's grid coordinates are displayed as interior text
        #   inside the room box
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->resetInteriorOffsetsCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap) {

            return undef;
        }

        # Update the world model
        $self->worldModelObj->setInteriorOffsets(0, 0);

        if ($self->worldModelObj->roomInteriorMode eq 'grid_posn') {

            # Redraw the current region
            $self->redrawRegions();
        }

        # Sensitise/desensitise menu bar/toolbar items, depending on current conditions
        $self->restrictWidgets();

        return 1;
    }

    sub toggleExclusiveProfileCallback {

        # Called by $self->enableRoomsColumn
        # Toggles the exclusivity for one or more selected rooms (specifically, toggles the rooms'
        #   ->exclusiveFlag IV)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $flagSetting, $mismatchFlag, $msg,
            @roomList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->toggleExclusiveProfileCallback',
                @_,
            );
        }

        # Standard callback check
        if (! $self->currentRegionmap || (! $self->selectedRoom && ! $self->selectedRoomHash)) {

            return undef;
        }

        # Get a list of selected rooms
        @roomList = $self->compileSelectedRooms();

        # Toggle their ->exclusive flags
        $self->worldModelObj->toggleRoomExclusivity(
            TRUE,           # Update Automapper windows now
            @roomList,
        );

        # Compose a message to display. Find out if every room in @roomList has its
        #   ->exclusiveFlag set to the same value
        OUTER: foreach my $roomObj (@roomList) {

            if (! defined $flagSetting) {

                # This is the first room in @roomList
                $flagSetting = $roomObj->exclusiveFlag;

            } elsif ($flagSetting != $roomObj->exclusiveFlag) {

                # The rooms in @roomList have their ->exclusiveFlag IV set to different values
                $mismatchFlag = TRUE;
                last OUTER;
            }
        }

        if ($mismatchFlag) {

            $msg = 'Toggled exclusivity for ';

            if ($self->selectedRoom) {
                $msg .= '1 room';
            } else {
                $msg .= scalar @roomList . ' rooms';
            }

        } else {

            $msg = 'Exclusivity for ';

            if ($self->selectedRoom) {
                $msg .= '1 room';
            } else {
                $msg .= scalar @roomList . ' rooms';
            }

            if ($flagSetting) {
                $msg .= ' turned on';
            } else {
                $msg .= ' turned off';
            }
        }

        $self->showMsgDialogue(
            'Toggle exclusive profiles',
            'info',
            $msg,
            'ok',
        );

        return 1;
    }

    sub addExclusiveProfileCallback {

        # Called by $self->enableRoomsColumn
        # Adds a profile to the selected room's exclusive profile hash
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the standard callback check fails
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $choice,
